#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import collections
from collections import namedtuple
from functools import reduce
import itertools
from operator import ior
import re

import Cell
import HumanReadable
import ProductAttributes as pa
import Tac
import Tracing
from .Types import (
   AutonegMode,
   COMPAT_AUTO,
   COMPAT_GBPS_10,
   COMPAT_GBPS_100,
   COMPAT_GBPS_25,
   COMPAT_GBPS_50,
   EthDuplex,
   EthFecEncoding,
   EthFecEncodingConfigSet,
   EthLaneCount,
   EthSpeed,
   EthSpeedApi,
   EthTypesApi,
)

traceHandle = Tracing.Handle( 'SysdbEthIntf' )
t0 = traceHandle.trace0

sliceNameCache = {}

phyIntfRe = re.compile( r'(Ethernet|Management|Switch)(\d+)' )

def sliceName( intfName, forceFixedSystem=False ):
   if forceFixedSystem and Cell.cellType() == 'fixed':
      return 'FixedSystem'
   if intfName in sliceNameCache:
      return sliceNameCache[ intfName ]
   if Cell.cellType() == 'supervisor':
      m = phyIntfRe.match( intfName )
      if m:
         intfType = m.group( 1 )
         modId = m.group( 2 )
         if modId == '0' and intfType == 'Management':
            # Special case for Management 0
            # We will stash this under the active cell slice dir
            sliceId = str( Cell.cellId() )
         elif intfType in [ 'Ethernet', 'Switch' ]:
            # Ethernet ports will be on Linecards.
            # Switch port slice management is yet to be determined, but for now put
            # them on Linecards.
            sliceId = 'Linecard%s' % modId
         else:
            sliceId = str( modId )
      else:
         t0( 'Interface type not handled:', intfName )
         return None
   else:
      sliceId = str( Cell.cellId() )

   sliceNameCache[ intfName ] = sliceId
   return sliceId

def lookupEthPhyIntfConfigDir( intfName,
                               ethPhyIntfConfigSliceDir ):
   if ( intfName.startswith( 'Ethernet' ) or
        intfName.startswith( 'Management' ) or
        intfName.startswith( 'UnconnectedEthernet' ) ):
      return ethPhyIntfConfigSliceDir.get( sliceName( intfName ) )
   return None

def ethPhyIntfConfigIter( ethPhyIntfConfigSliceDir ):
   allIntfConfigs = [ iter( ethPhyDir.intfConfig ) \
                      for ethPhyDir in ethPhyIntfConfigSliceDir.values() ]
   return itertools.chain.from_iterable( allIntfConfigs )

def defaultSwitchedIntfConfigName( linkMode ):
   return EthTypesApi.linkModeToDefaultSwitchedIntfConfig( linkMode )

# Tokens for error-correction encodings.
tokenToFecEncodingAttrs = {
   'reed-solomon' : [ 'fecEncodingReedSolomon', 'fecEncodingReedSolomon544' ],
   'fire-code' : [ 'fecEncodingFireCode' ],
   'disabled' : [ 'fecEncodingDisabled' ]
   }

# Error-correction encodings for tokens
fecEncodingToToken = {
   fecEncoding: token
   for token, fecEncodings in tokenToFecEncodingAttrs.items()
   for fecEncoding in fecEncodings
}

# returns the type of encoding
def encodingType( encoding=None ):
   if encoding.startswith( 'coherent' ):
      return "coherent"
   elif encoding in tokenToFecEncodingAttrs:
      return "non-coherent"
   else:
      return "invalid"

# This table depends on the fact that the enum values of
# EthFecBpassMode and the attributes of EthFecBypassCapabilitySet
# are the same. We use it to convert the token to the config value
# or to lookup support for bypass.
tokenToBypassAttr = {
   'correction' : 'fecBypassCorrection',
   'indication' : 'fecBypassIndication'
   }

pllSpeedGroupSettingStrMap = {
   '10g': COMPAT_GBPS_10,
   '25g': COMPAT_GBPS_25,
   '50g': COMPAT_GBPS_50,
   '100g': COMPAT_GBPS_100,
   'auto': COMPAT_AUTO,
}

def speedCompatTokenToEnum( token ):
   return pllSpeedGroupSettingStrMap.get( token.lower() )

linkModeToSpeedLanesDuplex = collections.OrderedDict( [
   ( 'linkModeUnknown', ( 'speedUnknown',
                          'laneCountUnknown',
                          'duplexUnknown' ) ),
   ( 'linkModeAutoneg', ( 'speedUnknown',
                          'laneCountUnknown',
                          'duplexUnknown' ) ),
   ( 'linkModeAuto40GbpsFull', ( 'speedUnknown',
                                 'laneCountUnknown',
                                 'duplexUnknown' ) ),
   ( 'linkMode10MbpsHalf', ( 'speed10Mbps',
                             'laneCount1',
                             'duplexHalf' ) ),
   ( 'linkMode10MbpsFull', ( 'speed10Mbps',
                             'laneCount1',
                             'duplexFull' ) ),
   ( 'linkMode100MbpsHalf', ( 'speed100Mbps',
                              'laneCount1',
                              'duplexHalf' ) ),
   ( 'linkMode100MbpsFull', ( 'speed100Mbps',
                              'laneCount1',
                              'duplexFull' ) ),
   ( 'linkModeForced10MbpsHalf', ( 'speed10Mbps',
                                   'laneCount1',
                                   'duplexHalf' ) ),
   ( 'linkModeForced10MbpsFull', ( 'speed10Mbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced100MbpsHalf', ( 'speed100Mbps',
                                    'laneCount1',
                                    'duplexHalf' ) ),
   ( 'linkModeForced100MbpsFull', ( 'speed100Mbps',
                                    'laneCount1',
                                    'duplexFull' ) ),
   ( 'linkModeForced1GbpsHalf', ( 'speed1Gbps',
                                  'laneCount1',
                                  'duplexHalf' ) ),
   ( 'linkModeForced1GbpsFull', ( 'speed1Gbps',
                                  'laneCount1',
                                  'duplexFull' ) ),
   ( 'linkModeForced2p5GbpsFull', ( 'speed2p5Gbps',
                                  'laneCount1',
                                  'duplexFull' ) ),
   ( 'linkModeForced10GbpsFull', ( 'speed10Gbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced25GbpsFull', ( 'speed25Gbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced40GbpsFull', ( 'speed40Gbps',
                                   'laneCount4',
                                   'duplexFull' ) ),
   ( 'linkModeForced50GbpsFull', ( 'speed50Gbps',
                                   'laneCount2',
                                   'duplexFull' ) ),
   ( 'linkModeForced50GbpsFull1Lane', ( 'speed50Gbps',
                                        'laneCount1',
                                        'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull', ( 'speed100Gbps',
                                    'laneCount4',
                                    'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull1Lane', ( 'speed100Gbps',
                                         'laneCount1',
                                         'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull2Lane', ( 'speed100Gbps',
                                         'laneCount2',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull8Lane', ( 'speed200Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull4Lane', ( 'speed200Gbps',
                                         'laneCount4',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull2Lane', ( 'speed200Gbps',
                                         'laneCount2',
                                         'duplexFull' ) ),
   ( 'linkModeForced400GbpsFull4Lane', ( 'speed400Gbps',
                                         'laneCount4',
                                         'duplexFull' ) ),
   ( 'linkModeForced400GbpsFull8Lane', ( 'speed400Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ( 'linkModeForced800GbpsFull8Lane', ( 'speed800Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ] )

nonExistentLinkStrToSpeedLanesDuplex = collections.OrderedDict( [
   ( '2p5G/full', ( 'speed2p5Gbps',
                    'laneCount1',
                    'duplexFull' ) ),
   ( '5G/full', ( 'speed5Gbps',
                  'laneCount1',
                  'duplexFull' ) )
   ] )

# Construct global map that maps speed string notation (e.g. 100g) to EthSpeed enums
_ethSpeedEnum = Tac.Type( "Interface::EthSpeed" )
_ethSpeedRe = re.compile( r"speed(?P<speed>\d+(p\d+)?[GM])bps" )
_speedStrToEthSpeed = {}
for attr in _ethSpeedEnum.attributes:
   if attr == "speedUnknown":
      continue

   # We don't really expect the CLI or most consumers to support non IEEE/Consortium
   # speeds. As such we don't allow the parsing of such modes until the need arises.
   if not EthSpeedApi.isIeeeSpeed( attr ):
      continue

   _speedStrToEthSpeed[ _ethSpeedRe.match( attr ).group( "speed" ).lower() ] = attr

# Construct global map that maps integer value to EthLaneCount enums
_laneCountEnum = Tac.Type( "Interface::EthLaneCount" )
_laneCountRe = re.compile( r"laneCount(?P<numLanes>\d+)" )
_lanesStrToEthLanes = {}
for attr in _laneCountEnum.attributes:
   if attr == "laneCountUnknown":
      continue
   _lanesStrToEthLanes[ int( _laneCountRe.match( attr ).group( "numLanes" ) ) ] = \
      attr

# Helper to expand Interface::EthLinkModeSet to list of Interface::EthSpeed
# and Interface::EthLaneCount tuples
def linkModeSetToSpeedLaneList( elms ):
   supportedSpeeds = []
   modeRe = re.compile( r"mode((\S+[GM])bps\S+)" )
   for _attr in elms.attributes:
      if not _attr.startswith( "mode" ):
         continue
      mode = getattr( elms, _attr )
      if mode:
         match = modeRe.match( _attr )
         assert match
         speedLaneDuplex, speed = match.groups()
         if speed == "5G":
            s, l, _ = nonExistentLinkStrToSpeedLanesDuplex[ "%s/full" % speed ]
         else:
            linkMode = "linkModeForced%s" % speedLaneDuplex
            s, l, _ = linkModeToSpeedLanesDuplex[ linkMode ]
         supportedSpeeds.append( ( s, l ) )
   return supportedSpeeds

# Mapping of autoneg mode string to AutonegMode
_autonegStrToAutonegMode = {
   'forced': AutonegMode.anegModeDisabled,
   'cl28': AutonegMode.anegModeClause28,
   'cl37': AutonegMode.anegModeClause37,
   'cl73': AutonegMode.anegModeClause73,
   'auto': AutonegMode.anegModeAuto,
}

# Mapping of FEC string to EthFecEncoding
_fecStrToEthFecEncoding = {
   "" : "fecEncodingUnknown",
   "NOFEC" : "fecEncodingDisabled",
   "RS544" : "fecEncodingReedSolomon544",
   "RS528" : "fecEncodingReedSolomon",
   "FCFEC" : "fecEncodingFireCode",
}

_autonegModeStrRegex = re.compile( r'(forced|cl\d+)' )

def modeStrToAutonegMode( mode ):
   """
   Returns Interface::AutonegMode given a string representation of a
   "aneg speed-lane/duplex fec" mode.

   Parameters
   ----------
   mode : str
      A "aneg speed-lane/duplex fec" mode string
      (e.g. forced 10G-1, f 25G-1 NOFEC, cl28 1G-1 NOFEC )
      Note: The autoneg string must be present in the first part of the mode string.

   Returns
   -------
   autonegMode : Interface::AutonegMode
   """
   mode = mode.lower()
   modeSplit = mode.split()

   # If it exists, the autoneg mode can only ever be in the first part.
   maybeAutonegModeStr = modeSplit[ 0 ] if modeSplit else ''
   autonegModeStrMatch = _autonegModeStrRegex.match( maybeAutonegModeStr )
   if not autonegModeStrMatch:
      return AutonegMode.anegModeUnknown

   return _autonegStrToAutonegMode[ autonegModeStrMatch.group( 1 ) ]

_speedLaneCountDuplexStrRegex = re.compile(
   r'([0-9p]+[gm])(?:-(\d+))?(?:\/(full|half|[fh]))?' )

def modeStrToEthSpeedAndLaneCountAndDuplex( mode ):
   """
   Returns Interface::EthSpeed, Interface::EthLaneCount and Interface::EthDuplex
   given a string representation of a "speed-lane/duplex fec" link mode.

   Parameters
   ----------
   mode : str
      A "speed-lane/duplex fec" link mode
      (e.g. 100g-4, 50g-1, 100m-1, 50g-1 RS544, 100m-1/h, 100m/full, 400g-8/f RS544)
      Note: duplex is optional: /h, /half, /f, /full, full duplex by default
            fec is optional: NOFEC, FCFEC, RS528, RS544.

   Returns
   -------
   ethSpeed : Interface::EthSpeed
   ethLaneCount : Interface::EthLaneCount
   ethDuplex : Interface::EthDuplex
   """

   # Depending on the mode string the speed/lanes/duplex string can exist in either
   # the first part or the second part. In order to simplify the logic we will not be
   # too fussy and just do a broad search across the entire string.
   speedLaneDuplexMatch = _speedLaneCountDuplexStrRegex.search( mode.lower() )

   if not speedLaneDuplexMatch:
      return ( EthSpeed.speedUnknown,
               EthLaneCount.laneCountUnknown,
               EthDuplex.duplexUnknown )

   speed = _speedStrToEthSpeed[ speedLaneDuplexMatch.group( 1 ) ]

   laneCount = EthLaneCount.laneCountUnknown
   laneCountStr = speedLaneDuplexMatch.group( 2 )
   if laneCountStr:
      laneCount = _lanesStrToEthLanes[ int( laneCountStr ) ]

   duplex = EthDuplex.duplexFull
   duplexStr = speedLaneDuplexMatch.group( 3 )
   if duplexStr:
      if duplexStr in [ 'h', 'half' ]:
         duplex = EthDuplex.duplexHalf
      elif duplexStr in [ 'f', 'full' ]:
         duplex = EthDuplex.duplexFull
      else:
         assert False, f'Unknown duplex {duplexStr}'

   return speed, laneCount, duplex

_fecStrRegex = re.compile( '(RS[0-9]+|[A-Z]+FEC)' )

def modeStrToFec( mode ):
   """
   Returns Interface::EthFecEncoding given a string representation of a
   speed-lane-fec link mode.

   Parameters
   ----------
   mode : str
      A speed-lane-fec link mode (e.g. 100g-4, 50g-1, 100m-1, 50g-1 RS544)

   Returns
   -------
   fec : Interface::EthFecEncoding
   """
   mode = mode.upper()
   modeSplit = mode.split()

   # If it exists, FEC can only ever be at the end of a mode string.
   maybeFecStr = modeSplit[ -1 ] if modeSplit else ''
   fecStrMatch = _fecStrRegex.match( maybeFecStr )
   if not fecStrMatch:
      return EthFecEncoding.fecEncodingUnknown

   return _fecStrToEthFecEncoding.get( fecStrMatch.group( 1 ) )

def modeStrToEthIntfMode( intfId, mode ):
   """
   Returns an EthIntfMode given a string representation of a "speed-lane/duplex fec"
   link mode and an interface.

   Parameters
   ----------
   intfId : Arnet::IntfId
   mode : str
      A "aneg speed-lane/duplex fec" link mode
      (e.g. 100g-4, 50g-1, 100m-1, 50g-1 RS544, 100m-1/h, 100m/full, 400g-8/f RS544)
      Note: aneg is optional: forced, cl28, cl37, cl73, unknown by default
            duplex is optional: /h, /half, /f, /full, full duplex by default
            fec is optional: NOFEC, FCFEC, RS528, RS544 unknown by default.

   Returns
   -------
   eim : Interface::EthIntfMode
   """
   eim = Tac.Value( "Interface::EthIntfMode", intfId )
   if not mode:
      return eim

   # Mode strings consist of anywhere between 1 to 3 parts. All parts are optional
   # but their relative ordering is fixed:
   #  1. Autonegotiation mode
   #  2. Speed/Lanes/Duplex
   #  3. FEC
   #
   # Each part has it's own parser and is responsible for determining if the part
   # exists as well as the value of the part.
   anegMode = modeStrToAutonegMode( mode )
   ethSpeed, ethLaneCount, ethDuplex = modeStrToEthSpeedAndLaneCountAndDuplex( mode )
   ethFec = modeStrToFec( mode )

   eim.speed = ethSpeed
   eim.laneCount = ethLaneCount
   eim.duplex = ethDuplex
   eim.fec = ethFec
   eim.anegMode = anegMode
   return eim

def modeStrToEthLinkMode( mode ):
   """
   Returns an EthLinkMode given a string representation of a "speed-lane/duplex" mode
   Parameters
   ----------
   mode : str
      A "speed-lane/duplex" link mode
      (e.g. 100g-4, 50g-1, 100m-1, 50g-1, 100m-1/h, 100m/full, 400g-8/f)
      Note: duplex is optional: /h, /half, /f, /full, full duplex by default
   Returns
   -------
   elm : Interface::EthLinkMode
   """
   return EthTypesApi.ethSpeedLaneToLinkMode(
             *modeStrToEthSpeedAndLaneCountAndDuplex( mode ) )

def modeStrsToEthLinkModeSet( modes ):
   """
   Converts a set of standard mode strings into an EthLinkModeSet.

   Parameters
   ----------
   modes : iter( str)
      A iterable of speed-lane-fec link mode strings. See 
      modeStrToEthSpeedAndLaneCountAndDuplex for exmaples of valid strings.

   Returns
   -------
   An Interface::EthLinkMode
   """
   return reduce(
      ior,
      ( EthTypesApi.ethSpeedLaneToLinkModeSet(
         *modeStrToEthSpeedAndLaneCountAndDuplex( mode ) )
        for mode in modes ) )

#-------------------------------------------------------------------------------
# Utility functions that help format EthIntf-related CLI commands
#-------------------------------------------------------------------------------
# Formatting mappings for EthIntf Enums. Using OrderedDict to help format some
# CLI commands (e.g. `show interfaces interactions`).
speedLanestoSpeedLanesStr = collections.OrderedDict(
      [ ( ( 'speed800Gbps', 'laneCount8' ), '800G-8' ),
        ( ( 'speed400Gbps', 'laneCount4' ), '400G-4' ),
        ( ( 'speed400Gbps', 'laneCount8' ), '400G-8' ),
        ( ( 'speed200Gbps', 'laneCount2' ), '200G-2' ),
        ( ( 'speed200Gbps', 'laneCount4' ), '200G-4' ),
        ( ( 'speed200Gbps', 'laneCount8' ), '200G-8' ),
        ( ( 'speed100Gbps', 'laneCount1' ), '100G-1' ),
        ( ( 'speed100Gbps', 'laneCount2' ), '100G-2' ),
        ( ( 'speed100Gbps', 'laneCount4' ), '100G-4' ),
        ( ( 'speed50Gbps', 'laneCount1' ), '50G-1' ),
        ( ( 'speed50Gbps', 'laneCount2' ), '50G-2' ),
        ( ( 'speed40Gbps', 'laneCount4' ), '40G' ),
        ( ( 'speed25Gbps', 'laneCount1' ), '25G' ),
        ( ( 'speed10Gbps', 'laneCount1' ), '10G' ),
        ( ( 'speed1Gbps', 'laneCount1' ), '1G' ),
        ( ( 'speed100Mbps', 'laneCount1' ), '100M' ),
        ( ( 'speed10Mbps', 'laneCount1' ), '10M' ),
      ] )

autonegToforcedMapping = collections.OrderedDict(
                   [ ( 'mode800GbpsFull8Lane' , 'linkModeForced800GbpsFull8Lane' ),
                     ( 'mode400GbpsFull4Lane' , 'linkModeForced400GbpsFull4Lane' ),
                     ( 'mode400GbpsFull8Lane' , 'linkModeForced400GbpsFull8Lane' ),
                     ( 'mode200GbpsFull2Lane' , 'linkModeForced200GbpsFull2Lane' ),
                     ( 'mode200GbpsFull4Lane' , 'linkModeForced200GbpsFull4Lane' ),
                     ( 'mode200GbpsFull8Lane' , 'linkModeForced200GbpsFull8Lane' ),
                     ( 'mode100GbpsFull1Lane' , 'linkModeForced100GbpsFull1Lane' ),
                     ( 'mode100GbpsFull2Lane' , 'linkModeForced100GbpsFull2Lane' ),
                     ( 'mode100GbpsFull' , 'linkModeForced100GbpsFull' ),
                     ( 'mode50GbpsFull1Lane' , 'linkModeForced50GbpsFull1Lane' ),
                     ( 'mode50GbpsFull' , 'linkModeForced50GbpsFull' ),
                     ( 'mode25GbpsFull' , 'linkModeForced25GbpsFull' ),
                   ] )

duplexStringsFormat1 = { 'duplexHalf' : 'Half-duplex',
                         'duplexFull' : 'Full-duplex',
                         'duplexUnknown' : 'Unconfigured' }

duplexStringsFormat2 = { 'duplexHalf' : 'half',
                         'duplexFull' : 'full',
                         'duplexUnknown' : 'unconfigured' }

duplexStringsFormat3 = { 'duplexHalf' : 'a-half',
                         'duplexFull' : 'a-full',
                         'duplexUnknown' : 'unconfigured' }

duplexStrings = { 'Format1' : duplexStringsFormat1,
                  'Format2' : duplexStringsFormat2,
                  'Format3' : duplexStringsFormat3 }

autonegDuplexStrings = { 'Format1' : 'Auto-duplex',
                         'Format2' : 'auto',
                         'Format3' : 'auto' }

anegStateIntfStrMap = { 'unknown' : 'off',
                        'off' : 'off',
                        'negotiating' : 'on',
                        'success' : 'on',
                        'failed' : 'fail' }
# This is the same format found in Xcvr::HwXcvrSlotType, for use with
# the internal model
slotTypeCamelCase = [
      'osfp400g',
      'qsfpDd400g',
      'osfp800g',
      'qsfpDd800g'
      ]

# This is the preferred format to be used in the Cli
slotTypeDashCase = [
      'osfp-400',
      'qsfp-dd-400',
      'osfp-800',
      'qsfp-dd-800'
      ]

# This dictionary is used to convert between the two formats above
slotTypeFormat = {
      'toCamelCase': dict( zip( slotTypeDashCase, slotTypeCamelCase ) ),
      'toDashCase': dict( zip( slotTypeCamelCase, slotTypeDashCase ) ),
      }

def _duplexStr( value, stringFormat ):
   """
   Returns formatted string for EthDuplex.

   Parameters
   ----------
   value : string representation of Interface::EthDuplex
   stringFormat : string describing desired format (one of "Format[123]")
   """
   d = duplexStrings.get( stringFormat )
   return d[ value ] if d else None


CapList = namedtuple( 'CapList', [ 'speed', 'lanes', 'showLanes', 'duplex',
                                   'default', 'pllIncompatible' ] )

FecCapList = namedtuple( 'FecCapList', [ 'encoding', 'speed', 'lanes',
                                         'showLanes', 'default', 'duplex' ] )

def cliNeverShowLanes( speedKey ):
   """
   Some speeds never show lanes in CLI output. Return, for a given
   speed, whether lanes should never be shown in the CLI.

   speedKey: Short-hand string shown in the output of show interface status.
   e.g. 10M, 2.5G, 400G.
   """
   return speedKey in [ "10M", "100M", "1G", "2.5G", "5G", "10G", "25G", "40G" ]

# input : Interface::EthLinkModeSet
# return : List of CapList
def linkModeCapabilitiesToList( caps, autonegCapable=False, defaultMode=None,
                                showLanes=False, imcompatElms=None ):

   if imcompatElms is None:
      imcompatElms = Tac.Value( 'Interface::EthLinkModeSet' )

   capLists = []
   for capable, speed, lanes, duplex, pllIncompatible, mode in (
      ( caps.mode10MbpsHalf, EthSpeed.speed10Mbps, EthLaneCount.laneCount1, 'half',
        False, 'linkModeForced10MbpsHalf' ),
      ( caps.mode10MbpsFull, EthSpeed.speed10Mbps, EthLaneCount.laneCount1, 'full',
        False, 'linkModeForced10MbpsFull' ),
      ( caps.mode100MbpsHalf, EthSpeed.speed100Mbps, EthLaneCount.laneCount1, 'half',
        False, 'linkModeForced100MbpsHalf' ),
      ( caps.mode100MbpsFull, EthSpeed.speed100Mbps, EthLaneCount.laneCount1, 'full',
        False, 'linkModeForced100MbpsFull' ),
      ( caps.mode1GbpsHalf, EthSpeed.speed1Gbps, EthLaneCount.laneCount1, 'half',
        False, 'linkModeForced1GbpsHalf' ),
      ( caps.mode1GbpsFull, EthSpeed.speed1Gbps, EthLaneCount.laneCount1, 'full',
        imcompatElms.mode1GbpsFull, 'linkModeForced1GbpsFull' ),
      ( caps.mode2p5GbpsFull, EthSpeed.speed2p5Gbps, EthLaneCount.laneCount1, 'full',
        False, 'linkModeForced2p5GbpsFull' ),

      # 5G doesn't have corresponding linkmodes. They are only used in autoneg.
      ( caps.mode5GbpsFull, EthSpeed.speed5Gbps, EthLaneCount.laneCount1, 'full',
        False, 'NON_EXISTENT_LINK_MODE' ),

      ( caps.mode10GbpsFull, EthSpeed.speed10Gbps, EthLaneCount.laneCount1, 'full',
        imcompatElms.mode10GbpsFull, 'linkModeForced10GbpsFull' ),
      ( caps.mode25GbpsFull, EthSpeed.speed25Gbps, EthLaneCount.laneCount1, 'full',
        imcompatElms.mode25GbpsFull, 'linkModeForced25GbpsFull' ),
      ( caps.mode40GbpsFull, EthSpeed.speed40Gbps, EthLaneCount.laneCount4, 'full',
        imcompatElms.mode40GbpsFull, 'linkModeForced40GbpsFull' ),
      ( caps.mode50GbpsFull, EthSpeed.speed50Gbps, EthLaneCount.laneCount2, 'full',
        imcompatElms.mode50GbpsFull, 'linkModeForced50GbpsFull' ),
      ( caps.mode50GbpsFull1Lane, EthSpeed.speed50Gbps, EthLaneCount.laneCount1,
        'full', imcompatElms.mode50GbpsFull1Lane, 'linkModeForced50GbpsFull1Lane' ),
      ( caps.mode100GbpsFull, EthSpeed.speed100Gbps, EthLaneCount.laneCount4, 'full',
        imcompatElms.mode100GbpsFull, 'linkModeForced100GbpsFull' ),
      ( caps.mode100GbpsFull2Lane, EthSpeed.speed100Gbps, EthLaneCount.laneCount2,
        'full', imcompatElms.mode100GbpsFull2Lane,
        'linkModeForced100GbpsFull2Lane' ),
      ( caps.mode100GbpsFull1Lane, EthSpeed.speed100Gbps, EthLaneCount.laneCount1,
        'full', imcompatElms.mode100GbpsFull1Lane,
        'linkModeForced100GbpsFull1Lane' ),
      ( caps.mode200GbpsFull8Lane, EthSpeed.speed200Gbps, EthLaneCount.laneCount8,
        'full', imcompatElms.mode200GbpsFull8Lane,
        'linkModeForced200GbpsFull8Lane' ),
      ( caps.mode200GbpsFull4Lane, EthSpeed.speed200Gbps, EthLaneCount.laneCount4,
        'full', imcompatElms.mode200GbpsFull4Lane,
        'linkModeForced200GbpsFull4Lane' ),
      ( caps.mode200GbpsFull2Lane, EthSpeed.speed200Gbps, EthLaneCount.laneCount2,
        'full', imcompatElms.mode200GbpsFull2Lane,
        'linkModeForced200GbpsFull2Lane' ),
      ( caps.mode400GbpsFull8Lane, EthSpeed.speed400Gbps, EthLaneCount.laneCount8,
        'full', imcompatElms.mode400GbpsFull8Lane,
        'linkModeForced400GbpsFull8Lane' ),
      ( caps.mode400GbpsFull4Lane, EthSpeed.speed400Gbps, EthLaneCount.laneCount4,
        'full', imcompatElms.mode400GbpsFull4Lane,
        'linkModeForced400GbpsFull4Lane' ),
      ( caps.mode800GbpsFull8Lane, EthSpeed.speed800Gbps, EthLaneCount.laneCount8,
        'full', imcompatElms.mode800GbpsFull8Lane,
        'linkModeForced800GbpsFull8Lane' ),
      ( autonegCapable, 'auto', EthLaneCount.laneCountUnknown, 'full', False,
        'auto' ) ):
      if capable:
         default = bool( mode == defaultMode )
         capList = CapList( speed, lanes, showLanes, duplex, default,
                            pllIncompatible )
         capLists.append( capList )

   return capLists

def generateOperDuplex( stringFormat, autonegActive, duplex ):
   """
   Returns formatted string for operational duplex

   Parameters
   ----------
   stringFormat : string describing desired format (one of "Format[123]")
   autonegActive : boolean describing whether autoneg is active or not
   duplex : string representation of Interface::EthDuplex
   """
   # If auto-negotiation is active and the duplex has not yet
   # been determined, just report auto-duplex
   if autonegActive and duplex == 'duplexUnknown':
      return autonegDuplexStrings[ stringFormat ]
   return _duplexStr( duplex, stringFormat )

def generateAutoNegotiation( autonegState ):
   """
   Helper function that returns the string representation of the interface's
   autonegotiation status

   Parameters
   ----------
   autonegState : string representation of autoneg state
   """
   return anegStateIntfStrMap[ autonegState ]

def getSpeedString( speed, autonegActive, stringFormat ):
   """
   Given a speed in bit/sec and if auto-negotiate this will produce a human
   readable string of the speed of the interface.

   Parameters
   ----------
   speed : floating point value representing speed in bit/sec
   autonegActive : boolean describing whether autoneg is active
   stringFormat : string corresponding to output formats. Example if given a
                  bit/s of 10000000
                  Format1 - 10Mb/s
                  Format2 - 10M
                  Format3 - a-10M
   """
   if autonegActive and speed == 0:
      return "Auto-speed" if stringFormat == 'Format1' else "auto"
   elif speed == 0:
      return "Unconfigured" if stringFormat == 'Format1' else "unconfigured"

   ( scaledBps, units ) = HumanReadable.scaleValueSi( speed )
   # We should not be doing a floating point comparison here
   if scaledBps % 1.0 == 0.0:
      fmt = "%d%s"
   else:
      fmt = "%.1f%s"

   if stringFormat == 'Format1':
      fmt = "%sb/s" % fmt
   elif stringFormat == 'Format2':
      fmt = "%s" % fmt
   else:
      fmt = "a-%s" % fmt

   return fmt % ( scaledBps, units )

def getLaneCountString( laneCount ):
   """
   Returns description of how many electrical lanes are used for a configuration

   Parameters
   ----------
   laneCount : integer representing number of electrical lanes
   """
   if laneCount == 0:
      return ""
   elif laneCount == 1:
      return " over 1 lane"
   else:
      return " over %d lanes" % laneCount

def showIntStatusSpeedStr( autonegActive, bandwidth ):
   """
   Returns formatted string for speed. Utility function used by
   "showInterfacesStatus()"

   Parameters
   ----------
   autonegActive : boolean describing whether autoneg is active
   bandwidth : floating point value representing bps
   """
   fmt = 'Format3' if autonegActive else 'Format2'
   return getSpeedString( bandwidth, autonegActive, fmt )

def showIntStatusDuplexStr( autonegActive, duplex ):
   """
   Returns formatted string for duplex. Utility function used by
   "showInterfacesStatus()"

   Parameters
   ----------
   autonegActive : boolean describing whether autoneg is active
   duplex : string representation of Interface::EthDuplex
   """
   fmt = 'Format3' if autonegActive else 'Format2'
   return generateOperDuplex( fmt, autonegActive, duplex )

def createEthIntfSysdbState( entMan, isSysdb=True ):
   """
   Sets up some reactors to be run by Sysdb related to 
   reacting to  EthPhyIntfConfig and EthPhyIntfStatus, and
   also sets up the reactors to populate state in the AllEthReactor.

   Parameters
   ----------
   entMan : The EntityManager object in Sysdb.
   isSysdb : Used to indicate whether this function is called in
   Sysdb as part of the SysdbPlugin, or if it is false, it 
   indicates this function is being called as part of the IntfAgent.
   """
   t0( "SysdbEthIntf.py plugin loading..." )
   # Create the Interface::AllEthPhyIntf{Config,Status}Dir's
   eicd = entMan.lookup( "interface/config/eth/phy*/all" )
   eisd = entMan.lookup( "interface/status/eth/phy*/all" )

   # Create the Interface::EthIntf{Config,Status}Dir's
   aecd = entMan.lookup( "interface/config/eth/intf" )
   aesd = entMan.lookup( "interface/status/eth/intf" )
   gicd = entMan.lookup( "interface/config/global" )

   eicsd = entMan.lookup( "interface/config/eth/phy*/slice" )
   eissd = entMan.lookup( "interface/status/eth/phy*/slice" )
   reicsd = None
   if pa.productAttributes().productAttributes.sysdbCreatesResolvedConfig:
      reicsd = entMan.lookup( "interface/resolvedconfig/eth/phy*/slice" )

   # these are the cell-specific Interface::EthIntfStatusLocal entities,
   # which track the per-supervisor state such as the namespace.
   eisld = entMan.lookup( Cell.path( "interface/status/eth/local" ) )
   episld = entMan.lookup( Cell.path( "interface/status/eth/phy/local" ) )

   portIdDir = entMan.lookup( "interface/eth/portid" )

   localRoot = entMan.root().parent
   allEthDir = localRoot.newEntity( "SysdbEthIntf::AllEthDir", "allethdir" )
   entMan.allEthDir = allEthDir

   # The dependency from the SysdbEthIntf plugin to the SysdbIntf plugin has
   # been removed. So it is possible that the plugin loader is loading the 
   # SysdbEthIntf plugin first before it loads the SysdbIntf plugin. 
   # The SysdbIntf plugin used to create the allIntfDir, which was then accessed 
   # by this plugin. Since there is no explicit dependency now, the allIntfDir 
   # object might not have yet been created. Create this object if it hasn't been
   # created.

   allIntfDir = localRoot.createEntity( "SysdbIntf::AllIntfDir", "allintfdir" )
   assert allIntfDir

   # Create locally-mounted reactors in sysdbEthIntfSm
   agent = localRoot.newEntity( "SysdbEthIntf::Agent", "SysdbEthIntf" )

   allIntfConfig = allIntfDir.config
   allIntfStatus = allIntfDir.status
   allEthIntfConfig = allEthDir.config
   allEthIntfStatus = allEthDir.status

   aisld = Cell.root( entMan.root() )[ 'interface' ][ 'status' ][ 'local' ]

   # Initialize reactors to copy EthPhyIntfConfigs and EthPhyIntfStatuses
   # to entMan.allIntfDir and entMan.allEthDir
   # For the EthPhyIntfConfigs, we first need a reactor to handle
   # the config requests.
   # This reactor populates interface/config/eth/phy/all
   # The second reactor, ethConfigReactor, in turn populates the allIntf and
   # allEthInt dirs
   agent.ethConfigReactor = ( eicd, allEthIntfConfig, allIntfConfig )

   # Setup a reactor that handles slice dirs added (by Fru) under 
   # interface/status/eth/phy/slice
   agent.ethPhyIntfStatusSliceDirReactor = ( eissd, eicsd, eisd, eicd, episld,
                                             allEthIntfStatus, allIntfStatus,
                                             eisld, aisld, portIdDir, gicd, reicsd )

   def _handleActive( active ):
      if active:
         t0( "Creating locally-mounted reactors" )
         # Create locally-mounted reactors under /ar/SysdbEthIntf

         # Instantiate an AllEthReactor to sync out our local collection of all 
         # EthIntfs.
         if not ( hasattr( agent, 'allEthReactor' ) and 
                  agent.allEthReactor ):
            agent.allEthReactor = ( aecd, aesd, allEthDir.config, allEthDir.status )

         # Setup the reactor for processing the intfConfig requests (from Sysdb
         # and Cli)
         agent.ethPhyIntfConfigSliceDirReactor = ( eicsd, eicd, reicsd, gicd )

         # EthPhyIntfStatusDirReactor should no longer be running in standby mode.
         agent.ethPhyIntfStatusSliceDirReactor.standbyMode = False
      else:
         # Note: this cleanup code is always a no-op except in testing.
         agent.ethPhyIntfStatusSliceDirReactor.standbyMode = True
         t0( "Destroying locally-mounted reactors" )
         agent.ethPhyIntfConfigSliceDirReactor = None
         agent.allEthReactor = None

   if isSysdb:
      entMan.registerActiveCallback( _handleActive )
   else:
      _handleActive( not entMan.locallyReadOnly() )

def fecEncodingsToEthFecEncodingConfigSet( fecEncodings ):
   """
   Given a collection of fec Encodings, populate a
   EthFecEncodingConfigSet with its values
   """
   efecs = EthFecEncodingConfigSet()
   if EthFecEncoding.fecEncodingDisabled in fecEncodings:
      efecs.fecEncodingDisabled = True
   if EthFecEncoding.fecEncodingReedSolomon544 in fecEncodings:
      efecs.fecEncodingReedSolomon544 = True
   if EthFecEncoding.fecEncodingReedSolomon in fecEncodings:
      efecs.fecEncodingReedSolomon = True
   if EthFecEncoding.fecEncodingFireCode in fecEncodings:
      efecs.fecEncodingFireCode = True
   return efecs
