# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

def varEnvName( varComps ):
   ''' 
   Takes a tuple of ( source, name, modifier ) for a variable
   returns a unique string that is suitable for a bash environment var name
   Examples:

       ( "bashCmd". "ls /tmp/EventBashPluginTest | wc | awk '{print $1}'", "" )
     Will  return:
       bashCmd_ls__tmp_EventBashPluginTest___wc___awk___print__1__

       ( "tcollector.demo.testvalue2". "param1=150,param2=/1/2/3", "delta" )
     Will  return:
       tcollector_demo_testvalue2_param1_150_param2__1_2_3_delta
   '''
   var = varComps[0]
   if varComps[1]:
      var += '_' + varComps[1]
   if varComps[2]:
      var += '_' + varComps[2]
   varName = ''
   for c in var:
      if c.isalnum():
         varName += c
      else:
         varName += '_'
   return varName

class ParsedExpression:
   def __init__( self ):
      self.vars = {}
      self.perSrcResult = ""
      self.expandedExp = ""
      self.hasWildcard = False

   def varsToLines( self ):
      lines = ''
      for var in self.vars: # pylint: disable=consider-using-dict-items
         lines += var[0] + '\n' 
         lines += var[1] + '\n' 
         lines += var[2] + '\n' 
         lines += self.vars[var] + '\n'
         lines += varEnvName( var ) + '\n' 
      return lines

def splitVar( var ):
   ''' 
   Split a variable name into its 3 components: source, counter name, modifier
   returns a tuple with 3 items
   Examples:

       "bashCmd.VAR1"
     Will return:
       ( "bashCmd", "VAR1", "" )

       "tcollector.demo.testvalue2.param1=150,param2=/1/2/3.delta"
     Will  return:
       ( "tcollector.demo.testvalue2", "param1=150,param2=/1/2/3", "delta" )
   '''
   parts = var.split( "." )
   count = len( parts )
   if count < 2:
      return None
   for part in parts:
      if not part:
         return None
   if parts[-1] == 'delta':
      modifier = parts[-1]
      del parts[-1]
      count -= 1
   else:
      modifier = ''
   if count == 0:
      return ( var, '', '' )
   if parts[0] == 'tcollector':
      separator = ''
      src = ''
      for part in parts:
         count -= 1
         if '=' in part:
            # Make sure that this is the last part
            if count:
               t0( "splitVar: counter name not last", var )
               return ( var, '', '' )
            return ( src, part, modifier )
         src = src + separator + part
         separator = '.'
      return ( src, '', modifier )
   elif count == 1:
      return ( parts[0], '', modifier )
   elif count == 2:
      return ( parts[0], parts[1], modifier )
   return ( var, '', '' )

def combineVarComps( varComps ):
   ''' 
   Does the opposite of splitVar
   Examples:

       ( "bashCmd", "VAR1", "" )
     Will return:
       "bashCmd.VAR1"

       ( "tcollector.demo.testvalue2", "param1=150,param2=/1/2/3", "delta" )
     Will  return:
       "tcollector.demo.testvalue2.param1=150,param2=/1/2/3.delta"
   '''
   var = varComps[0]
   if varComps[1]:
      var += '.' + varComps[1]
   if varComps[2]:
      var += '.' + varComps[2]
   return var

def verifyExpression( exp, parsedResult=None, counterSources=None ):
   import re # pylint: disable=import-outside-toplevel

   t0( "verifyExpression: " + exp )
   varMap = {}

   # Since the tcollector counters can have counter names with the = char,
   # we need to distinguish between this case and the ==, >=, and <=  operators.
   # Add space before and after since the counter names do not include spaces
   expandedExp = exp.replace( '==', ' == ' )
   expandedExp = expandedExp.replace( '>=', ' >= ' )
   expandedExp = expandedExp.replace( '<=', ' <= ' )
   expandedExp = expandedExp.replace( '!=', ' != ' )

   # Since the bashCmd source can have commands with any character,
   # we will extract these commands and replace them with a simple var name
   if "bashCmdFloat" in expandedExp:
      validBashVar = re.compile( r'bashCmdFloat\."((?:\\.|[^"\\])+?)"' )
      cmdStr = "bashCmdFloat."
   else:
      validBashVar = re.compile( r'bashCmd\."((?:\\.|[^"\\])+?)"' )
      cmdStr = "bashCmd."
   i = 0
   bashVarMap = {}
   for match in validBashVar.finditer( expandedExp ):
      repName = "VAR%s" % i # pylint: disable=consider-using-f-string
      bashVarMap[repName] = match.group(1).replace( r'\"', r'"' )
      expandedExp = expandedExp.replace( match.group( 0 ), cmdStr + repName )
      i += 1

   #   Make sure that it only includes valid characters
   validCharExp = re.compile( r'[^a-zA-Z0-9_,\.\+\-\/*<>=!\s\(\)\"]+' )
   if validCharExp.search( expandedExp ):
      return "Invalid expression"

   #   Extract the list of list of SUM calls and expand them
   sums = {}
   validSumExp = re.compile(r'SUM\s*\(\s*(.+?)\s*\)')
   for match in validSumExp.finditer( expandedExp ):
      var = match.group(1)
      sumExp = match.group(0)
      sums[ sumExp ] = var
      parts = splitVar( var )
      if not parts:
         # pylint: disable-next=consider-using-f-string
         return "Invalid counter name: %s" % var
      source = parts[0]
      if parts[1]:
         name = "." + parts[1]
      else:
         name = ""
      if parts[2]:
         mod = "." + parts[2]
      else:
         mod = ""
      srcs = expandWildcard( source, counterName=parts[1],
                             counterSources=counterSources )
      expandedSum = "(  "
      for src in srcs:
         expandedSum += src + name + mod + " + "
      expandedSum = expandedSum[:-2] + ")"
      expandedExp = expandedExp.replace( sumExp, expandedSum )

   #   Extract the list of variables
   validVarExp = re.compile(r'[a-zA-Z0-9_,=\-\.\*/]+')

   # To extract all variable names we need to take care of the <= and >= operators
   # since the = character can be part of the tcollector variable name. The regular
   # expression is assuming the the = part of >= is a variable. We need to remove >=
   # and <= operators for the purpose of variable name extraction
   newExp = expandedExp.replace( '<=', '' )
   newExp = newExp.replace( '>=', '' )
   newExp = newExp.replace( '!=', '' )
   variables = validVarExp.findall( newExp )

   #   Verify that all variables have standard names
   wildcard = ""
   validOps = [ "SUM", "and", "or", "not", "==", '<=', '>=', '!=' ]
   i = 0
   for var in variables:
      if var in validOps:
         continue
      # check if the token is a number
      try:
         float( var )
         continue
      except ValueError:
         parts = splitVar( var )
      if not parts:
         # pylint: disable-next=consider-using-f-string
         return "Invalid counter name: %s" % var
      wildcardVarList = []
      if '*' in parts[0]:
         # Variable name has a wildcard. Check for the invalid cases
         if wildcard and parts[0] != wildcard:
            return "More than  one wildcard was used"
         if '*' in parts[0][:-1]:
            return "Wildcard can only be at the end of the counter source"
         wildcard = parts[0]
         srcs = expandWildcard( parts[0], counterName=parts[1],
                                counterSources=counterSources )
         name = parts[1]
         if parts[2]:
            name += '.' + parts[2]
         for src in srcs:
            varNameParts = ( src, parts[1], parts[2] )
            wildcardVarList.append( varNameParts )
      else:
         wildcardVarList.append( parts )

      for var1 in wildcardVarList:
         if var1 not in varMap:
            # pylint: disable-next=consider-using-f-string
            replacementVarName = "VAR%s" % i
            varMap[ var1 ] = replacementVarName
            i += 1
            if parsedResult:
               if var1[ 0 ] == 'bashCmd' or var1[ 0 ] == 'bashCmdFloat' :
                  var1 = ( var1[0], bashVarMap[var1[1]], var1[2] )
               parsedResult.vars[ var1 ] = replacementVarName

   perSrcResult = "srcExpResult = {};\n"
   srcs = [ var[0] for var in varMap ]
   newExp = ""
   for src in srcs:
      # Expand the expression if it has wild cards
      if wildcard:
         srcExpResult = expandedExp.replace( wildcard, src )
      else:
         srcExpResult = expandedExp
      perSrcResult += \
         "try:\n" + \
         "   srcExpResult[ '" + src + "' ] = ( " + \
               srcExpResult + " )\n" + \
         "except NameError:\n" + \
         "   srcExpResult[ '" + src + "' ] = False\n"
      newExp += "( srcExpResult[ '" + src + "' ] ) or\\\n"
   if newExp:
      expandedExp = newExp[:-5]
   else:
      expandedExp = "0"

   #   Replace the variables with simple var names
   newExp = expandedExp
   # Do the ones with a modifier first
   for var in varMap: # pylint: disable=consider-using-dict-items
      if var[2]:
         newExp = newExp.replace( combineVarComps( var ), varMap[ var ] )
         perSrcResult = perSrcResult.replace( combineVarComps( var ), varMap[ var ] )
   # Do the rest
   for var in varMap: # pylint: disable=consider-using-dict-items
      if not var[2]:
         newExp = newExp.replace( combineVarComps( var ), varMap[ var ] )
         perSrcResult = perSrcResult.replace( combineVarComps( var ), varMap[ var ] )

   # Compile the expression
   try:
      compile( perSrcResult, 'EventMgrPreProcessing', 'exec' )
      compile( newExp, 'EventMgrCond', 'eval' )
   except SyntaxError as e:
      return e.msg

   if parsedResult:
      parsedResult.perSrcResult = perSrcResult
      parsedResult.expandedExp = newExp
      parsedResult.hasWildcard = ( wildcard != "" )

   return None

sources = {}

def initSources( counterSources, pluginStatus ):
   for plugin in pluginStatus.values():
      for source, sourceEntity in plugin.counterSources.items():
         counterSources[ source ] = list( sourceEntity.counterNames.values() )

def getCounterSources():
   t0( "getCounterSources: ", list( sources ) )
   return list( sources )

def getCounterNames( source ):
   return sources.get( source )

def expandWildcard( source, counterName=None, counterSources=None ):
   if source.endswith( "*" ):
      source = source[:-1]
   if counterSources:
      if counterName is None:
         return [ x for x in counterSources if x.startswith( source ) ]
      else:
         return [ x for x in counterSources if x.startswith( source ) and
                  counterName in counterSources.get( x ) ]
   else:
      if counterName is None:
         return [ x for x in getCounterSources() if x.startswith( source ) ]
      else:
         return [ x for x in getCounterSources() if x.startswith( source ) and
                  counterName in getCounterNames( x ) ]

def getConditionForSubhandler( condition, source ):
   import re # pylint: disable=import-outside-toplevel
   validVarExp = re.compile(r'[a-zA-Z0-9_,=\-\.\*/]+')
   variables = validVarExp.findall( condition )
   validOps = [ "SUM", "and", "or", "not", "==", '<=', '>=', '!=' ]
   newCondition = condition
   for var in variables:
      if var in validOps:
         continue
      try:
         float( var )
         continue
      except ValueError:
         parts = splitVar( var )
      if '*' in parts[0]: # pylint: disable=used-before-assignment
         wildcard = parts[0]
         if source.startswith( wildcard[ : -1 ] ):
            newCondition = condition.replace( wildcard, source )
            return newCondition
   return None
