#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable-msg=W0702
# pkgdeps : library EventMon

# pylint: disable=consider-using-f-string
import Tac
import Agent
import SharedMem
from Task import ( Task, TaskRunResult )
import math
import os
import sqlite3
import Tracing
import re
import Logging
import Plugins
import glob

from EventMonUtils import qtTableConfigs

EVENTMON_DB_WRITE_FAILED = Logging.LogHandle(
              "EVENTMON_DB_WRITE_FAILED",
              severity=Logging.logError,
              fmt="A sqlite %s exception occurred "
              "when writing to the EventMon database",
              explanation="An error occurred when trying to write to "
              "EventMon Database",
              recommendedAction=Logging.CONTACT_SUPPORT )

__defaultTraceHandle__ = Tracing.Handle( "EventMon" )
th = Tracing.defaultTraceHandle()
t0 = th.trace0 # Errors
t1 = th.trace1 # Major lifecycle events
t8 = th.trace8 # Noise
t9 = th.trace8 # Frequent noise
eventTypeDict = { '1': "updated", '2': "added", '3': "deleted",
            '4': "aging deletion", '5': "aging deletion(hw not programmed)",
            '6': "aging deletion(tcp closed)", '7': "peer deletion" }

QT_PARSE_RE = re.compile( r"\(.*\)" )

# Timeout value when invoking qtcat: if the commands takes that amount of time to
# run, we'll abort invoking qtcat.
# Go with a timeout of 10 seconds: at most it's possible that EventMon ends up
# syncing more than 6 files, so an average runtime of 10 secondes per qt file would
# result in a SIGQUIT.
QTTAIL_TIMEOUT = 10

# Minimum number of rows to wait in between calls to maybeDeleteRows.
MAYBE_DELETE_DELAY = 10

# To ensure that deleting does not take longer than a quantum, always delete from
# a table that is more than MAYBE_DELETE_THRESHOLD above its max table size.
MAYBE_DELETE_THRESHOLD = 400000 # 400kb

# The percentage of a table that is written before tableSize is queried again
REPLACED_BEFORE_SIZE_QUERY = 0.9

def removeFile( filePath ):
   try:
      if os.path.isdir( filePath ): # pylint: disable=no-else-raise
         raise OSError( 'EventMonAgent attempting to delete a directory' )
      else:
         os.remove( filePath )
   except FileNotFoundError:
      t0( f'delete failed; {filePath} does not exist' )
   except OSError:
      t0( 'could not delete', filePath )

def runQtTail( qtFileName ):
   return Tac.run( [ '/usr/bin/qttail', '-c', qtFileName ],
         stdout=Tac.CAPTURE,
         timeout=QTTAIL_TIMEOUT )

def QTFileIterator( qtFileName ):
   ''' returns msg lines for qt file '''
   try:
      os.stat( qtFileName )
   except OSError:
      t0( 'could not find qt file', qtFileName )
      return

   # typical output line looks as below:
   # 2012-05-18 09:13:18 0 00051.325776031, +965174 "1 %
   # (1.1.1.0/32,,,,removed,11)
   #
   # space split output looks as follows:
   # ['2012-05-18', '09:13:18', '0', '00051.325776031,',
   # '+965174', '"1', '%', '(1.1.1.0/32,,,,removed,11)', '"']
   #
   # first,second fields are date,time
   # seventh field is msg - use QT_PARSE_RE to strip the braces
   #   and quotes to get the user msg
   try:
      DATE_FIELD = 0
      TIME_FIELD = 1

      before = Tac.now()
      output = runQtTail( qtFileName )
      delta = Tac.now() - before
      # If running qtcat took more than 50% of our timeout, log a message.
      if delta > QTTAIL_TIMEOUT / 2:
         t0( 'Invoking qtcat on ', qtFileName, 'took', delta, 'seconds' )
      if not output:
         return

      traceLines = output.split( '\n' )
      for traceLine in traceLines:
         if not traceLine or "first msg" in traceLine:
            continue

         traceAttrArr = traceLine.split( " " )
         msgMatch = QT_PARSE_RE.search( traceLine )
         if msgMatch:
            msgLine = msgMatch.group( 0 )
            timestamp = traceAttrArr[ DATE_FIELD ] \
                  + " " + traceAttrArr[ TIME_FIELD ]
            msg = ''.join( c for c in msgLine if c not in '()\'\\ ' )
            yield ( timestamp, msg )
   except Exception as e: # pylint: disable=broad-except
      t0( "qtcat failed on:", qtFileName, "with:", e )

def foreverLogEnabled( config, tableId=None ):
   if not tableId:
      return config.foreverLogEnabled
   table = config.table[ tableId ]
   if ( table and ( ( table.foreverLogOverride and table.foreverLogEnabled ) or
      ( not table.foreverLogOverride and config.foreverLogEnabled ) ) ):
      return True
   return False

def foreverLogPath( config, tableId ):
   table = config.table[ tableId ]
   if table.foreverLogOverride:
      foreverPath = "{}/{}".format( table.foreverLogDirectory,
                                    tableId + table.foreverLogFileName )
   else:
      foreverPath = "{}/{}".format( config.foreverLogDirectory,
                                    tableId + config.foreverLogFileName )
   return foreverPath

class EventMonDbMgr:
   class TableSizeInfo:
      def __init__( self, maxTableSize, pageSize ):
         percentagePadding = 0.1

         # A running estimate of memory occupied by an average row in the table.
         self.rowSizeEstimate = 0

         # The number of rows in the table last time maybeDeleteRows was called.
         self.lastRowCount = 0

         # Number of rows written since last time tableSize was queried. Used to
         # amortize getTableSize calls across maybeDeletes.
         self.rowsSinceTableSize = 0

         # The number of rows to write before calling maybeDeleteRows again.
         self.rowsUntilMaybeDelete = MAYBE_DELETE_DELAY
         if maxTableSize == 0:
            self.maxSize = 0
            self.extraSize = 0
            self.totalSize = 0
            self.percentageToDel = 0
            self.writeBetweenQuery = 0
            return

         # Compute max size based on pagination, to improve page utilization.
         self.maxSize = math.ceil( maxTableSize / pageSize ) * pageSize

         # The amount of memory written to this table before requerying tableSize
         # to update rowSizeEstimate. Used to amortize the getTableSize cost.
         self.writeBetweenQuery = self.maxSize * REPLACED_BEFORE_SIZE_QUERY

         # Deletes are performed at a threshold above the max size.
         threshold = min( MAYBE_DELETE_THRESHOLD, maxTableSize * percentagePadding )

         # Compute padding size based on pagination, to improve page utilization.
         self.extraSize = math.ceil( threshold / pageSize ) * pageSize

         self.totalSize = self.maxSize + self.extraSize
         self.percentageToDel = self.extraSize / self.totalSize

   def __init__( self, dbLocation, tableSchema, tableConfig, rowCount, tableFull ):
      self.initialized = False
      self.dbLocation = dbLocation
      self.tableSchema = tableSchema
      self.tableConfig = tableConfig
      self.rowCount = rowCount
      self.tableSizeInfo = {}
      self.createDb()
      if not self.initialized:
         t0( "db creation failed, flushing old db" )
         self.flushDb()
      self.pageSize = self.getPageSize()
      self.tableFull = tableFull
      for tableName in tableConfig:
         self.tableSizeInfo[ tableName ] = \
            self.TableSizeInfo( tableConfig[ tableName ].maxTableSize.val,
               self.pageSize )

         # Run a sql query to get the row counts in the case that the db already
         # exists.
         rowCount = self.getRowCount( tableName )
         self.rowCount[ tableName ] = 0 if rowCount is None else rowCount

   def flushDb( self, ):
      self.deleteDb()
      self.createDb()

      if not self.initialized:
         t0( "db creation attempt failed, deleting empty eventMon.db" )
         self.deleteDb()

   def flushTable( self, tableName ):
      t1( "flushTable", tableName )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return

      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            columnNames = ",".join( [ ( f"{attr} {attrType}" )
               for attr, attrType, _ in self.tableSchema[ tableName ] ] )
            db.execute( "drop table if exists %s" % tableName )
            db.execute( f"create table {tableName} ({columnNames})" )
            for attr, attrType, indexed in self.tableSchema[ tableName ]:
               if indexed:
                  db.execute( "create index idx_%s_%s on %s (%s)" %
                        ( tableName, attr, tableName, attr ) )

      except sqlite3.Error as e:
         t0( "flushTable: exception", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

   def createDb( self, ):
      t1( "createDb at", self.dbLocation )
      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            for tableName in self.tableSchema:
               columnNames = ",".join( [ ( f"{attr} {attrType}" )
                  for attr, attrType, _ in self.tableSchema[ tableName ] ] )
               db.execute( "create table if not exists %s (%s)" %
                     ( tableName, columnNames ) )
               # Index needs to be created after the table is,
               # meaning we have to go though again
               for attr, attrType, indexed in self.tableSchema[ tableName ]:
                  if indexed:
                     db.execute( "create index if not exists "
                                f"idx_{ tableName }_{ attr } on "
                                f"{ tableName } ({ attr })" )
            self.initialized = True
      except sqlite3.Error as e:
         t0( "createDb: exception", e )
         self.initialized = False
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

   def deleteDb( self, ):
      t1( "deleteDb at", self.dbLocation )
      removeFile( self.dbLocation )
      for tableName in self.tableConfig:
         if tableName in self.rowCount:
            self.rowCount[ tableName ] = 0

   # To conserve space in QT, we packed ( target, fullCone, upnpIgd, addrOnly,
   # twiceNat, estalished ) into an U8 boolArr for sfe and nat, now we
   # need to decode them before flushing into the database.
   def processNatMsg( self, evtMsg ):
      decodedList = list( f'{int( evtMsg[ -3 ] ):08b}' )
      boolArr = [ "T" if x == "1" else "F" for x in decodedList[ : -1 ] ]
      boolArr[ 0 ] = "src" if boolArr[ 0 ] == "F" else "dst"
      eventType = eventTypeDict[ evtMsg[ -2 ] ]
      # HwStatusPresent is not relevant if the event is added in add/delConnection
      boolArr[ -1 ] = "N/A" if eventType in ( "added", "deleted" ) else boolArr[ -1 ]
      evtMsg = evtMsg[ : -3 ] + boolArr + [ eventType ] + evtMsg[ -1 : ]
      return evtMsg

   def getPageSize( self ):
      t8( "getPageSize" )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return None

      size = None
      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            sql = "pragma page_size;"
            cursor = db.execute( sql )
            size = cursor.fetchone()[ 0 ]
            t8( sql, size )
      except sqlite3.Error as e:
         t0( "getPageSize: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return None
      finally:
         db.close()
      return size

   # Returns the size of a table in bytes.
   def getTableSize( self, tableName ):
      t8( "getTableSize", tableName )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return None

      if ( not tableName ) or ( tableName not in self.tableSchema ):
         t0( "table", tableName, "not found" )
         return None

      size = None
      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            sql = f'select payload from "dbstat" where name="{tableName}" and ' + \
               'aggregate=true;'
            cursor = db.execute( sql )
            size = cursor.fetchone()[ 0 ]
            t8( sql, size )
      except sqlite3.Error as e:
         t0( "getTableSize: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return None
      finally:
         db.close()
      return size

   # Returns the number of rows in a table. This relies on the invariant that our
   # eviction scheme within the table always deletes the oldest rows first, i.e. that
   # rowIds will be contiguous and that the table is effectively a circular buffer.
   # We rely on this invariant to avoid a very costly 'select count(*) from table'
   def getRowCount( self, tableName ):
      t8( "getRowCount", tableName )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return 0

      if ( not tableName ) or ( tableName not in self.tableSchema ):
         t0( "table", tableName, "not found" )
         return 0

      size = 0
      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            sql = "select maxRowId - minRowId + 1 from" + \
               f"(select max(rowId) as maxRowId from {tableName}) join" + \
               f"(select min(rowId) as minRowId from {tableName});"
            cursor = db.execute( sql )
            size = cursor.fetchone()[ 0 ]
            t8( sql, size )
      except sqlite3.Error as e:
         t0( "getRowCount: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return 0
      finally:
         db.close()
      return size

   # Deletes numRows oldest rows from a given table.
   def deleteOldTableRows( self, tableName, numRows ):
      t8( "deleteOldTableRows", tableName, numRows )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return False

      if ( not tableName ) or ( tableName not in self.tableSchema ):
         t0( "table", tableName, "not found" )
         return False

      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            sql = f"delete from {tableName} where rowId in ( " + \
               f"select rowId from {tableName} order by rowId asc limit {numRows} )"
            db.execute( sql )
            t8( sql )
            self.rowCount[ tableName ] = \
               max( self.rowCount[ tableName ] - numRows, 0 )
      except sqlite3.Error as e:
         t0( "deleteOldTableRows: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return False
      finally:
         db.close()

      return True

   def setRowsUntilMaybeDelete( self, tableName, tableSize ):
      tableSizeInfo = self.tableSizeInfo[ tableName ]

      # Find memory left to write until deletion threshold is hit.
      memoryUntilDeletion = tableSizeInfo.totalSize - tableSize

      # Divide the memory left by the row size estimate.
      tableSizeInfo.rowsUntilMaybeDelete = math.ceil( memoryUntilDeletion
         / tableSizeInfo.rowSizeEstimate )

      # If rowsUntilMaybeDelete is very small, clamp it to MAYBE_DELETE_DELAY to
      # avoid too many subsequent checks.
      tableSizeInfo.rowsUntilMaybeDelete = \
         max( tableSizeInfo.rowsUntilMaybeDelete, MAYBE_DELETE_DELAY )


   # Deletes oldest rows from a table if it has grown beyond its maxSize limit.
   def maybeDeleteRows( self, tableName ):
      t8( "maybeDeleteRows", tableName )
      tableSizeInfo = self.tableSizeInfo[ tableName ]
      numRows = self.rowCount[ tableName ]

      # Haven't hit the threshold to check for delete yet.
      if tableSizeInfo.rowsUntilMaybeDelete + tableSizeInfo.lastRowCount > numRows:
         return False

      # No deletion if maxSize is unbounded, i.e. == 0, or if there are no rows.
      if tableSizeInfo.maxSize == 0 or numRows == 0:
         return False

      tableSize = None
      # Amortize the cost of getTableSize by only calling it every time half of the
      # rows in the table are replaced.
      if tableSizeInfo.rowSizeEstimate == 0 or tableSizeInfo.rowsSinceTableSize * \
         tableSizeInfo.rowSizeEstimate > tableSizeInfo.writeBetweenQuery:
         tableSize = self.getTableSize( tableName )
         tableSizeInfo.rowsSinceTableSize = 0

         # Get average row size and store it in rowSizeEstimate.
         tableSizeInfo.rowSizeEstimate = math.ceil( tableSize / numRows )
      else:
         tableSize = self.rowCount[ tableName ] * tableSizeInfo.rowSizeEstimate
         tableSizeInfo.rowsSinceTableSize += self.rowCount[ tableName ] - \
            tableSizeInfo.lastRowCount

      if tableSize is None or tableSizeInfo.totalSize > tableSize:
         tableSizeInfo.lastRowCount = numRows
         self.setRowsUntilMaybeDelete( tableName, tableSize )
         return False

      self.tableFull[ tableName ] = True

      # Compute how many rows over the totalSize were added beyond padding.
      numRowsOverage = math.floor( numRows *
         ( tableSize - tableSizeInfo.totalSize ) / tableSize )

      # Compute how many rows over the maxSize were added as padding.
      numRowsPadding = math.floor( ( numRows - numRowsOverage ) *
                                   tableSizeInfo.percentageToDel )

      self.deleteOldTableRows( tableName, numRowsOverage + numRowsPadding )

      tableSizeInfo.lastRowCount = self.rowCount[ tableName ]
      tableSizeEstimate = tableSizeInfo.rowSizeEstimate * self.rowCount[ tableName ]
      self.setRowsUntilMaybeDelete( tableName, tableSizeEstimate )
      return True

   def syncQTToDb( self, qtFileName, tableName ):
      t8( "syncQTToDb", qtFileName, "to", tableName )
      if not self.initialized:
         t1( "eventMon.db not initialized..attempting flush" )
         # attempt creating db
         self.flushDb()
         if not self.initialized:
            t0( "eventMon.db recreation failed. not syncing" )
            return False
         else:
            t1( "eventMon.db recreate successful..attempting sync" )

      if ( not tableName ) or ( tableName not in self.tableSchema ):
         t0( "table", tableName, "not found" )
         return True

      db = sqlite3.connect( self.dbLocation )
      rowsWritten = 0
      try:
         with db:
            fileIter = QTFileIterator( qtFileName )
            if not fileIter:
               t0( "no qt records found in ", qtFileName )
               return True
            for timeStamp, evtMsg in fileIter:
               # evtMsg will typically look as below:
               # abc,Ethernet/1,[foo,bar],,,removed,12
               evtMsg = re.split( r',\s*(?![^[]*\])', evtMsg )
               if tableName in ( "nat", "sfe" ): # unpack boolArr
                  evtMsg = self.processNatMsg( evtMsg )
                  t8( "evtMsg: ", evtMsg )
               attrTuple = ( timeStamp, ) + tuple(
                  str.replace( "[", "" ).replace( "]", "" ) for str in evtMsg )
               numFields = len( self.tableSchema[ tableName ] )
               if len( attrTuple ) != numFields:
                  t0( "len(attrTuple) is", len( attrTuple ), "expected", numFields )
                  continue
               sql = ( "insert into %s values ( %s )" %
                     ( tableName, ",".join( "?" * numFields ) ) )
               try:
                  db.execute( sql, attrTuple )
                  rowsWritten += 1
                  t9( sql, attrTuple )
               except sqlite3.IntegrityError as e:
                  t1( "syncQtToDb: IntegrityError exception ", e )
      except sqlite3.Error as e:
         t0( "syncQtToDb: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return False
      finally:
         # Update table row count with number of rows successfully written.
         self.rowCount[ tableName ] += rowsWritten
         db.close()
         self.maybeDeleteRows( tableName )
      return True

   def removeFromDb( self, table, fileName ):
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return

      db = sqlite3.connect( self.dbLocation )
      fileIter = QTFileIterator( fileName )
      if not fileIter:
         t0( "no qt records found in ", fileName )
         db.close()
         return

      msgs = [ msg for _, msg in fileIter ]
      if not msgs:
         db.close()
         return
      splitLastMsg = msgs[ -1 ].split( ',' )
      if not splitLastMsg:
         db.close()
         return
      counter = splitLastMsg[ -1 ]
      sql = ( f"delete from {table} where counter <= {counter}" )
      t8( sql )
      try:
         with db:
            before = Tac.now()
            t8( 'Deleting from table, now = ', before, 'seconds' )
            db.execute( sql )
            delta = Tac.now() - before
            t8( 'Delete from table took ', delta, 'seconds' )
      except sqlite3.Error as e:
         t0( "removeFromDb exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

# This class manages the the sync of the qt files to the sqlite db and
# buffer flush. Both the events are triggered from Cli
class EventMonEpochReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::EpochMarker"
   MAX_SYNCS_PER_ROUND = 1

   def __init__( self, epochMarker, agent ):
      self.agent_ = agent
      Tac.Notifiee.__init__( self, epochMarker )
      self.timer_ = Tac.ClockNotifiee( self.syncFiles, timeMin=Tac.endOfTime )
      self.tableChangeList = []
      for table in self.agent_.config.table:
         self.agent_.syncedFile[ table ] = {}

      # create qt directory if it doesn't exist
      qtDir = self.agent_.config.qtDir
      if not os.path.exists( qtDir ):
         os.makedirs( qtDir )
      self.handleSyncEpoch()

   def foreverDirQueue( self, eventMonTableId ):
      ''' queue the files in the forever dir for later synchronization '''
      t8( "foreverDirQueue", eventMonTableId )
      if not foreverLogEnabled( self.agent_.config, eventMonTableId ):
         return

      foreverFullPath = foreverLogPath( self.agent_.config, eventMonTableId )
      for fn in glob.iglob( "%s.[0-9]*" % foreverFullPath ):
         if not re.search( r'\.\d+$', fn ):
            continue

         if fn not in self.agent_.syncedFile[ eventMonTableId ]:
            self.agent_.syncedFile[ eventMonTableId ][ fn ] = False

   def eventMonTableChangeList( self, ):
      t8( "eventMonTableChangeList" )
      changeList = []
      for tableName in self.agent_.config.table:
         if( tableName in self.agent_.statusMount ) and (
               self.agent_.status.tableSyncCount[ tableName ] !=
               self.agent_.statusMount[ tableName ].count() ):
            t8( tableName, " eventCount : ",
                  self.agent_.statusMount[ tableName ].count(), " syncCount : ",
                  self.agent_.status.tableSyncCount[ tableName ] )
            changeList.append( tableName )
      return changeList

   @Tac.handler( 'bufferFlushEpoch' )
   def handleBufferFlush( self, name=None ):
      ''' remove eventmon current and forever logs '''
      self.agent_.eventMonDbMgr.flushDb()
      pathname = "{}/*{}*".format( self.agent_.config.qtDir,
                                   self.agent_.config.qtFileName )
      for fn in glob.iglob( pathname ):
         removeFile( fn )

      if self.agent_.config.foreverLogEnabled:
         pathname = "{}/*{}*".format( self.agent_.config.foreverLogDirectory,
                                      self.agent_.config.foreverLogFileName )
         for fn in glob.iglob( pathname ):
            if re.search( r'\.\d+$', fn ):
               removeFile( fn )

      for _, tableConfig in qtTableConfigs( self.agent_.config ):
         if tableConfig.foreverLogOverride and tableConfig.foreverLogEnabled:
            pathname = "{}/*{}*".format(
                   self.agent_.config.table[ tableConfig.name ].foreverLogDirectory,
                   self.agent_.config.table[ tableConfig.name ].foreverLogFileName )
            for fn in glob.iglob( pathname ):
               if re.search( r'\.\d+$', fn ):
                  removeFile( fn )
         self.agent_.syncedFile[ tableConfig.name ] = {}

      # We cleanup Forever directories here, the inotify watcher must watch new
      # directories. Hence we reinstantiate the config reactor
      self.agent_.configReactor = EventMonConfigReactor( self.agent_.config,
            self.agent_ )
      self.agent_.status.newBufferFlushEpoch = self.notifier().bufferFlushEpoch

   def syncFiles( self ):
      t8( 'syncFiles', self.agent_.syncedFile )

      syncSuccess = True
      t1( "syncing foreverLog files" )
      for tableName in self.tableChangeList:
         # files in the collection that have not yet been synced
         # take 1 at a time
         unSyncedFiles = (
              sorted( ( f for ( f, synced ) in ( self.agent_.syncedFile[
            tableName ] ).items() if not synced ),
            key=lambda x: int( x.split( '.' )[ -1 ] ) )[ :
                  EventMonEpochReactor.MAX_SYNCS_PER_ROUND ] )
         t8( 'Unsynced', unSyncedFiles )
         if unSyncedFiles:
            for fn in unSyncedFiles:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( fn, tableName )
               if syncSuccess:
                  self.agent_.syncedFile[ tableName ][ fn ] = True
               else:
                  t0( f"Failed syncing {fn} {tableName} " )
                  break

            if not syncSuccess: # pylint: disable=no-else-break
               break
            else:
               t1( "Scheduling timer for next round" )
               self.timer_.timeMin = Tac.now()
               return
         else:
            t1( "no unsynced files..disabling timer" )
            self.timer_.timeMin = Tac.endOfTime

      # sync qt files
      t1( "syncing qt files" )
      if syncSuccess:
         qtDir = self.agent_.config.qtDir
         qtFileName = self.agent_.config.qtFileName
         for tableName in self.tableChangeList:
            t8( "event count for ", tableName, " ",
                  self.agent_.statusMount[ tableName ].count() )
            qtPathName = qtDir + tableName + qtFileName
            oldQtFileList = sorted( ( int( fn.split( "." )[ -1 ] ), fn )
                                    for fn in glob.iglob( qtPathName + ".[0-9]" ) )
            for _, fn in oldQtFileList:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( fn, tableName )
               if not syncSuccess:
                  t0( f"Failed syncing {fn} {tableName} " )
                  break

            if syncSuccess:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( qtPathName,
                     tableName )
               if not syncSuccess:
                  t0( f"Failed syncing {qtPathName} {tableName} " )

      # update status counts on successful sync
      if syncSuccess:
         for tableName in self.tableChangeList:
            if tableName in self.agent_.statusMount:
               self.agent_.status.tableSyncCount[ tableName ] = \
                   self.agent_.statusMount[ tableName ].count()

      # sync epoch is updated anyways to signify the end of this sync round
      # note that the syncSuccess doesn't play a part in finishing a sync
      # epoch round, so if the error condition causing sync failure is resolved
      # user can issue another sync request to get the latest state
      self.agent_.status.newSyncEpoch = self.notifier().syncEpoch
      t8( 'done syncing' )

   @Tac.handler( 'syncEpoch' )
   def handleSyncEpoch( self, ):
      t8( "handleSyncEpoch" )
      # turn off existing timer
      t1( "turning off existing timer" )
      self.timer_.timeMin = Tac.endOfTime
      self.tableChangeList = self.eventMonTableChangeList()

      for tableName in self.tableChangeList:
         self.agent_.syncedFile[ tableName ] = {}
         self.agent_.eventMonDbMgr.flushTable( tableName )
         if foreverLogEnabled( self.agent_.config, tableName ):
            self.foreverDirQueue( tableName )
      self.syncFiles()

class ForeverCleanupReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::ForeverFileTableCounter"
   MAX_DELETES_PER_ROUND = 5

   def __init__( self, config, foreverFileTableCounter, agent ):
      Tac.Notifiee.__init__( self, foreverFileTableCounter )
      self.timer_ = Tac.ClockNotifiee( self.handleForeverCleanup,
            timeMin=Tac.endOfTime )
      self.agent_ = agent
      self.config = config
      self.deletedFileCountPerRound = 0
      self.handleForeverCleanup()

   def foreverCleanupExtraLogs( self, tableName ):
      t8( "foreverCleanupExtraLogs for ", tableName )
      pathname = "{}/{}*".format( self.config.foreverLogDirectory,
                                  tableName + self.config.foreverLogFileName )
      foreverLogFileList = [ ( int( fn.split( "." )[ -1 ] ), fn )
            for fn in glob.iglob( pathname ) if re.search( r'\.\d+$', fn ) ]
      excess = len( foreverLogFileList ) - self.config.foreverLogMaxSize.val
      if excess > 0:
         t8( "excess files for ", tableName, " is ", excess )
         self.deletedFileCountPerRound = 0
         for ( _, foreverLogName ) in sorted( foreverLogFileList )[ : excess ]:
            if( self.deletedFileCountPerRound >
                  ForeverCleanupReactor.MAX_DELETES_PER_ROUND ):
               t8( "deferring deletion ", tableName )
               self.timer_.timeMin = Tac.now()
               return False
            else:
               self.timer_.timeMin = Tac.endOfTime

            self.agent_.eventMonDbMgr.removeFromDb( tableName, foreverLogName )
            t8( "removing forever log from synced file list ", foreverLogName )
            if foreverLogName in self.agent_.syncedFile[ tableName ]:
               del self.agent_.syncedFile[ tableName ][ foreverLogName ]

            removeFile( foreverLogName )
            self.deletedFileCountPerRound += 1
      return True

   @Tac.handler( 'fileCount' )
   def handleForeverCleanup( self, tableName=None ):
      # pylint: disable-next=unidiomatic-typecheck
      if type( self.config ) == Tac.Type( "EventMon::Config" ):
         if tableName is None:
            # pylint: disable-next=redefined-argument-from-local
            for tableName, _ in qtTableConfigs( self.config ):
               if not self.foreverCleanupExtraLogs( tableName ):
                  return
         else:
            if tableName not in self.config.table:
               t8( "unknown table name %s ignoring" % tableName )
               return
            self.foreverCleanupExtraLogs( tableName )
      else:
         if tableName and tableName != self.config.name:
            t8( "unknown table name %s ignoring" % tableName )
            return
         self.foreverCleanupExtraLogs( self.config.name )

class EventMonTableConfigReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::TableConfig"

   def __init__( self, config, agent ):
      Tac.Notifiee.__init__( self, config )
      self.config = config
      self.agent_ = agent

      # Shaq specific init.
      if not self.notifier_.eventConfig.containsQt():
         return

      # Qt specific init.
      self.foreverDirWatcher = None
      self.foreverCleanupReactor = None
      self.setForeverDirWatcher()
      # Could use captureNotifyValues in C++. Not sure if GenericIf bindings support
      # it.

   @Tac.handler( 'maxTableSize' )
   def handleMaxTableSize( self ):
      dbMgr = self.agent_.eventMonDbMgr
      tableName = self.config.name
      dbMgr.tableSizeInfo[ tableName ] = \
         dbMgr.TableSizeInfo( self.notifier_.maxTableSize.val, dbMgr.pageSize )
      self.agent_.status.tableFull[ tableName ] = False
      dbMgr.maybeDeleteRows( tableName )

   @Tac.handler( 'eventConfig' )
   def handleConfig( self ):
      # Shaq eventConfigs do not need this reactor.
      if not self.notifier_.eventConfig.containsQt():
         return

      self.setForeverDirWatcher()

      # This is idempotent since there will not be excess logs if the log max size
      # did not change.
      if self.foreverCleanupReactor:
         self.handleForeverLogMaxSize()

   def handleForeverLogMaxSize( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogMaxSize {} {} ".format(
         self.config.fullName, self.config.foreverLogMaxSize ) )
      if self.foreverCleanupReactor:
         self.foreverCleanupReactor.handleForeverCleanup()

   def setForeverDirWatcher( self, ):
      t1( "EventMonTableConfigReactor: setForeverDirWatcher" )
      if self.config.foreverLogEnabled and self.config.foreverLogOverride:

         # This check makes setForeverDirWatcher idempotent when the data that
         # foreverDirWatcher depends on has not changed.
         if self.foreverCleanupReactor is not None and \
            self.foreverDirWatcher is not None and\
            self.config.foreverLogFileName == self.foreverDirWatcher.qtFileName and \
            self.config.foreverLogDirectory == self.foreverDirWatcher.path:
            return

         try:
            os.makedirs( self.config.foreverLogDirectory )
         except OSError:
            pass
         self.foreverDirWatcher = Tac.newInstance( "EventMon::ForeverDirWatcher",
            self.config.foreverLogDirectory, self.config.foreverLogFileName )
         self.foreverCleanupReactor = ForeverCleanupReactor( self.config,
              self.foreverDirWatcher.foreverFileTableCounter, self.agent_ )
      else:
         self.foreverCleanupReactor = None
         self.foreverDirWatcher = None

def shaqPath( tableName ):
   return 'eventMon/' + tableName

def shaqMountParams( tableName ):
   # TODO: Should we just shove the type name into the schema or the config?
   eventClass = tableName.capitalize()
   typeName = f'EventMon_{eventClass}::{eventClass}EventShaq'

   return ( shaqPath( tableName ), typeName,
            Tac.Type( 'TacShaq::MountInfo' )( 'reader' ) )

class EventMonConfigReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::Config"

   def __init__( self, config, agent ):
      Tac.Notifiee.__init__( self, config )
      self.config = config
      self.agent_ = agent
      self.tableConfigReactor = {}
      self.foreverCleanupReactor = None
      self.foreverDirWatcher = None
      self.handleTableConfig()
      self.setForeverDirWatcher()

   def handleTableConfig( self, ):
      for table, tableConfig in self.config.table.items():
         self.tableConfigReactor[ table ] = \
            EventMonTableConfigReactor( tableConfig, self.agent_ )

         # Qt specific tableConfig setup.
         if tableConfig.eventConfig.containsQt():
            continue

         # Shaq specific tableConfig setup.
         shmemMg = self.agent_.shmemEm.getMountGroup()
         shmemMg.doMount( *shaqMountParams( table ) )
         shmemMg.doClose()
         # TODO: Shaq is moving to async mounts. The 'right' thing to do here
         #       would be to install a reactor on the mount group's readiness, but
         #       our reactor already no-ops if the mount status isn't attached.
         #       So as long as that transition is sane, we should be fine -- the
         #       reader entity will be initialized (even if not attached)
         #       synchronously, meaning the pipe and control references will be
         #       valid to pass to our SMs. But! If we ever access any pipe or
         #       control APIs outside of the reactor, we will need to do the right
         #       thing and wait for MountGroup::ready!
         e = self.agent_.shmemEm.getEntity[ shaqPath( table ) ]
         self.agent_.shaqEntity[ table ] = e
         self.agent_.shaqSm[ table ] =\
         EventMonShaqPipeReactor( e, self.agent_.eventMonDbMgr,
                                  self.agent_.tableSchema[ table ], table )

   @Tac.handler( 'foreverLogEnabled' )
   def handleForeverLogEnabled( self, ):
      t8( "EventMonConfigReactor: handleForeverLogEnabled {} {} ".format(
         self.config.fullName, self.config.foreverLogEnabled ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogFileName' )
   def handleForeverLogFileName( self, ):
      t8( "EventMonConfigReactor: handleForeverLogFileName {} {}".format(
         self.config.fullName, self.config.foreverLogFileName ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogDirectory' )
   def handleForeverLogDir( self, ):
      t8( "EventMonConfigReactor: handleForeverLogDir {} {}".format(
         self.config.fullName, self.config.foreverLogDirectory ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogMaxSize' )
   def handleForeverLogMaxSize( self, ):
      t8( "EventMonConfigReactor: handleForeverLogMaxSize {} {} ".format(
         self.config.fullName, self.config.foreverLogMaxSize ) )
      if self.foreverCleanupReactor:
         self.foreverCleanupReactor.handleForeverCleanup()

   def setForeverDirWatcher( self, ):
      if self.config.foreverLogEnabled:
         try:
            os.makedirs( self.config.foreverLogDirectory )
         except OSError:
            pass

         self.foreverDirWatcher = Tac.newInstance( "EventMon::ForeverDirWatcher",
            self.config.foreverLogDirectory, self.config.foreverLogFileName )
         self.foreverCleanupReactor = ForeverCleanupReactor( self.config,
              self.foreverDirWatcher.foreverFileTableCounter, self.agent_ )
      else:
         self.foreverCleanupReactor = None
         self.foreverDirWatcher = None

class EventMonShaqPipeReactor( Tac.Notifiee ):
   notifierTypeName = "TacShaq::ShaqControl"

   def __init__( self, entity, dbMgr, schema, tableName ):
      Tac.Notifiee.__init__( self, entity.pipeControl )
      self.dbLocation = dbMgr.dbLocation
      self.pipe = entity.pipe
      self.tableName = tableName
      self.dbMgr = dbMgr
      self.sqlInsert = ( "insert into %s values ( %s )" %
                       ( tableName, ",".join( "?" * len( schema ) ) ) )
      self.drainTask = Task( f'{self.tableName}ShaqPipeDrain', self.handleTaskRun )

      # Some types, like Tac::Time, need to be explicitly converted before sqlite can
      # bind them. Let's explicitly convert them all, ignoring qualifiers like
      # UNIQUE.
      def _sqlConversion( sqlType ):
         if 'float' in sqlType:
            return float
         if 'integer' in sqlType:
            return int
         if 'text' in sqlType:
            return str
         if 'boolean' in sqlType:
            return bool
         assert False, f'unsupported sqlType:{sqlType}'
      self.schema = tuple( ( attrName, _sqlConversion( attrType ) )
                           for attrName, attrType, _ in schema )

   @Tac.handler( 'readyTrigger' )
   def handleTrigger( self ):
      if self.notifier_.mountStatus == 'none' or self.notifier_.readyTrigger == 0:
         return

      t8( f'Scheduling pipe sync for {self.tableName}; ' +
          f'triggers:{self.notifier_.readyTrigger}' )
      self.drainTask.schedule()

   def handleTaskRun( self, shouldYield ):
      t8( f'{len( self.pipe )} values to sync' )
      rowsWritten = 0
      try:
         db = sqlite3.connect( self.dbLocation )
         with db:
            while len( self.pipe ) > 0:
               # TODO: Should maybe only peek here and pop if and only if the sql
               #       insert succeeds.
               event = self.pipe.deq()
               # Using the schema here instead of event.attributes, to guarantee
               # ordering and to allow for the possibility that clients may add attrs
               # we don't log.
               values = tuple( conversion( event.getRawAttribute( attrName ) )
                               for attrName, conversion in self.schema )
               db.execute( self.sqlInsert, values )
               rowsWritten += 1
               t8( self.sqlInsert, values )

               if shouldYield():
                  break

      except sqlite3.Error as e:
         t0( 'handleTrigger: exception ", e' )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         # TODO: Don't swallow these for now.
         #       Might consider switching to peek instead of pop so we can try again
         #       instead of missing events.
         raise
      finally:
         # Update number of table rows with the number of successful writes.
         self.dbMgr.rowCount[ self.tableName ] += rowsWritten
         db.close()

      # TODO: ensure this isn't too costly to call on every update. If it is, we will
      # need some logic to only call this occasionally. This logic will likely need
      # to be dynamic. Calling every N updates is not sufficient.
      #
      # TODO: Should maybeDeleteRows get its own task? We can schedule it here.
      self.dbMgr.maybeDeleteRows( self.tableName )

      return TaskRunResult( len( self.pipe ) > 0, rowsWritten )

class EventMon( Agent.Agent ):
   def __init__( self, entityManager, blocking=False ):
      Agent.Agent.__init__( self, entityManager )
      self.tableSchema = {}
      self.statusMount = {}
      self.statusReactor = {}
      self.epochReactor = None
      self.configReactor = None
      self.syncedFile = {}
      self.eventMonDbMgr = None
      self.shaqEntity = {}
      self.shaqSm = {}
      self.shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

      class Context:
         def __init__( self, agent ):
            self.registerDb = agent.registerDb

      ctx = Context( self )
      Plugins.loadPlugins( 'EventMonPlugin', ctx, None, None )

      def _finish():
         t1( "mounts finished" )
         # Populate status tables with default values.
         for tableName in self.tableSchema:
            # add new table status
            self.status.tableSyncCount[ tableName ] = 0
            self.status.rowCount[ tableName ] = 0
            self.status.tableFull[ tableName ] = False

         self.eventMonDbMgr = EventMonDbMgr( self.config.dbLocation,
               self.tableSchema, self.config.table, self.status.rowCount,
               self.status.tableFull )

         # Register reactors.
         self.epochReactor = EventMonEpochReactor( self.epochMarker, self )
         self.configReactor = EventMonConfigReactor( self.config, self )

      # do the initial mounts
      mg = entityManager.mountGroup()

      for tableName in self.tableSchema:
         self.statusMount[ tableName ] = mg.mountPath(
            "eventMon/%s/status" % tableName )

      self.config = mg.mountPath( "eventMon/config" )
      self.status = mg.mountPath( "eventMon/status" )
      self.epochMarker = mg.mountPath( "eventMon/epochMarker" )

      if blocking:
         mg.close( blocking=True )
         _finish()
      else:
         mg.close( _finish )

   def registerDb( self, tableSchema ):
      name = tableSchema.name()
      t0( 'Register event monitor for', name )
      schema = tableSchema.schema()
      self.tableSchema[ name ] = schema

def main():
   # Enable level 0 tracing for EventMonAgent by default
   th.facility.levelEnabled[ 0 ] = True
   container = Agent.AgentContainer( [ EventMon ] )
   container.runAgents()
