# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import os
import pwd
import signal
import sys
import tty

import BasicCli
import BasicCliSession
import Cli
import CliArgParser
import EntityManager
import Tac
import TacSigint
import TerminalUtil

GLOBAL_CLI = None

def startActivityThread():
   # The main thread (the cli readline thread) is responsible for
   # appropriately handling SIGINT and SIGTSTP signals. When a signal
   # is received, the thread that receives the signal is random. If
   # the run activities thread receives the signal, it will use the
   # signal handler installed by the other thread, which longjump into the
   # context of the other thread's execution.
   # We must therefore ensure signals are only handled by the appropriate
   # thread. We block all signals here, before spawning off multiple threads,
   # so that all threads will start with these signals blocked. If a thread
   # wishes to handle a specific signal, it must first unblock the signal.
   # All the signals below are handled by libedit. We do not want the other
   # threads to receive them.

   signals = ( signal.SIGINT, signal.SIGTSTP, signal.SIGQUIT, signal.SIGHUP,
               signal.SIGTERM, signal.SIGCONT, signal.SIGWINCH )
   for s in signals:
      Tac.threadsigmask( s, True )

   Tac.activityThread().start()
   # The following waitFor is a temporary fix for BUG12195.  Really, the call
   # to activityThread().start() should block until the activityThread()
   # sets inExecTime to something.
   Tac.waitFor( lambda: not Tac.activityManager.inExecTime.isZero, sleep=True )
   Tac.activityManager.enableSmartPtrLockCheck = True

   for s in signals:
      Tac.threadsigmask( s, False )

def createEntityManager( options ):
   # Only print extra information if we are in interactive mode
   verbose = not options.input_file and not options.command
   if options.standalone:
      return EntityManager.Local( sysname=options.sysname )
   else:
      return EntityManager.Sysdb( sysname=options.sysname,
                                 sysdbsockname=options.sysdbsockname,
                                 waitForSysdbToInitialize=not options.startup_config,
                                 mountProfileAgentString='ConfigAgent',
                                 verbose=verbose )

def createSession( options, entityManager, cli ):
   aaaAuthnId = os.environ.get( 'AAA_AUTHN_ID' )
   aaaAuthnId = int( aaaAuthnId ) if aaaAuthnId else None

   try:
      user = pwd.getpwuid( os.getuid() ).pw_name
   except KeyError:
      user = None

   aaaUser = BasicCliSession.AaaUser( user, aaaAuthnId,
                                      os.getuid(), os.getgid() )
   autoComplete = not options.disable_autocomplete
   if options.standalone:
      # standalone cli sessions, by default, disable guards, unless
      # the --standalone-guards argument was also passed.
      options.disable_guards = not options.standalone_guards
   return BasicCliSession.Session( BasicCli.UnprivMode,
                                   entityManager,
                                   privLevel=options.privilege,
                                   disableAaa=options.disable_aaa,
                                   disableAutoMore=options.disable_automore,
                                   disableGuards=options.disable_guards,
                                   standalone=options.standalone,
                                   startupConfig=options.startup_config,
                                   autoComplete=autoComplete,
                                   interactive=( not options.startup_config ),
                                   cli=cli,
                                   aaaUser=aaaUser )

def _execCli( entityManager, session, options, cli ):
   termFd = TerminalUtil.getTerminalFileno()

   try:
      if termFd >= 0:
         termAttr = tty.tcgetattr( termFd )

      TacSigint.setDelayedMode()

      # Ignore SIGTSTP so that Ctrl-Z doesn't suspend the process when the CLI is
      # run manually (during testing).  Note that the default behavior of SIGTSTP
      # does nothing when the CLI is run as the shell.
      signal.signal( signal.SIGTSTP, signal.SIG_IGN )

      # We changing terminal settings. Sometimes there is a window when we are
      # not the controlling process of the terminal and we don't want to get an
      # error, so just ignore SIGTTOU (similar to bash etc).
      signal.signal( signal.SIGTTOU, signal.SIG_IGN )

      # Ignore SIGQUIT so that Ctrl-\ doesn't terminate the li/
      # that we must do this after importing the Tac library because the
      # backtrace module overwrites the SIGQUIT handler.
      signal.signal( signal.SIGQUIT, signal.SIG_IGN )

      # Add the sbin dirs to PATH.  This is necessary because Cli is the user's
      # shell and /bin/login only puts the sbin dirs in root's path.  We want to
      # avoid every single CliPlugin that runs external commands needing to
      # manipulate the path or specify an absolute path.
      extraPathDirs = ( '/usr/local/sbin', '/usr/sbin', '/sbin' )
      currPath = os.environ.get( 'PATH', '' )
      currPathDirs = currPath.split( ':' )
      for d in extraPathDirs:
         if not d in currPathDirs:
            currPathDirs.insert( 0, d )
      os.environ[ 'PATH' ] = ':'.join( currPathDirs )

      try:
         startActivityThread()
         returnCode = Cli.main( cli, entityManager, session, options )
         if returnCode != 0:
            sys.exit( returnCode )
         return returnCode
      finally:
         Tac.activityThread().stop()
         # if KeyboardInterrupt, exit gracefully, else raise the original
         # exception
         TacSigint.check()
   except KeyboardInterrupt:
      # This can happen if someone hits Ctrl-C before the CLI has finished
      # initializing.
      return errno.EINTR
   except OSError as e:
      # This can happen if Cli is not invoked interactively
      if e.errno == errno.EPIPE:
         return 141
      else:
         raise
   except SystemExit as e:
      return e.code
   finally:
      # Explicitly flush stdout and stderr before exiting, as the impending
      # os._exit() doesn't do that for us. This fixes BUG27575.
      sys.stdout.flush()
      sys.stderr.flush()
      if termFd >= 0:
         # pylint: disable-next=used-before-assignment
         tty.tcsetattr( termFd, tty.TCSAFLUSH, termAttr )
   return 0

def cliInitialized( cliInstance ):
   global GLOBAL_CLI
   GLOBAL_CLI = cliInstance

def execCli( entityManager=None, session=None ):
   Tac.setproctitle( 'Cli [interactive]' )
   options = CliArgParser.parseArgs()

   if not entityManager:
      entityManager = createEntityManager( options )

   if GLOBAL_CLI is None:
      Cli.initCli( entityManager, callback=cliInitialized, block=True,
                   runSessionManager=False,
                   plugins=options.plugins,
                   noPlugins=options.no_plugins,
                   standalone=options.standalone )
      assert GLOBAL_CLI

   if not session:
      # if we don't have a sssion that means that this is not being run as part
      # of the CliServer (since that always createsa session). Instead we are
      # being run as part of a test. In this case we might not actually have a
      # a real user, so we will just create a session and try create a aaaUser
      session = createSession( options, entityManager, GLOBAL_CLI )

   return _execCli( entityManager, session, options, GLOBAL_CLI )
