#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# Common utility functions dealing with the underlying rpm
# library. Used by ExtensionMgrLib as well as the Yum and Pypi
# specific implementation libraries in this package.

import errno
import os
import re
import rpm
import shutil
import struct
import tempfile
import yaml
from collections import defaultdict, namedtuple, OrderedDict

import EosVersionValidator
import KernelVersion
import SimpleConfigFile
import Tac
import Tracing

from ExtensionMgr import errors
from Swix import schema

# This is the most useful (quietest) form of RPM output;
# the lower integer value RPMLOG_EMERG still emits faulty
# RPM format errors.
rpm.setVerbosity( rpm.RPMLOG_CRIT )

__defaultTraceHandle__ = Tracing.Handle( 'ExtensionMgr' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3


# Current image information
ImgInfo = namedtuple( "ImgInfo", "version blessed flavor" )


# Directories for OverlayFS.
upperDirName = '/tmp/apps/upper{}{}'
overlayDirs = ( '{}', '/mnt/apps{}', upperDirName, '/tmp/apps/work{}{}' )


class UninstallError( Exception ):
   pass


def newTransactionSet( verifySignatures=False ):
   ts = rpm.TransactionSet()
   if not verifySignatures:
      flags = rpm._RPMVSF_NOSIGNATURES  # pylint: disable=protected-access
      ts.setVSFlags( flags )
   return ts


def addRpm( info, filename, header ):
   t1( "Adding rpm", header[ 'name' ], "for extension", filename )
   rpmObj = info.package.newMember( filename )
   props = {
      'packageName': 'name',
      'version': 'version',
      'release': 'release',
      'vendor': 'vendor',
      'summary': 'summary',
      'description': 'desc',
      'url': 'url',
      'license': 'license',
      'installedSize': 'size' }
   for attr, key in props.items():
      val = header[ key ]
      if val:
         setattr( rpmObj, attr, val )


def readRpmHeader( ts, path ):
   """Takes an rpm.TransactionSet and a path to an RPM file, reads the header,
   and returns an rpm.hdr object.  Raises an RpmReadError if there is a problem
   reading from the file."""
   t1( "Reading rpm header from", path )
   try:
      fd = os.open( path, os.O_RDONLY )
   except OSError as e:
      raise errors.RpmReadError( str( e ) )
   try:
      return ts.hdrFromFdno( fd )
   except rpm.error as e:
      raise errors.RpmReadError( str( e ) )
   finally:
      try:
         os.close( fd )
      except OSError:
         pass


def isRpmInstalled( ts, hdr ):
   """Takes a rpm.ts and a rpm.hdr and returns True if the exact version
   of the rpm is already installed, otherwise False."""
   mi = ts.dbMatch( 'name', hdr[ rpm.RPMTAG_NAME ] )
   # There can be 2 rpms with the same name as we do support multilib in 32bit
   assert mi.count() <= 2
   for h in mi:
      match = True
      for tag in ( rpm.RPMTAG_EPOCH, rpm.RPMTAG_VERSION, rpm.RPMTAG_RELEASE,
                   rpm.RPMTAG_ARCH ):
         if h[ tag ] == hdr[ tag ]:
            continue
         match = False
         break
      if match:
         return True
   return False


def _getImgInfo():
   """Return a tuple of SWI version and blessed status of current image"""
   swiInfo = SimpleConfigFile.SimpleConfigFileDict( "/etc/swi-version" )
   version = swiInfo.get( "SWI_VERSION" )
   blessed = swiInfo.get( "BLESSED" ) == "1"
   flavor = swiInfo.get( "SWI_FLAVOR" )
   return ImgInfo( version=version, blessed=blessed, flavor=flavor )


def _rpmsInDir( path, yml, compFilter=False, toMount=None ):
   """Given a path to a directory, returns a list of the full paths of the RPMs
   in that directory. If the compatibility filter is applied, return compatible
   RPMs only"""
   ents = os.listdir( path )
   t1( path, "contents:", ents )
   # Get the current image info
   imgInfo = _getImgInfo()
   imgVersion, imgBlessed = imgInfo.version, imgInfo.blessed
   # Get the yaml file info
   yamlVersions = yml.get( 'version' )
   yamlBlessed = yml.get( 'installOnBlessedImage' )
   yamlSwiFlavors = yml.get( 'swiFlavor' )
   # Get candidate RPMs
   # Use OrderedDict instead of set to preserve insertion order. This ensures a
   # deterministic installation order
   rpms = OrderedDict()
   allRpms = OrderedDict( ( e, None ) for e in ents if e.endswith( ".rpm" ) )
   if not compFilter or ( yamlVersions is None and yamlBlessed is None ):
      rpms = allRpms
   else:
      # If blessed status doesn't match the value, don't install
      # pylint: disable-next=singleton-comparison
      if yamlBlessed != None and ( yamlBlessed ^ imgBlessed ):
         raise errors.InstallInfoError(
                         "Blessed status doesn't match "
                         "installOnBlessedImage attribute in YAML file." )
      if yamlSwiFlavors and imgInfo.flavor not in yamlSwiFlavors:
         raise errors.InstallInfoError(
                         "Image flavor doesn't match "
                         "swiFlavor attribute list '%s' "
                         "in YAML file." % ",".join( yamlSwiFlavors ) )

      # Get compatible RPMs
      if imgVersion:
         compVersions = []
         try:
            # An example of v is { '4.20, 4.20.{1-12}': [ 'Test2.rpm' ] }
            for version in yamlVersions:
               if EosVersionValidator.validateVersion( version, imgVersion ):
                  compVersions.append( version )
               elif toMount:
                  # Remove irrelevant mountpoints.
                  toMount.pop( version, None )
         except ValueError as err:
            raise errors.InstallInfoError( "Error in YAML file: %s" % err )
         for v in compVersions:
            files = yamlVersions[ v ]
            if "all" in files: # pylint: disable=no-else-break
               rpms = allRpms
               break
            else:
               rpms.update( ( f, None ) for f in files if f.endswith( ".rpm" ) )
         if not rpms:
            raise errors.InstallInfoError(
                            "No RPMs are compatible with current EOS version." )
      else:
         rpms = allRpms
   # Get full path
   rpms_ = []
   for p in rpms:
      full = os.path.join( path, p )
      # We expect the contents to be flat
      if os.path.isfile( full ):
         rpms_.append( full )
   return rpms_

def safeEnsureStr( val ):
   """Ensure the val is a string if its not None"""
   if isinstance( val, bytes ):
      val = val.decode()
   return val

def readRpmHeaderIntoDict( ts, path ):
   """Takes an rpm.TransactionSet and a path to an RPM file and reads the
   header info into a dict, which is returned.  Raises an RpmReadError if
   there is a problem reading from the file."""
   h = readRpmHeader( ts, path )
   r = {}
   r[ 'name' ] = safeEnsureStr( h[ rpm.RPMTAG_NAME ] )
   r[ 'desc' ] = safeEnsureStr( h[ rpm.RPMTAG_DESCRIPTION ] )
   r[ 'vendor' ] = safeEnsureStr( h[ rpm.RPMTAG_VENDOR ] )
   r[ 'epoch' ] = h[ rpm.RPMTAG_EPOCH ]
   r[ 'version' ] = safeEnsureStr( h[ rpm.RPMTAG_VERSION ] )
   r[ 'release' ] = safeEnsureStr( h[ rpm.RPMTAG_RELEASE ] )
   r[ 'summary' ] = safeEnsureStr( h[ rpm.RPMTAG_SUMMARY ] )
   r[ 'url' ] = safeEnsureStr( h[ rpm.RPMTAG_URL ] )
   r[ 'size' ] = h[ rpm.RPMTAG_SIZE ]
   r[ 'license' ] = safeEnsureStr( h[ rpm.RPMTAG_LICENSE ] )

   # xxx todo get dependencies information from yum
   return r


def _addInstalledRpms( installed, info ):
   assert not info.installedPackage
   for i in installed:
      info.installedPackage.addMember( info.package[ i ] )

def _addAgentsToRestart( yml, info ):
   agents = yml.get( 'agentsToRestart', [] )
   info.affectedAgents.clear()
   for i, a in enumerate( agents ):
      info.affectedAgents[ i ] = a

def installYum( status, info, force ):
   ts = newTransactionSet()
   if force:
      setForceProblemFlags( ts )
   else:
      ts.setProbFilter( 0 )

   basepath = os.path.dirname( info.filepath ) + '/'
   paths = [ basepath + x for x in info.package ]
   try:
      installed = _installRpms( paths, ts, force )
      _addInstalledRpms( installed, info )
      installedStr =  'installed' if not force else 'forceInstalled'
      # Set the dependencies as installed
      # Iterating the entire list of infos is not ideal but it's better than
      # than messing around with finding keys because of generaitons.
      for i in status.info.values():
         if i.filename in info.package:
            i.status = installedStr
            i.statusDetail = ''

   except errors.RpmInstallError as e:
      info.statusDetail = str( e )
      raise errors.InstallError( "RPM install error: " + str( e ) )


def installRpm( status, info, force ):
   ts = newTransactionSet()
   if force:
      setForceProblemFlags( ts )
   else:
      ts.setProbFilter( 0 )

   path = info.filepath
   try:
      installed = _installRpms( [ path ], ts, force )
      assert len( installed ) == 1
      _addInstalledRpms( installed, info )
      info.status = 'installed' if not force else 'forceInstalled'
      info.statusDetail = ''
   except errors.RpmInstallError as e:
      info.statusDetail = str( e )
      raise errors.InstallError( "RPM install error: " + str( e ) )


def installArchiveDir( path, ts, info, force, yml, versionedMounts ):
   # Apply the compatibility filter so only RPMs compatible with current SWI
   # version will be returned
   try:
      compFilter = True
      rpms = _rpmsInDir( path, yml, compFilter, versionedMounts )
      installed = _installRpms( rpms, ts, force )
      _addInstalledRpms( installed, info )
      for mounts in versionedMounts.values():
         for i, mount in enumerate( mounts ):
            # Store the mountpoints, so we can later unmount them.
            # A SWIX may have overlapping or nested mountpoints,
            # so we want to remember the mounting order so we can backtrack.
            info.mountpoints[ i ] = mount.mount( i )
      info.status = 'installed' if not force else 'forceInstalled'
      info.statusDetail = ''
      _addAgentsToRestart( yml, info )
   except ( errors.RpmInstallError, errors.InstallInfoError ) as e:
      info.statusDetail = str( e )
      raise errors.InstallError( "SWIX install error: " + str( e ) )


def installSwix( status, info, force ):
   assert info.format == 'formatSwix'
   ts = newTransactionSet()
   if force:
      setForceProblemFlags( ts )
   else:
      ts.setProbFilter( 0 )

   try:
      _doInSwix( info.filepath,
                 lambda mp, yml, toMount: installArchiveDir( mp, ts, info,
                                                             force, yml, toMount ) )
   except errors.SwixReadError as e:
      info.presenceDetail = str( e )


def _readManifest( path ):
   t1( "reading manifest", path )
   f = open( path ) # pylint: disable=consider-using-with
   manifest = {}
   pat = re.compile( "(\\S+)\\s*:\\s*(\\S+)" )
   while True:
      line = f.readline()
      if line == '':
         break
      m = pat.match( line )
      if m is None:
         t0( "Error parsing line in manifest:", line )
         continue
      key, value = m.group( 1 ), m.group( 2 )
      t1( key, ":", value )
      manifest[ key ] = value
   f.close()
   return manifest


def _readArchiveDir( path, ts, yml ):
   """Pass in a path to the mountpoint where the zip archive is mounted,
   and an rpm.TransactionSet. Returns ( primaryRpm, dictionary mapping rpm name to 
   rpm header information ). Throws SwixReadError."""
   manifestPath = os.path.join( path, "manifest.txt" )
   if not os.path.exists( manifestPath ):
      t0( "No manifest found at", manifestPath )
      raise errors.SwixReadError( "Missing manifest" )
   manifest = _readManifest( manifestPath )
   primaryRpm = manifest.get( 'primaryRpm' )
   if not primaryRpm:
      t0( "No primaryRpm found in manifest", manifestPath )
      raise errors.SwixReadError( "Manifest does not specify primaryRpm" )
   rpms = _rpmsInDir( path, yml )
   failedRpms = []
   rpmHeaderInfo = {}
   for rpmfile in rpms:
      try:
         header = readRpmHeaderIntoDict( ts, rpmfile )
      except errors.RpmReadError as e:
         t0( "failed to read", rpmfile, ": ", e )
         failedRpms.append( rpmfile )
      else:
         rpmHeaderInfo[ os.path.basename( rpmfile ) ] = header
   if failedRpms: # pylint: disable=no-else-raise
      raise errors.SwixReadError( 'Failed to read rpm(s): ' +
                                  ', '.join( failedRpms ) )
   elif primaryRpm not in rpmHeaderInfo:
      t0( "primaryRpm", primaryRpm, "not found in archive" )
      raise errors.SwixReadError( "primaryRpm not found in archive" )
   return ( primaryRpm, rpmHeaderInfo )


def _writeArchiveDirInfo( path, ts, info, yml ):
   """Pass in a path to the mountpoint where the zip archive is mounted,
   an rpm.TransactionSet, and the Extension::Info to populate.  Doesn't
   return anything."""
   try:
      primaryRpm, rpmHeaderInfo = _readArchiveDir( path, ts, yml )
   except errors.SwixReadError as e:
      info.presence = 'absent'
      info.presenceDetail = str( e )
   else:
      info.presence = 'present'
      info.presenceDetail = ''
      info.primaryPkg = primaryRpm
      for rpmName, header in rpmHeaderInfo.items():
         addRpm( info, rpmName, header )
      _addAgentsToRestart( yml, info )

def _unmountSwix( mountpoint ):
   try:
      shutil.rmtree( mountpoint )
   except OSError as e:
      t0( "Error removing mountpoint dir:", e )

class Mount:
   '''Objects to be mounted instead of extracted/installed from a SWIX file.
   '''
   # Zipfile constants.
   HEADER_SIZE = 30
   HEADER_STRUCT_FMT = '<4s2B4H3l2H'
   FILENAME_FIELD_LENGTH = 10
   EXTRA_FIELD_LENGTH = 11

   def __init__( self, zf, filename, where ):
      filename = filename.strip()
      where = where.strip()
      try:
         info = zf.getinfo( filename )
      except KeyError:
         error = "Could not find %r in SWIX file." % filename
         raise errors.InstallInfoError( error ) # pylint: disable=raise-missing-from

      zf.fp.seek( info.header_offset )
      headerData = zf.fp.read( self.HEADER_SIZE )
      headerData = struct.unpack( self.HEADER_STRUCT_FMT, headerData )
      self.params = {
            'offset': ( info.header_offset
                      + self.HEADER_SIZE
                      + headerData[ self.FILENAME_FIELD_LENGTH ]
                      + headerData[ self.EXTRA_FIELD_LENGTH ] ),
            'size': info.file_size,
            'file': zf.fp.name,
            'where': where,
      }

   def _makeDirs( self, *dirs ):
      for d in dirs:
         try:
            os.makedirs( d )
         except OSError as e:
            if e.errno != errno.EEXIST:
               self._safeRemoveDirs( dirs )
               raise e

   def _safeRemoveDirs( self, dirs ):
      for d in dirs:
         try:
            os.rmdir( d )
         except OSError:
            pass

   def _mountLowerDir( self, lowerDir ):
      cmd = 'mount -o loop,offset={offset},sizelimit={size} {file} {lower}'
      cmd = cmd.format( lower=lowerDir, **self.params ).split()
      maxTries = 3
      while maxTries:
         try:
            Tac.run( cmd, asRoot=True, stderr=Tac.CAPTURE )
            break
         except Tac.SystemCommandError as e:
            maxTries -= 1
            if 'Resource temporarily unavailable' not in e.output or not maxTries:
               raise errors.InstallError( e.output )

   def _mountOverlay( self, lowerDir, upperDir, workDir, where ):
      # Need to check kernel xino=off support until we stop building on
      # AroraKernel4.9 (see BUG652321)
      xinoOpt = ''
      if KernelVersion.supports( KernelVersion.OVL_XINO_OPT ):
         xinoOpt = ',xino=off'

      cmd = ( 'mount -t overlay overlay '
              '-o lowerdir={},upperdir={},workdir={}{} {}' )
      cmd = cmd.format( lowerDir, upperDir, workDir, xinoOpt, where ).split()
      try:
         Tac.run( cmd, asRoot=True, stderr=Tac.CAPTURE )
      except Tac.SystemCommandError as e:
         Tac.run( [ 'umount', lowerDir ], asRoot=True, stderr=Tac.CAPTURE )
         raise errors.InstallError( e.output )

   def mount( self, index ):
      where = self.params[ 'where' ]
      where, lowerDir, upperDir, workDir = ( d.format( where, index )
                                             for d in overlayDirs )
      self._makeDirs( where, lowerDir, upperDir, workDir )
      self._mountLowerDir( lowerDir )
      self._mountOverlay( lowerDir, upperDir, workDir, where )
      return where

   def __repr__( self ):
      return 'Mount( %s )' % self.params

def _parseMounts( zf, yml ):
   ''' Given a zipfile and its manifest.yaml, process any mountpoints.
   The `Mount` class computes the necessary mount parameters.
   '''
   result = defaultdict( list )
   versions = yml.get( 'version', {} )
   for version, files in versions.items():
      for f, instructions in files.items():
         for instruction in instructions:
            if instruction == 'mount':
               mountpoint = instructions[ instruction ]
               result[ version ].append( Mount( zf, f, mountpoint ) )

   return result

def _extractYaml( zf ):
   '''Extract a single yaml file in the root of a zipfile (SWIX).
   Return an empty dictionary if a YAML file wasn't found.
   '''
   for f in zf.namelist():
      if f.endswith( '.yaml' ):
         try:
            yml = yaml.safe_load( zf.read( f ) )
            schema.checkInfoSchema( yml )
            return yml
         except ( yaml.YAMLError, ValueError ) as e:
            raise errors.InstallInfoError( f"Error reading from {f}: {e}" )
   return {}

def _simplifyYaml( yml ):
   '''
   It will do you good not to try and read this code.
   Simplify the 'manifest.yaml' so that it is easier to work with.
   Sample result:
   { 'metadataVersion': 1.0,
     'version': { '{4.23.1}': { 'file1.squashfs': { 'mount': '/tmp/foo' },
                                'bar.rpm': {} },
                  '{4.20.7}': { 'all': {} }
                }
   }
   '''
   # pylint: disable=too-many-nested-blocks
   result = {}
   for option, value in yml.items():
      if option == 'version':
         result.setdefault( option, {} )
         for versions in value: # 'versions' list.
            for version, files in versions.items():
               # Use OD here because YAML had a list and we'd lose order.
               result[ option ].setdefault( version, OrderedDict() )
               for entry in files:
                  if isinstance( entry, dict ):
                     for f, instructionsList in entry.items():
                        result[ option ][ version ].setdefault( f, {} )
                        for instructions in instructionsList:
                           for instruction, data in instructions.items():
                              result[ option ][ version ][ f ][ instruction ] = data
                  else: # `entry` is simple string - no instructions.
                     result[ option ][ version ][ entry ] = {}
      else:
         result[ option ] = value

   return result

def _parseYaml( zf ):
   '''Try to extract and simplify a YAML file from the zipfile (SWIX).
   '''
   yml = _extractYaml( zf )
   if yml:
      return _simplifyYaml( yml )
   return yml

def _doInSwix( path, func ):
   """Takes a swix path and a callable, mounts the swix, and
   invokes the callable, passing the mountpoint as a parameter.
   Returns None if the mount fails, otherwise returns whatever func
   returns."""
   # This import must be delayed until here, not done at module
   # level, due to zipfile being blacklisted in CliTests.py.
   import zipfile # pylint: disable=import-outside-toplevel
   mountpoint = None
   try:
      mountpoint = tempfile.mkdtemp( prefix='ExtensionMgr-', dir="/tmp" )
      with zipfile.ZipFile( path ) as zf:
         yml = _parseYaml( zf )
         toMount = _parseMounts( zf, yml )
         # Separate files; we don't extract .squashfs files.
         toExtract = ( f for f in zf.namelist() if not f.endswith( '.squashfs' ) )
         zf.extractall( mountpoint, members=toExtract )
   except zipfile.BadZipfile as e:
      t0( "Bad zip file", path, ":", e )
      raise errors.SwixReadError( "Bad extension file" )
   except OSError as e:
      t0( "Error mounting zip file", path, ":", e )
      raise errors.SwixReadError( "Failed to unzip file. %s" % e )
   else:
      return func( mountpoint, yml, toMount )
   finally:
      if mountpoint:
         _unmountSwix( mountpoint )


def readSwix( path, ts ):
   """Reads a swix for info and returns ( primaryRpm name, dictionary of rpm info ).
   Throws SwixReadError."""
   return _doInSwix( path, lambda mp, yml, _: _readArchiveDir( mp, ts, yml ) )


def saveSwixInfo( info, ts ):
   assert info.format == 'formatSwix'
   try:
      _doInSwix( info.filepath,
                 lambda mp, yml, _: _writeArchiveDirInfo( mp, ts, info, yml ) )
   except errors.SwixReadError as e:
      info.presenceDetail = str( e )


def _tsProblemsAsString( ts ):
   problems = ts.problems()
   errorDetails = []
   for p in problems:
      errorDetails.append( str( p ) )
   if errorDetails:
      reason = "\n".join( errorDetails )
   else:
      reason = "no details provided"
   return reason


def _runTransactionSet( transactionSet, rpms, excType, force=False ):
   """Actually runs the given transaction set, and handle errors.

   We land here either when installing or uninstalling one or more RPMs.

   Args:
     - transactionSet: The transaction set to execute to install/uninstall RPMs.
     - rpms: Sequence of paths to RPMs we're in the process of installing.
       Not used in the uninstall code path (in which case this should be None).
     - excType: The class of exception to raise when we fail (because we want
       to raise different kinds of exceptions for installation failures vs
       uninstallation failures).  Must be one of:
         o RpmInstallError
         o RpmUninstallError
     - force: If True, force the rollback of the RPMs.  Only used during install.
   """
   opened = {} # fd objects keyed by rpm name
   errs = []
   def runCb( reason, amount, total, key, client_data ):
      t2( "transaction callback: key=", key, "amount=", amount,
          "total=", total, "reason=", reason )
      if reason == rpm.RPMCALLBACK_INST_OPEN_FILE:
         t2( "Opening file descriptor for", key )
         fd = os.open( key, os.O_RDONLY )
         opened[ key ] = fd
         return fd
      elif reason == rpm.RPMCALLBACK_INST_CLOSE_FILE:
         t2( "Closing file descriptor for", key )
         fd = opened.pop( key )
         os.close( fd )
      elif reason == rpm.RPMCALLBACK_SCRIPT_ERROR:
         # For script error, the error is only lethal if "total" is true.
         if total:
            t2( "Fatal script error for", key )
            errs.append( "A script error occurred when installing %s"
                           % os.path.basename( key ) )
         else:
            t2( "Non-fatal script error for", key, "continuing" )
      elif reason == rpm.RPMCALLBACK_CPIO_ERROR:
         t2( "cpio error for", key )
         errs.append( "A cpio error occurred when installing %s"
                        % os.path.basename( key ) )
      elif reason == rpm.RPMCALLBACK_UNPACK_ERROR:
         t2( "unpack error for", key )
         errs.append( "An unpack error occurred when installing %s"
                        % os.path.basename( key ) )
      return None

   result = transactionSet.run( runCb, 1 )
   t0( "ts.run returned", result )
   assert not opened, f"Left files opened: {opened}"

   if result:
      reason = _tsProblemsAsString( transactionSet )
      msg = "Transaction failed: %s" % reason
      raise excType( msg )

   if errs:
      toRollback = []
      if excType is errors.RpmInstallError:
         # If we get here, at least one RPM encountered a fatal error while
         # getting installed.  Unforuntately part of the RPMs in the
         # transactionSet have been installed, so we need to roll them back
         # manually now.
         for r in rpms:
            try:
               h = readRpmHeader( transactionSet, r )
            except errors.RpmReadError:
               continue
            if isRpmInstalled( transactionSet, h ):
               toRollback.append( h.name )
      try:
         if toRollback:
            t1( "Rolling back", toRollback )
            uninstallRpms( toRollback, force=force )
      finally:
         msg = "Transaction failed: %s" % ", ".join( errs )
         raise excType( msg )


def _installRpms( rpms, transactionSet=None, force=False ):
   """Pass me a list of full paths to RPM package files that I will install in
   the rpm.ts that you pass me.  If you don't pass me a rpm.ts I will create a
   new rpm.ts and use it."""
   if not rpms:
      raise errors.RpmInstallError( "No packages to install!" )
   if transactionSet is None:
      transactionSet = newTransactionSet()
   toInstall = []
   for r in rpms:
      try:
         h = readRpmHeader( transactionSet, r )
      except errors.RpmReadError as e:
         msg = "Error reading rpm: " + str( e )
         raise errors.RpmInstallError( msg )
      if isRpmInstalled( transactionSet, h ):
         t1( "Skipping rpm because it's already installed:", r )
         continue
      t1( "Adding rpm for upgrade/install:", r )
      transactionSet.addInstall( h, r, 'u' )
      toInstall.append( os.path.basename( r ) )
   if not toInstall:
      if len( rpms ) == 1:
         msg = "The %s package is already installed" % ( rpms[ 0 ] )
      else:
         msg = "All packages in the extension are already installed"
      raise errors.RpmInstallError( msg )
   for te in transactionSet:
      t2( te.NEVR(), "installs these files:" )
      for fi in te.FI( 'Basenames' ):
         t2( " ", fi )
   def checkCb( ts, tagN, N, EVR, flags ):
      # Since we don't attempt to find missing rpms and add them to the
      # transaction there isn't much to do in this function.
      t3( "transaction check callback: tagN=%s N=%s EVR=%s flags=%s" %(
             tagN, N, EVR, flags ) )
   if not force:
      unresolved = transactionSet.check( checkCb )
      if unresolved:
         # Instead of rummaging through unresolved, get the problem descriptions
         # in a nice human-readable format.
         reason = _tsProblemsAsString( transactionSet )
         msg = "Transaction check failed: " + reason
         t1( msg )
         raise errors.RpmInstallError( msg, unresolved )
      transactionSet.order()

   _runTransactionSet( transactionSet, rpms, errors.RpmInstallError, force=force )
   # At this point the packages in toInstall have been installed
   return toInstall


def uninstallRpms( packageNames, transactionSet=None, force=False ):
   """Pass me a list of RPM package names that I will uninstall in the rpm
   transaction set that you pass me.  If you don't pass me a transaction set I
   will create a one and use it."""
   if not packageNames:
      raise errors.RpmUninstallError( "No packages to uninstall!" )
   if transactionSet is None:
      transactionSet = newTransactionSet()
   for p in packageNames:
      if transactionSet.dbMatch( rpm.RPMDBI_LABEL, p ):
         t1( "Adding rpm for uninstall:", p )
         transactionSet.addErase( p )
   for te in transactionSet:
      t2( te.NEVR(), "removes these files:" )
      for fi in te.FI( 'Basenames' ):
         t2( " ", fi )
   def checkCb( ts, tagN, N, EVR, flags ):
      # Since we don't attempt to add dependent rpms to the transaction
      # there isn't much to do in this function.
      t3( "transaction check callback: tagN=%s N=%s EVR=%s flags=%s" %(
             tagN, N, EVR, flags ) )
   if not force:
      unresolved = transactionSet.check( checkCb )
      if unresolved:
         # Instead of rummaging through unresolved, get the problem descriptions
         # in a nice human-readable format.
         reason = _tsProblemsAsString( transactionSet )
         msg = "Transaction check failed: " + reason
         t1( msg )
         raise errors.RpmUninstallError( msg, unresolved )
      transactionSet.order()
   _runTransactionSet( transactionSet, None, errors.RpmUninstallError )


def setForceProblemFlags( ts ):
   problemFilter = ( rpm.RPMPROB_FILTER_REPLACENEWFILES
      | rpm.RPMPROB_FILTER_REPLACEOLDFILES
      | rpm.RPMPROB_FILTER_REPLACEPKG
      | rpm.RPMPROB_FILTER_OLDPACKAGE )
   ts.setProbFilter( problemFilter )
