#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import tempfile
import threading

import Tac
import Tracing
import ExtensionMgr.errors as errors # pylint: disable=consider-using-from-import
import ExtensionMgr.rpmutil as rpmutil # pylint: disable=consider-using-from-import
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'ExtensionMgrYum' )
t3 = Tracing.trace3

Presence = TacLazyType( 'Extension::Info::Presence' )

# This module provides the package manager interface implementation
# for Yum repositories/packages.

YUM_REPO_TEMPLATE = """
[{name}]
name={description}
baseurl={url}
skip_if_unavailable=True
enabled=1
"""

YUM_REPO_DIR = '/etc/yum.repos.d'
RPM_FINAL_DEST = '/mnt/flash/.extensions/'
_name_prefix = 'eos-'

RE_YUM_PKG_NOT_AVAIL = re.compile( r'\nNo package (.+?) available\.' )

threadLock = threading.Lock()

def _repo_fn( repo ):
   return os.path.join( YUM_REPO_DIR, f'{_name_prefix}{repo.name}.repo' )

def updateRepository( repo, timeout=60 ):
   t3( 'update repo', repo )
   reponame = repo.name.replace( ' ', '_' )
   contents = YUM_REPO_TEMPLATE.format(
      name=reponame,
      description=repo.description or reponame,
      url=repo.url )
   with tempfile.NamedTemporaryFile( mode="w" ) as f:
      f.write( contents )
      f.flush()

      try:
         cmd = [ 'cp', '-f', f.name, _repo_fn( repo ) ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         cmd = [ 'chmod', 'og+r', _repo_fn( repo ) ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      except Tac.SystemCommandError as e:
         # pylint: disable-next=consider-using-f-string
         t3( 'Failed to update repository filename %s - output: %s'
             % ( _repo_fn( repo ), e.output ) )
         # pylint: disable-next=consider-using-f-string
         raise errors.Error( 'Failed to update repository %s' % repo.name )

def deleteRepository( repo ):
   t3( 'delete repo', repo )

   fn = _repo_fn( repo )
   try:
      cmd = [ 'rm', '-f', fn ]
      Tac.run( cmd, asRoot=True )
   except Tac.SystemCommandError as e:
      t3( 'failed to delete repo file', fn, 'error', str( e ) )

def dnf( *cmd ):
   '''
   A wrapper for DNF commands.
   It runs `'dnf ' + ' '.join( cmd )`, ignores stderr, and returns output.
   '''
   # pkgdeps: rpm dnf
   command = [ 'dnf' ]
   command.extend( cmd )
   return Tac.run( command, ignoreReturnCode=True,
                   stdout=Tac.CAPTURE, stderr=Tac.DISCARD )

def packageSearch( query, repo ):
   '''
   Search DNF for packages from `repo` respository that match the name `query`.
   Returns a list of Extension::Info.
   '''
   # Output for `dnf search mysql` looks like this:
   # =============================== Name & Summary Matched: mysql ===============...
   # python2-mysqlclient.x86_64 : MySQL/mariaDB database connector for Python
   # python3-mysqlclient.x86_64 : MySQL/mariaDB database connector for Python
   # ================================== Summary Matched: mysql ===================...
   # VmDut-vmdbmanager.x86_64 : Module to track VM hosts in mysql database
   # mytest.noarch : Helpers for writing tests involving mysql
   #
   # We are only interested in the name matches. The splits work ok on '' too.
   lines = dnf( 'search', query )
   lines = lines.partition( 'Name & Summary Matched:' )[ -1 ]
   lines = lines.partition( 'Summary Matched:' )[ 0 ]
   lines = lines.splitlines()
   packages = []
   for line in lines:
      try:
         package, _ = line.partition( ' : ' )
         packages.append( package )
      except ValueError:
         pass

   # Now we query them each individually and make Ext::Info out of them.
   infos = []
   for package in packages:
      # Output from `dnf info python2-mysqlclient` looks like this:
      # Installed Packages
      # Name         : python2-mysqlclient
      # Version      : 1.4.6
      # Release      : 27824191.adityabenablegcc11artic46.1
      # Architecture : x86_64
      # Size         : 280 k
      # Source       : python-mysqlclient-1.4.6-27824191.adityabenablegcc11artic...
      # Repository   : @System
      # Summary      : MySQL/mariaDB database connector for Python
      # URL          : https://github.com/PyMySQL/mysqlclient-python
      # License      : GPLv2
      # Description  : MySQLdb is an interface to the popular MySQL database ...
      #              : the Python database API.
      #
      lines = dnf( 'info', package )
      try:
         if re.search( r'\nRepository\s+: (\S+)\n', lines ).group( 1 ) == repo.name:
            name = re.search( r'\nName\s+: (\S+)\n', lines ).group( 1 )
            infoKey = Tac.Value( 'Extension::InfoKey', name, 'formatRpm', 1 )
            info = Tac.newInstance( 'Extension::Info', infoKey )
            info.presence = Presence.present
            info.status = 'notInstalled'
            info.primaryPkg = name

            infos.append( info )
      except AttributeError:
         pass

   return infos

def packageDownload( name, repo, status ):
   """
   Downloads the package and dependencies to a temporary directory.
   We then go through and create the Info object to return.
   Finally we copy the files to /flash/.extensions.
   """

   # XXX This function could do with some better error checking.
   # I.e. can we screen scrape to see if dependent RPMs weren't
   # downloaded. What should we do in that case?

   t3( 'package download', name, 'from repo', repo )
   tmpDir = tempfile.mkdtemp()
   # pkgdeps: rpm dnf
   # pkgdeps: rpm dnf-plugins-core
   cmd = [ 'dnf', 'download', '--resolve', '--color', 'never', '--downloaddir',
           tmpDir, name ]

   try:
      output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      t3( 'package download output', output )
   except Tac.SystemCommandError as e:
      if 'There are no enabled repos' in e.output:
         # This is a potential issue; repos listed in system config but not
         # written to disk for yum to know of.
         t3( 'dnf has no enabled repos but we acted for repo:', repo )
         raise errors.NoRepoError( repo )

      match = RE_YUM_PKG_NOT_AVAIL.search( e.output )
      if match:
         pkgname = match.group( 1 )
         # pylint: disable-next=consider-using-f-string
         t3( 'dnf package %r not available' % pkgname )
         raise errors.PackageNotAvailableError( pkgname )

      t3( f'Package {name} failed to install - output: {e.output}' )
      raise errors.InstallError(
         f'Failed to download package {name}: {e}' )

   # Get the filename of the main RPM to create the infoKey
   primaryPkg = None
   primaryInfo = None
   for _, _, files in os.walk( tmpDir ):
      for f in files:
         fPath = os.path.join( tmpDir, f )
         header = rpmutil.readRpmHeaderIntoDict( rpmutil.newTransactionSet(), fPath )
         if header.get( 'name' ).lower() == name.lower():
            primaryPkg = os.path.basename( f )
            primaryInfo = _addPkgToStatus( status, fPath, None )

   if primaryPkg is None:
      raise errors.PackageNameDownloadMismatchError(
         # pylint: disable-next=consider-using-f-string
         "Package named '%s' was downloaded but could not be read for "
         "installation." % name )

   # call addRpm for each dependency then move the files to the final location
   for _, _, files in os.walk( tmpDir ):
      # pylint: disable-next=consider-using-f-string
      t3( 'Adding %d RPM to package %s' % ( len( files ), name ) )
      for f in files:
         fPath = os.path.join( tmpDir, f )
         _addPkgToStatus( status, fPath, primaryInfo )
         t3( 'moving', fPath, 'to', RPM_FINAL_DEST )
         cmd = [ 'mv', '-f', fPath, RPM_FINAL_DEST ]
         finalDest = os.path.join( RPM_FINAL_DEST, f )
         try:
            Tac.run( cmd, asRoot=True )
            # The Cli expects to have control over those files. With vfat
            # this was garanteed by mount options (gid=88, umask=007). With
            # ext4, this has to be manually done
            Tac.run( [ 'chown', 'root:eosadmin', finalDest ], asRoot=True,
                     ignoreReturnCode=True )
            Tac.run( [ 'chmod', '660', finalDest ], asRoot=True,
                     ignoreReturnCode=True )
         except Tac.SystemCommandError as e: # pylint: disable=unused-variable
            # We didn't successfully move this RPM to its install location.
            raise errors.InstallError( f'Failed to download RPM {f}' )

# pylint: disable-next=inconsistent-return-statements
def _addPkgToStatus( status, fPath, primaryInfo ):
   name = os.path.basename( fPath )
   if primaryInfo is not None and name == primaryInfo.primaryPkg:
      # We have already added this package
      return
   key = Tac.Value( 'Extension::InfoKey', name, 'formatYum', 1 )
   info = status.info.newMember( key )
   info.presence = 'present'
   info.status = 'notInstalled'
   info.primaryPkg = name
   info.filepath = RPM_FINAL_DEST + name
   header = rpmutil.readRpmHeaderIntoDict( rpmutil.newTransactionSet(), fPath )
   # For dependencies we do not keep track of all the packages downloaded. This
   # will be trakced via the info of the primary package downloaded only.
   rpmutil.addRpm( info, name, header )
   if primaryInfo is not None:
      primaryInfo.package.newMember( name )
   return info


