#!/usr/bin/env python3
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import errno
import hashlib
import os
import re
import rpm
import shutil
from datetime import datetime

import Cell
import Logging
import SwiSignLib
import Tac
import TpmGeneric.Defs as TpmDefs
from TpmGeneric.Tpm import TpmGeneric
import Tracing
import Url

from ExtensionMgr import (
      errors,
      logs,
      pypilib,
      rpmutil,
      yumlib,
)

DEFAULT_STATUS_FILE = "/var/run/extension-status"

# pkgdeps: library MgmtSecuritySsl
MgmtSslConstants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

BootStatus = Tac.Type( "Extension::BootStatus" )
PkgStatus = Tac.Type( "Extension::PkgStatus" )
Presence = Tac.Type( "Extension::Info::Presence" )

# This is the most useful (quietest) form of RPM output;
# the lower integer value RPMLOG_EMERG still emits faulty
# RPM format errors.
rpm.setVerbosity( rpm.RPMLOG_CRIT )

__defaultTraceHandle__ = Tracing.Handle( 'ExtensionMgr' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3

VALID_REPO_FORMATS = ( 'yum', 'pypi' )

# Presence error messages for Extension::Info records
errUnsupportedFormat = 'Unsupported format'
errInvalidSwix = 'Invalid SWIX file'

INSTALL_FUNCS =  { 'formatRpm': rpmutil.installRpm,
                   'formatYum': rpmutil.installYum,
                   'formatSwix': rpmutil.installSwix,
                   # 'formatPyPi': rpmutil.installPyPi,
                }

# File to log to, syslog if None
LOGFILE = None

def log( *args ):
   if LOGFILE:
      with open( LOGFILE, 'a' ) as logf:
         msg = datetime.now().isoformat() + ' ' + str( args[0] ) +\
               ' ' + ' '.join( args[ 1: ] ) + '\n' 
         logf.write( msg )
   # Always log to syslog
   Logging.log( *args )

# Sysdb paths
def configPath():
   return Cell.path( 'sys/extension/config' )

def repoConfigPath():
   return 'sys/extension/repoConfig'

def statusPath():
   return Cell.path( 'sys/extension/status' )


def sha1sum( filename ):
   """Computes a sha1sum of the file contents and returns it as a hex string.
   Does no error checking: any exceptions that occur propagate to the caller."""
   with open( filename, "rb" ) as f:
      h = hashlib.sha1()  # pylint: disable=no-member
      while True:
         chunk = f.read( 65536 )
         if chunk == b'':
            break
         h.update( chunk )
   return h.hexdigest()

def latestExtensionForName( filename, extensionStatus ):
   """Returns the extension with the given name that has the highest
   generation id, or None if no extension with that name has been
   added."""
   best = None
   for info in extensionStatus.info.values():
      # If file system is case-insensitive, then do case-insensitive compare
      fs = Url.getFilesystem( "extension:" )
      if ( ( fs.ignoresCase() and info.filename.lower() == filename.lower() ) or
           ( not fs.ignoresCase() and info.filename == filename ) ):
         if best is None or best.generation < info.generation:
            best = info 
   return best

def installedExtensionForName( filename, extensionStatus ):
   """Returns the extension with the given name that is installed.  Only one
   extension with a given name may be installed. Returns None if no extension
   with that name has been installed."""
   for info in extensionStatus.info.values():
      if ( info.filename == filename
           and info.status in ( 'installed', 'forceInstalled' ) ):
         return info 
   return None

def getPackageFormat( path ):
   prefixes = { 'yum:': 'formatYum',
                'pypi:': 'formatPyPi',
             }
   suffixes = { '.rpm': 'formatRpm',
                '.tar.gz': 'formatPyPi',
                '.swix': 'formatSwix',
             }
   for prefix in prefixes: # pylint: disable=consider-using-dict-items
      if path != prefix and path.startswith( prefix ):
         return prefixes[ prefix ]
   for suffix in suffixes: # pylint: disable=consider-using-dict-items
      if path != suffix and path.endswith( suffix ):
         return suffixes[ suffix ]
   return 'formatUnknown'

def readExtensionRpmData( path, fmt=None ):
   """Pass in the path to an extension file and the type of file it is (in case of 
   temporary file name) and return a dictionary mapping
   rpm name to rpm info. May raise ExtensionReadError."""
   transactionSet = rpmutil.newTransactionSet()
   fn = os.path.basename( path )
   if not fmt:
      fmt = getPackageFormat( path )
   if fmt == 'formatRpm':
      header = rpmutil.readRpmHeaderIntoDict( transactionSet, path )
      return { fn: header }
   elif fmt == 'formatSwix':
      import zipfile # pylint: disable=import-outside-toplevel
      if zipfile.is_zipfile( path ):
         _, rpmHeaderInfo = rpmutil.readSwix( path, transactionSet )
         return rpmHeaderInfo
   assert False, 'Unsupported extension format: %d' % fmt
   return {}

def readExtensionFile( path, extensionStatus, transactionSet=None, pType=None, 
                       deps=None ):
   """Pass me the path to an extension file, a rpm.TransactionSet instance,
   and an instance of Extension::Status and I will examine the file and
   populate an entry in Extension::Status::info.  The status of the entry
   will indicate whether the extension is valid or not."""
   if transactionSet is None:
      transactionSet = rpmutil.newTransactionSet()
   fn = os.path.basename( path )
   if pType:
      fmt = pType
   else:
      fmt = getPackageFormat( path )
   latest = latestExtensionForName( fn, extensionStatus )
   if latest is None:
      generation = 1
   else:
      generation = latest.generation + 1

   key = Tac.Value( 'Extension::InfoKey', fn, fmt, generation )
   info = extensionStatus.info.newMember( key )
   info.filepath = path
   info.presence = 'absent'
   if not fmt or fmt == 'formatUnknown':
      t0( path, 'is not an extension! Skipping...' )
      info.presenceDetail = errUnsupportedFormat
   elif fmt == 'formatRpm' or fmt == 'formatYum': # pylint: disable=consider-using-in
      try:
         header = rpmutil.readRpmHeaderIntoDict( transactionSet, path )
         rpmutil.addRpm( info, fn, header )
      except errors.RpmReadError as e:
         t0( 'Error reading rpm', path, ':', e )
         info.presenceDetail = str( e )
         log( logs.EXTENSION_INSTALL_ERROR, info.filename, str( e ) )
         return

      if deps:
         basepath = os.path.dirname( path ) + '/'
         for dep in deps:
            # skip the pimary rpm
            if dep == fn or dep == '': # pylint: disable=consider-using-in
               continue
            fpath = basepath + dep 
            header = rpmutil.readRpmHeaderIntoDict( transactionSet, fpath )
            rpmutil.addRpm( info, dep, header )

      info.primaryPkg = fn
      info.presence = 'present'
      info.presenceDetail = ''
   elif fmt == 'formatSwix':
      # This import must be delayed until here, not done at module
      # level, due to zipfile being blacklisted in CliTests.py.
      import zipfile # pylint: disable=import-outside-toplevel
      if not zipfile.is_zipfile( path ):
         t0( path, 'is not a zipfile! Skipping...' )
         info.presenceDetail = errInvalidSwix
         return
      rpmutil.saveSwixInfo( info, transactionSet )
   elif fmt == 'formatPyPi':
      # XXX handle pypi archives
      pass
   else: # unknown format type
      t0( path, 'has unsupported type', fmt )
      info.presenceDetail = errUnsupportedFormat

def readExtensionsDir( dirpath, extensionStatus, continueOnError=False ):
   """Pass me a directory path and an instance of Extension::Status and I will
   look for extension files in the directory, examine them, and populate the
   Extension::Status instance.  The caller is responsible for handling any
   exception raised during reading the directory."""
   t0( "readExtensionsDir for", dirpath )
   ents = os.listdir( dirpath )
   ts = rpmutil.newTransactionSet()
   for ext in ents:
      path = os.path.join( dirpath, ext )
      if not os.path.isfile( path ):
         continue
      t0( "processing possible extension file", path )
      try:
         readExtensionFile( path, extensionStatus, ts )
      except Exception as e:  # pylint: disable=broad-except
         # We really want to catch the user top level exception
         t0( "Caught exception while processing", path, ":", e )
         if not continueOnError:
            raise

def _internalCheckInstallPrerequisites( extensionInfo ):
   ei = extensionInfo
   if ei.presence != 'present':
      msg = "Installation failed"
      if ei.presenceDetail:
         msg += ": " + ei.presenceDetail
      raise errors.InstallError( msg )
   if ei.status in ( 'installed', 'forceInstalled' ):
      raise errors.InstallError( "Extension is already installed" )
   pr = ei.package.get( ei.primaryPkg )
   if pr is None:
      raise errors.InstallError( "Extension %s cannot be installed: missing "
                                 "primary RPM\n" % ei.filename )

def checkInstallPrerequisites( extensionInfo ):
   try:
      _internalCheckInstallPrerequisites( extensionInfo )
      return ( True, None )
   except errors.InstallError as e:
      return ( False, str( e ) )

def _doPcrExtend( filename, filePath, sha1Fingerprint, action, event ):
   # The extension of TPM PCRs is a security feature that allows post boot
   # attestation. We precisely measure every step of EOS that can load code. We
   # record hashes that can be extracted and validated by an external controller
   # in a secure manner. This can used as part of a security model to verify that
   # no tampering of any kind has happened on a device.
   #
   # Extensions being one way to load code at boot time in EOS, it's critical
   # to have a record of this execution.
   #
   # This is what the piece of code does below, if we detect a TPM, support the
   # feature on the platform, and it's enabled (via the 'measuredboot' bit in the
   # secure boot toggle)
   tpm = TpmGeneric()
   try:
      if not tpm.isToggleBitSet( TpmDefs.SBToggleBit.MEASUREDBOOT ):
         return
   except ( TpmDefs.NoTpmDevice, TpmDefs.NoTpmImpl, TpmDefs.NoSBToggle ):
      return
   except TpmDefs.Error as e:
      t0( "Failed to check measured boot status: %s", str( e ) )
      return

   try:
      pcrFingerprint = sha1Fingerprint
      if not pcrFingerprint or tpm.pcrHashAlg() != TpmDefs.HashAlg.SHA1:
         try:
            pcrFingerprint = tpm.hashFile( filePath ).hexdigest()
         except Exception as e: # pylint: disable=broad-except
            t0( 'Failed to compute fingerprint of %s: %s' %
                ( filePath, e ) )
            return

      tpm.pcrExtendFromData( TpmDefs.PcrRegisters.EOS_EXTENSIONS,
                             event,
                             f'EOS extension {action} {pcrFingerprint}',
                             log=[ '%s extension %s (%s)' %
                                   ( action, filename, pcrFingerprint ) ] )
   except TpmDefs.Error as e:
      t0( "Failed to extend PCR registers for %s (%s): %s" %
          ( filename, pcrFingerprint, str( e ) ) )

def installExtension( status, info, force=False ):
   """Installs an extension.

   Provide an instance of Extension::Status and  Extension::Info and 
   I will install the extension by following the procedure described in 
   AID 522.  The process in which I am called must have an effective uid 
   of 0 because I must have permission to manipulate the RPM database.

   Raises:
     - InstallError: if installation fails.
   """
   assert os.geteuid() == 0, 'Not running with EUID of 0'
   _internalCheckInstallPrerequisites( info )
   pr = info.package.get( info.primaryPkg )
   try:
      fingerprint = sha1sum( info.filepath )
   except Exception as e:
      raise errors.InstallError( "Error computing SHA-1 fingerprint of %s: %s"
                                 % ( info.filename, e ) )
   _doPcrExtend( info.filename, info.filepath, fingerprint, 'install',
                 TpmDefs.PcrEventType.EOS_EXTENSION_INSTALL )
   log( logs.EXTENSION_INSTALLING, info.filename, pr.version, fingerprint )
   try:
      install = INSTALL_FUNCS.get( info.format )
      if install is not None:
         install( status, info, force )
      else:
         raise errors.InstallError( "Unsupported extension format '%s'" %
                                    info.format )
      log( logs.EXTENSION_INSTALLED, info.filename )
   except ( errors.InstallError, Exception ) as e:  # pylint: disable=broad-except
      log( logs.EXTENSION_INSTALL_ERROR, info.filename, str( e ) )
      _doPcrExtend( info.filename, info.filepath, fingerprint, 'install error',
                    TpmDefs.PcrEventType.EOS_EXTENSION_INSTALL_FAILURE )
      raise

def _internalUninstallCheck( extensionInfo ):
   ei = extensionInfo
   if ei.status not in ( 'installed', 'forceInstalled' ):
      raise errors.UninstallError( "Only installed extensions may be uninstalled" )

def checkUninstallPrerequisites( extensionInfo ):
   try:
      _internalUninstallCheck( extensionInfo )
      return ( True, None )
   except errors.UninstallError as e:
      return ( False, str( e ) )

def unmountAndRmDirs( extensionInfo, mountIdx, mountPoint ):
   # Unmount the overlay, then the lower dir for it.
   for dirPrefix in ( '', '/mnt/apps' ):
      Tac.run( [ 'umount', dirPrefix + mountPoint ],
               asRoot=True, stderr=Tac.CAPTURE )

   del extensionInfo.mountpoints[ mountIdx ]

   for dirPrefix in rpmutil.overlayDirs:
      directory = dirPrefix.format( mountPoint, mountIdx )
      if dirPrefix == rpmutil.upperDirName:
         # Empty the upper dir, or it retains files across SWIX reinstallations.
         shutil.rmtree( directory, ignore_errors=True )
      else:
         try:
            os.rmdir( directory )
         except OSError as e:
            # ENOTEMPTY (39) could happen if we've mounted on non-empty dir.
            # EBUSY (16) could happen if we've mounted on the same dir twice.
            if e.errno not in ( errno.ENOTEMPTY, errno.EBUSY ):
               error = f'Failed to remove directory {mountPoint!r}: {e}'
               raise errors.UninstallError( error )

def uninstallExtension( extensionInfo, extensionStatus, force=False ):
   """Uninstall an extension.

   The process in which I am called must have an effective uid of 0 because
   I must have permission to manipulate the RPM database.

   Args:
     - extensionInfo: The instance of Extension::Info of the extension to
       uninstall by following the procedure described in AID 522.
     - extensionStatus: Our Extension::Status.
     - force: Whether or not to force the uninstall.

   Raises:
     - UninstallError: if I fail.
   """
   assert os.geteuid() == 0, "Must run as root, not UID %d" % os.geteuid()
   _internalUninstallCheck( extensionInfo )
   filename = extensionInfo.filename
   log( logs.EXTENSION_UNINSTALLING, filename )

   ts = rpmutil.newTransactionSet()

   if force:
      rpmutil.setForceProblemFlags( ts )
   else:
      ts.setProbFilter( 0 )

   try:
      # We want to unmount in the reverse mounting order in case it matters.
      mountpoints = sorted( extensionInfo.mountpoints.items(), reverse=True )
      for mountIdx, mountPoint in mountpoints:
         try:
            unmountAndRmDirs( extensionInfo, mountIdx, mountPoint )
         except Tac.SystemCommandError as e:
            error = 'Failed to unmount %r' % mountPoint

            # If the unmounting failed due to files open, find the procs and report.
            if 'target is busy' in e.output:
               # '+c0' means don't truncate the procs' names.
               output = Tac.run( [ 'lsof', '+c0', mountPoint ], stdout=Tac.CAPTURE )
               procs = output.splitlines()[ 1: ] # Behead the output.
               procs = { tuple( line.split()[ :2 ] ) for line in procs }
               fmt = '%s (pid %s)'
               error += ' due to files kept open by '
               error += ', '.join( fmt % procAndPid for procAndPid in procs )

            raise errors.UninstallError( error )

      _doPcrExtend( extensionInfo.filename, extensionInfo.filepath, None,
                    'uninstall', TpmDefs.PcrEventType.EOS_EXTENSION_UNINSTALL )
      packages = [ r.packageName for r in extensionInfo.installedPackage.values() ]
      try:
         rpmutil.uninstallRpms( packages, ts, force )
      except errors.RpmUninstallError as e:
         _doPcrExtend( extensionInfo.filename, extensionInfo.filepath, None,
                       'uninstall error',
                       TpmDefs.PcrEventType.EOS_EXTENSION_UNINSTALL_FAILURE )
         extensionInfo.statusDetail = str( e )
         raise errors.UninstallError( "RPM uninstall error: %s" % e )

      extensionInfo.installedPackage.clear()
      extensionInfo.status = PkgStatus.notInstalled
      extensionInfo.statusDetail = ''
      # If this extension is absent (e.g. its file was removed) then
      # also garbage collect its associated status as there is no way
      # it can be re-installed anyway.
      if extensionInfo.presence == 'absent':
         del extensionStatus.info[ extensionInfo.key ]

      log( logs.EXTENSION_UNINSTALLED, filename )
   except Exception as e:  # pylint: disable=broad-except
      log( logs.EXTENSION_UNINSTALL_ERROR, filename, str( e ) )
      raise

def addToConfig( config, name, pFormat, forced, signatureIgnored=False ):
   """Takes an Extension::Config, an extension name, an Extension::Format, a 
   a boolean indicating whether the extension was forcibly installed, and 
   a boolean indicating whether to ignore the signature during verification,
   then adds it to the config installation collection with an index higher 
   than any index currently in the collection."""
   highest = 0
   installed = list( config.installation.values() )
   for extension in installed:
      if extension.filename == name:
         # This is unexpected -- why was this extension already in the config?
         # There's no point to adding the same extension twice.
         return
      if extension.index > highest:
         highest = extension.index
   index = highest + 1
   item = config.installation.newMember( name, pFormat, index )
   item.force = forced
   item.signatureIgnored = signatureIgnored

def removeFromConfig( config, name ):
   """Takes an Extension::Config and an extension name, then removes the entry
   with that name from the config installation collection."""
   installed = list( config.installation.values() )
   for extension in installed:
      if extension.filename == name:
         try:
            del config.installation[ extension.index ]
         except Exception as e:  # pylint: disable=broad-except
            # We really want to catch the user top level exception
            t0( "Failed to remove item from config.installation:", e )
         break

def updateStatus( status, primaryPkg, install=True ):
   pkgs = primaryPkg.split( "." )
   if install:
      status.newInstalledPrimaryPkg( pkgs[ 0 ] )
   else:
      del status.installedPrimaryPkg[ pkgs[ 0 ] ]

def updateInstalledExtensionsForBoot( sysdbRoot ):
   """Update installed extensions booStatus in Sysdb.
   If any installed extensions do not have valid signatures and were
   not installed with "signature-verification ignored", SignatureVerificationError
   is raised with error message saying which extensions are affected.
   """
   config = sysdbRoot.entity[ configPath() ]
   status = sysdbRoot.entity[ statusPath() ]
   mgmtSecConfig = sysdbRoot.entity[ 'mgmt/security/config' ]
   mgmtSslConfig = sysdbRoot.entity[ 'mgmt/security/ssl/config' ]
   checkSigs = signatureVerificationEnabled( mgmtSecConfig )
   warningMsg = ''
   signingCerts = []
   if checkSigs:
      try:
         signingCerts = signatureVerificationCerts( mgmtSecConfig, mgmtSslConfig )
      except errors.SignatureVerificationError as e:
         warningMsg = "Skipping SWIX signature verification: " + str( e )
         checkSigs = False

   installed = sorted( config.installation.values(), key=lambda a: a.index )
   invalidSwixSigs = []

   for installCfg in installed:
      info = latestExtensionForName( installCfg.filename, status )
      # Don't add swix to boot-extensions file if it needs to be signed properly
      # and isn't
      if ( checkSigs and not installCfg.signatureIgnored and
           installCfg.format == 'formatSwix' ):
         swixPath = info.filepath
         sigValid, _, _ = SwiSignLib.verifySwixSignature( swixPath, signingCerts )
         if not sigValid:
            invalidSwixSigs.append( installCfg.filename )
            continue

      if info:
         if info.boot == BootStatus.notBoot:
            info.boot = BootStatus.bootByCopy

   if invalidSwixSigs:
      warningMsg = ( "%s not copied to boot-extensions because of missing or"
                     " invalid signature." ) % ", ".join( invalidSwixSigs )
   if warningMsg:
      raise errors.SignatureVerificationError( warningMsg )

def saveBootExtensions( sysdbRoot, dstFile ):
   """Writes the installed extensions to dstFile in the boot-extensions
   format. 
   """
   config = sysdbRoot.entity[ configPath() ]
   status = sysdbRoot.entity[ statusPath() ]
   installed = sorted( config.installation.values(), key=lambda a: a.index )

   # Lines in boot-extensions file have four fields seperated by space
   # "extension-name flags extension-format [dependencies]"
   # flags field is used to indicate both force and boot information.
   # Possible values of flags field is as follows:
   #    If an extension is added to boot-extensions file by:
   #    - "copy installed-extensions boot-extensions" : "force" or "no"
   #    - "boot extension EXTENSION" : "boot"
   #    - both the above commands: "force,boot" or "no,boot"
   for installCfg in installed:
      info = latestExtensionForName( installCfg.filename, status )
      if info and info.boot == BootStatus.notBoot:
         continue

      forceStr = "force" if installCfg.force else "no"
      if info and info.boot == BootStatus.bootByConfig:
         forceStr += ",boot"

      bootExtLine = "{} {} {}".format( installCfg.filename,
                                       forceStr,
                                       installCfg.format )

      # For yum/rpm packages we need to write their dependencies that exist
      # in /mnt/flash/.extensions. The RPM transactionSet requires
      # them for proper installation.
      # pylint: disable-next=consider-using-in
      if installCfg.format == 'formatRpm' or installCfg.format == 'formatYum':
         if info:
            bootExtLine += ' ' + str( list( info.package ) )
         else:
            t3( 'Error! Installed pkg %s found but no info found' %
                installCfg.filename )

      dstFile.write( bootExtLine.encode() + b'\n' )

   for info in status.info.values():
      if info.status != PkgStatus.notInstalled:
         #  installed extensions have already been handled
         continue
      if info.boot == BootStatus.bootByCopy:
         # If an extension was added to boot-extensions by
         # "copy installed-extensions boot-extensions", but now
         # the extension has been uninstalled, it should be removed
         # from boot-extensions
         info.boot = BootStatus.notBoot
      elif info.boot == BootStatus.bootByConfig:
         # Write boot extensions that are configured by "boot extension" command
         bootExtLine = "{} boot {}".format( info.filename,
                                            info.format )
         dstFile.write( bootExtLine.encode() + b'\n' )

def checkIfUpgrading( status, newInfo ):
   """
   If we are upgrading an extension to a new version we have to mark the
   old extension as not installed. RPM will take care of actually upgrading
   the system itself. This only takes care of Sysdb state.
   """
   pkgNew = newInfo.package[ newInfo.primaryPkg ]
   for info in status.info.values():
      if info.status in ( 'installed', 'forceInstalled' ):
         pkg = info.package[ info.primaryPkg ]
         # packageName is the actual RPM package name
         if pkg.packageName == pkgNew.packageName:
            return info
   return None

def getInstalledVersion( info ):
   pkg = info.package[ info.primaryPkg ]
   if info.key.format == 'formayPypi':
      # BUG94891 once we support pip packages we need to get this
      # from the actual system. Right now I'm returning
      # what will always match so the warning does not mprint.
      return ( pkg.version, pkg.release )
   else:
      # yum, rpm, and swix types
      ts = rpmutil.newTransactionSet()
      hdrs = ts.dbMatch( rpm.RPMTAG_NAME, pkg.packageName )
      if len( hdrs ) != 1:
         # RPM may have been removed under the covers
         return ( None, None )
      hdr = next( hdrs )
      return ( hdr[ rpm.RPMTAG_VERSION ], hdr[ rpm.RPMTAG_RELEASE ] )

# Package manager interface

# Implementation modules for package formats
impls = { 'formatYum': yumlib,
          'formatPyPi': pypilib }


def updateRepository( repo ):
   if repo is None:
      return
   impl = impls.get( repo.format )
   if impl is not None:
      impl.updateRepository( repo )


def deleteRepository( repo ):
   if repo is None:
      return
   impl = impls.get( repo.format )
   if impl is not None:
      impl.deleteRepository( repo )


def packageDownload( name, repos, repotype, status ):
   t0( 'packageDownload', name, 'from type', repotype )
   impl = impls.get( repotype )
   if impl is not None:
      impl.packageDownload( name, repos, status )


def packageSearch( query, repos ):
   t0( 'packageSearch', query )

   yumRepos = []
   pypiRepos = []
   for repo in repos:
      if repo.format == 'formatYum':
         yumRepos.append( repo )
      elif repo.format == 'formatPyPi':
         pypiRepos.append( repo )

   pkgs = []
   for repo in yumRepos:
      # TODO: This is suboptimal.
      # We query all packages for each repo name.
      # Should query once, then compare against a set of repos.
      pkgs.extend( impls[ 'formatYum' ].packageSearch( query, repo ) )
   for repo in pypiRepos:
      pkgs.extend( impls[ 'formatPyPi' ].packageSearch( query, repo ) )
   return pkgs

repoTypeStrMap = {
   'formatYum': 'yum',
   'formatPyPi': 'pypi',
   'formatSwix': 'swix',
   'formatUnknown': 'unknown'
   }

def signatureVerificationEnabled( mgmtSecConfig ):
   ''' Returns true if signature-verification is enabled '''
   return mgmtSecConfig and mgmtSecConfig.enforceSignature 

def signatureVerificationCerts( mgmtSecConfig, mgmtSslConfig ):
   ''' Returns the list of potential signing certs to use for signature
   verification '''
   sslProfileName = mgmtSecConfig.sslProfile
   sslProfile = mgmtSslConfig.profileConfig.get( sslProfileName )
   if not sslProfile:
      msg = "Unable to verify SWIX signatures. SSL profile '%s' does not exist"
      raise errors.SignatureVerificationError( msg % sslProfileName )
   certs = [ MgmtSslConstants.certPath( cert ) for cert in sslProfile.trustedCert ]
   if not certs:
      msg = "Unable to verify SWIX signatures. No trusted certificates defined"
      raise errors.SignatureVerificationError( msg )
   return certs

def verifySignature( swixFile, mgmtSecConfig, mgmtSslConfig ):
   ''' Returns false if SWIX signature is invalid '''
   signingCerts = signatureVerificationCerts( mgmtSecConfig, mgmtSslConfig )
   sigValid, _, _ = SwiSignLib.verifySwixSignature( swixFile, signingCerts )
   return sigValid

def printExtensionInfo( output, info, sigValid=None ):
   signedStr = ''
   if sigValid:
      signedStr = 'signed'
   elif sigValid is False:
      # False (the SWIX is not signed properly) != None (this isn't a swix)
      signedStr = 'notSigned'

   def _safestr( s ):
      return re.sub( "[\t\n]", " ", s )

   installed = " ".join( info.installedPackage )
   mountpoints = " ".join( info.mountpoints.values() )
   row = [ info.key.filename, info.boot, info.key.format,
      info.presence, _safestr( info.presenceDetail ), info.status,
      _safestr( info.statusDetail ), signedStr, installed, mountpoints ]
   line  = '\t'.join( row )
   t0( "status:", line )
   output.write( line + '\n' )

# save current extensions to a file
def saveInstalledExtensionStatus( sysdbRoot, dstFile ):
   """This is similar to saveInstalledExtensions, but the file is for
   LoadExtensionStatus instead of LoadExtension."""
   if not dstFile:
      return
   t0( "save status for all extensions:", dstFile )
   config = sysdbRoot.entity[ configPath() ]
   status = sysdbRoot.entity[ statusPath() ]

   try:
      tmp = dstFile + '.tmp'
      with open( tmp, "w" ) as output:
         for installCfg in sorted( config.installation.values(),
                                   key=lambda a: a.index ):
            info = latestExtensionForName( installCfg.filename, status )
            if info:
               # Here is a bit of inconsistency:
               #
               # printExtensionInfo expects sigValid to be 3 values: None/True/False
               # which translate to ''/'signed'/'notSigned'. But LoadExtensionStatus
               # will turn it into signatureIgnored which is a boolean (notSigned
               # means True). We try to convert the boolean to the 3 values but
               # essentially None and True are the same.
               if info.format == "formatSwix":
                  sigValid = not installCfg.signatureIgnored
               else:
                  sigValid = None
               printExtensionInfo( output, info, sigValid=sigValid )
            else:
               t3( "Error: no info found for installed package",
                   installCfg.filename )

      os.rename( tmp, dstFile )
   except OSError as e:
      # ignore the error and continue
      log( logs.EXTENSION_STATUS_SAVE_ERROR, dstFile, e.strerror )
