# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the fabric interface name matcher.

Unlike other matchers, this one will only match on fabric interfaces that exist.
'''

from __future__ import absolute_import, division, print_function
import CliMatcher
import CliPlugin.VirtualIntfRule as VirtualIntfRule
import Tac

class IntfStatusNumberMatcher( object ):
   '''An interface status matcher that is designed to match any interface name
   registered in a Interface::*StatusDir collection.

   The matcher expects the fully aggregated interface status path.

   Note:
      unlike the physical matcher defined in EntityMib this matcher does not match on
      interfaces that don't yet exist.
   '''

   def __init__( self, basename, allIntfStatusPath ):
      self.basename = basename
      self.allIntfStatusPath = allIntfStatusPath

   def intfStatus( self, mode ):
      '''Returns an aggregated list of interfaces that exist in the interface status
      collections.
      '''
      allIntfStatusDir = mode.entityManager.lookup( self.allIntfStatusPath )
      if allIntfStatusDir is None:
         return []

      return allIntfStatusDir.intfStatus

   def intfExists( self, mode, token ):
      '''Checks if a given token matches a known interface status.'''
      # fa..::.. is a valid ipv6 address which tries to match here.
      # Checking if the token is a valid IntfId before intfExists in intfStatusDir. 
      try:
         intfId = Tac.Value( "Arnet::IntfId", self.basename + token )
      except IndexError:
         return False

      inStartupConfig = mode.session and mode.session.startupConfig()
      return inStartupConfig or ( intfId in self.intfStatus( mode ) )

   def match( self, mode, context, token ):
      '''Match on the <intfNumber> part of the interface ID.'''
      if self.intfExists( mode, token ):
         return CliMatcher.MatchResult( token, token )

      return CliMatcher.noMatch

   def completions( self, mode, context, token ):
      '''TODO BUG529146 Add completion support.'''
      return []

class FabricIntfStatusMatcher( VirtualIntfRule.VirtualIntfMatcherBase ):
   '''This matcher is meant to match only on interface names that exist inthe fully
   aggregated fabric interface status "all" path
   '''

   def __init__( self, basename, allFabricIntfStatusPath, helpdesc=None,
                 alternates=None, **kwargs ):
      self.basename = basename
      intfNumberMatcher = IntfStatusNumberMatcher( basename,
                                                   allFabricIntfStatusPath )

      # For <baseName> matching we don't need to re-invent the wheel,
      # _VirtualIntfMatcher meets all our needs.
      # pylint: disable=protected-access
      intfNameMatcher = VirtualIntfRule._VirtualIntfMatcher( basename,
                                                             intfNumberMatcher,
                                                             alternates=alternates )
      # pylint: enable=protected-access

      super( FabricIntfStatusMatcher, self ).__init__( basename,
                                                       intfNameMatcher,
                                                       intfNumberMatcher,
                                                       helpdesc=helpdesc,
                                                       alternates=alternates,
                                                       **kwargs )
