# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, FdlInventoryInitializer

class Cooling1RUInventoryDirInitializer( FdlInventoryInitializer.DirInitializer ):
   '''Used to create fdl for cooling system in 1RUs where fans in system
      can run at different speeds.
   '''
   managedTypeName = "Inventory::CoolingDomainDir"

   def initInventoryEntry( self, **kwargs ):
      overheatThresholdRatioBase = kwargs[ 'overheatThresholdRatioBase' ]
      minCoolingDomainFanSpeed = kwargs[ 'minCoolingDomainFanSpeed' ]
      slopeOffsetBasedFanControl = kwargs[ 'slopeOffsetBasedFanControl' ]
      fanSlopeOffsetInfo = kwargs[ 'fanSlopeOffsetInfo' ]
      baseFanPriority = kwargs[ 'baseFanPriority' ]
      powerSupplySlotDescs = kwargs[ 'powerSupplySlotDescs' ]

      mainCoolingDomain = self.invEntity_.newCoolingDomain(
                              "FixedSystem",
                              overheatThresholdRatioBase,
                              minCoolingDomainFanSpeed )

      # check if fan slope/offset based control needs to be initialized
      if slopeOffsetBasedFanControl:
         self.initSlopeOffsetBasedFanControl( mainCoolingDomain, fanSlopeOffsetInfo,
                                              baseFanPriority )

      # check if domain slots need to be initialized
      if powerSupplySlotDescs:
         self.initDomainSlots( mainCoolingDomain, powerSupplySlotDescs )


   def initSlopeOffsetBasedFanControl( self, mainCoolingDomain,
                                       fanSlopeOffsetInfo, baseFanPriority ):
      def getFanKey( fanName, direction ):
         return Tac.Value( 'Environment::Thermostat::FanSpeedModifierKey',
                           fanName, direction )
      # Define base fan whose speed is set to domain speed. Domain speed is
      # changed according to the current temperature. The speed of other fans
      # is calculated based on domain speed and offset. We maintain the list
      # of base fans with priority. In the system with J and S, S should be
      # selected as base fan.
      mainCoolingDomain.baseFansDefined = True
      for ( fanName, direction, priority ) in baseFanPriority:
         fanKey = getFanKey( fanName, direction )
         mainCoolingDomain.baseFanKeyPriority[ fanKey ] = priority

      for ( fanName, slope, offset, direction ) in fanSlopeOffsetInfo:
         fanKey = getFanKey( fanName, direction )
         mainCoolingDomain.fanSpeedSlope[ fanKey ] = slope
         mainCoolingDomain.fanSpeedOffset[ fanKey ] = offset
   
      # Make sure fans in the system must be in above list
      mainCoolingDomain.fanSlopeOffsetDefined = True
      
   def initDomainSlots( self, mainCoolingDomain, powerSupplySlotDescs ):
      # Add domain slots
      slotToCoolingDomain = self.invEntity_.slotToCoolingDomain
      slotToCoolingDomain[ "FixedSystem" ] = mainCoolingDomain
      for i in range( 1, 5 ):                   # 4 system fan slots
         # pylint: disable-next=consider-using-f-string
         slotToCoolingDomain[ "Fan%d" %  i ] = mainCoolingDomain
      for slotDesc in powerSupplySlotDescs:     # 2 power supply slots
         # pylint: disable-next=consider-using-f-string
         slotToCoolingDomain[ "%s%d" % ( slotDesc.slotType,
                                         slotDesc.label ) ] = mainCoolingDomain
         # pylint: disable-next=consider-using-f-string
         slotToCoolingDomain[ "FanP%d" % ( slotDesc.label ) ] = mainCoolingDomain

class Fans1RUInventoryDirInitializer( FdlInventoryInitializer.DirInitializer ):
   '''Used to create fdl for system fans in 1RUs. It supports to create 
      J and S fans.
   '''
   managedTypeName = "Inventory::SupportedSystemFans"
   
   def initInventoryEntry( self, **kwargs ):
      fanProperties = kwargs[ 'fanProperties' ]
      fanIdToFanTrayDict = kwargs[ 'fanIdToFanTrayDict' ]
      supportedFans = kwargs[ 'supportedFans' ]
      outOfDateFans = kwargs[ 'outOfDateFans' ]
      powerSupplyVerificationNeeded = kwargs[ 'powerSupplyVerificationNeeded' ]
      supportedPsuFans = kwargs[ 'supportedPsuFans' ]
      outOfDatePsuFans = kwargs[ 'outOfDatePsuFans' ]
      powerSupplySlotDir = kwargs[ 'powerSupplySlotDir' ]

      # We just support a valid set of system fans
      validFanNames = [ value[ 0 ] for value in fanProperties.values() ]
      assert set( supportedFans ).issubset( validFanNames )
      assert set( outOfDateFans ).issubset( validFanNames )

      def createFanTray( fanId ):
         def isFanSupported( fanId ):
            if fanId not in fanProperties:
               return False
            if fanProperties[ fanId ][ 0 ] in supportedFans:
               return True
            return False

         def isFanOutOfDate( fanId ):
            if fanId not in fanProperties:
               return False
            if fanProperties[ fanId ][ 0 ] in outOfDateFans:
               return True
            return False
         
         fanSupportedInSystem = isFanSupported( fanId )
         fanOutOfDateInSystem = isFanOutOfDate( fanId )
         # If a fan tray is not supported, it shouldn't be considered out-
         # of-date one because we are issuing log messages for unsupported
         # fan and out-of-date fan seperatedly. So, if a fanId is in list
         # outOfDateFans, it shouldn't be in supportedFanIdsInProduct.
         # But if a fan tray is supported, it could be out-of-date, i.e.,
         # fans can still run, but don't achieve high performance.
         if fanOutOfDateInSystem and not fanSupportedInSystem:
            assert False, "Out-of-date fan tray should be still supported."

         if fanSupportedInSystem:
            _fanId = fanId
         else:
            # If we see a fan which is not supported, we pick J fan with the same
            # direction to describe its properties. E.g., if forward S fan is
            # inserted, and it is not supported in this system, it can still run
            # with maxRPM of 18000 like forward J fan.
            _fanId = 3 if fanId <= 3 else 7

         ( fanTrayName, fanTrayModelName ) = fanIdToFanTrayDict[ _fanId ]
         # pylint: disable-next=consider-using-f-string
         fanTray = self.invEntity_.newFanTray( "%s.%d" % ( fanTrayName, fanId ), 1 )
         fanTray.modelName = fanTrayModelName
         fanTray.fanSupportedInSystem = fanSupportedInSystem
         fanTray.fanOutOfDateInSystem = fanOutOfDateInSystem

         # Create fans in fan tray
         fan = fanTray.newFan( 1 )
         fan.physicalProperties = fanProperties[ _fanId ]
         return fanTray

      for i in range ( 8 ):
         self.invEntity_.fanIdToFanTray[ i ] = createFanTray( i )

      # In some systems that use slope and offset for setting speed, we support
      # a valid set of Psu fans
      if powerSupplyVerificationNeeded:
         assert powerSupplySlotDir
         powerSupplySlotDir.powerSupplyVerificationNeeded = \
               powerSupplyVerificationNeeded
         powerSupplySlotDir.supportedPsuFans.clear()
         powerSupplySlotDir.outOfDatePsuFans.clear()
         for psuFan in supportedPsuFans:
            powerSupplySlotDir.supportedPsuFans.enq( psuFan )
         for psuFan in outOfDatePsuFans:
            powerSupplySlotDir.outOfDatePsuFans.enq( psuFan )
              
def Plugin( context ):
   context.registerInitializer( Cooling1RUInventoryDirInitializer )
   context.registerInitializer( Fans1RUInventoryDirInitializer )
