# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
from ClassificationLib import (
   computePortRangeSet,
   rangeSetToNumericalRange
)

th = Tracing.Handle( "FieldSetOpenConfigLib" )
t0 = th.trace0
t9 = th.trace9

maxL4Port = Tac.Value( 'Classification::Constants' ).maxL4Port

def comparePortNumRangeAndPortRange( portNumRangeList, portRangeList ):
   extPortRange = portNumRangeToPortRange( portNumRangeList )
   return all( p in portRangeList for p in extPortRange )

def rangeListToPortNumRange( rangeList ):
   '''
   @rangeList: Python list containing mix of integers, strings describing range in
   'x..y' format or "ANY".

   Given a list of elements ( mix of integer & strings ), return a list containing
   Acl::OpenConfig::PortNumRange objects.
   '''
   portNumRangeList = []
   for p in rangeList:
      portNumRange = Tac.Value( 'Acl::OpenConfig::PortNumRange' )
      if isinstance( p, int ):
         # This is a single port
         portNumRange.portNumber = p
      if isinstance( p, str ) and p == "ANY":
         # Any/All ports are included.
         portNumRange.any = 'any'
      if isinstance( p, str ) and '..' in p:
         # Range of ports in the x..y format.
         portNumRange.port = p
      portNumRangeList.append( portNumRange )
   return portNumRangeList

def portNumRangeToRangeSetString( portNumRangeList ):
   '''
   portNumRangeList contains Acl::OpenConfig::PortNumRange elements.
   PortNumRange is a union of 3 types - int, string and enumType; they are
   interpreted as below -
      int = Single port number
      string = Range of ports in the '10..20' format
      enumType = Only 1 enumType ANY is supported, which represents all ports.
   Example: [ 1, 5, '10..20', 'ANY' ]

   This API converts portNumRangeList into a set represented as a string
   ( ex: '1,5,10-20' ) and returns it.
   '''
   rangeSetString = ""
   for p in portNumRangeList:
      rangeSetString += "," if rangeSetString != "" else ""
      if p.containsPortNumber():
         rangeSetString += str( p.portNumber )
      if p.containsPort():
         # Port from OpenConfig is sent in the form "x..y", replace it with
         # "x-y" to make it compatible with existing api.
         rangeSetString += str( p.port.replace( "..", "-" ) )
      if p.containsAny():
         # ANY option is used when all ports are included in the field-set
         return "ANY"
   t0( f"rangeSetString: {rangeSetString}" )
   return rangeSetString

def portNumRangeToPortRange( portNumRangeList ):
   '''
   portNumRangeList contains Acl::OpenConfig::PortNumRange elements. This
   API converts them into Classification::PortRange that can be used in the Native
   field-set configuration of L4-Ports.
   '''
   rangeSetString = portNumRangeToRangeSetString( portNumRangeList )
   if rangeSetString == "ANY":
      # 'Any' suggests that all ports are included in the field-set
      anyPortRange = Tac.Value( 'Classification::PortRange', 0, maxL4Port )
      return [ anyPortRange ]
   # Convert rangeSetString into a set of integers
   portRangeSet = computePortRangeSet( rangeSetString )
   # Convert this portRangeSet into Numberical Range to be used in the
   # Native entity.
   return rangeSetToNumericalRange( portRangeSet, "Classification::PortRange" )
