# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# This module provides a mechanism whereby CliPlugins can register additional
# urls that can be used with certain commands provided by FileCli, such as copy
# and diff.  FileCli itself uses this module to register urls for startup-config
# and running-config.  The module also contains some functions useful for
# implementing file-related Cli commands.

import errno
import os
import shutil
import sys

import CliExtensions
import CliCommand
import CliCommon
import CliParser
import Logging
import PyClient
import Tac
import TacSigint
import Tracing
import Url

t0 = Tracing.trace0

DIFF_MAX_FILE_SIZE = 1024 * 1024 * 8

SYS_FILE_COPY_ERROR = Logging.LogHandle( "SYS_FILE_COPY_ERROR",
      severity=Logging.logError,
      fmt="An error occurred when copying from %s to %s (%s).",
      explanation="An error occurred when copying a file in the file system.",
      recommendedAction="This may indicate a serious hardware or "
                        "software problem. Please check free space and integrity of "
                        "the file system. If the problem persists, please contact "
                        "technical support." )

def logLocalFileCopyError( surl, durl, error ):
   # We only want to log serious errors
   allowedErrno = [ errno.EFAULT, errno.EFBIG, errno.ELOOP,
                       errno.EMFILE, errno.ENOMEM, errno.ENOSPC,
                       errno.EOVERFLOW, errno.EROFS, errno.EBADF, errno.EIO ]

   # We cannot compare the types of filesystem of urls because doing so
   # would cause cyclic depencies between Cli and FileCli;
   # localFilename() should be a good way to go, as it should return None if
   # there is no local file that correspond to the url.
   fsIsLocal = surl.localFilename() or durl.localFilename()
   if fsIsLocal and ( error.errno in allowedErrno ):
      Logging.log( SYS_FILE_COPY_ERROR, surl.url, durl.url, error.strerror )

def checkUrl( url ):
   """Checks a Url to make sure it isn't the special object indicating that a
   relative Url was entered but the current filesystem no longer exists.  If it is,
   prints an error message and aborts the current command."""
   if url == Url.currentFilesystemNoLongerExists:
      print( "% Error: current filesystem no longer exists" )
      raise CliParser.AlreadyHandledError

def showFile( mode, url ):
   """A Cli value function that prints the contents of a file to stdout."""
   checkUrl( url )
   bufferSize = 1024

   try:
      f = url.open()
      if hasattr( f, 'name' ):
         if f.name.endswith( ".gz" ):
            f.close()
            import gzip # pylint: disable=import-outside-toplevel
            f = gzip.GzipFile( f.name, f.mode )
         elif f.name.endswith( ".bz2" ):
            f.close()
            import bz2 # pylint: disable=import-outside-toplevel
            f = bz2.BZ2File( f.name, f.mode )
      try:
         data = f.read( bufferSize )
         while data:
            sys.stdout.write(
                  data if isinstance( data, str ) else data.decode( 'utf-8' ) )
            data = f.read( bufferSize )
            TacSigint.check()
      finally:
         f.close()
   except EOFError:
      mode.addError( "Error displaying %s (End of file)" % ( url.url ) )
   except OSError as e:
      # don't commit a double fault (sometimes that kills the cli session)
      if e.errno != errno.EPIPE:
         mode.addError( f"Error displaying {url.url} ({e.strerror})" )
   except PyClient.RpcError:
      mode.addError( "Error displaying %s" % ( url.url ) )
      raise
   except ValueError:
      mode.addError( "Error displaying %s" % ( url.url ) )


# copyFileCheckers extension is called before the copy is performed.
# It returns a boolean to indicate if the copy is allowed.
#
# copyFileNotifiers extension is called after the copy succeeds.
#
# Both are called by:
#
#      callback( mode, surl, durl, commandSource )

copyFileCheckers = CliExtensions.CliHook()
copyFileNotifiers = CliExtensions.CliHook()

def copyFile( cliStatus, mode, surl, durl, commandSource="commandLine",
              ignoreENOENT=False ):
   checkUrl( surl )
   checkUrl( durl )

   copyStagingDirPath = None
   try:
      if surl == durl:
         raise OSError( 0, "Source and destination are the same file" )

      if durl.isdir():
         d = durl.child( surl.basename() )
      else:
         d = durl

      for checkFunc in copyFileCheckers.extensions():
         if not checkFunc( mode, surl, durl, commandSource ):
            return "not allowed by hook"

      # now generate extra files
      if durl.copyHook and durl.copyHook.filter( mode, surl, durl ):
         copyDirPath = os.path.join( durl.fs.location_, durl.copyHook.path )
         # create a staging sub dir
         copyStagingDirPath = durl.tempfilename( copyDirPath )
         os.mkdir( copyStagingDirPath )
         for copyFunc in durl.copyHook.hook.extensions():
            copyFunc( mode, surl, durl, commandSource,
                      copyStagingDirPath )
         # fsync all files
         for configFile in os.listdir( copyStagingDirPath ):
            Url.syncfile( os.path.join( copyStagingDirPath, configFile ) )

      d.copyfrom( surl )

      # Now move the staging dir to final destination.
      # It's not expected to fail.
      if copyStagingDirPath:
         # remove the target directory
         shutil.rmtree( copyDirPath, ignore_errors=True )
         os.rename( copyStagingDirPath, copyDirPath )
         # fsync the dir
         Url.syncfile( copyDirPath )
         # fsync the parent dir
         Url.syncfile( os.path.dirname( copyDirPath ) )

      for n in copyFileNotifiers.extensions():
         n( mode, surl, durl, commandSource )

      if commandSource == "commandLine":
         mode.addMessage( "Copy completed successfully." )
   except CliCommon.LoadConfigError as e:
      return e.msg
   except OSError as e:
      if copyStagingDirPath and os.path.isdir( copyStagingDirPath ):
         shutil.rmtree( copyStagingDirPath, ignore_errors=True )
         Url.syncfile( os.path.dirname( copyStagingDirPath ) )

      if not ( ignoreENOENT and e.errno == errno.ENOENT ):
         mode.addError( "Error copying {} to {} ({})".format(
            surl.url, durl.url, e.strerror ) )
         logLocalFileCopyError( surl, durl, e )
         # make sure the strerror is not empty in case the URL plugin
         # raises an EnvironmentError with empty string.
         return e.strerror or "Unknown error"

   # success
   return ""

urlExpressions_ = []

class UrlExpressionFactory( CliCommand.OrExpressionFactory ):
   def __init__( self, description, fsFunc, notAllowed, allowAllPaths=False ):
      CliCommand.OrExpressionFactory.__init__( self )
      self.urlMatcher_ = Url.UrlMatcher( fsFunc, description,
                                         notAllowed=notAllowed,
                                         allowAllPaths=allowAllPaths )
      self |= ( "BASE", self.urlMatcher_ )

   def registerExtension( self, name, expr, token ):
      self |= ( name, expr )
      if token:
         self.urlMatcher_.notAllowed_.append( token )

def registerUrlExpression( description, fsFunc, notAllowed, allowAllPaths=False ):
   t0( "registerUrlExpression", description )
   expr = UrlExpressionFactory( description, fsFunc, notAllowed, allowAllPaths )
   urlExpressions_.append( expr )
   return expr

def registerUrlExprExtension( exprName, expr ):
   # Url rule exteions can only be registered after registerUrlRule()
   # therefore, check Url rules have been registerd first
   assert urlExpressions_
   for r in urlExpressions_:
      r |= ( exprName, expr )

copySourceUrlExpr_ = registerUrlExpression( 'Source file name',
                                            lambda fs: fs.supportsRead(),
                                            notAllowed=[], allowAllPaths=True )
copyDestUrlExpr_ = registerUrlExpression( 'Destination file name',
                                          lambda fs: fs.supportsWrite(),
                                          notAllowed=[], allowAllPaths=True )

def registerCopySource( name, expr, token=None ):
   """Register a CliParser rule that will be included in the rule that specifies
   source URLs for the copy command.  If token is not None it will be included
   in the set of tokens that UrlMatcher will not match."""
   copySourceUrlExpr_.registerExtension( name, expr, token )

def registerCopyDestination( name, expr, token=None ):
   """Add a CliParser rule that matches destination URLs for the copy command.
   The given token will be included in the set of tokens that UrlMatcher
   will not match.
   """
   # py lint: disable=protected-access
   copyDestUrlExpr_.registerExtension( name, expr, token )
   # py lint: enable=protected-access

def copySourceUrlExpr():
   return copySourceUrlExpr_

def copyDestinationUrlExpr():
   return copyDestUrlExpr_

def _firstLineIsHeader( url, filename ):
   try:
      with open( filename ) as f:
         line = f.readline( 1024 )
         return url.isHeader( line )
   except UnicodeDecodeError:
      return False
   except OSError as e:
      if e.errno == errno.ENOENT:
         return False
      raise

def _diffFileParam( url, filename ):
   if _firstLineIsHeader( url, filename ):
      # skip the first line
      return "<( tail -n +2 %s 2>/dev/null )" % filename
   else:
      return filename

def diffFile( mode, aUrl, bUrl ):
   checkUrl( aUrl )
   checkUrl( bUrl )

   aFilename = bFilename = None

   try:
      aFilename = aUrl.makeLocalFile( ignoreENOENT=True )
      bFilename = bUrl.makeLocalFile( ignoreENOENT=True )

      # if none of the files exist, just return
      if ( not os.path.exists( aFilename ) and
           not os.path.exists( bFilename ) ):
         return

      aParam = _diffFileParam( aUrl, aFilename )
      bParam = _diffFileParam( bUrl, bFilename )

      # run diff on the two files
      #
      # -N: treat one non-existent file as empty
      # -Z: ignore trailing whitespaces
      # -b: ignore change in the amount of whitespaces
      if ( aUrl.ignoreTrailingWhitespaceInDiff() and
           bUrl.ignoreTrailingWhitespaceInDiff() ):
         cmdOpts = "Zb"
      else:
         cmdOpts = ""

      cmd = "diff -uN{} --label {} --label {} {} {}".format( cmdOpts,
                                                             aUrl.url,
                                                             bUrl.url,
                                                             aParam,
                                                             bParam )
      Tac.run( [ 'bash', '-c', cmd ], ignoreReturnCode=True )
   except OSError as e:
      mode.addError( "Error comparing {} to {} ({})".format(
         aUrl.url, bUrl.url, e.strerror ) )
   finally:
      aUrl.cleanupLocalFile( aFilename )
      bUrl.cleanupLocalFile( bFilename )

diffFirstUrlExpr_ = registerUrlExpression( 'First file path',
                                           lambda fs: fs.supportsRead(),
                                           notAllowed=[], allowAllPaths=True )
diffSecondUrlExpr_ = registerUrlExpression( 'Second file path',
                                            lambda fs: fs.supportsRead(),
                                            notAllowed=[], allowAllPaths=True )

def diffFirstUrlExpr():
   return diffFirstUrlExpr_

def diffSecondUrlExpr():
   return diffSecondUrlExpr_

def sizeFmt( num ):
   for x in [ 'bytes', 'KB', 'MB', 'GB', 'TB', 'PB' ]:
      if num < 1024.0:
         return f"{num:3.1f} {x}"
      num /= 1024.0
   return None

class PersistentConfig:
   def __init__( self ):
      self.info_ = {}

   def _register( self, keyword, handler, description, default ):
      self.info_[ keyword ] = ( handler, description, default )

   def info( self ):
      return self.info_

   def cleanup( self, mode, delete, preserve ):
      for keyword, ( handler, _description, default ) in self.info_.items():
         try:
            handler( mode,
                     bool( ( delete and keyword in delete ) or
                           ( ( ( preserve is None ) or
                             ( keyword not in preserve ) ) and
                            default ) ) )
         # pylint: disable-msg=W0703
         except Exception as e:
            mode.addError( f"Error erasing {keyword} configuration: {e}" )

__persistentConfigInfo__ = PersistentConfig()

def registerPersistentConfiguration( keyword, handler, description,
                                     default=True ):
   """Register the existence of persistent configuration info so that the
   'write erase' command is aware of it, and can be able to delete it.
   Provide a keyword that will be used in the Cli command, a description
   string that will be part of the help string for the keyword, specify
   whether it will be erased as part of 'write erase' by default, or not, and
   finally provide a handler function that takes 2 args: mode, and a
   boolean arg, specifying whether to delete (True) or preserve (False)
   the configuration"""
   # pylint: disable-msg=W0212
   __persistentConfigInfo__._register( keyword, handler, description, default )

def chunkedHashCompute( url, hashInitializer ):
   """Computes the hash of a file pointed to by url without reading the whole file
   at once into memory (does it in chunks)."""
   chunkSize = 16384
   f = url.open( 'rb' )
   try:
      hashFunc = hashInitializer()
      data = f.read( chunkSize )
      while data:
         hashFunc.update( data )
         data = f.read( chunkSize )
   finally:
      f.close()
   return hashFunc.hexdigest()
