#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import ReloadConsts
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "AcpiBert" )
t0 = Tracing.trace0
syncFloodSignature = b"SYNCFLOOD_RESET"
bertDataPath = "/sys/firmware/acpi/tables/data/BERT"

def Plugin( context ):
   '''
   BERT plugin that processes the ACPI BERT data.
   We propagate the reload cause info if a BERT error is detected.
   '''

   # No BERT data info available
   if not os.path.isfile( bertDataPath ):
      return

   try:
      with open( bertDataPath, mode="rb" ) as f:
         bertData = f.read()
   except OSError as err:
      t0( f"Unable to read BERT data file: {err.strerror}" )
      return

   # Empty BERT data, this means no Boot Error Region. BERT data should be bigger
   # than 4 bytes, we need at least 4 bytes to check the Boot Error Region. If data
   # is smaller (because of some BIOS bug?) just ignore it.
   if len( bertData ) < 4:
      return

   # No error reported in the Boot Error Region (Block Status == 0)
   if all( b == 0 for b in iter( bertData[ 0 : 4 ] ) ):
      return

   ReloadCauseConstants = Tac.Type( "ReloadCause::ReloadCauseConstants" )
   RCC = ReloadCauseConstants()

   # A syncflood reset is signaled on merlinfalcon CPU with Aboot9 with a
   # "SYNCFLOOD_RESET" signature in the Processor Generic Error Section
   if syncFloodSignature in bertData:
      RCC.writeLocalReloadCause( ReloadConsts.cpuSyncFloodDescription,
                                 bertDataPath, False )
   else:
      RCC.writeLocalReloadCause( ReloadConsts.cpuMachineCheckDescription,
                                 bertDataPath, False )
