# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Tracing
import ReloadConsts
import os

__defaultTraceHandle__ = Tracing.Handle( "Fru.AgentConditions" )
t0 = Tracing.trace0 
t3 = Tracing.trace3

thermostatStatusReactor = None
em = None

class ThermostatStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Environment::Thermostat::Status"
   temperatureOkPath = 'temperatureOk'
   thermostatCookieFile = "/mnt/flash/debug/criticalTempAction"

   def __init__( self ):
      self.envThermostatConfig = None
      self.agentStartConditionsPath = em.lookup( 'sys/agentStartConditions' )
      self.envThermostatStatus = em.lookup( 'environment/thermostat/status' )
      self.agentConditionsProcessed = False
      self.setupThermostatDone = False
      self.isRebootOnOverheat = self.isRebootDueToOverheat()
      self.isRebootOnAsu = self.isAsuReboot()
      Tac.Notifiee.__init__( self, self.envThermostatStatus )
      self.handleInletTemperature()

   def checkAgentConditionPath( self, conditionPath="" ):
      try:
         return self.agentStartConditionsPath[ conditionPath ]
      except KeyError:
         return False

   def isAsuReboot( self ):
      asuReboot = False
      try:
         # pylint: disable-next=consider-using-with
         cmdline = open( '/proc/cmdline' ).read()
         asuReboot = ( "arista.asu_reboot" in cmdline or
                       "arista.asu_hitless" in cmdline )
      except OSError:
         pass
      t3( "AgentConditions plugin found isAsuReboot %s" % asuReboot )
      return asuReboot

   def isRebootDueToOverheat( self ):
      if "SIMULATION_VMID" not in os.environ:
         found = os.system( "grep %r %s" %
                            ( ReloadConsts.overTempPowercycleDescription,
                            self.thermostatCookieFile ) )
         self.isRebootOnOverheat = not found # Misleading. Ret code arithmetic.
      else:
         self.isRebootOnOverheat = False
      t0( 'AgentConditions plugin found isRebootonOverheat %s' %
          self.isRebootOnOverheat )
      return self.isRebootOnOverheat
      
   def createAgentConditionPath( self, conditionPath="" ):
      self.agentStartConditionsPath.mkdir( conditionPath )
      t0( 'conditionPath %r created.' % conditionPath )
      if "SIMULATION_VMID" not in os.environ:
         os.system( "rm %s" % self.thermostatCookieFile )

   def setupThermostatEnvironment( self  ):
      self.envThermostatConfig = em.lookup( 'environment/thermostat/config' )
      return True

   @Tac.handler( 'inletTemperature' )
   def handleInletTemperature( self ):
      actionOnOverheat = Tac.Type( "Environment::Thermostat::ActionOnOverheat" )
      if self.agentConditionsProcessed:
         t3( 'agentConditions are already processed so returning ' )
         return
      if self.checkAgentConditionPath( self.temperatureOkPath ):
         # This can happen when Fru restarts
         t3( 'conditionPath %r exists, just returning.' % self.temperatureOkPath )
         self.agentConditionsProcessed = True
         return
      if not self.setupThermostatDone:
         self.setupThermostatDone = self.setupThermostatEnvironment()
      if self.isRebootOnAsu:
         t0( "System rebooted as ASU Boot, creating a conditionPath" )
         self.createAgentConditionPath( self.temperatureOkPath )
         self.agentConditionsProcessed = True
      elif self.envThermostatConfig.actionOnOverheat != \
            actionOnOverheat.actionPowercycle:
         t0( "power-cycle on overheat is not configured, creating a conditionPath" )
         self.createAgentConditionPath( self.temperatureOkPath )
         self.agentConditionsProcessed = True
      elif ( self.envThermostatConfig.recoveryModeOnOverheat !=
            'recoveryModeRestricted' ):
         t0( "recoveryMode is not restricted, creating a conditionPath" )
         self.createAgentConditionPath( self.temperatureOkPath )
         self.agentConditionsProcessed = True
      elif not self.isRebootOnOverheat:
         t0( "reboot is not due to overheat, creating a conditionPath" )
         self.createAgentConditionPath( self.temperatureOkPath )
         self.agentConditionsProcessed = True
      elif str( self.envThermostatStatus.inletTemperature ) == '-inf':
         t3( 'inletTemperature is not set so returning' )
         return
      elif ( self.envThermostatStatus.inletTemperature < 
             self.envThermostatConfig.ambientThreshold ):
         # no need to check for reboot reason
         t0( "inletTemperature is below threshold, creating a conditionPath" )
         self.createAgentConditionPath( self.temperatureOkPath )
         self.agentConditionsProcessed = True

def setupAgentConditionsReactors():
   global thermostatStatusReactor
   thermostatStatusReactor = ThermostatStatusReactor()

def Plugin( ctx ):
   global em

   em = ctx.entityManager
   mountGroup = em.mountGroup()
   mountGroup.mount( 'sys/agentStartConditions', 'Tac::Dir', 'w' )
   mountGroup.mount( 'environment/thermostat/status',
                     'Environment::Thermostat::Status', 'r' )
   mountGroup.mount( 'environment/thermostat/config',
                     'Environment::Thermostat::Config', 'r' )

   mountGroup.close( setupAgentConditionsReactors )
