#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import DesiredTracing
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'CliTuningSm' )
DesiredTracing.desiredTracingIs( 'CliTuningSm/13' )

t1 = Tracing.trace1
t3 = Tracing.trace3
t7 = Tracing.trace7

RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )
TopologyGenerationWatcher = TacLazyType(
   'Hardware::L1Topology::TopologyGenerationWatcher' )
topoGenWatcher = None
topoGenStatus = None

class RedundancyStatusReactor( Tac.Notifiee ):
   '''
   This Sm is responsible for listening to the RedundancyStatus of the system, and
   calling into the parent sm to try and start the CliTuningSm when it
   changes.
   '''

   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, starter, redundancyStatus ):
      super().__init__( redundancyStatus )
      self.starter = starter

   @Tac.handler( 'mode' )
   def handleRedundancyMode( self ):
      t7( 'Handling redundancy mode' )  
      self.starter.manageTuningSm()

class TopoGenStatusReactor( Tac.Notifiee ):
   '''This Sm is responsible for listening to the TopologyGenerationStatus'''

   notifierTypeName = 'Hardware::L1Topology::TopologyGenerationStatus'

   def __init__( self, starter, generationStatus ):
      super().__init__( generationStatus )
      self.starter = starter

   @Tac.handler( 'topologyGeneration' )
   def handleTopoGenChange( self ):
      t7( 'Handling generation change' ) 
      self.starter.manageTuningSm()
   
class CliTuningSmStarter:

   def __init__( self, redundancyStatus, tuningDir,
                 phyFeatureConfigDir, *args ):
      self.redundancyStatus = redundancyStatus
      self.tuningDir = tuningDir
      self.phyFeatureConfigDir = phyFeatureConfigDir
      self.topoDir = args[ 0 ]
      self.phyTxEqProfileDir = args[ 1 ]
      self.topoGenStatus = args [ 2 ]
      self.fruTuningHelper = Tac.newInstance(
         'Hardware::L1Topology::TuningFruPluginHelper', tuningDir )
      self.cliTuningGeneration = tuningDir.newEntity(
         'Hardware::L1Topology::CliTuningGeneration', 'cliGen' )
      
      self.sliceDir = self.topoDir[ 'slice' ]
      self.cliTuningSm = {}
      # Setup the reactors for later triggers, and attempt to immediately manage the
      # TuningSm as no reactors will fire if the conditions are already met.
      self.reactorList = [
         RedundancyStatusReactor( self, redundancyStatus ),
         TopoGenStatusReactor( self, topoGenStatus ),
      ]
      self.manageTuningSm()
   
   def manageTuningSm( self ):
      # only care about FixedSystem and Linecards and exclude Xcvr adapters
      filteredSliceDir = [ sliceId for sliceId in self.sliceDir if 
                           sliceId.startswith( ( 'FixedSystem', 'Linecard' ) ) 
                           and 'Xcvr' not in sliceId ]
      self.cleanupTuningSm( filteredSliceDir )

      # We are the on the active supervisor
      if self.redundancyStatus.mode == RedundancyMode.active and \
            topoGenStatus.topologyGeneration.valid:
         self.startAllTuningSms( filteredSliceDir )
      
   def startAllTuningSms( self, sliceDir ):
      for sliceId in sliceDir:
         self.startTuningSm( sliceId )
         
   def startTuningSm( self, sliceId ):
      if sliceId in self.cliTuningSm:
         t7( f'CliTuningSm is already running for { sliceId }' )
         return

      t1( f'Starting the CliTuningSm for { sliceId }' )
      self.cliTuningSm[ sliceId ] = Tac.newInstance(
         'Hardware::L1Topology::CliTuningSm', 
         sliceId, self.topoDir, self.phyFeatureConfigDir[ sliceId ],
         self.phyTxEqProfileDir, self.topoGenStatus, self.fruTuningHelper,
         self.cliTuningGeneration )

   def cleanupTuningSm( self, sliceDir ):
      t7( 'cleaning TuningSms' )
      tuningSmsSet = set( self.cliTuningSm.keys() )
      slicesRemoved = tuningSmsSet.difference( set( sliceDir ) )
      for sliceId in slicesRemoved:
         t7( f'Stopping TuningSm for { sliceId }' )
         del self.cliTuningSm[ sliceId ]

tuningSmStarter = None
def Plugin( context ):
   '''Mount the neccessary Sysdb paths and load the TuningSm.'''
   global topoGenStatus
   mg = context.entityManager.mountGroup()
   
   # Paths required for the CliTuningSmStarter
   redundancyStatus = mg.mountPath( Cell.path( 'redundancy/status' ) )
   tuningDir = mg.mountPath( 'hardware/l1/tuning' )

   # Paths requird for TuningSm
   topoDir = mg.mountPath( 'hardware/l1/topology' )
   phyFeatureConfigDir = mg.mountPath(
      'hardware/archer/phy/config/cli/feature/slice' )
   phyTxEqProfileDir = mg.mountPath(
      'hardware/archer/phy/config/cli/phyTxEqProfiles' )
   topoGenStatus = Tac.newInstance(
      'Hardware::L1Topology::TopologyGenerationStatus' )
   
   def mountsCompleteHook():
      global topoGenWatcher, tuningSmStarter
      topoGenWatcher = TopologyGenerationWatcher( topoDir, topoGenStatus )
      tuningSmStarter = CliTuningSmStarter(
         redundancyStatus,
         tuningDir,
         phyFeatureConfigDir,
         topoDir,
         phyTxEqProfileDir,
         topoGenStatus,
      )

   mg.close( mountsCompleteHook )
