#!/usr/bin/env python3
# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CEosHelper
import EntityMib
from EntityMib import IndexAllocator
import Fru
from FruPlugin import TempSensorFru
from Toggles.SwagCoreToggleLib import toggleMultiSwagBringupEnabled

class CpuComplexDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::CpuComplex"
   managedApiRe = ".*$"

   # XXX_APECH This seems kind of hacky to have the cpu complex driver
   # require environmentInit. The problem is that if any components
   # downstream of the cpu complex requires an init step, then
   # the cpu complex driver has to require it in order for ordering
   # to actually work.
   requires = [ Fru.FruDriver.systemInit, Fru.FruDriver.environmentInit ]

   def __init__( self, cpuComplex, parentMibEntity, parentDriver, driverCtx ):
      driverCtx = driverCtx.clone()
      Fru.FruDriver.__init__( self, cpuComplex, parentMibEntity,
                              parentDriver, driverCtx )

      # mounting stuff for temperature sensor XXX_DSEBASTIAN
      cell = Fru.fruBase( cpuComplex ).managingCellId
      # TODO: BUG 998464: remove after CpuComplex is swagified
      # We hardcode cellId = 3 in bi-swag where we assume the swag worker has a
      # member ID of 3. It has no cellId because its fruBase is a non-supe card
      if toggleMultiSwagBringupEnabled() and not cell:
         cell = Fru.fruBase( cpuComplex ).smid
      hwCell = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ str( cell ) ]
      hwCcDir = hwCell.mkdir( "cpuComplex" )

      # Create CpuComplexConfig
      hwCcConfig = hwCcDir.newEntity( "Hardware::CpuComplexConfig",
                                      "config" )

      if cpuComplex.cpu:
         self.instantiateChildDrivers( cpuComplex.cpu,
                                       parentMibEntity,
                                       skipRoot=False )
         hwCcConfig.tccActivationTemp = cpuComplex.cpu.tccActivationTemp

      # Is this the default
      if cpuComplex.southBridge:
         self.instantiateChildDrivers( cpuComplex.southBridge,
                                       parentMibEntity,
                                       skipRoot=False )
      if cpuComplex.tempSensor:
         # Setup temp sensors. Assign localId
         localId = 0
         self.envTempDir_ = driverCtx.sysdbRoot[ 'environment' ][ 'temperature' ]

         for invSensor in cpuComplex.tempSensor.values():
            sensorName = EntityMib.componentNameFromParent(
               parentMibEntity, "TempSensor", invSensor.sensorId )
            envConfig = TempSensorFru.createSensorEnvConfig( self.envTempDir_,
                                                       sensorName,
                                                       invSensor )

            Fru.Dep( hwCcConfig.tempSensor, invSensor ).newMember(
                  localId,
                  invSensor.sensorId,
                  envConfig,
                  invSensor.cpuId,
                  CEosHelper.devFolderPath() )
            hwSensor =  hwCcConfig.tempSensor[ invSensor.cpuId ]
            hwSensor.offset = invSensor.offset
            localId += 1   
            # Entity Mib registration
            sensorMib = self.parentMibEntity_.sensor.get( invSensor.sensorId )
            if sensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( self.parentMibEntity_.sensor, invSensor.sensorId )
               sensorMib = Fru.Dep( self.parentMibEntity_.sensor, invSensor ). \
                           newMember( physicalIndex, invSensor.sensorId,
                           "TempSensor" )
            sensorMib.label = str( invSensor.sensorId )
            # Copy description from Config
            sensorMib.description = invSensor.description
            EntityMib.populateMib( sensorMib, invSensor )
            sensorMib.groupName = invSensor.groupName
            # Must be very last
            sensorMib.initStatus = "ok"

class SouthBridgeDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::SouthBridge"
   managedApiRe = ".*$"

   def __init__( self, pch, parentMibEntity, parentDriver, driverCtx ):
      driverCtx = driverCtx.clone()
      Fru.FruDriver.__init__( self, pch, parentMibEntity,
                              parentDriver, driverCtx )
      self.instantiateChildDrivers( pch, parentMibEntity, skipRoot=True )

class CpuDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::Cpu"
   managedApiRe = ".*$"

   def __init__( self, cpu, parentMibEntity, parentDriver, driverCtx ):
      driverCtx = driverCtx.clone()
      Fru.FruDriver.__init__( self, cpu, parentMibEntity,
                              parentDriver, driverCtx )

def Plugin( context ):
   context.registerDriver( CpuComplexDriver )
   context.registerDriver( SouthBridgeDriver )
   context.registerDriver( CpuDriver )
