# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac, Logging, Fru, Fdl, EntityMib
import FruPlugin.Smbus, FruPlugin.TempSensorFru, FruPlugin.Gpio
import sys
from EntityMib import IndexAllocator, formatEntityMibName
from FruPlugin.Health import registerHealthSource
from Toggles.AristaSAIToggleLib import toggleAristaSAIModeEnabled
import EntityMibUtils
import Tracing
__defaultTraceHandle__ = Tracing.Handle( "Fru.Fan" )
t0 = Tracing.trace0 
t3 = Tracing.trace3

# Log message definitions

FRU_FAN_INSERTED = Logging.LogHandle(
              "FRU_FAN_INSERTED",
              severity=Logging.logInfo,
              fmt="%s has been inserted. model: %s",
              explanation="The specified fan has been inserted.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

FRU_FAN_REMOVED = Logging.LogHandle(
              "FRU_FAN_REMOVED",
              severity=Logging.logInfo,
              fmt="%s has been removed. model: %s",
              explanation="The specified fan has been removed or is not properly "
                          "seated.",
              recommendedAction="Reinsert the fan, or check that the fan is "
                                "properly inserted." )

FRU_FAN_UNSUPPORTED = Logging.LogHandle(
              "FRU_FAN_UNSUPPORTED",
              severity=Logging.logError,
              fmt="%s is unsupported",
              explanation="The specified is not supported by the running software.",
              recommendedAction=("Check that the version of software you are "
                                 "running supports the specified fan type. Upgrade "
                                 "software or replace the fan.") )

# Fru driver initialization steps to sort things properly among the
# various fan-related drivers
supportedFansInit = "supportedFansInit"

def copyFanProperties( coolingConfig, fp, invertFanDirection=False ):
   # For products that use the same fan for both the chassis' front and
   # back, we need to invert the direction on one of those orientations.
   if invertFanDirection:
      if fp.name[ -1 ] == "F":
         invertedName = fp.name[ :-1 ] + "R"
      elif fp.name[ -1 ] == "R":
         invertedName = fp.name[ :-1 ] + "F"
      else:
         invertedName = fp.name
      envFanProperties = coolingConfig.newSupportedFanProperties( invertedName )
      tacAirFlowDirection = Tac.Type(
         "Environment::Cooling::AirflowDirection" )
      if fp.airflowDirection == tacAirFlowDirection.frontToBackAirflow:
         envFanProperties.airflowDirection = tacAirFlowDirection.backToFrontAirflow
      elif fp.airflowDirection == tacAirFlowDirection.backToFrontAirflow:
         envFanProperties.airflowDirection = tacAirFlowDirection.frontToBackAirflow
      else:
         envFanProperties.airflowDirection = fp.airflowDirection
   else:
      envFanProperties = coolingConfig.newSupportedFanProperties( fp.name )
      envFanProperties.airflowDirection = fp.airflowDirection
   envFanProperties.maxRpm = fp.maxRpm
   envFanProperties.minRpm = fp.minRpm
   envFanProperties.tachPulsesPerRevolution = fp.tachPulsesPerRevolution
   for ( i, maxRpm ) in fp.alternateMaxRpm.items():
      envFanProperties.alternateMaxRpm[ i ] = maxRpm
   return envFanProperties

class SupportedSystemFansDriver( Fru.FruDriver ):

   provides = [ supportedFansInit ]

   managedTypeName = "Inventory::SupportedSystemFans"
   managedApiRe = "$"

   def __init__( self, supportedSystemFans, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, supportedSystemFans,
                              parentMibEntity, parentDriver, ctx )

      coolingConfig = ctx.sysdbRoot[ 'environment' ][ 'cooling' ][ 'config' ]
      hwFanConfig  = ctx.sysdbRoot[ 'hardware' ][ 'fan' ][ 'config' ]

      def _createSupportedFanTray( invFanTray ):
         supportedFanTray = hwFanConfig.newSupportedFanTray( invFanTray.name )
         supportedFanTray.modelName = invFanTray.modelName
         supportedFanTray.fanSupportedInSystem = invFanTray.fanSupportedInSystem
         supportedFanTray.fanOutOfDateInSystem = invFanTray.fanOutOfDateInSystem
         for ( label, fan ) in invFanTray.fan.items():
            envFanProperties = copyFanProperties( coolingConfig,
                                                  fan.physicalProperties )
            supportedFanTray.fan[ label ] = envFanProperties
         return supportedFanTray
         

      if supportedSystemFans.defaultFanTray:
         supportedFanTray = _createSupportedFanTray(
            supportedSystemFans.defaultFanTray )
         hwFanConfig.defaultFanTray = supportedFanTray
      for ( fanId, invFanTray ) in supportedSystemFans.fanIdToFanTray.items():
         supportedFanTray = _createSupportedFanTray( invFanTray )
         hwFanConfig.fanIdToFanTray[ fanId ] = supportedFanTray

class FanTrayDriver( Fru.FruDriver ):
   
   provides = [ supportedFansInit ]
   requires = [ FruPlugin.Gpio.gpioInit, Fru.FruDriver.environmentInit ]

   managedTypeName = "Inventory::FanTray"
   managedApiRe = ".*$"

   def __init__( self, invFanTray, fanTraySlotMib, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invFanTray, fanTraySlotMib, parentDriver, ctx )

      coolingConfig = ctx.sysdbRoot[ 'environment' ][ 'cooling' ][ 'config' ]
      entmib = ctx.entity( 'hardware/entmib' ) # pylint: disable=unused-variable

      # Create the fan tray entity mib object
      if fanTraySlotMib.fanTray is None:
         physicalIndex = IndexAllocator.physicalIndex( fanTraySlotMib,
                                                       "FanTray",
                                                       fanTraySlotMib.relPos )
         Fru.Dep( fanTraySlotMib, invFanTray ).fanTray = (
            physicalIndex, fanTraySlotMib.relPos, "FanTray" )
      fanTraySlotMib.fanTray.description = \
          'Fan Tray %s' % fanTraySlotMib.relPos
      fanTraySlotMib.secondarySerialNum = invFanTray.secondarySerialNum
      if invFanTray.parent.detectionType == "dmIntegrated":
         fanTraySlotMib.fanTray.swappability = "notSwappable"
      else:
         fanTraySlotMib.fanTray.swappability = "hotSwappable"
      fanTrayMib = fanTraySlotMib.fanTray
      EntityMib.populateMib( fanTrayMib, invFanTray )

      fanTrayMib.numFans = len( invFanTray.fan )

      for invFan in invFanTray.fan.values():
         if invFan.physicalProperties:
            properties = copyFanProperties( coolingConfig,
                                            invFan.physicalProperties )
         else:
            # Unsupported fan
            properties = None

         Fru.newDependentSet( invFan )

         # Create the fan entity mib object
         fanMib = fanTrayMib.fan.get( invFan.label )
         if fanMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( fanTrayMib.fan, invFan.label )
            fanMib = fanTrayMib.newFan( physicalIndex, invFan.label, "Fan" )
         fanMib.description = '%s Fan %s' % (
            fanTrayMib.description, str(invFan.label) )
         fanName = EntityMib.componentName( fanMib )
         fanMib.fanName = fanName
         fanMib.secondary = invFan.secondary

         fanSensorMib = fanMib.sensor.get( invFan.label )
         if fanSensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( fanMib.sensor, invFan.label )
            fanSensorMib = fanMib.sensor.newMember \
                           ( physicalIndex, invFan.label, "FanSensor" )
         fanSensorMib.description = "%s Sensor %s" % (
            fanMib.description, str( invFan.label ) )

         registerHealthSource( invFan, fanName )
         fanTrayName = EntityMib.componentName( fanTraySlotMib )

         # Create the associated Environment::FanConfig object
         fanEnvConfig = Fru.Dep( coolingConfig.fan, invFan ).newMember( fanName )
         fanEnvConfig.properties = properties
         fanEnvConfig.fanTrayName = fanTrayName
         fanEnvConfig.generationId = Fru.powerGenerationId( invFan )

         coolingDomainName = getattr( invFan.fanInterface, 'coolingDomainName', '' )
         if coolingDomainName:
            fanEnvConfig.coolingDomainName = coolingDomainName

         if invFan.controllerVars:
            fanEnvConfig.prop = invFan.controllerVars.p
            fanEnvConfig.integ = invFan.controllerVars.i
            fanEnvConfig.deriv = invFan.controllerVars.d
            fanEnvConfig.pband = invFan.controllerVars.pband
            fanEnvConfig.pidOverride = True

         Fru.Dep( coolingConfig.fanPtr, invFan ).addMember( fanEnvConfig )

         # BUG730815 EntityMib::Fan should be deleted before deletion of
         # Environment::Cooling::FanConfig from Environment::Cooling::Config.fan[Ptr]
         Fru.Dep( fanTrayMib.fan, invFan ).addDep( fanMib )

         # BUG2898 this should really be done by the fan controller agent
         if invFan.fanInterface.enableSecondaryFanTachGpo:
            gpo = invFan.fanInterface.enableSecondaryFanTachGpo.fruGpo
            assert gpo
            if invFan.secondaryFanPhysicalProperties:
               # Fan has a secondary fan, enable tach reading
               gpo.asserted = True
            else:
               # Fan has no secondary fan, disable tach reading
               gpo.asserted = False
            # Dual fans was used in some Napa fixed systems, but treated as 
            # single fans because we didn't use the Tach of the secondary fan.

         if properties:
            # Create the associated Hardware::FanController::FanConfig object if
            # the fan is supported. We leave unsupported fans alone, as we can't
            # even be sure if they are blowing air in the right direction
            fanInterface = invFan.fanInterface

            ecbFanOnName = ''
            invEcbFanOnGpio = invFanTray.parent.ecbFanOn
            if invEcbFanOnGpio is not None:
               ecbFanOnName = invEcbFanOnGpio.systemName
               assert ecbFanOnName

            fanConfig = \
               Fru.Dep( fanInterface.controllerInterface.hwFanController.fanConfig,
                        invFan ).newMember( fanName, fanInterface.id,
                                            fanInterface.configGroupId,
                                            properties.maxRpm,
                                            properties.tachPulsesPerRevolution,
                                            fanEnvConfig,
                                            fanInterface.dualFan,
                                            fanInterface.inletFanTachId,
                                            fanInterface.outletFanTachId,
                                            ecbFanOnName,
                                            fanInterface.pwmBaseOffset,
                                            fanInterface.tachBaseOffset,
                                            fanInterface.fanOffset,
                                            invFan.ignoreCheckingFanFaultReg )
            fanConfig.disableSpeedControl = fanInterface.disableSpeedControl
            if fanInterface.dualFan:
               # For dual fan, verify fan failure by checking fan speed in range
               # pylint: disable-next=unused-variable
               for pwm, rpmRange in ( fanInterface.expectedFanRpmRange
                                    ).items():
                  fanConfig.expectedFanRpmRange.addMember( rpmRange )

         # Declare success for the fan. This MUST be done after the
         # FanEnvConfig object has been created
         fanSensorMib.initStatus = "ok"
         fanMib.initStatus = "ok"

      # Declare success for the fan tray.
      fanTrayMib.initStatus = "ok"

def _createFanTray( fanTraySlot ):
   Fru.Dep( fanTraySlot, fanTraySlot ).fanTray = ( "FanTray%d" % fanTraySlot.label,
                                                   fanTraySlot.label )
   fanTray = fanTraySlot.fanTray
   fanTray.component = ( "component", )
   fanTray.managingCellId = 0
   if not fanTray.generationId:
      fanTraySlotDir = fanTraySlot.parent
      fanTraySlotDir.generationIdMarker += 1
      fanTray.generationId = fanTraySlotDir.generationIdMarker

   # pylint: disable-next=redefined-builtin
   for ( id, fanInterface ) in fanTraySlot.fanInterface.items():
      fan = fanTray.newFan( id )
      fan.fanInterface = fanInterface

   # Create a new dependent set for the fan tray
   Fru.newDependentSet( fanTray )

   return fanTray

def fanTrayFactory( fanTraySlot, fdl, idInParent ):
   """ Create the fan tray fru within the given slot."""
   fanTray = _createFanTray( fanTraySlot )
   fdl.execFdl( "fru", fanTray )
   return fanTray

class HwFanStatusReactor( Tac.Notifiee ):

   notifierTypeName = "Hardware::Fan::Status"

   def __init__( self, status, hwSlotConfig, invFanTraySlot,
                 fanTraySlotMib, driverCtx ):
      self.hwSlotConfig_ = hwSlotConfig
      self.invFanTraySlot_ = invFanTraySlot
      self.fanTraySlotMib_ = fanTraySlotMib
      self.driverCtx_ = driverCtx
      self.fanTraySlotStatusReactor_ = None
      Tac.Notifiee.__init__( self, status )
      self.handleSlotStatus( key=hwSlotConfig.name )

   @Tac.handler( "slotStatus" )
   def handleSlotStatus( self, key ):
      if key != self.hwSlotConfig_.name:
         return
      hwSlotStatus = self.notifier_.slotStatus.get( self.hwSlotConfig_.name )
      if ( hwSlotStatus and
           ( self.hwSlotConfig_.generationId == hwSlotStatus.generationId ) ):
         assert not self.fanTraySlotStatusReactor_
         self.fanTraySlotStatusReactor_ = FanTraySlotStatusReactor(
            hwSlotStatus, self.hwSlotConfig_, self.invFanTraySlot_,
            self.fanTraySlotMib_, self.driverCtx_ )
      else:
         self.fanTraySlotStatusReactor_ = None

class FanTraySlotStatusReactor( Tac.Notifiee ):

   notifierTypeName = "Hardware::Fan::SlotStatus"

   def __init__( self, hwSlotStatus, hwSlotConfig, invFanTraySlot,
                 fanTraySlotEntmib, driverCtx ):
      self.hwSlotConfig_ = hwSlotConfig
      self.invFanTraySlot_ = invFanTraySlot
      self.fanTrayDriver_ = None
      self.fanTraySlotEntmib_ = fanTraySlotEntmib
      # Customer description of the fan tray
      self.fanTrayDesc_ = "Fan tray %s" % formatEntityMibName(
         EntityMib.componentName( fanTraySlotEntmib ) )
      self.driverCtx_ = driverCtx

      self.state_ = "unknown"
      Tac.Notifiee.__init__( self, hwSlotStatus )
      self.handleState()  # BUG1869 Required since we're doing immediate
                          # notifications

   def __del__( self ):
      # on systems where the fanTraySlot can get removed we can get deleted
      # without any notification
      self._handleFanTrayRemoved( logMessage=( self.state_=="present" ) )
      Tac.Notifiee.__del__( self )

   # Helper function for logging FanTray insertion and removal.
   def _logFanChange( self, logMsg ):
      # Defaults for if the FanTray or its attributes do not exist
      modelName = 'unknown'

      fanTray = self.invFanTraySlot_.fanTray 

      if fanTray:
         modelName = fanTray.modelName or modelName

      # now check if log is for REMOVED and we still don't have model name
      if modelName == 'unknown' and logMsg == FRU_FAN_REMOVED:
         # if so, then check if we remembered model name
         if self.invFanTraySlot_.modelName != "":
            # if so, use it and reset the memory with empty model name
            modelName = self.invFanTraySlot_.modelName
            self.invFanTraySlot_.modelName = ""

      # pylint: disable-next=consider-using-in
      if logMsg != FRU_FAN_INSERTED and logMsg != FRU_FAN_REMOVED:
          # pylint: disable-next=bad-indentation
          assert False, "FanFru logMsg %s is not handled in _logFanChange" % logMsg

      Logging.log( logMsg, self.fanTrayDesc_, modelName )

   def _isTrayIntegrated( self ):
      return self.hwSlotConfig_.detectionType == "dmIntegrated"

   def _handleFanTrayRemoved( self, logMessage=True ):
      t0( self.notifier_.name, "handleFanTrayRemoved" )
      # if this is not a removal fan tray don't remove it
      if self._isTrayIntegrated():
         return
      # Fan was present (maybe), now removed (maybe)
      if logMessage:
         # Fan was removed. Log a message.
         self._logFanChange( FRU_FAN_REMOVED )

      self.fanTraySlotEntmib_.fanTray = None
      Fru.deleteDependents( self.invFanTraySlot_.fanTray )
      self.invFanTraySlot_.fanTray = None
      self.fanTrayDriver_ = None

   @Tac.handler( "state" )
   def handleState( self ):
      oldState = self.state_
      self.state_ = self.notifier_.state

      t3( self.notifier_.name, "handleState(", self.state_, ")" )
      
      if self.state_ == "unknown":
         # Should only be in this state at the beginning of time
         # if the fan is integrated it must be present, otherwise it should not
         # be present
         assert bool( not self.invFanTraySlot_.fanTray ) is not \
                     bool( self.invFanTraySlot_.detectionType == "dmIntegrated" )
      elif self.state_ == "notPresent":
         if self.invFanTraySlot_.fanTray:
            self._handleFanTrayRemoved()
         self.fanTraySlotEntmib_.lastChangeTime = Tac.now()

      # pylint: disable-next=consider-using-in
      elif self.state_ == "present" or self.state_ == "notPresentOrNoSeeprom":
         # A fan is inserted (maybe)
         
         if self.invFanTraySlot_.fanTray:
            oldFdlHash = self.invFanTraySlot_.fanTray.fdlHash
            newFdlHash = Fdl.fdlHash( self.notifier_.fdl )
            if oldFdlHash != newFdlHash:
               # A new fan has been inserted while we were away
               self._handleFanTrayRemoved( logMessage=(oldState=="present") )
            else:
               # We've already handled this fan's insertion
               return
                  
         t0( self.notifier_.name, "a new fan has been detected" )
      
         if self._isTrayIntegrated():
            # don't need to do anything for this fan type
            pass
         elif self.notifier_.fdl == b"":
            assert not self.invFanTraySlot_.fanTray
            # Special case for an unsupported fan
            Logging.log( FRU_FAN_UNSUPPORTED, self.fanTrayDesc_ )
            # We need to populate the inventory fan tray object anyway,
            # to ensure that the appropriate Environment::Cooling::FanConfig
            # and EntityMib objects are created appropriately.
            _createFanTray( self.invFanTraySlot_ )
         else:
            assert not self.invFanTraySlot_.fanTray
            fdl = Fdl.Rfc822Fdl( self.notifier_.fdl )
            try:
               self.driverCtx_.fruFactoryRegistry.newFru(
                  self.invFanTraySlot_, fdl )
               # Even if the fdl is empty, we still exec the fdl
               # so that the factory gets called (which sets up
               # a bunch of other things)
            except ( Fru.FruFactoryError, NotImplementedError ):
               sys.excepthook( *sys.exc_info() )
               Logging.log( FRU_FAN_UNSUPPORTED, self.fanTrayDesc_ )
               # We need to populate the inventory fan tray object anyway,
               # to ensure that the appropriate Environment::Cooling::FanConfig
               # and EntityMib objects are created appropriately.
               _createFanTray( self.invFanTraySlot_ )
         invFanTray = self.invFanTraySlot_.fanTray
         invFanTray.fdlHash = Fdl.fdlHash( self.notifier_.fdl )

         # A fan is now inserted
         if self.state_ == "present":
            # Fan was inserted. Log a message.
            self._logFanChange( FRU_FAN_INSERTED )

         # instantiate the associated driver class
         try:
            self.fanTrayDriver_ = self.driverCtx_.driverRegistry.newDriver(
               invFanTray, self.fanTraySlotEntmib_, None, self.driverCtx_ )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            Logging.log( FRU_FAN_UNSUPPORTED, self.fanTrayDesc_ )

         entmibRoot = \
               self.driverCtx_.sysdbRoot[ 'hardware' ][ 'entmib' ].root 
         EntityMibUtils.updateRootExtendedFields( entmibRoot )

         self.fanTraySlotEntmib_.lastChangeTime = Tac.now()
      else:
         assert False
   
   @Tac.handler( "compressedFdl" )
   def handleFdl( self ):
      t3( self.notifier_.name, "handleFdl(", self.notifier_.fdl,  ")" )
      fdlString = self.notifier_.fdl

      if self.state_ == "present":
         # A fan has already been inserted, and fdl has changed, e.g. bug 87431.
         
         assert( self.invFanTraySlot_.fanTray ) # pylint: disable=superfluous-parens
         oldFdlHash = self.invFanTraySlot_.fanTray.fdlHash
         newFdlHash = Fdl.fdlHash( fdlString )
         assert( oldFdlHash != newFdlHash ) # pylint: disable=superfluous-parens
         
         # The fan has not been removed, so don't log a message,
         # but, save the modelName so that when logging a REMOVED syslog,
         # we can use this modelName
         self.invFanTraySlot_.modelName = self.invFanTraySlot_.fanTray.modelName
         self._handleFanTrayRemoved( logMessage=False )
                  
         t0( self.notifier_.name, "a new fan has been detected" )
      
         if self._isTrayIntegrated():
            # don't need to do anything for this fan type
            pass
         elif fdlString == b"":
            # This case is only possible if we're in 'diags' mode. When a fan is
            # removed in 'diags' mode, then fdl will first become an empty string,
            # then FanSlot status will change from 'present' to 'notPresent'. The
            # 'diags' mode is utilized in product tests.

            # We need to populate the inventory fan tray object anyway,
            # to ensure that the appropriate Environment::Cooling::FanConfig
            # and EntityMib objects are created.
            _createFanTray( self.invFanTraySlot_ )
         else:
            fdl = Fdl.Rfc822Fdl( fdlString )
            try:
               self.driverCtx_.fruFactoryRegistry.newFru(
                  self.invFanTraySlot_, fdl )
               # Even if the fdl is empty, we still exec the fdl
               # so that the factory gets called (which sets up
               # a bunch of other things)
            except ( Fru.FruFactoryError, NotImplementedError ):
               sys.excepthook( *sys.exc_info() )
               Logging.log( FRU_FAN_UNSUPPORTED, self.fanTrayDesc_ )
               # We need to populate the inventory fan tray object anyway,
               # to ensure that the appropriate Environment::Cooling::FanConfig
               # and EntityMib objects are created appropriately.
               _createFanTray( self.invFanTraySlot_ )

            # Fan was inserted. Log a message.
            self._logFanChange( FRU_FAN_INSERTED )

         invFanTray = self.invFanTraySlot_.fanTray
         invFanTray.fdlHash = Fdl.fdlHash( fdlString )

         # Instantiate the associated driver class.
         try:
            self.fanTrayDriver_ = self.driverCtx_.driverRegistry.newDriver(
               invFanTray, self.fanTraySlotEntmib_, None, self.driverCtx_ )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            Logging.log( FRU_FAN_UNSUPPORTED, self.fanTrayDesc_ )

         self.fanTraySlotEntmib_.lastChangeTime = Tac.now()

class FanTraySlotReactor( Tac.Notifiee ):
   notifierTypeName = "Inventory::FanTraySlot"

   def __init__( self, invFanTraySlot, fanTraySlotDriver ):
      Tac.Notifiee.__init__( self, invFanTraySlot )
      self.invFanTraySlot = invFanTraySlot
      self.fanTraySlotDriver = fanTraySlotDriver
      if invFanTraySlot.slotDesc:
         self.handleSlotDesc()

   @Tac.handler( 'slotDesc' )
   def handleSlotDesc( self ):
      t0( self.notifier_.name, "slotDesc populated " )
      self.fanTraySlotDriver.handleSlotDesc()

class FanTraySlotDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::FanTraySlot"
   managedApiRe = "$"

   requires = [ Fru.FruDriver.systemInit, supportedFansInit,
                FruPlugin.Gpio.gpioInit ]

   fanAllSuiteCreated = False

   def __init__( self, invFanTraySlot, parentMibEntity, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invFanTraySlot, parentMibEntity,
                              parentDriver, ctx )

      self.hwFanStatusReactor_ = None
      self.invFanTraySlotReactor = FanTraySlotReactor( invFanTraySlot, self )

   def handleSlotDesc( self ):
      invFanTraySlot = self.invEntity_
      ctx = self.driverCtx_
      assert invFanTraySlot.slotDesc

      entmib = ctx.entity( 'hardware/entmib' )
      config = ctx.sysdbRoot[ 'hardware' ][ 'fan' ][ 'config' ]
      status = ctx.entity( 'hardware/fan/status' )
      
      # The EntityMib for the fan tray slot is on the FixedSystem/Chassis,
      # which may not be our parentMibEntity. It is created by the
      # FixedSystem/ModularSystem fru drivers.
      fanTraySlotMib = entmib.root.fanTraySlot[ invFanTraySlot.slotDesc.id ]

      fanTrayName = EntityMib.componentName( fanTraySlotMib )
      
      # Create the associated Hardware::Fan::{SlotConfig,SlotStatus} objects
      hwSlotConfig = Fru.Dep( config.slotConfig, invFanTraySlot ).newMember(
         fanTrayName, Fru.fruBase( invFanTraySlot ).generationId )

      detectionType = invFanTraySlot.detectionType
      if invFanTraySlot.detectionType == "dmLegacy":
         # for legacy fdls figure out what the detection type
         # is from the other attributes
         if invFanTraySlot.seeprom:
            detectionType = "dmSeeprom"
         elif invFanTraySlot.present:
            detectionType = "dmPresentPin"
         else:
            detectionType = "dmSystem"
         
      if detectionType == "dmIntegrated":
         hwSlotConfig.detectionType = "dmIntegrated"
      elif detectionType == "dmPresentPin":
         hwSlotConfig.detectionType = "dmPresentPin"
         assert invFanTraySlot.present
         # Detection is through the present gpo + fan id bits
         hwSlotConfig.presentGpo = invFanTraySlot.present.systemName
         for ( fanId, invFanId ) in invFanTraySlot.fanId.items():
            assert invFanId.systemName
            hwSlotConfig.fanIdGpo[ fanId ] = invFanId.systemName
      elif detectionType == "dmPresentGpio":
         hwSlotConfig.detectionType = "dmPresentGpio"
         assert invFanTraySlot.presentGpio
         hwSlotConfig.presentGpioName = invFanTraySlot.presentGpio.systemName
         for ( pinId, invFanId ) in invFanTraySlot.fanIdGpio.items():
            assert invFanId.systemName
            hwSlotConfig.fanIdGpioName[ invFanId.systemName ] = True
            hwSlotConfig.fanIdGpioBit[ invFanId.systemName ] = pinId
      elif detectionType == "dmFanIdPresentGpio":
         hwSlotConfig.detectionType = "dmFanIdPresentGpio"
         for ( pinId, invFanId ) in invFanTraySlot.fanIdGpio.items():
            assert invFanId.systemName
            hwSlotConfig.fanIdGpioName[ invFanId.systemName ] = True
            hwSlotConfig.fanIdGpioBit[ invFanId.systemName ] = pinId
      elif detectionType == "dmSystem":
         hwSlotConfig.detectionType = "dmSystem"
      elif detectionType == "dmSeeprom":
         hwSlotConfig.detectionType = "dmSeeprom"
         assert invFanTraySlot.seeprom
         # Detection is through a seeprom
         hwSlotConfig.seepromEightBitAddr = invFanTraySlot.seeprom.eightBitAddr
         topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
         ahamArgs = [ fanTrayName ]
         ahamArgs.extend(
            FruPlugin.Smbus.ahamDesc( topology, invFanTraySlot.seeprom ) )         
         seepromAhamDesc = Fru.Dep( config.ahamDesc,
                                    invFanTraySlot ).newMember( *ahamArgs )
         hwSlotConfig.seepromAhamDesc = seepromAhamDesc

      if invFanTraySlot.ecbFanOn:
         hwSlotConfig.ecbFanOnName = invFanTraySlot.ecbFanOn.systemName
         hwSlotConfig.powerFanOnDelay = invFanTraySlot.powerFanOnDelay

      hwSlotConfig.ready = True

      # Setup a reactor to the Hardware::Fan::SlotStatus object being created
      # (by the FanDetector in response to the config being created)
      self.hwFanStatusReactor_ = HwFanStatusReactor(
            status, hwSlotConfig, invFanTraySlot, fanTraySlotMib, self.driverCtx_ )

def Plugin( context ):
   # This should really be "FanTray", but some fans have already
   # shipped with "Fan", so it's easier to keep it as is.
   context.registerFruFactory( fanTrayFactory, "Fan" )
   context.registerDriver( SupportedSystemFansDriver )

   # We let SONiC control the fans in AristaSAI, we have disabled the fan
   # controllers  and shouldn't register the the FanTraySlotDriver or FanTrayDrivers
   # We need SupportedSystemFansDriver to prevent FRU from crashing
   if toggleAristaSAIModeEnabled():
      return
   context.registerDriver( FanTraySlotDriver )
   context.registerDriver( FanTrayDriver )

   mountGroup = context.entityManager.mountGroup()
   mountGroup.mount( 'hardware/fan/config',
                     'Hardware::Fan::Config', 'w' )
   mountGroup.mount( 'hardware/fan/status',
                     'Hardware::Fan::Status', 'r' )
   mountGroup.mount( 'environment/cooling/config',
                     'Environment::Cooling::Config', 'w' )
   mountGroup.mount( 'environment/temperature/config',
                     'Environment::Temperature::Config', 'w' )

   import Cell # pylint: disable=import-outside-toplevel
   if Cell.cellType() != "supervisor":
      # some fans use GPOs which have hotplug hams (via the scd they reside on)
      # We mount the pcie cell specific pcieDeviceStatusDir so the
      # hamImpl can get the pcie address of the scd.
      # BUG2898 tracks fixing this for real such that the Fru agent never
      # tries to write hardware directly.
      mountGroup.mount( 'cell/%s/hardware/pciDeviceStatusDir' % Cell.cellId(),
                        'Hardware::PciDeviceStatusDir', 'r' )
   
   mountGroup.close( None )
