# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Fru
import Tracing
from TypeFuture import TacLazyType

from HwL1TopologyComponentLib import IntfSlotTypes, Components

__defaultTraceHandle__ = Tracing.Handle( "PhyTopoFru" )

t0 = Tracing.trace0
t1 = Tracing.trace1

InterfaceSlotType = TacLazyType( "Hardware::L1Topology::InterfaceSlotType" )
InterfaceRoot = TacLazyType( "Hardware::L1Topology::InterfaceRoot" )
InterfaceSlotLaneGroupMode = TacLazyType(
   "Hardware::L1Topology::InterfaceSlotLaneGroupMode" )
InterfaceSlotFru = TacLazyType( "Hardware::L1Topology::InterfaceSlotFru" )
TraceEndpointHelper = TacLazyType( "Hardware::L1Topology::DescriptorHelper" )
XcvrLaneDescriptor = TacLazyType( "Hardware::L1Topology::XcvrLaneDescriptor" )
SerdesDescriptor = TacLazyType( "PhyEee::SerdesDescriptor" )
InterfaceSlotModule = TacLazyType(
   "Hardware::L1Topology::InterfaceSlotModule" )

class InterfaceSlotDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::L1InterfaceSlotTopologyDir"
   managedApiRe = "$"
   
   provides = [ "L1Topology.InterfaceSlots" ]
   requires = [ Fru.FruDriver.interfaceInit ]

   def __init__( self, invIntfSlotTopoDir, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, invIntfSlotTopoDir, fruEntMib,
                              parentDriver, driverCtx )
      sliceId = Fru.fruBase( invIntfSlotTopoDir ).sliceId
      if not sliceId:
         assert Cell.cellType() == "fixed"
         sliceId = "FixedSystem"

      # pylint: disable-next=consider-using-f-string
      t0( "Handling Inventory::L1InterfaceSlotTopologyDir for %s" % sliceId )

      intfSlotTypeDir = driverCtx.entity( "hardware/l1/topology/intfSlotType" )
      hwFruTopoSliceDir = driverCtx.entity( "hardware/l1/fru/topology/slice" )
      hwFruIntfSlotDir = hwFruTopoSliceDir[ sliceId ].newEntity(
         "Hardware::L1Topology::InterfaceSlotFruDir", "intfSlot" )
      hwIntfTopoSliceDir = driverCtx.entity( "hardware/l1/topology/slice" )
      hwIntfTopoSliceDir[ sliceId ].newEntity(
         "Hardware::L1Topology::InterfaceSlotTopologyDir", "intfSlot" )

      self.supportedPrefixes = [ "Ethernet", "Fabric" ]

      self.populateStaticIntfSlotTypeHwModel( invIntfSlotTopoDir, intfSlotTypeDir )
      self.populateIntermediaryHwFruModel(
         invIntfSlotTopoDir, hwFruIntfSlotDir, sliceId )

   def populateStaticIntfSlotTypeHwModel( self,
                                          invIntfSlotTopoDir,
                                          hwIntfSlotDir ):
      for intfSlotTopo in invIntfSlotTopoDir.interfaceSlotTopology.values():
         invIntfSlotTypeStr = intfSlotTopo.intfSlotType
         invIntfSlotPrefix = intfSlotTopo.descriptor.slotPrefix
   
         assert invIntfSlotPrefix in self.supportedPrefixes, \
            f"Unsupported interface slot prefix {invIntfSlotPrefix}."

         # Grab intfSlotType from Component Lib
         intfSlotTypeCls = IntfSlotTypes.getIntfSlotType( invIntfSlotTypeStr )
         intfSlotTypeInfo = intfSlotTypeCls()
         intfSlotTypeInfo.generateIntfSlotTypeModel( hwIntfSlotDir )


   def populateIntermediaryHwFruModel( self,
                                       invIntfSlotTopoDir,
                                       hwFruTopoDir,
                                       sliceId ):
      # Populate hw fru model for intf slot types
      for intfSlotDesc, intfSlotTopo in \
         invIntfSlotTopoDir.interfaceSlotTopology.items():

         invIntfSlotTypeStr = intfSlotTopo.intfSlotType
         invIntfSlotPrefix = intfSlotDesc.slotPrefix 
         assert invIntfSlotPrefix in self.supportedPrefixes, \
            f"Unsupported interface slot prefix {invIntfSlotPrefix}."
         
         # Grab intfSlotType from Component Lib
         intfSlotTypeCls = IntfSlotTypes.getIntfSlotType( invIntfSlotTypeStr )
         intfSlotTypeInfo = intfSlotTypeCls()
         
         # Translate slot type to something known by enum
         intfSlotType = intfSlotTypeInfo.intfSlotTypeEnum
        
         # Create Hardware::L1Topology::InterfaceSlotFru model 
         hwinterfaceSlotFru = hwFruTopoDir.newInterfaceSlotFru(
            intfSlotDesc, intfSlotType )

         for intfRootId, ethPort in intfSlotTopo.rootToPort.items():
            if not ethPort.bound:
               continue
            hwinterfaceSlotFru.interfaceRootsToIntfIds[ intfRootId ] = \
               ethPort.intfId

         # Create TraceEndpointDescriptors for components
         for component in intfSlotTopo.components:
            # TODO: BUG694889 Add support for other component types
            if component.componentType == "XcvrSlot":
               xcvrLaneDesc = XcvrLaneDescriptor(
                  sliceId,
                  component.componentId,
                  component.componentConnectionNodeId,
                  component.tx )

               traceEndpointDesc = TraceEndpointHelper.traceEndpointFromXcvrLane(
                  xcvrLaneDesc )
               hwinterfaceSlotFru.traceEndpointDescriptor.add( traceEndpointDesc )
            else:
               componentClass = Components.getComponent( component.componentType )
               componentDef = componentClass( mode=component.mode or None,
                                              coreModes=component.coreMode or None )
               ( coreId, physicalLaneId ) = componentDef.getSerdes(
                  component.componentConnectionNodeId )
               # coreId is the local core ID, so we translate it here
               # to the global core ID.
               phyCoreId = ( component.componentId *
                             componentDef.getComponentCoreIdOffset() + coreId )
               phyType = componentDef.getComponentType( coreId )
               serdesDesc = SerdesDescriptor(
                  sliceId,
                  component.componentType,
                  component.componentId,
                  phyType,
                  phyCoreId,
                  component.phyScope,
                  physicalLaneId,
                  component.tx )

               traceEndpointDesc = TraceEndpointHelper.traceEndpointFromSerdes(
                  serdesDesc )
               hwinterfaceSlotFru.traceEndpointDescriptor.add( traceEndpointDesc )

def Plugin( context ):
   """Register the plugins."""
   context.registerDriver( InterfaceSlotDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( "hardware/l1/topology", "Tac::Dir", "wi" )
   mg.mount( "hardware/l1/fru/topology", "Tac::Dir", "wi" )
   mg.close( None )
