# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import EntityMib
from EntityMib import IndexAllocator
import Fru
import FruPlugin.Gpio
from FruPlugin import TempSensorFru
from Toggles.SwagCoreToggleLib import toggleMultiSwagBringupEnabled

import os

class KabiniDriver( Fru.FruDriver ):

   requires = [ Fru.FruDriver.pciInit ]
   provides = [ "Kabini", FruPlugin.Gpio.gpioInit ]

   managedTypeName = "Inventory::Kabini::Kabini"

   def __init__( self, invKabini, parentMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, invKabini, parentMib, parentDriver, driverCtx )

      cell = Fru.fruBase( invKabini ).managingCellId
      # TODO: BUG 998464: remove after CpuComplex is swagified
      # We hardcode cellId = 3 in bi-swag where we assume the swag worker has a
      # member ID of 3. It has no cellId because its fruBase is a non-supe card
      if toggleMultiSwagBringupEnabled() and not cell:
         cell = Fru.fruBase( invKabini ).smid
      hwCell = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ str( cell ) ]
      hwKabiniDir = hwCell.mkdir( "kabini" )
      gpoDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpo' ]

      # Create KabiniConfig & Ham
      hwKabiniConfig = hwKabiniDir.newEntity( "Hardware::Kabini::KabiniConfig",
                                            "config" )

      # Setup each Gpo and Led
      for gpo in invKabini.kabiniGpo.values():
         systemName = f"{invKabini.name}-{gpo.name}"
         fruGpo = Fru.Dep( gpoDir, invKabini ).newEntity(
            "Hardware::Kabini::KabiniGpo",
            systemName )
         fruGpo.direction = gpo.direction
         fruGpo.offset = gpo.offset
         fruGpo.polarity = gpo.polarity
         Fru.Dep( gpo, invKabini ).fruGpo = fruGpo
         gpo.systemName = systemName

      # Setup temp sensors. Assign localId
      localId = 0
      self.sysdbRoot_ = driverCtx.sysdbRoot
      self.envTempDir_ = self.sysdbRoot_[ 'environment' ][ 'temperature' ]

      for invSensor in invKabini.tempSensor.values():
         sensorName = EntityMib.componentNameFromParent(
            parentMib, "TempSensor", invSensor.sensorId )
         envConfig = TempSensorFru.createSensorEnvConfig( self.envTempDir_,
                                                          sensorName,
                                                          invSensor )
         Fru.Dep( hwKabiniConfig.tempSensor,
                  invKabini ).newMember( localId, invSensor.sensorId, envConfig,
                                         invSensor.tempInput )
         hwSensor = hwKabiniConfig.tempSensor[ invSensor.tempInput ]
         hwSensor.offset = invSensor.offset
         localId += 1
         # Entity Mib registration
         sensorMib = self._createEntityMib( invSensor )
         EntityMib.populateMib( sensorMib, invSensor )
         sensorMib.groupName = invSensor.groupName
         # Must be very last
         sensorMib.initStatus = "ok"

      # Populate TDP to max CPU frequency mapping
      for tdp in invKabini.tdpToCpuFreq:
         self._checkTdpToCpuFreqSupported()
         hwKabiniConfig.tdpToCpuFreq[ tdp ] = invKabini.tdpToCpuFreq[ tdp ]

      # Signal Ready
      hwKabiniConfig.configReady = True

   def _createEntityMib( self, sensorFru ):
      sensorMib = self.parentMibEntity_.sensor.get( sensorFru.sensorId )
      if sensorMib is None:
         physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
            self.parentMibEntity_.sensor, sensorFru.sensorId )
         sensorMib = self.parentMibEntity_.newSensor(
            physicalIndex, sensorFru.sensorId, "TempSensor" )
      sensorMib.label = str( sensorFru.sensorId )
      # Copy description from Config
      sensorMib.description = sensorFru.description
      return sensorMib

   def _checkTdpToCpuFreqSupported( self ):
      # WARNING: DO NOT add any product in this list unless you have verified lower
      # TDP is required on the product as this could cause performance degradation.
      whitelist = [ 'alcatrazIsland' ]
      if 'SIMULATION_VMID' in os.environ:
         whitelist += [ 'kabiniTestProduct' ]
      assert Cell.product() in whitelist

def Plugin( context ):
   context.registerDriver( KabiniDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/archer/gpo', 'Tac::Dir', 'wi' )
   mg.close( None )
