# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the FRU driver used to marshal card default L1 profiles from
the card / fixed system inventory to Sysdb hardware models.
'''

from ArPyUtils.Types import ArException
import Cell
import DesiredTracing
import Fru
from Fru.Driver import FruDriver
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'L1Profile::DefaultDriver' )
DesiredTracing.desiredTracingIs( 'L1Profile::DefaultDriver/2' )

TWARN = Tracing.trace2

CardDefaultDir = TacLazyType( 'L1Profile::CardDefaultDir' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
InvCardDefault = TacLazyType( 'Inventory::L1Profile::CardDefault' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

class L1ProfileCardDefaultDriver( FruDriver ):
   managedTypeName = InvCardDefault.tacType.fullTypeName
   managedApiRe = '.*$'

   # The L1 profile config driver does not depend on anything, it's just concerned
   # with L1 profile's own sysdb state. Having this driver depend on anything else
   # may cause dependency cycles for other FruPlugins that depend on on it.
   requires = []
   provides = [ 'L1Profile::CardDefault' ]

   def __init__( self, invCardDefault, parentMib, parentDriver, driverCtx ):
      super().__init__( invCardDefault, parentMib, parentDriver, driverCtx )
      if not invCardDefault.cardProfile:
         return

      fruBase = Fru.fruBase( invCardDefault )
      cardSlotId = fruBase.sliceId
      if Cell.cellType() != 'supervisor':
         cardSlotId = 'FixedSystem'

      # Specifying non builtin card profile defaults does not make sense as the FDL
      # infrastrucure should only know about card profiles builtin to EOS.
      if invCardDefault.cardProfile.source != CardProfileSource.builtin:
         raise ArException( 'Encountered unsupported default card profile source',
                             cardSlotId=cardSlotId,
                             source=invCardDefault.cardProfile.source )

      cardDefaultDir = driverCtx.entity( MountConstants.cardDefaultDirPath() )
      cardDefault = Fru.Dep( cardDefaultDir.cardDefault, fruBase ).newMember(
         cardSlotId )

      cardDefault.cardProfile = invCardDefault.cardProfile

def Plugin( context ):
   context.registerDriver( L1ProfileCardDefaultDriver )

   mg = context.entityManager.mountGroup()

   mg.mount( MountConstants.cardDefaultDirPath(),
             CardDefaultDir.tacType.fullTypeName,
             'w' )

   mg.close( None )
