# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the FRU driver responsible for translating the L1 profile
product configuration inventories into the hardware models which are used to
determine the behaviour / capabilities of L1 profiles on the product.
'''

import DesiredTracing
from Fru.Driver import FruDriver
from TypeFuture import TacLazyType
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'L1Profile::ProductConfigDriver' )
DesiredTracing.desiredTracingIs( 'L1Profile::ProductConfigDriver/2' )

TWARN = Tracing.trace2

MountConstants = TacLazyType( 'L1Profile::MountConstants' )
ProductConfig = TacLazyType( 'L1Profile::ProductConfig' )
InvProductConfig = TacLazyType( 'Inventory::L1Profile::ProductConfig' )

class L1ProfileProductConfigDriver( FruDriver ):
   managedTypeName = InvProductConfig.tacType.fullTypeName
   managedApiRe = '.*$'


   # The L1 profile config driver does not depend on anything, it's just concerned
   # with L1 profile's own sysdb state. Having this driver depend on anything else
   # may cause dependency cycles for other FruPlugins that depend on on it.
   requires = []
   provides = [ 'L1Profile::ProductConfig' ]

   def __init__( self, invProductConfig, parentMib, parentDriver, driverCtx ):
      super().__init__( invProductConfig, parentMib, parentDriver, driverCtx )

      productConfig = driverCtx.entity( MountConstants.productConfigPath() )

      productConfig.enabled = invProductConfig.enabled
      if productConfig.enabled:
         TWARN( 'L1 profiles enabled on product' )

def Plugin( context ):
   context.registerDriver( L1ProfileProductConfigDriver )

   mg = context.entityManager.mountGroup()

   mg.mount( MountConstants.productConfigPath(),
             ProductConfig.tacType.fullTypeName,
             'w' )

   mg.close( None )
