#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Fru
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'Fru.LedPolicyDirDriver' )
t0 = Tracing.trace0

class LedPolicyDirDriver( Fru.FruDriver ):
   managedTypeName = 'Inventory::InterfaceLedPolicyDir'
   provides = [ 'ledPolicyDirInit' ]

   def __init__( self, invLedPolicyDir, parentMibEnt,
                 parentDriver, ctx ):
      super().__init__( invLedPolicyDir, parentMibEnt, parentDriver, ctx )
      t0( 'Initializing LedPolicyDirDriver' )
      fruBase = Fru.fruBase( invLedPolicyDir )
      sliceId = fruBase.sliceId
      hwLedPolicyDirBase = ctx.entity( 'hardware/led/slice/' )
      self.hwLedPolicyDir = Fru.Dep( hwLedPolicyDirBase,
                                     invLedPolicyDir ).\
                                     newEntity( 'Hardware::Led::LedPolicyDir',
                                                sliceId )

      self.invLedPolicyDir = invLedPolicyDir
      self.parentDriver = parentDriver
      self.parentMibEnt = parentMibEnt
      self.ctx = ctx

      # handle case of policy deletion between Fru restart
      self.hwLedPolicyDir.ledPolicy.clear()

      for policyName, policy in self.invLedPolicyDir.ledPolicy.items():
         hwPolicy = self.hwLedPolicyDir.ledPolicy.newMember( policyName )
         for intfSlot in policy.ledIntfSlot:
            t0( f'HW Model Added intfSlot {intfSlot} to policy {policyName}' )
            hwPolicy.ledIntfSlot.add( intfSlot )

class LedSinglePolicyDriver( Fru.FruDriver ):
   managedTypeName = 'Inventory::InterfaceLedPolicy'
   provides = [ 'ledPolicyInit' ]

   def __init__( self, invLedPolicy, parentMibEnt,
                 parentDriver, ctx ):
      super().__init__( invLedPolicy, parentMibEnt, parentDriver, ctx )
      fruBase = Fru.fruBase( invLedPolicy )
      sliceId = fruBase.sliceId
      t0( 'Initializing LedSinglePolicyDriver' )
      self.hwLedPolicyDir = ctx.entity( f'hardware/led/slice/{sliceId}' )
      self.invLedPolicy = invLedPolicy
      self.parentDriver = parentDriver
      self.parentMibEnt = parentMibEnt
      self.ctx = ctx
      for intfSlot in invLedPolicy.ledIntfSlot:
         t0( f'HW Model Added intfSlot {intfSlot} to policy { invLedPolicy.name }' )
         if invLedPolicy.name not in self.hwLedPolicyDir.ledPolicy.keys():
            hwPolicy = self.hwLedPolicyDir.ledPolicy.newMember( invLedPolicy.name )
         hwPolicy = self.hwLedPolicyDir.ledPolicy[ invLedPolicy.name ]
         hwPolicy.ledIntfSlot.add( intfSlot )

def Plugin( ctx ):
   ctx.registerDriver( LedPolicyDirDriver )
   ctx.registerDriver( LedSinglePolicyDriver )
   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/led/slice', 'Tac::Dir', 'wi' )
   mg.close( None )
