# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Fru
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.Olive" )
t0 = Tracing.trace0

class OliveDriver( Fru.FruDriver ):

   managedTypeName = 'Inventory::Olive'

   def __init__( self, invOlive, parentMib, parentDriver, driverCtx ):
      t0( 'Handling Inventory::Olive for', Fru.fruBaseName( invOlive ) )
      Fru.FruDriver.__init__( self, invOlive, parentMib, parentDriver, driverCtx )

      # In case of SSO, we will get called, on the active Supe, with invOlive
      # corresponding to the standby Supe as well. We want to ignore that update.
      # We want to update information only for the active Supe. When the standby
      # becomes active, we will get called again - and then we will update the
      # hardware config corresponding to that Supe/cellId.
      managingCellId = Fru.fruBase( invOlive ).managingCellId
      if managingCellId != Cell.cellId():
         t0( 'Update corresponds to a different cell,', managingCellId,
             ', current cell', Cell.cellId(),
             ' - ignoring update for', Fru.fruBaseName( invOlive ) )
         return

      sysdbRoot = driverCtx.sysdbRoot
      hwOliveConfig = sysdbRoot.entity[ Cell.path( 'hardware/olive/config' ) ]
      hwOliveConfig.isOliveHw = True
      hwOliveConfig.opModeOffset = invOlive.block.opModeOffset
      hwOliveConfig.opStatusOffset = invOlive.block.opStatusOffset
      hwOliveConfig.tpmBufferAddrOffset = invOlive.block.tpmBufferAddrOffset
      hwOliveConfig.tpmBufferDataOffset = invOlive.block.tpmBufferDataOffset
      hwOliveConfig.tpmSpiCmdFifoOffset = invOlive.block.tpmSpiCmdFifoOffset
      hwOliveConfig.tpmSpiRegStride = invOlive.block.tpmSpiRegStride

      # For TPM reset
      hwOliveConfig.tpmResetGpoName = invOlive.tpmResetGpo.systemName

      # Ham to access the Olive registers in SCD
      dep = invOlive
      Fru.Dep( hwOliveConfig, dep ).ham = invOlive.block.ham

      # And the recursive ham for the TPM Spi Accelerator, whose offset 0 corresponds
      # to the CmdFifo
      Fru.Dep( hwOliveConfig, dep ).tpmSpiHam = (
            'OliveTpmSpiHam', 'hamTypeRecursive', invOlive.block.ham,
            invOlive.block.tpmSpiCmdFifoOffset, 0x30, '' )

      # Set up the launcherConfig Tac::Dir which is used as the qualPath for
      # Olive agent
      lcDir = sysdbRoot.entity[ Cell.path( 'hardware/olive/launcherConfig' ) ]
      lcDir.newEntity( 'Tac::Dir', 'olive' )

def Plugin( context ):
   context.registerDriver( OliveDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( Cell.path( 'hardware/olive/config' ), 'Hardware::Olive::Config', 'w' )
   mg.mount( Cell.path( 'hardware/olive/launcherConfig' ), 'Tac::Dir', 'wi' )
   mg.close( None )
