#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Cell
import EntityMib
import Fru
import FruPlugin.Gpio as Gpio # pylint: disable=consider-using-from-import
import FruPlugin.TempSensorFru as TempHelper
import FruPlugin.XcvrCtrl as XcvrHelper
import FruPlugin.Led as LedHelper
import FruPlugin.Poe as Poe # pylint: disable=consider-using-from-import
import Tracing
from EntityMib import ChipType, IndexAllocator
from FruPlugin.PLSlot import defaultSliceId
import sys
from FruPlugin.Health import registerHealthSource
import EntityMibUtils

__defaultTraceHandle__ = Tracing.Handle( "Fru.PLDevice" )
t0 = Tracing.trace0

class PLDeviceDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::PLDevice"
   requires = [ 'plutoLibraryInit', Gpio.gpioInit ]

   def __init__( self, device, parentEntityMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, device, parentEntityMib, parentDriver,
                              driverCtx )
      self.device_ = device
      fruBase = Fru.fruBase( device )
      fruBaseName = Fru.fruBaseName( device )
      sliceId = fruBase.sliceId
      cellId = fruBase.managingCellId
      modular = not cellId

      # special case for the XcvrController that will be handled by PLXcvr
      if sliceId == "XcvrController":
         assert not device.led
         assert not device.tempSensor
         assert not device.powerSupply
         assert not device.fanTray
         xcvrDevicePath = "hardware/platform/xcvr/device"
         hwDevice = driverCtx.sysdbRoot.entity[ xcvrDevicePath ]
      else:
         assert not device.xcvrController
         configSlicePath = "hardware/platform/device/config/slice"
         configSliceDir = driverCtx.sysdbRoot.entity[ configSlicePath ]
         hwDevice = configSliceDir.get( sliceId )

      if hwDevice is None:
         hwDevice = configSliceDir.newEntity(
            "Hardware::PLDevice::Device",
            sliceId )
      hwDevice.valid = False
      envTemperature = driverCtx.sysdbRoot.entity[ "environment/temperature" ]
      envCoolingConfig = driverCtx.sysdbRoot.entity[ "environment/cooling/config" ]

      topologySmbus = None
      for xcvrControllerBlock in device.xcvrController.values():
         xcvrName = xcvrControllerBlock.name
         xcvrType = xcvrControllerBlock.xcvrController.xcvrType
         # skip unsupported xcvr causing other agents to crash
         if xcvrType not in [ "qsfpPlus", "sfpPlus", "qsfpCmis", "qsfpDd" ]:
            t0( f"Skipping unsupported xcvr {xcvrName} of type {xcvrType}" )
            continue
         t0( f"Processing xcvr {xcvrName} of type {xcvrType}" )

         xcvrControllerId = xcvrControllerBlock.id
         xcvrConfig = XcvrHelper.addXcvr( xcvrControllerBlock.xcvrController,
                                          fruBase, device, driverCtx,
                                          xcvrType, modular )
         # An L1 profile may remove all interfaces from an XCVR slot. In that case
         # the XCVR slot is said to be unbound and should be skipped.
         if not xcvrConfig:
            t0( "Skipping unbound XCVR", xcvrControllerBlock.xcvrController.name )
            continue

         xcvrCtrlName = xcvrConfig.name
         plXcvrController = Fru.Dep(
               hwDevice.xcvrConfig,
               xcvrControllerBlock ).newMember( xcvrCtrlName, xcvrControllerId )

         plXcvrController.xcvrType = xcvrType
         plXcvrController.libPath = xcvrControllerBlock.libPath
         plXcvrController.devArgs = xcvrControllerBlock.devArgs
         plXcvrController.generationId = xcvrConfig.generationId

         if xcvrControllerBlock.xcvrController.mgmtType == 'smbusMgmt':
            if not topologySmbus:
               topologySmbus = \
                  driverCtx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
            import FruPlugin.Smbus # pylint: disable=import-outside-toplevel
            allConfigDir = XcvrHelper.getAllConfigDir( fruBase, driverCtx )
            ahamDesc = allConfigDir.newAhamDesc( xcvrCtrlName,
                                *FruPlugin.Smbus.ahamDesc(
                                   topologySmbus,
                                   xcvrControllerBlock.xcvrController.smbusDeviceBase
                                ) )
            xcvrConfig.ahamDesc = ahamDesc
      for led in device.led.values():
         ledConfig = LedHelper.addLed( led.name, fruBaseName, device, driverCtx )
         ledConfig.ledAttribute = Tac.Value( "Led::LedAttribute",
                                             blue=led.capability.blue )
         hwLed = hwDevice.newLed( led.name )
         hwLed.libPath = led.libPath
         hwLed.devArgs = led.devArgs

      for poePort in device.poePort.values():
         # Path to be created for policy agent LLDP
         hwPoe = hwDevice.newPoePort( poePort.name )
         hwPoe.portId = poePort.portId
         hwPoe.capacity = poePort.capacity
         hwPoe.libPath = poePort.libPath
         hwPoe.devArgs = poePort.devArgs
      if device.poePort:
         hwDevice.continuousPoeSupported = device.continuousPoeSupported
         hwDevice.continuousPoePLSystem = device.continuousPoePLSystem
         hwDevice.firmwareUpgradable = device.firmwareUpgradable
         # PoE chip ID start at offset in entityMib to avoid conflict
         chipId = IndexAllocator.getChipId( ChipType.poe, 0 )
         chipModel = "BCM59121"
         poeChipMib = parentEntityMib.chip.get( chipId )
         if poeChipMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( parentEntityMib.chip, chipId )
            poeChipMib = parentEntityMib.newChip( physicalIndex,
                                                  chipId,
                                                  chipModel + "-" +
                                                  str( chipId % 20 ) )
         poeChipMib.label = str( chipId )
         poeChipMib.description = "PSE Chip %s" % chipId
         poeChipMib.modelName = chipModel
         EntityMib.populateMib( poeChipMib, device )

         Poe.controllerNameToChipMib[ "PlutoPoe" ] = poeChipMib

         statusDir = driverCtx.sysdbRoot.entity[ "environment/power/status/poe" ]
         if not Poe.dirReactor:
            Poe.dirReactor = Poe.PoeStatusDirReactor( statusDir )

      for tempSensor in device.tempSensor.values():
         tempSensorId = tempSensor.sensorId
         tempSensorName = EntityMib.componentNameFromParent( parentEntityMib,
                                                             "TempSensor",
                                                             tempSensorId )
         tsEnvConfig = TempHelper.createSensorEnvConfig(
            envTemperature, tempSensorName, tempSensor )
         ts = Fru.Dep( hwDevice.tempSensor, tempSensor ).newMember( tempSensor.name,
                                                                    tsEnvConfig )
         sensorMib = parentEntityMib.sensor.get( tempSensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
               parentEntityMib.sensor, tempSensorId )
            sensorMib = Fru.Dep(
               parentEntityMib.sensor, tempSensor ).newMember(
               physicalIndex, tempSensorId, "TempSensor" )

         sensorMib.description = tempSensor.description
         sensorMib.label = str( tempSensorId )
         sensorMib.groupName = tempSensor.groupName

         # Setting device information and library to handle this device
         ts.devArgs = tempSensor.devArgs
         ts.libPath = tempSensor.libPath
         ts.offset = tempSensor.offset

         # Declare success for the sensor. This MUST be done after the
         # TempSensorEnvConfig and TempSensorEnvStatus have been created
         sensorMib.initStatus = "ok"

      if device.fanTray:
         fanTray = device.fanTray
         name = fanTray.name
         slotId = fanTray.slotId
         fanTraySlotEntMib = parentEntityMib.fanTraySlot[ slotId ]
         if fanTraySlotEntMib.fanTray is None:
            physicalIndex = IndexAllocator.physicalIndex(
               fanTraySlotEntMib,
               "FanTray",
               fanTraySlotEntMib.relPos )
            Fru.Dep( fanTraySlotEntMib, fanTray ).fanTray = (
               physicalIndex,
               fanTraySlotEntMib.relPos,
               "FanTray" )
            fanTraySlotEntMib.fanTray.description = "Fan Tray %d" % slotId
            if device.fixed:
               fanTraySlotEntMib.fanTray.swappability = "notSwappable"
            else:
               fanTraySlotEntMib.fanTray.swappability = "hotSwappable"
         fanTrayEntMib = fanTraySlotEntMib.fanTray
         fanTrayEntMib.label = fanTraySlotEntMib.label
         EntityMib.populateMib( fanTrayEntMib, fanTray )

         for fan in fanTray.fan.values():
            fanId = fan.fanId
            fanName = EntityMib.componentNameFromParent( fanTrayEntMib,
                                                         "Fan",
                                                         fanId )
            registerHealthSource( fan, fanName )
            fanModel = fanTray.modelName
            fanProperties = envCoolingConfig.newSupportedFanProperties( fanModel )
            fanEnvConfig = Fru.Dep( envCoolingConfig.fan, fanTray ).newMember(
               fanName )
            fanEnvConfig.managed = fan.managed
            fanEnvConfig.readOnly = fan.readOnly
            fanEnvConfig.properties = fanProperties
            fanEnvConfig.fanTrayName = EntityMib.componentName( fanTrayEntMib )
            fanEnvConfig.generationId = Fru.powerGenerationId( fanTray ) 
            fanProperties.maxRpm = fan.maxRpm
            fanProperties.airflowDirection = fan.airflowDirection
            fanEnvConfig.prop = fan.p
            fanEnvConfig.integ = fan.i
            fanEnvConfig.pband = fan.pband
            Fru.Dep( envCoolingConfig.fanPtr, fanTray ).addMember( fanEnvConfig )
            hwFan = Fru.Dep( hwDevice.fan, fanTray ).newMember(
               f"{fanTray.name} {fan.name}", fanEnvConfig )
            hwFan.libPath = fan.libPath
            hwFan.devArgs = fan.devArgs
            fanMib = fanTrayEntMib.fan.get( fanId )
            if fanMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  fanTrayEntMib.fan, fanId )
               fanMib = fanTrayEntMib.newFan( physicalIndex,
                                              fanId,
                                              "Fan" )
               fanMib.description = "{} Fan {}".format( fanTrayEntMib.description,
                                                        fanId )
               fanMib.fanName = EntityMib.componentName( fanMib )

            fanSensorMib = fanMib.sensor.get( fanId )
            if fanSensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  fanMib.sensor, fanId )
               fanSensorMib = fanMib.newSensor( physicalIndex, fanId,
                                                "FanSensor" )
            fanSensorMib.description = "%s Sensor %d" % ( fanMib.description, fanId )
            fanSensorMib.initStatus = "ok"
            fanMib.initStatus = "ok"

         fanTrayEntMib.initStatus = "ok"
         EntityMibUtils.updateRootExtendedFields( parentEntityMib )
         
      if device.powerSupply:
         powerSupply = device.powerSupply
         slotId = powerSupply.slotId
         name = "PowerSupply%d" % powerSupply.slotId
         powerSupply.sliceId = name
         registerHealthSource( powerSupply, name )
         psu = Fru.Dep( hwDevice.powerSupply, powerSupply ).newMember( name )
         psu.slotId = slotId
         psu.libPath = powerSupply.libPath
         psu.devArgs = powerSupply.devArgs
         psu.generationId = int( Tac.now() )
         psu.capacity = powerSupply.capacity
         psu.inputVoltageSensor = powerSupply.inputVoltageSensor
         psu.outputVoltageSensor = powerSupply.outputVoltageSensor
         psu.inputCurrentSensor = powerSupply.inputCurrentSensor
         psu.outputCurrentSensor = powerSupply.outputCurrentSensor
         powerSupplySlotEntMib = parentEntityMib.powerSupplySlot[ slotId ]
         psu.parentLabel = powerSupplySlotEntMib.label
         if powerSupplySlotEntMib.powerSupply is None:
            physicalIndex = IndexAllocator.physicalIndex(
               powerSupplySlotEntMib,
               "PowerSupply",
               powerSupplySlotEntMib.relPos )
            Fru.Dep( powerSupplySlotEntMib, powerSupply ).powerSupply = (
               physicalIndex,
               powerSupplySlotEntMib.relPos,
               "PowerSupply" )
            powerSupplySlotEntMib.powerSupply.description = name
            if device.fixed:
               powerSupplySlotEntMib.powerSupply.swappability = "notSwappable"
            else:
               powerSupplySlotEntMib.powerSupply.swappability = "hotSwappable"
         powerSupplyEntMib = powerSupplySlotEntMib.powerSupply
         powerSupplyEntMib.label = powerSupplySlotEntMib.label
         EntityMib.populateMib( powerSupplyEntMib, powerSupply )

         for tempSensor in powerSupply.tempSensor.values():
            tempSensorId = int( tempSensor.sensorId )
            sensorName = EntityMib.componentNameFromParent( powerSupplyEntMib,
                                                            "TempSensor",
                                                            tempSensorId )
            tsEnvConfig = TempHelper.createSensorEnvConfig(
               envTemperature, sensorName, tempSensor )
            tsEnvConfig.offOnPowerLoss = tempSensor.offOnPowerLoss
            ts = psu.newTempSensor( tempSensor.name, tsEnvConfig )
            ts.libPath = tempSensor.libPath
            ts.devArgs = tempSensor.devArgs
            ts.offset = tempSensor.offset
            sensorMib = powerSupplyEntMib.sensor.get( tempSensorId )
            if sensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  powerSupplyEntMib.sensor, tempSensorId )
               sensorMib = powerSupplyEntMib.newSensor(
                  physicalIndex, tempSensorId, "TempSensor" )
               # Copy description from config
               sensorMib.description = f"{name} {tsEnvConfig.description}"
            sensorMib.initStatus = "ok"

         for fan in powerSupply.fan.values():
            fanId = fan.fanId
            fanName = EntityMib.componentNameFromParent( powerSupplyEntMib,
                                                         "Fan",
                                                         fanId )
            registerHealthSource( fan, fanName )
            fanProperties = envCoolingConfig.newSupportedFanProperties( fanName )
            fanEnvConfig = Fru.Dep( envCoolingConfig.fan, powerSupply ).newMember(
               fanName )
            fanEnvConfig.managed = fan.managed
            fanEnvConfig.readOnly = fan.readOnly
            fanEnvConfig.offOnPowerLoss = fan.offOnPowerLoss
            fanEnvConfig.properties = fanProperties
            fanEnvConfig.fanTrayName = EntityMib.componentName(
                  powerSupplySlotEntMib )
            fanEnvConfig.generationId = Fru.powerGenerationId( powerSupply ) 
            fanProperties.maxRpm = fan.maxRpm
            fanProperties.airflowDirection = fan.airflowDirection
            fanEnvConfig.prop = fan.p
            fanEnvConfig.integ = fan.i
            fanEnvConfig.pband = fan.pband
            Fru.Dep( envCoolingConfig.fanPtr, powerSupply ).addMember( fanEnvConfig )
            hwFan = psu.newFan( fan.name, fanEnvConfig )
            hwFan.libPath = fan.libPath
            hwFan.devArgs = fan.devArgs
            fanMib = powerSupplyEntMib.fan.get( fanId )
            if fanMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  powerSupplyEntMib.fan, fanId )
               fanMib = powerSupplyEntMib.newFan( physicalIndex,
                                                  fanId,
                                                  "Fan" )
               fanMib.description = f"{name} Fan {fanId}"
               fanMib.fanName = EntityMib.componentName( fanMib )

            fanSensorMib = fanMib.sensor.get( fanId )
            if fanSensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  fanMib.sensor, fanId )
               fanSensorMib = fanMib.newSensor( physicalIndex, fanId,
                                                "FanSensor" )
            fanSensorMib.description = "%s Sensor %d" % ( fanMib.description, fanId )
            fanSensorMib.initStatus = "ok"
            fanMib.initStatus = "ok"

         powerEnvConfig = driverCtx.sysdbRoot.entity[ 'environment/power/config' ]
         id_ = len( powerSupply.tempSensor ) + 1
         label = 1
         sensorList = []
         if psu.inputCurrentSensor:
            sensorList.append(
               ( id_, label, "CurrentSensor", "input current sensor" ) )
            id_ += 1
            label += 1
         if psu.outputCurrentSensor:
            sensorList.append(
               ( id_, label, "CurrentSensor", "output current sensor" ) )
            id_ += 1
         label = 1
         if psu.inputVoltageSensor:
            sensorList.append(
               ( id_, label, "VoltageSensor", "input voltage sensor" ) )
            id_ += 1
            label += 1
         if psu.outputVoltageSensor:
            sensorList.append(
               ( id_, label, "VoltageSensor", "output voltage sensor" ) )
         for ( sensorId, sensorLabel, sensorType, description ) in sensorList:
            sensorName = EntityMib.componentNameFromParent( powerSupplyEntMib,
                                                            sensorType,
                                                            sensorLabel )
            if description == "output voltage sensor":
               marginLow = 0
               marginHigh = 0
               vsEnvConfig = Fru.Dep( powerEnvConfig.voltageSensor,
                                      powerSupply ).newMember( sensorName,
                                                               marginLow,
                                                               marginHigh )
               vsEnvConfig.description = description
               vsEnvConfig.expectedVoltage = powerSupply.expectedOutputVoltage
               vsEnvConfig.sliceName = "PowerSupply%d" % powerSupply.slotId
            sensorMib = powerSupplyEntMib.sensor.get( sensorId )
            if sensorMib is None:
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  powerSupplyEntMib.sensor, sensorId )
               sensorMib = powerSupplyEntMib.newSensor( physicalIndex,
                                                        sensorId,
                                                        sensorType )
            sensorMib.label = str( sensorLabel )
            sensorMib.description = f"{name} {description}"
            sensorMib.initStatus = "ok"

         powerSupplyEntMib.initStatus = "ok"

      gpoDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpo' ]
      libConfig = driverCtx.sysdbRoot.entity[ 'hardware/pluto/library/config' ]
      assert libConfig.valid

      for gpio in device.gpio.values():
         systemName = f"{device.name}-{gpio.name}"
         fruGpo = Fru.Dep( gpoDir, device ).newEntity(
            "Hardware::Pluto::Gpo",
            systemName )
         fruGpo.activeType = gpio.activeType
         fruGpo.devArgs = gpio.devArgs
         fruGpo.libPath = gpio.libPath
         fruGpo.libConfig = ( libConfig.name, )
         fruGpo.libConfig.libPath = libConfig.libPath
         fruGpo.libConfig.devArgs = libConfig.devArgs
         fruGpo.libConfig.valid = libConfig.valid
         gpio.fruGpo = fruGpo
         gpio.systemName = systemName

      if device.poeResetGpo:
         hwDevice.poeResetGpoName = device.poeResetGpo.systemName
      if device.pseResetGpo:
         hwDevice.pseResetGpoName = device.pseResetGpo.systemName
      if device.pseEnableGpo:
         hwDevice.pseEnableGpoName = device.pseEnableGpo.systemName
      if device.pseUnlockGpo:
         hwDevice.pseUnlockGpoName = device.pseUnlockGpo.systemName
      if device.userRebootActionGpo:
         hwDevice.userRebootActionGpoName = device.userRebootActionGpo.systemName
      if device.watchdogRebootActionGpo:
         hwDevice.watchdogRebootActionGpoName = \
            device.watchdogRebootActionGpo.systemName
      for ( psu, mpss ) in device.psuToMpssMap.items():
         hwDevice.psuToMpssMap[ psu ] = mpss
      hwDevice.valid = True

class PLDeviceDirDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::PLDeviceDir"
   requires = [ 'plutoLibraryInit' ]
   provides = [ 'PLDeviceDirInit' ]

   def __init__( self, invPLDeviceDir, parentMibEnt, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invPLDeviceDir,
                              parentMibEnt, parentDriver, ctx )
      self.deviceDriver_ = {}
      for ( deviceName, deviceInv ) in invPLDeviceDir.device.items():
         deviceInv.component = ( "component", )
         deviceInv.managingCellId = Cell.cellId()
         if not deviceInv.sliceId:
            deviceInv.sliceId = defaultSliceId
         t0( "PLDeviceDriver looking for driver for PLDevice", deviceName )
         try:
            self.deviceDriver_[ deviceName ] = PLDeviceDriver(
               deviceInv, parentMibEnt, parentDriver, ctx )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            raise

def Plugin( ctx ):
   ctx.registerDriver( PLDeviceDriver )
   ctx.registerDriver( PLDeviceDirDriver )

   mg = ctx.entityManager.mountGroup()
   mg.mount( "hardware/platform/device/config/slice", "Tac::Dir", "wi" )
   mg.mount( "hardware/platform/xcvr/device", "Hardware::PLDevice::Device", "w" )
   mg.mount( 'hardware/archer/gpo', 'Tac::Dir', 'wi' )
   mg.close( None )
