# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module contains a plugin that manages kernel-managed Ethernet phys."""

import Cell
import Tac, Tracing, Fru
from EthIntf import MAX_MGMT_MTU, MAX_SUPPORTED_MTU
from Toggles.SwagCoreToggleLib import toggleMultiSwagBringupEnabled

__defaultTraceHandle__ = Tracing.Handle( "Fru.PhyEthtool" )
traceDetail = Tracing.trace9
traceNormal = Tracing.trace8
traceAlert = Tracing.trace1
traceError = Tracing.trace0

# Constant from Constants.h4
ARISTA_CPU_OVERHEAD = 128

# XXX_LWR: figure out what to do with hotpluggable transceivers, eg
#          SFPs. Maybe, we'll have to run an Activity that polls every
#          so often.... which means a HAM...?

class PhyEthtoolDriver( Fru.FruDriver ):
   """This Fru plugin manages any object of type Inventory::Phy::EthtoolDir."""

   requires = [ Fru.FruDriver.systemInit, Fru.FruDriver.interfaceInit ]

   managedTypeName = "Inventory::Phy::EthtoolDir"
   managedApiRe = "$"

   def __init__( self, phyDir, parentMib, parentDriver, driverCtx ):
      traceAlert( "Creating a Fru driver for the PhyDir" )
      Fru.FruDriver.__init__( self, phyDir, parentMib, parentDriver, driverCtx )

      # TODO: BUG 998475: remove after swagifying PhtEthtool
      # We hardcode cellId = 3 in bi-swag where we assume the swag worker has a
      # member ID of 3. It has no cellId because its fruBase is a non-supe card
      cellId = Fru.fruBase( phyDir ).managingCellId
      if toggleMultiSwagBringupEnabled() and not cellId:
         cellId = Fru.fruBase( phyDir ).smid

      hwCellDir = driverCtx.sysdbRoot[ "hardware" ][ "cell" ][ str(cellId) ]

      peDir = hwCellDir.mkdir( "phy/ethtool" )
      self.phyConfig_ = peDir.newEntity( "Hardware::Phy::EthtoolConfig",
                                         "config" )

      traceDetail( "Initializing PhyDir" )
      # For each Phy found in Fru's PhyDir collection, create
      # a Phy in the PhyAgent's collection, and hook it up to its
      # EthIntfStatus.
      traceDetail( "Creating Hardware::Phy::PhyEthtools for ",
                   list( phyDir.phy ) )
      for fruPhy in phyDir.phy.values():
         port = fruPhy.port

         # We don't want the cEOS container to have access to the management ports
         # as the container's host may want to manipulate the interface.
         import CEosHelper # pylint: disable=import-outside-toplevel
         if port.role == 'Management' and CEosHelper.isCeos() and \
            not CEosHelper.isCeosLab():
            if not CEosHelper.enableCeosMgmtInterface(): 
               # pylint: disable-next=consider-using-f-string
               traceError( "Skip %s" % fruPhy.name )
               continue

         ## 'Phyet' are ethernet interfaces used in vEOS
         if 'Phyet' in fruPhy.name or port.role in [ 'Switched', 'Cpu' ]:
            maxMtu = MAX_SUPPORTED_MTU
            if Cell.product().startswith( 'Cottesloe' ) and port.role == 'Cpu':
               maxMtu += ARISTA_CPU_OVERHEAD
               # pylint: disable-next=consider-using-f-string
               traceDetail( "Adjusting CPU mtu to %d" % maxMtu )
         else:
            maxMtu = MAX_MGMT_MTU
         traceDetail( f"Phy {fruPhy.name} --> {port.intfId}" )
         phyPhy = Fru.Dep( self.phyConfig_.phy, fruPhy ).newMember(
            fruPhy.name, port.intfId, maxMtu, port.role )
         phyPhy.ethtoolConfigEnabled = fruPhy.ethtoolConfigEnabled
         phyPhy.ptpHardwareTimestampingSupported = \
            fruPhy.ptpHardwareTimestampingSupported

def Plugin( context ):
   traceDetail( "PhyEthtoolFru plugin registering with Fru..." )
   context.registerDriver( PhyEthtoolDriver )
