# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Cell
import ProductAttributes

class AgentEventConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Stage::EventConfig'

   def __init__( self, agentName, eventConfig, daemonMonitor ):
      self.agentName_ = agentName
      self.eventConfig_ = eventConfig
      self.daemonMonitor_ = daemonMonitor
      Tac.Notifiee.__init__( self, eventConfig )
      self.handleComplete()

   @Tac.handler( 'complete' )
   def handleComplete( self ):
      if self.notifier_.complete[ 'default' ]:
         self.daemonMonitor_.markAgentReady( self.agentName_ )

class DaemonMonitor:
   def __init__( self, daemonDir, agentConfig ):
      self.daemonDir_ = daemonDir
      self.agentConfig_ = agentConfig
      self.reactors_ = {}

   def populate( self ):
      daemons = ProductAttributes.productAttributes().daemonAttributes
      for name, daemon in daemons.items():
         config = self.daemonDir_.config.newMember( name, daemon.exe )
         for i, arg in daemon.argv.items():
            config.argv[ i ] = arg

         event = self.agentConfig_[ name ].event.get( 'PrimaryAgent', None )
         if event is not None:
            self.reactors_[ name ] = AgentEventConfigReactor( name, event, self )
         else:
            self.markAgentReady( name )

      if not len( daemons ): # pylint: disable=use-implicit-booleaness-not-len
         self.daemonDir_.warm = True

   def markAgentReady( self, agentName ):
      self.daemonDir_.config[ agentName ].warm = True
      ready = True
      for daemon in self.daemonDir_.config.values():
         if not daemon.warm:
            ready = False
      if ready:
         self.daemonDir_.warm = True

daemonManager = None

def Plugin( context ):
   mg = context.entityManager.mountGroup()

   daemonDir = mg.mount( 'hardware/pluto/daemons',
                         'Hardware::PlutoDaemon::Dir', 'w' )

   agentConfig = {}
   for name in ProductAttributes.productAttributes().daemonAttributes:
      # pylint: disable-next=consider-using-f-string
      agentConfig[ name ] = mg.mount( Cell.path( 'stageInput/boot/%s' % name ),
                                      'Stage::AgentConfig', 'r' )

   def _doMountsComplete():
      global daemonManager

      daemonManager = DaemonMonitor( daemonDir, agentConfig )
      daemonManager.populate()

   mg.close( _doMountsComplete )
