#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import EntityMib
from EntityMib import IndexAllocator
from collections import defaultdict

__defaultTraceHandle__ = Tracing.Handle( "Fru.Poe" )
t0 = Tracing.trace0

dirReactor = None
controllerNameToChipMibArgs = {}
controllerNameToChipMib = {}
controllerNameToSensorMib = defaultdict( list )

class PoeStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::Dir"
   controllerStatusType = Tac.Type( "Environment::Power::PoeStatus" )

   def __init__( self, statusDir ):
      Tac.Notifiee.__init__( self, statusDir )
      self.statusDir_ = statusDir
      self.controllerNameToChipReactor = {}
      for key in self.notifier_.keys():
         self.handleController( key )

   @Tac.handler( 'entityPtr' )
   def handleController( self, key ):
      status = self.statusDir_.entityPtr.get( key )
      statusEntry = self.statusDir_.entryState.get( key )

      if ( not status and
           not statusEntry and
           key in self.controllerNameToChipReactor ):
         t0( "Deleting reactor for: '%s' " % key )
         del self.controllerNameToChipReactor[ key ]
      elif ( status and
             statusEntry and
             isinstance( status, self.controllerStatusType ) and
             key not in self.controllerNameToChipReactor ):
         t0( "Creating reactor for: '%s' " % key )
         self.controllerNameToChipReactor[ key ] = PoeChipReactor( status )

class PoeChipReactor( Tac.Notifiee ):
   """propagates PoE chip version from hardware to entmib"""
   notifierTypeName = "Environment::Power::PoeStatus"

   def __init__( self, status ):
      Tac.Notifiee.__init__( self, status )

      for chipName in self.notifier_.chipModel:
         self.updateModelName( chipName )
      for chipName in self.notifier_.firmwareRevision:
         self.updateFirmwareRev( chipName )

   def updateModelName( self, chipName ):
      modelName = self.notifier_.chipModel[ chipName ]
      if chipName in controllerNameToChipMibArgs:
         if chipName not in controllerNameToChipMib:
            parentMibEntity, device, chipId = controllerNameToChipMibArgs[ chipName ]
            poeChipMib = parentMibEntity.chip.get( chipId )
            if poeChipMib is None:
               # create EntityMib object for PoE chip
               physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                  parentMibEntity.chip, chipId )
               poeChipMib = parentMibEntity.newChip( physicalIndex,
                                                     chipId,
                                                     modelName + "-" +
                                                     str( chipId % 100 ) )
               EntityMib.populateMib( poeChipMib, device )
            t0( "Publishing chip model", modelName, "for", chipName )
            poeChipMib.modelName = modelName
            poeChipMib.label = str( chipId )
            poeChipMib.description = "PSE Chip %s" % chipId

            controllerNameToChipMib[ chipName ] = poeChipMib
         if self.notifier_.firmwareRevision[ chipName ]:
            self.updateFirmwareRev( chipName )

   def updateFirmwareRev( self, chipName ):
      firmwareRev = self.notifier_.firmwareRevision[ chipName ]
      if chipName in controllerNameToChipMib:
         controllerNameToChipMib[ chipName ].firmwareRev = firmwareRev
         t0( "Publishing firmware revision", firmwareRev, "for", chipName )

   @Tac.handler( 'chipModel' )
   def handleChipModel( self, key ):
      t0( "Chip model changed" )
      self.updateModelName( key )

   @Tac.handler( 'firmwareRevision' )
   def handleFirmwareVersion( self, key ):
      t0( "Firmware revision changed" )
      self.updateFirmwareRev( key )
