#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from dataclasses import dataclass
from functools import cache
from itertools import (
   chain,
   permutations,
)

# psuIncompatibilities is a list containing pairs of incompatible PSUs - to add a new
# incompatibility, add a pair of tuples of the form (skuName,revision).
# If several PSUs are incompatible, add one entry for each incompatible pair
# i.e. if PSUs A2, B2, and C2 all do not work with each other, you would add the
# following entries:
#
# ( ( A, 2 ), ( B, 2 ) ),
# ( ( A, 2 ), ( C, 2 ) ),
# ( ( B, 2 ), ( C, 2 ) )
#
# Here A, B, C represent the skuNames and 2 represents their revision. For
# incompatible PSUs regardless of revision, you may omit the revision entirely (i.e.
# ( ( A, ), ( B, ) ) 
#
# You do not need to add the inverse relationship, ( B, 2 ), ( A, 2 ) gets
# automatically included in incompatibilityMap below.

psuIncompatibilities = [
   ( ( 'PWR-1511-AC-RED', 'S3' ), ( 'PWR-1511-AC-RED', 'S4' ) ),
   ( ( 'PWR-1511-DC-RED', 'S3' ), ( 'PWR-1511-DC-RED', 'S4' ) ),
   ( ( 'PWR-1511-AC-BLUE', 'S3' ), ( 'PWR-1511-AC-BLUE', 'S4' ) ),
   ( ( 'PWR-1511-DC-BLUE', 'S3' ), ( 'PWR-1511-DC-BLUE', 'S4' ) ),
   ( ( 'PWR-1900AC-F', ), ( 'PWR-2401-AC-RED', ) ),
   ( ( 'PWR-1900AC-R', ), ( 'PWR-2401-AC-BLUE', ) ),
]

@dataclass( frozen=True )
class PowerSupplyDesc:
   skuName: str
   revision: str = ''

@cache
def incompatibilityMap():
   ''' Returns a map detailing all incompatibilities for each PSU model (based on
       'psuIncompatibilites' above).
   '''
   incompatibilities = {}
   for modelArgs, otherArgs in psuIncompatibilities:
      for modelDesc, otherDesc in permutations( ( PowerSupplyDesc( *modelArgs ),
                                                  PowerSupplyDesc( *otherArgs ) ) ):
         incompatibilities.setdefault( modelDesc, [] ).append( otherDesc )
   return incompatibilities

def modelIncompatibilities( modelDesc ):
   ''' Returns all the incompatibilities for a given PSU model. '''
   return chain.from_iterable( incompatibilityMap().get( desc, [] ) for desc in
                               ( modelDesc, PowerSupplyDesc( modelDesc.skuName ) ) )

@cache
def checkCompatibility( modelDesc, otherDesc ):
   ''' Returns True if the two given PSUs are compatible with one another. '''
   for incompatModel in modelIncompatibilities( modelDesc ):
      revisionMatch = ( incompatModel.revision == otherDesc.revision or
                        not incompatModel.revision )
      if incompatModel.skuName == otherDesc.skuName and revisionMatch:
         return False
   return True
