# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Fru, ArbusAgent
import FruPlugin.Gpio as Gpio  # pylint: disable=consider-using-from-import
import Toggles.FruToggleLib

import Tracing
__defaultTraceHandle__ = Tracing.Handle( "Fru.SmbusAccelFru" )
t4 = Tracing.trace4

class SmbusAccelDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::PciScd"
   requires = [ Gpio.gpioInit ]
   primaryDriver = False

   def __init__( self, scd, parentMibEntity, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, scd, parentMibEntity, parentDriver, driverCtx )

      if not scd.smbusAccelBlock and not scd.smbusPlutoProxy:
         # Nothing to do
         return

      # Determine the correct location of the 'SmbusEngine' in Sysdb
      # for this particular Scd. For accelerators located on a cell,
      # this is 'hardware/cell/<cellId>'. For accelerators located on
      # a slice, this is 'hardware/smbus/slice/<sliceId>'.
      sysdbRoot = driverCtx.sysdbRoot

      # If not on a slice, the system agent ID is used.
      agentId = ArbusAgent.sliceIdToAgentId( Fru.fruBaseName( scd ) )
      # Use scd sliceName if supplied
      if scd.sliceName:
         if Toggles.FruToggleLib.toggleFixedSliceEnabled():
            agentId = ArbusAgent.sliceIdToAgentId( scd.sliceName )

      # Create the qualPath for slice agent
      hwSmbus = Fru.newStyleHardwareDir( sysdbRoot[ 'hardware' ][ 'smbus' ],
                                         'Hardware::SmbusEngine',
                                         scd )

      t4( "Wiring up the scd driver ham on scd ", scd.chipId,
          "to ", scd.ham )

      for smbusAccel in scd.smbusAccelBlock.values():
         # Interrupt controller (if present)
         intCtrl = None
         if smbusAccel.interruptController:
            intCtrl = smbusAccel.interruptController.fruIntCtrl
         hwAccel = Fru.Dep( hwSmbus.accel, scd ).newMember( smbusAccel.id,
                                                            smbusAccel.version,
                                                            smbusAccel.offset,
                                                            intCtrl )
         hwAccel.registerOffsets = scd.smbusAccelBlockRegisterOffset
         hwAccel.transient = smbusAccel.transient
         hwAccel.ham = scd.ham

         if smbusAccel.scannerResetGpo:
            hwAccel.scannerResetGpoName = smbusAccel.scannerResetGpo.systemName

         # Set up the bus timeout mappings
         for ( timeout, bits ) in smbusAccel.busTimeoutToBits.items():
            hwAccel.busTimeoutToBits[ timeout ] = bits
         hwAccel.ready = True

      for accel in scd.smbusPlutoProxy.values():
         Fru.Dep( hwSmbus.proxyAccel, scd ).newMember( accel.id )

      hwSmbus.agentId = agentId
      hwSmbus.engineGenerationId = Tac.Value( "Ark::Generation",
                                              Fru.powerGenerationId( scd ),
                                              True )

def Plugin( context ):
   context.registerDriver( SmbusAccelDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/smbus', 'Tac::Dir', 'wi' )
   mg.close( None )
