# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pylint: disable=F0401

import Cell
import EntityMib
from EntityMib import ChipType, IndexAllocator
import FpgaUtil
import Fru
import FruPlugin.Gpio, FruPlugin.Smbus
import FruPlugin.Led as LedHelper
import Tac
import Tracing

import sys

__defaultTraceHandle__ = Tracing.Handle( "Fru.StandbyCpld" )
t0 = Tracing.trace0

class CpldDriver( Fru.FruDriver ):
   managedApiRe = ".*$"

   provides = [ "StandbyCpld", FruPlugin.Gpio.gpioInit ]

   def aham( self, cpld, configDir, ctx ):
      raise NotImplementedError

   def gpoAham( self, cpld, gpo, ctx ):
      raise NotImplementedError

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx, chipType ):
      Fru.FruDriver.__init__( self, cpld, parentMibEntity, parentDriver, driverCtx )
      fruBase = Fru.fruBase( cpld )
      cellId = fruBase.managingCellId
      if cellId == 0:
         # LC or FC CPLD. Use slotId. Supe StandbyCpld agent will handle this CPLD
         cellId = Fru.slotId()
      hardwareDir = driverCtx.sysdbRoot.entity[ "hardware/cell/%d" % cellId ]
      cpldDir = hardwareDir.mkdir( "cpld" )
      configDir = cpldDir.newEntity( "Hardware::StandbyCpld::SystemConfig",
                                     "config" )

      #-----------------------------------
      # Create the cpld hw config
      ahamDesc = self.aham( cpld, configDir, driverCtx )

      hwCpldConfig = Fru.Dep( configDir.cpldConfig, cpld ).newMember( cpld.name,
                                                                      ahamDesc )
      hwCpldConfig.customerName = Fru.fruBaseName( cpld )

      hwCpldConfig.firmwareRevReg = cpld.firmwareRevReg
      hwCpldConfig.hardwareMajorRevReg = cpld.hardwareMajorRevReg

      if fruBase.managingCellId == 0:
         hwCpldConfig.slotId = fruBase.slot

      if cpld.suicideRegisterOffset:
         hwCpldConfig.suicideRegisterOffset = cpld.suicideRegisterOffset

      if cpld.cpldBasedReloadCause:
         hwCpldConfig.cpldBasedReloadCause = cpld.cpldBasedReloadCause
      if cpld.reloadCauseRegister:
         hwCpldConfig.reloadCauseRegister = cpld.reloadCauseRegister
         hwCpldConfig.reloadCauseRegisterMask = cpld.reloadCauseRegisterMask
      if cpld.reloadCauseResetRegister:
         hwCpldConfig.reloadCauseResetRegister = cpld.reloadCauseResetRegister
      if cpld.reloadCauseRtcBaseRegister:
         hwCpldConfig.reloadCauseRtcBaseRegister = cpld.reloadCauseRtcBaseRegister
      hwCpldConfig.railFaultEnableRegister = cpld.railFaultEnableRegister
      hwCpldConfig.railFaultEnableBit = cpld.railFaultEnableBit
      hwCpldConfig.isMainController = cpld.isMainController
      hwCpldConfig.altIndicatorSource = cpld.altIndicatorSource
      for errVal, cause in cpld.reloadCauseMap.items():
         hwCpldConfig.reloadCauseMap[ errVal ] = cause

      for offset, value in cpld.registerInitializer.items():
         hwCpldConfig.registerInitializer[ offset ] = value

      for addr, status in cpld.powerGoodStatusInfo.items():
         s = hwCpldConfig.newPowerGoodStatusInfo( addr, status.name )
         for offset, name in status.statusNames.items():
            s.statusNames[ offset ] = name

      gpoDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpo' ]
      for block in cpld.registerGpoBlock.values():
         for bit in block.bit.values():
            systemName = '%s-%s-%d' % ( cpld.name, block.name, bit.bitpos )
            fruGpo = Fru.Dep( gpoDir, cpld ).newEntity(
               'Hardware::CpldRegisterGpo',
               systemName )
            fruGpo.ahamDesc = self.gpoAham( cpld, fruGpo, driverCtx )
            fruGpo.offset = block.offset
            fruGpo.bit = bit.bitpos
            fruGpo.activeType = bit.activeType
            bit.fruGpo = fruGpo
            bit.systemName = systemName

      self.hwSystemCpldConfig = hwCpldConfig

      chipId = IndexAllocator.getChipId( chipType, cpld.chipId )
      chipMib = self.parentMibEntity_.chip.get( chipId )
      if chipMib is None:
         physicalIndex = (
            IndexAllocator.collectionItemPhysicalIndex( self.parentMibEntity_.chip,
                                                        chipId ) )
         chipMib = self.parentMibEntity_.newChip( physicalIndex, chipId,
                                                  f"{chipType}-{cpld.chipId}")
         chipMib.label = str( chipId )
         chipMib.description = f"{chipType} {chipId}"

         EntityMib.populateMib( chipMib, cpld )

         status = driverCtx.entity( f"hardware/cpld/status/cell/{Cell.cellId()}" )

         self.cpldReactor = (
            Tac.collectionChangeReactor(status.cpldStatus, SystemCpldStatusReactor,
                                        reactorArgs=( chipMib, ),
                                        reactorFilter=lambda x: x == cpld.name ) )

class SystemCpldStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Hardware::StandbyCpld::SystemCpldStatus"

   def __init__( self, cpldStatus, chipMib ):
      Tac.Notifiee.__init__( self, cpldStatus )
      self.chipMib = chipMib
      self.handleFirmwareRevision()

   @Tac.handler( "firmwareRev" )
   @Tac.handler( "hardwareMajorRev" )
   def handleFirmwareRevision( self ):
      major = self.notifier_.firmwareRev
      minor = self.notifier_.hardwareMajorRev
      self.chipMib.firmwareRev = f"{major:#04x}.{minor:02x}"

class I2cCpldDriver( CpldDriver ):
   managedTypeName = "Inventory::I2cCpld"

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx ):
      CpldDriver.__init__( self, cpld, parentMibEntity, parentDriver, driverCtx,
                           ChipType.i2cCpld )
      self.hwSystemCpldConfig.ready = True

   def aham( self, cpld, configDir, ctx ):
      ahamDesc = Fru.Dep( configDir.i2cAhamDesc, cpld ).newMember(
         cpld.name, cpld.master.id, cpld.deviceId )
      return ahamDesc

   def gpoAham( self, cpld, gpo, ctx ):
      gpo.i2cAhamDesc = ( cpld.name, cpld.master.id, cpld.deviceId )
      return gpo.i2cAhamDesc

class SmbusCpldDriver( CpldDriver ):
   managedTypeName = "Inventory::SmbusCpld"

   def __init__( self, cpld, parentMibEntity, parentDriver, driverCtx ):
      if cpld.enablePowerCycleOnRailFault:
         cpld.registerInitializer[ cpld.powerCycleRegister ] = \
            cpld.powerCycleRegisterValue
      CpldDriver.__init__( self, cpld, parentMibEntity, parentDriver, driverCtx,
                           ChipType.smbusCpld )

      # Create ArcherPinInit
      t0( "creating archerpinInit for smbuscpld " )
      gpioInitDir = driverCtx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      for( pinBlockSubAddr, invPinBlock ) in cpld.pinBlock.items():
         hwPinBlockConfig = self.hwSystemCpldConfig.newPinBlock( pinBlockSubAddr,
                                                                 invPinBlock.count )
         for ( pinId, invPin ) in invPinBlock.gpioPin.items():
            assert ( 1 <= pinId + invPin.count <= invPinBlock.count ), (
               "Bit location of pin (pin id: %d) plus its count (%d) are outside "
               "the range of [ 1, %d ]."
               % ( pinId, invPin.count, invPinBlock.count ) )
            systemName = "%s-%d-%d" % ( cpld.name, pinBlockSubAddr, pinId )
            t0( "Creating system cpld gpioPins for %s and id %d with systemName %s"
               % ( invPin.pinName, pinId, systemName ) )
            pinInit = Fru.Dep( gpioInitDir, cpld ).newEntity(
               "Hardware::Gpio::ArcherPinInit", systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwPinBlockConfig.gpioPinName[ systemName ] = True

      if cpld.busReadyGpioPin:
         self.hwSystemCpldConfig.busReadyGpioPinName = \
               cpld.busReadyGpioPin.systemName

      self.hwSystemCpldConfig.ready = True

   def aham( self, cpld, configDir, ctx ):
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = FruPlugin.Smbus.ahamDesc( topology, cpld )
      ahamDesc = Fru.Dep( configDir.smbusAhamDesc, cpld ).newMember(
         cpld.name, *ahamArgs )
      return ahamDesc

   def gpoAham( self, cpld, gpo, ctx ):
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      ahamArgs = FruPlugin.Smbus.ahamDesc( topology, cpld )
      gpo.smbusAhamDesc = ( cpld.name, ) + ( ahamArgs )
      return gpo.smbusAhamDesc

class StandbyCpldDriver( Fru.FruDriver ):

   managedTypeName = "Inventory::StandbyCpld"
   managedApiRe = ".*"

   def __init__( self, invStandbyCpld, parentMibEnt, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStandbyCpld, parentMibEnt, parentDriver, ctx )

      standbyCpldConfig = ctx.sysdbRoot[ 'hardware' ][ 'standbyCpld' ][ 'config' ]
      standbyCpldStatus = ctx.entity( 'hardware/standbyCpld/status' )
      topology = ctx.sysdbRoot[ 'hardware' ][ 'smbus' ][ 'topology' ]
      gpioInitDir = ctx.sysdbRoot.entity[ 'hardware/archer/gpio/init' ]

      name = "%d-%s" % ( parentMibEnt.relPos, invStandbyCpld.name )
      hwCpldConfig = Fru.Dep( standbyCpldConfig.cpldConfig,
                              invStandbyCpld ).newMember( name )
      hwCpldConfig.generationId = Fru.fruBase( invStandbyCpld ).generationId
      hwCpldConfig.ahamDesc = ( "cpldAhamDesc", ) + \
                              FruPlugin.Smbus.ahamDesc( topology, invStandbyCpld )
      hwCpldConfig.customerName = Fru.fruBaseName( invStandbyCpld )

      hwCpldConfig.revisionRegister = invStandbyCpld.revisionRegister
      hwCpldConfig.minorRevRegister = invStandbyCpld.minorRevRegister

      # Pin blocks to define gpio registers
      for ( pinBlockSubAddr, invPinBlock ) in invStandbyCpld.pinBlock.items():
         hwPinBlockConfig = hwCpldConfig.newPinBlock( pinBlockSubAddr,
                                                      invPinBlock.count )
         # Pins within the pin blocks (i.e. bits within gpio registers)
         for ( pinId, invPin ) in invPinBlock.gpioPin.items():
            # When writing pin values, we don't want them to wrap around
            assert ( 1 <= pinId + invPin.count <= invPinBlock.count ), \
                   "Bit location of pin (pin id: %d) plus its count (%d) are " \
                   "outside the range of [ 1, %d ]." \
                   % ( pinId, invPin.count, invPinBlock.count )
            t0( "  Creating gpioPins for %s and id %d " % (invPin.pinName, pinId ) )
            # Archer GPIO initialization
            systemName = "%s-%d-%d" % ( name, pinBlockSubAddr, pinId )
            pinInit = Fru.Dep( gpioInitDir, invStandbyCpld ).newEntity(
               "Hardware::Gpio::ArcherPinInit",
               systemName )
            pinInit.pinName = invPin.pinName
            pinInit.pinId = pinId
            pinInit.count = invPin.count
            pinInit.activeType = invPin.activeType
            pinInit.setToInputIfInactive = invPin.setToInputIfInactive
            pinInit.setToInputIfActive = invPin.setToInputIfActive
            pinInit.direction = invPin.direction
            pinInit.defaultVal = invPin.defaultVal
            invPin.systemName = systemName
            hwPinBlockConfig.gpioPinName[ systemName ] = True
      if invStandbyCpld.controllerInterface:
         hwCpldConfig.fanController = ( 'fanController', )
         invStandbyCpld.controllerInterface.hwFanController = \
               hwCpldConfig.fanController
      hwCpldConfig.pwmRegisterBase = invStandbyCpld.pwmRegisterBase
      hwCpldConfig.tachRegisterBase = invStandbyCpld.tachRegisterBase
      hwCpldConfig.pwmRegisterStride = invStandbyCpld.pwmRegisterStride
      hwCpldConfig.tachRegisterStride = invStandbyCpld.tachRegisterStride
      hwCpldConfig.fanFaultRegister = invStandbyCpld.fanFaultRegister

      # Leds
      fruBaseName = Fru.fruBaseName( invStandbyCpld )
      for ( ledName, cpldLedBlock ) in invStandbyCpld.cpldLedBlock.items():
         t0( "Create Led %s" % ledName )
         # For each Led, create a ledConfig that will be updated by LedPolicyAgent.
         ledConfig = LedHelper.addLed( ledName, fruBaseName,
                                       invStandbyCpld, ctx )
         # Right now we use LedAttribute defined in InvScd to store blueLedOffset,
         # which is defined as U32 (since Scd's register is 32 bits), to keep the 
         # implementation consistent with Leds controlled by Scd.
         # But the offset for Leds controlled by StandbyCpld is 8 bit address (U8).
         # So we count on fdl implemetator to set the right address (0x00 to 0xff),
         # but it's worth to have an assert here.
         assert 0x00 <= cpldLedBlock.capability.blueLedOffset <= 0xff
         ledBlock = Tac.Value( "Hardware::StandbyCpld::CpldLedBlock", 
                               cpldLedBlock.offset,
                               cpldLedBlock.greenBit,
                               cpldLedBlock.redBit,
                               cpldLedBlock.capability.blueLedOffset, 
                               cpldLedBlock.blueBit )
         hwCpldConfig.cpldLedBlock[ ledName ] = ledBlock
         ledAttribute = Tac.Value( "Led::LedAttribute",
                                    blue=cpldLedBlock.capability.blue )
         ledConfig.ledAttribute = ledAttribute
      hwCpldConfig.maxSequentialTachFailures = \
          invStandbyCpld.maxSequentialTachFailures

      if invStandbyCpld.kernelPanicEeprom:
         hwCpldConfig.kernelPanicEepromI2cAham = ( name,
                  invStandbyCpld.kernelPanicEeprom.master.id,
                  invStandbyCpld.kernelPanicEeprom.deviceId )
         hwCpldConfig.kernelPanicEepromI2cAham.waitBetweenTransactions = \
               invStandbyCpld.kernelPanicEeprom.waitBetweenTransactions
         hwCpldConfig.kernelPanicEepromAham = hwCpldConfig.kernelPanicEepromI2cAham
         if invStandbyCpld.kernelPanicEepromWriteProtect:
            # not every device has a write protect pin
            hwCpldConfig.kernelPanicEepromWpGpo = \
                  invStandbyCpld.kernelPanicEepromWriteProtect.systemName
      
      hwCpldConfig.ready = True

      # Mendocino doesn't have a separate standby cpld to control its fans,
      # so it doesn't make sense to track its revision
      if invStandbyCpld.revisionRegisterValid:
         chipId = IndexAllocator.getChipId( ChipType.standbyCpld, 
                                            invStandbyCpld.chipId )

         chipMib = self.parentMibEntity_.chip.get( chipId )
         if chipMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( self.parentMibEntity_.chip, chipId )
            chipMib = self.parentMibEntity_.newChip( physicalIndex, chipId,
                                                     f"stdbycpld-"
                                                     f"{invStandbyCpld.chipId}" )
            chipMib.label = str( chipId )
            chipMib.description = "Standby Cpld Chip %d" % invStandbyCpld.chipId
            EntityMib.populateMib( chipMib, invStandbyCpld )

         self.cpldReactor = (
            Tac.collectionChangeReactor(standbyCpldStatus.cpldStatus,
                                        StandbyCpldStatusReactor,
                                        reactorArgs=( chipMib, ),
                                        reactorFilter=lambda x: x == name ) )

class StandbyCpldDirDriver( Fru.FruDriver ):

   # Technically, StandbyCpldDriver provides gpioInit. However, because the
   # provides-requires mechanism is only effective within one stage/generation
   # of FruPlugin's, we need to put it here. In other words, standbyDomain
   # won't look for "provides" in *Cpld* since it is a grandchild. But it will
   # know to look for "provides" in *CpldDir* since it is a direct child.
   provides = [ FruPlugin.Gpio.gpioInit ]

   managedTypeName = "Inventory::StandbyCpldDir"
   managedApiRe = ".*"

   # TODO: rework this part

   def __init__( self, invStandbyCpldDir, parentMibEnt, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invStandbyCpldDir,
                              parentMibEnt, parentDriver, ctx )
      self.standbyCpldDriver_ = {}
      for invStandbyCpld in invStandbyCpldDir.standbyCpld.values():
         t0( "StandbyCpldDriver looking for driver for cpld", invStandbyCpld.name )
         try:
            self.standbyCpldDriver_[ invStandbyCpld.name ] = ctx.driverRegistry.\
                  newDriver( invStandbyCpld, parentMibEnt, parentDriver, ctx )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            #XXX What should be done here?
            raise

class StandbyCpldStatusReactor( Tac.Notifiee ):
   """propagates StandbyCpld version from hardware to entmib"""
   notifierTypeName = "Hardware::StandbyCpld::CpldStatus"

   def __init__( self, cpldStatus, chipMib ):
      Tac.Notifiee.__init__( self, cpldStatus )
      self.chipMib = chipMib
      self.handleFirmwareRevision()
      self.handleFirmwareMinorRev()

   @Tac.handler( 'revision' )
   def handleFirmwareRevision( self ):
      major = self.notifier_.revision
      minor = FpgaUtil.parseVersion( self.chipMib.firmwareRev ).minor
      self.chipMib.firmwareRev = "0x%02x.%02x" % ( major, minor )

   @Tac.handler( 'minorRev' )
   def handleFirmwareMinorRev( self ):
      minor = self.notifier_.minorRev
      major = FpgaUtil.parseVersion( self.chipMib.firmwareRev ).major
      self.chipMib.firmwareRev = "0x%02x.%02x" % ( major, minor )

def Plugin( ctx ):
   ctx.registerDriver( StandbyCpldDriver )
   ctx.registerDriver( StandbyCpldDirDriver )
   ctx.registerDriver( I2cCpldDriver )
   ctx.registerDriver( SmbusCpldDriver )

   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/standbyCpld/config', 'Hardware::StandbyCpld::Config', 'w' )
   mg.mount( 'hardware/standbyCpld/status', 'Hardware::StandbyCpld::Status', 'r' )
   mg.mount( f'hardware/cpld/status/cell/{Cell.cellId()}',
             'Hardware::StandbyCpld::SystemStatus', 'r' )
   mg.mount( 'hardware/archer/gpo', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/archer/gpio/init', 'Tac::Dir', 'wi' )
   mg.close( None )
