#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This plugin provides a FruFactory that is used when EOS in running
# in a VM (vEOS). We synthesize a FDL based on what we can read out of
# the Linux kernel. We assume that we're on a fixed system. Interfaces
# that begin with 'ma' are assumed to be Management Ethernet
# Interfaces, while interfaces that begin with 'vmnicet' are assumed
# to be virtual nic interfaces for front-panel switchports. All other
# interfaces are ignored.

import Tac
import Tracing
from Fru.FruBaseVeosDriver import (
      parseVeosConfig,
      handleEtbaOverride,
)
from FruPlugin.GenericPc import renameKernelDevices
from FruPlugin.EosInitFruPluginLib import (
      VEosLabCEosLabBaseDriver,
      Config,
      JsonIntfConfig,
      IntfType,
)
from EbraTestBridgeConstants import bridgingEtbaConfigPath, bridgingEtbaConfigType
import VeosHypervisor

__defaultTraceHandle__ = Tracing.Handle( "Fru.vEOScEOSLab" )
t0 = Tracing.trace0

class VEosConfig( Config ):

   def __init__( self ):
      self.veosConfig = parseVeosConfig()

   def getSerialNumber( self ):
      return self.veosConfig[ 'SERIALNUMBER' ]

   def getSystemMacAddr( self ):
      return self.veosConfig[ 'SYSTEMMACADDR' ]

   def getMode( self ):
      return self.veosConfig[ 'MODE' ]

   def handleEtbaOverride( self ):
      return handleEtbaOverride( self.veosConfig )

class VEosLabDriver( VEosLabCEosLabBaseDriver ):
   managedTypeName = "Eos::GenericPcFru"
   managedApiRe = "veos$"
   # Higher priority than the default GenericPcDriver
   driverPriority = 3

   def __init__( self, genericPc, parentMibEntity, parentDriver, driverCtx ):
      t0( "VEosLabDriver: Initializing" )
      config = VEosConfig()
      self.ethDevPfxs = { 'et', 'vmnicet' }
      VEosLabCEosLabBaseDriver.__init__( self,
                                         genericPc,
                                         parentMibEntity,
                                         parentDriver,
                                         driverCtx,
                                         modelName="vEOS-lab",
                                         description="vEOS for lab use",
                                         config=config )

   def getValidDevices( self ):
      # VEosLab accepts any interface (it will rename them)
      return self.getDevices()

   def parseKernelDevices( self, devNames, config ):
      devConfig = []
      intfConfig = JsonIntfConfig()
      nameMapping = intfConfig.getNameMapping()
      if nameMapping is None:
         devNames = renameKernelDevices( devNames, config.getMode() )
         for devName in devNames:
            labels = self.getLabels( devName )
            if labels is None:
               # Can't get labels, unsupported format (ex. et2.4096 kernel subintf)
               continue
            devType, label, subLabel, subSubLabel = labels
            intfType = self.intfTypeFromDevType( devType, config, self.ethDevPfxs )
            devConfig.append( ( devName, intfType, label, subLabel, subSubLabel ) )
      else:
         for devName in devNames:
            desiredIntfName = nameMapping.get( devName, None )
            if not desiredIntfName:
               # Device isn't mapped, just ignore.
               continue
            labels = self.getLabels( desiredIntfName )
            if labels is None:
               # Can't get labels, malformed desired interface name
               continue
            intfPfx, label, subLabel, subSubLabel = labels
            intfType = self.intfTypeFromDesiredIntfPrefix( intfPfx, config )
            devConfig.append( ( devName, intfType, label, subLabel, subSubLabel ) )
      return devConfig

   def handleDevice( self, devName, intfType, label, subLabel, subSubLabel,
                     mac, portId, etbaConfig, ethPortDir, pciPortDir,
                     phyEthtoolPhyDir ):
      # We support 3 types of interfaces:
      #    - maN, which we assume are management interfaces
      #    - vmnicetN, which we assume are test front-panel interfaces
      #      created as part of vEOS which will be managed by Etba.
      #    - etN, which we assume are test front-panel interfaces
      #      created as part of vEOS which will be managed by
      #      PhyEthtool
      # All interfaces are injested and renamed to one of these 3, so
      # this should be exhaustive. If the JsonIntfConfig exists, no interface
      # renaming takes place and we ignore any kernel devices that aren't
      # explicitly mapped to an interface.
      if intfType is IntfType.management:
         assert not subSubLabel
         port = pciPortDir.newPort( portId )
         port.description = "Management Ethernet Port %d" % label
         port.role = "Management"
         # Don't set a pci address, the EthPciPort fru driver will assign
         # things correctly based on the device name

         # Create an Inventory::PhyEthtoolPhy
         phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" % ( intfType, label ) )
         phy.port = port

      elif intfType is IntfType.ethernetLinuxFwd:
         port = ethPortDir.newPort( portId )
         port.description = "Front-Panel Port %d" % label
         port.role = "Switched"

         # Create an Inventory::PhyEthtoolPhy
         phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" % ( intfType, label ) )
         phy.port = port

      elif intfType is IntfType.ethernet:
         tapDevice = Tac.Value( "Bridging::Etba::Config::TapDevice" )
         tapDevice.name = devName
         tapDevice.fileno = 0
         if subSubLabel:
            etbaConfig.extTapDevice[ "Ethernet%d/%d/%d" %
                  ( label, subLabel, subSubLabel ) ] = tapDevice
         elif subLabel:
            etbaConfig.extTapDevice[ "Ethernet%d/%d" %
                  ( label, subLabel ) ] = tapDevice
         else:
            etbaConfig.extTapDevice[ "Ethernet%d" % label ] = tapDevice

         port = ethPortDir.newPort( portId )
         port.description = "Front-Panel Port %d" % label
         port.role = "Switched"

         # Force the MTU of these ports to 10000. This is how
         # real hardware ports behave.
         try:
            Tac.run( [ "ifconfig", devName, "mtu", "10000" ] )
         except Tac.SystemCommandError:
            # Shoot, we're seem to be using a virtual network
            # adapter that does not support jumbo frames
            # (ie hyper-v).
            # Fall back to 1500 bytes
            Tac.run( [ "ifconfig", devName, "mtu", "1500" ] )

         # Make sure our vmnicet interfaces don't respond to arp
         # requests. This is done by the etN interface created on
         # top of it
         with open( "/proc/sys/net/ipv4/conf/%s/arp_ignore" % devName,
                    "w" ) as f:
            f.write( "1" )
         # Make sure the device is actually up. This is required
         # before the Etba agent attempts to bind to it
         Tac.run( [ "ifconfig", devName, "promisc", "up" ] )

      else:
         # This only happens during testing - we have interfaces
         # that appear in all network namespaces, so we just
         # ignore them.
         t0( "Unknown port", devName )
         return

      port.macAddr = mac
      port.label = label
      if subLabel:
         port.subLabel = subLabel
      if subSubLabel:
         port.subSubLabel = subSubLabel

   def setDeviceName( self, phyEthtoolPhyDir ):
      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            intfStatus = invPhy.port.intfStatus
            if not intfStatus.deviceName:
               intfStatus.deviceName = invPhy.port.intfId.replace( "Ethernet", "et" )

   def setBridging( self, etbaConfig, linuxConfig, config ):
      # Clone vmnicetX mac to etX on Azure
      if VeosHypervisor.platformAzure():
         etbaConfig.useRoutedPortAddr = True
      # Etba is always presumed configured on VEoslab even with linux forwarding,
      # and I'm not sure why.
      linuxConfig.enabled = config.getMode() == 'linux'
      etbaConfig.complete = True

   def setHostname( self, netConfig ):
      pass

def Plugin( context ):
   context.registerDriver( VEosLabDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( bridgingEtbaConfigPath, bridgingEtbaConfigType, 'w' )
   mg.mount( 'bridging/linux/config', 'Bridging::Linux::Config', 'w' )
   mg.close( None )
