#!/usr/bin/env python3
# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Fru
import Tac
import Tracing
from TypeFuture import TacLazyType
import XcvrLib
from XcvrLib import isCmisTypeStr
import Toggles.XcvrToggleLib

__defaultTraceHandle__ = Tracing.Handle( "XcvrFru" )

t0 = Tracing.trace0
t1 = Tracing.trace1

xcvrInit = "xcvr"

RS528 = 'ReedSolomon'
RS544 = 'ReedSolomon544'
FCFEC = 'FireCode'
NOFEC = 'Disabled'

TopologyGenerationStatus = TacLazyType(
   'Hardware::L1Topology::TopologyGenerationStatus' )
TopologyGenerationWatcher = TacLazyType(
   'Hardware::L1Topology::TopologyGenerationWatcher' )
XcvrLaneMgrHwModelGenerator = TacLazyType(
   'Xcvr::XcvrLaneMgrHwModelGenerator' )

topologyGenerationStatus = None
topologyGenerationWatcher = None

class XcvrSlotDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::Xcvr::SlotDir"
   # managedApiRe = "$"
   provides = [ xcvrInit ]
   requires = [ Fru.FruDriver.interfaceInit, Fru.FruDriver.systemInit, 'L1Topology' ]

   def __init__( self, slotDir, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, slotDir, fruEntMib,
                              parentDriver, driverCtx )
      # pylint: disable-next=consider-using-f-string
      t0( "Handling Inventory::Xcvr::SlotDir for %s" % Fru.fruBaseName( slotDir ) )

      xcvrDir = driverCtx.entity( "hardware/xcvr/config" ).parent
      envDir = driverCtx.sysdbRoot[ "environment" ]
      envTempConfig = envDir[ "temperature" ][ "config" ]
      topoDir = driverCtx.entity( "hardware/l1/topology" )
      mappingDir = driverCtx.entity( "hardware/l1/mapping" )

      invModularSystem = driverCtx.sysdbRoot[ 'hardware' ][ 'inventory' ].get(
         'modularSystem' )
      invFixedSystem = driverCtx.sysdbRoot[ 'hardware' ][ 'inventory' ].get(
         'fixedSystem' )
      invL1Topo = None

      sliceId = Fru.fruBaseName( slotDir )
      cardId = Fru.fruBaseSlotId( slotDir )

      chassisModelName = ""
      linecardModelName = ""
      if invModularSystem:
         invL1Topo =\
               invModularSystem.card[ Cell.cellId() ].component.get( 'L1Topology' )
         chassisModelName, linecardModelName = \
            self._getLinecardChassisModelNameModular( invModularSystem, cardId )
      elif invFixedSystem:
         invL1Topo = invFixedSystem.component.get( 'L1Topology' )
         chassisModelName = invFixedSystem.modelName

      self.subdomain = None
      if invL1Topo and invL1Topo.get( 'XcvrSlot' ):
         self.subdomain = invL1Topo[ 'XcvrSlot' ].subdomain


      # Now that we're know that we have xcvrs in the system, load the
      # attentuation map into sysdb
      attnTable = xcvrDir[ "attenuation" ]
      XcvrLib.populateAttenuationMap( attnTable )

      fruHelper = Tac.newInstance( "Xcvr::FruPluginHelper", "" )
      fruHelper.slotDir = slotDir
      fruHelper.xcvrDir = xcvrDir
      fruHelper.envTempConfig = envTempConfig
      fruHelper.attnTable = attnTable
      fruHelper.xgc = driverCtx.entity( "hardware/xcvr/xgc" )
      fruHelper.fruEntMib = fruEntMib
      fruHelper.fruUtil = Fru.fruUtil()
      fruHelper.generationId = Fru.powerGenerationId( slotDir )
      fruHelper.sliceName = sliceId
      fruHelper.statusRootDir = driverCtx.entity( "hardware/archer/xcvr/status" )
      fruHelper.coherentSliceDir = driverCtx.entity(
                             "hardware/archer/phy/config/data/coherent/slice" )
      fruHelper.perfMonSliceDir = driverCtx.entity(
         "hardware/xcvr/config/perfmon/slice" )
      fruHelper.perfMonCellDir = driverCtx.entity(
         "hardware/xcvr/config/perfmon/cell" )
      if Toggles.XcvrToggleLib.toggleXcvrMappingEnabled():
         fruHelper.xcvrMappingSliceDir = driverCtx.entity(
            "hardware/xcvr/config/mapping/slice" )
         fruHelper.xcvrMappingCellDir = driverCtx.entity(
            "hardware/xcvr/config/mapping/cell" )
         fruHelper.channelSliceDir = driverCtx.entity(
            "hardware/xcvr/config/channel/slice" )
         fruHelper.channelCellDir = driverCtx.entity(
            "hardware/xcvr/config/channel/cell" )
      configSliceDir = driverCtx.entity(
         "hardware/xcvr/config/slice" )
      fruHelper.moduleRequestDir = driverCtx.entity(
         "hardware/l1/topology/request/module/Fru-Xcvr" )
      fruHelper.xcvrL1TopoFruHelper = Tac.newInstance(
                             "Hardware::L1Topology::XcvrFruPluginHelper", topoDir )
      fruHelper.blancoL1TopoFruHelper = Tac.newInstance(
                             "Hardware::L1Topology::BlancoL1TopologyHelper",
                             topoDir, mappingDir )
      self.maybeUpdateCoherentConfig( fruHelper )
      fruHelper.doPopulateDerivedNames()
      self.maybeUpdatePerfMonConfigDir( fruHelper )
      self.maybeUpdateXcvrMappingConfigDir( fruHelper )
      self.maybeUpdateChannelConfigDir( fruHelper )
      fruHelper.doPopulateHardware()
      self.fruHelper = fruHelper
      self.maybeUpdateTuningParameters( fruHelper )

      if invModularSystem and sliceId in configSliceDir:
         configSliceDir[ sliceId ].chassisModelName = chassisModelName
         configSliceDir[ sliceId ].linecardModelName = linecardModelName

      fruHelper.allConfigDir.chassisModelName = chassisModelName

      topoHelper = Tac.newInstance(
         "Hardware::L1Topology::XcvrFruPluginHelper", topoDir )
      xcvrTopoDir = topoHelper.xcvrTopologyDir( sliceId )
      if xcvrTopoDir:
         for i, xcvrId in slotDir.xcvrIdOrder.items():
            xcvrTopoDir.xcvrIdOrder[ i ] = xcvrId

         t0( "xcvrIdOrder:", list( xcvrTopoDir.xcvrIdOrder.values() ) )

      if slotDir.l1TopoBasedLaneMgr:
         t0( 'Generating XCVR lane manager HW models based on L1 Topology for',
              sliceId )
         self.xcvrLaneMgrModelGenerator = XcvrLaneMgrHwModelGenerator(
            topoDir,
            mappingDir,
            topologyGenerationStatus,
            slotDir,
            Fru.fruUtil(),
            driverCtx.entity( 'hardware/xcvrLaneMgr/config' ) )

   @staticmethod
   def _unbound( xcvrSlotInv ):
      """
      Determines if all interfaces on the corresponding XCVR slot have been unbound.

      Parameters
      ----------
      xcvrSlotInv : Inventory::Xcvr::Slot

      Returns
      -------
      True if all interfaces on the XCVR slot are unbound else False
      """

      for port in xcvrSlotInv.port.values():
         # If any interface on the XCVR slot is still bound then the entire XCVR slot
         # is bound.
         if port.bound:
            return False

      return True

   def _getLinecardChassisModelNameModular( self, invModularSystem, cardId ):
      # This function retrieves the chassisModelName and linecardModelName from an
      # Inventory::ModularSystem
      #
      # Dual supervisor cards with shared uplinks (such as Tundra) also has Xcvr
      # ports that are managed by the sliced XcvrAgent. (XcvrAgent-Linecard1 ...)
      # These cards are created under ModularSystem::supeUplinkCard instead of
      # ModularSystem::card. Since a XcvrAgent linecard slice is created for both
      # supervisors, this function will return linecardModelName as the supervisor
      # model name.
      chassisModelName = ""
      linecardModelName = ""

      if not invModularSystem.chassis:
         # If Inventory::Chassis has not been created yet, Fru has not encountered
         # the first linecard slice yet and we can not retrieve the modelNames
         return chassisModelName, linecardModelName

      if cardId in invModularSystem.supeUplinkCard:
         linecardModelName = invModularSystem.supeUplinkCard[ cardId ].modelName
      elif cardId in invModularSystem.card:
         linecardModelName = invModularSystem.card[ cardId ].modelName

      return chassisModelName, linecardModelName

   def maybeUpdateTuningParameters( self, fruHelper ):
      t0( "maybeUpdateTuningParameters" )
      for slotKey in fruHelper.slotDir.xcvrSlot:
         slot = fruHelper.slotDir.xcvrSlot[ slotKey ]
         if self._unbound( slot ):
            continue

         if slot.slotType == 'qsfp':
            intfName = slot.derivedName
            conf = fruHelper.allConfigDir.xcvrConfig[ intfName ]
            self.updateQsfpTuningParameters( conf, slot )
         elif isCmisTypeStr( slot.slotType ):
            intfName = slot.derivedName
            conf = fruHelper.allConfigDir.xcvrConfig[ intfName ]
            self.updateCmisTuningParameters( conf, slot )

   def updateQsfpTuningParameters( self, xcvrConfig, slot ):
      # pylint: disable-next=consider-using-f-string
      t0( "updateQsfpTuningParameters for %s" % xcvrConfig.name )
      for lane in range( 0, 4 ):
         if lane in slot.scaledXcvrRxTraceLength:
            pre, amp = self.translateRxTrace( slot.scaledXcvrRxTraceLength[ lane ] )
         else:
            pre = Tac.Value( "Xcvr::TuningValue", False, 0, False )
            amp = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         xcvrConfig.rxOutputPreEmphasis[ lane ] = pre
         xcvrConfig.rxOutputAmplitude[ lane ] = amp

      for lane in range( 0, 4 ):
         if lane in slot.scaledXcvrTxTraceLength:
            eq = self.translateTxTrace( slot.scaledXcvrTxTraceLength[ lane ] )
         else:
            eq = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         xcvrConfig.txInputEqualization[ lane ] = eq

   def updateCmisTuningParameters( self, xcvrConfig, slot ):
      # pylint: disable-next=consider-using-f-string
      t0( "updateCmisTuningParameters for %s" % xcvrConfig.name )
      for lane in range( 0, 8 ):
         # TODO: Currently all fdl osfp/qsfpDd tuning config are not valid.
         #       We don't know how to translate them yet. Update the logic
         #       of translating trace length to tuning value. Bug 345549
         pre = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         post = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         amp = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         adaEq = Tac.Value( "Xcvr::TuningValue", False, 0, False )
         fixEq = Tac.Value( "Xcvr::TuningValue", False, 0, False )

         laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )
         laneTuningParam.rxOutputPreEmphasis = pre
         laneTuningParam.rxOutputPostEmphasis = post
         laneTuningParam.rxOutputAmplitude = amp
         laneTuningParam.txInputAdaptiveEqEnable = adaEq
         laneTuningParam.txInputEqualization = fixEq
         xcvrConfig.fdlTuningParam[ lane ] = laneTuningParam

   def maybeUpdatePerfMonConfigDir( self, fruHelper ):
      t0( "maybeUpdatePerfMonConfigDir" )
      inv = Fru.fruBase( fruHelper.slotDir )
      if inv.managingCellId:
         # Mount the perfMonConfigDir using the cellId for fixed
         # systems and modular supervisors with managed xcvrs. This is because both
         # supervisors share the same 'FixedSystem' sliceId path during SSO, which
         # causes the XcvrAgent to crash when the path gets overwritten. The cellId
         # will always be unique among supervisors and will avoid a shared reference
         # to perfMonConfigDir. Including fixed systems here for consistency
         # although their use of a single cellId already avoids the crash.
         cellId = inv.managingCellId
         perfMonConfigDir = Fru.Dep(
            fruHelper.perfMonCellDir, inv ).newEntity( "Xcvr::PerfMonConfigDir",
                                                       str( cellId ) )
      else:
         sliceId = inv.sliceId
         perfMonConfigDir = Fru.Dep(
            fruHelper.perfMonSliceDir, inv ).newEntity( "Xcvr::PerfMonConfigDir",
                                                        sliceId )
      for slot in fruHelper.slotDir.xcvrSlot.values():
         # For now, only populate dco, qsfp, and CMIS slots
         if slot.slotType in ( 'cfp2', 'qsfp' ) or isCmisTypeStr( slot.slotType ):
            perfMonConfigDir.xcvrSlot.add( slot.derivedName )
      perfMonConfigDir.generation = Tac.Value( "Ark::Generation",
         Fru.powerGenerationId( inv ) + 1, True )

   def maybeUpdateXcvrMappingConfigDir( self, fruHelper ):
      t0( "maybeUpdateXcvrMappingDir" )
      if not Toggles.XcvrToggleLib.toggleXcvrMappingEnabled():
         return
      inv = Fru.fruBase( fruHelper.slotDir )
      if inv.managingCellId:
         # Mount the xcvrMappingConfigDir using the cellId for fixed
         # systems and modular supervisors with managed xcvrs. This is because both
         # supervisors share the same 'FixedSystem' sliceId path during SSO, which
         # causes the XcvrAgent to crash when the path gets overwritten. The cellId
         # will always be unique among supervisors and will avoid a shared reference
         # to xcvrMappingConfigDir. Including fixed systems here for consistency
         # although their use of a single cellId already avoids the crash.
         xcvrMappingConfigDir = Fru.Dep(
            fruHelper.xcvrMappingCellDir, inv ).newEntity(
               "Xcvr::XcvrMappingConfigDir", str( inv.managingCellId ) )
      else:
         xcvrMappingConfigDir = Fru.Dep(
            fruHelper.xcvrMappingSliceDir, inv ).newEntity(
               "Xcvr::XcvrMappingConfigDir", inv.sliceId )
      for slot in fruHelper.slotDir.xcvrSlot.values():
         if isCmisTypeStr( slot.slotType ):
            xcvrMappingConfigDir.xcvrSlotName.add( slot.derivedName )
      xcvrMappingConfigDir.generation = Tac.Value(
         "Ark::Generation",
         Fru.powerGenerationId( inv ) + 1, True )

   def maybeUpdateChannelConfigDir( self, fruHelper ):
      t0( "maybeUpdateChannelConfigDir" )
      if not Toggles.XcvrToggleLib.toggleXcvrMappingEnabled():
         return
      inv = Fru.fruBase( fruHelper.slotDir )
      if inv.managingCellId:
         channelConfigDir = Fru.Dep(
            fruHelper.channelCellDir, inv ).newEntity( "Xcvr::ChannelConfigDir",
                                                       str( inv.managingCellId ) )
      else:
         channelConfigDir = Fru.Dep(
            fruHelper.channelSliceDir, inv ).newEntity( "Xcvr::ChannelConfigDir",
                                                        inv.sliceId )
      for slot in fruHelper.slotDir.xcvrSlot.values():
         if isCmisTypeStr( slot.slotType ):
            channelConfigDir.xcvrSlotName.add( slot.derivedName )
      channelConfigDir.generation = Tac.Value( "Ark::Generation",
         Fru.powerGenerationId( inv ) + 1, True )

   def maybeUpdateCoherentConfig( self, fruHelper ):
      for _, slot in fruHelper.slotDir.xcvrSlot.items():
         if ( slot.slotType in [ 'osfp', 'qsfpDd' ] or
              ( slot.slotType == 'cfp2' and
                slot.xcvrSlotCapabilities.cfp2DcoSupported ) ):
            inv = Fru.fruBase( fruHelper.slotDir )
            cellId = Fru.fruBase( fruHelper.slotDir ).managingCellId
            sliceId = Fru.fruBase( fruHelper.slotDir ).sliceId
            if cellId:
               coherentConfig = Fru.Dep( fruHelper.coherentSliceDir, inv ).newEntity(
                 "Phy::Coherent::PhyCoherentConfig", "FixedSystem" )
            else:
               coherentConfig = Fru.Dep( fruHelper.coherentSliceDir, inv ).newEntity(
                  "Phy::Coherent::PhyCoherentConfig", sliceId )

            coherentConfig.generation = Tac.Value( "Ark::Generation",
                                                Fru.powerGenerationId( inv ) + 1,
                                                True )

   # Returns tuple ( rxOutputPreEmphasis, rxOutputAmplitude )
   @staticmethod
   def translateRxTrace( traceLength ):
      # Configure tuning, and don't rely on any module defaults
      return ( Tac.Value( "Xcvr::TuningValue", bool( traceLength ), 0, False ),
               Tac.Value( "Xcvr::TuningValue", True, 2, False ) )

   # Returns txInputEqualization
   @staticmethod
   def translateTxTrace( traceLength ):
      if traceLength:
         # Configure tuning, and don't rely on any module defaults
         return Tac.Value( "Xcvr::TuningValue", True, 0, False )
      else:
         return Tac.Value( "Xcvr::TuningValue", False, 0, False )

# ----------------------------------------------------------------------
#
# Mount the relevant state from Sysdb.
#
# ----------------------------------------------------------------------
def Plugin( context ):
   context.registerDriver( XcvrSlotDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/xcvr/attenuation', 'Xcvr::ModelToAttenuation', 'w' )
   mg.mount( 'hardware/xcvr/config', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/xcvr/xgc', 'Xcvr::Xgc', 'r' )
   # NOTE - we don't need to mount any other subdirectories of
   # hardware/xcvr/{config,status} (like the all subdirectories), as
   # we're doing an immediate mount above. This has to be
   # mount-immediate to ensure that Fru mounts everything that it has
   # previously created in the agent restart case.
   mg.mount( 'environment/temperature/config',
             'Environment::Temperature::Config', 'w' )
   mg.mount( 'hardware/l1/topology/request/module/Fru-Xcvr',
             'Hardware::L1Topology::ModuleRequestDir', 'wc' )
   topoDir = mg.mount( 'hardware/l1/topology', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/mapping', 'Tac::Dir', 'ri' )
   mg.mount( 'hardware/xcvr/config/perfmon/slice', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/xcvr/config/perfmon/cell', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/xcvrLaneMgr/config', 'Tac::Dir', 'wi' )
   if Toggles.XcvrToggleLib.toggleXcvrMappingEnabled():
      mg.mount( 'hardware/xcvr/config/mapping/slice', 'Tac::Dir', 'wi' )
      mg.mount( 'hardware/xcvr/config/mapping/cell', 'Tac::Dir', 'wi' )
      mg.mount( 'hardware/xcvr/config/channel/slice', 'Tac::Dir', 'wi' )
      mg.mount( 'hardware/xcvr/config/channel/cell', 'Tac::Dir', 'wi' )
   global topologyGenerationStatus
   topologyGenerationStatus = TopologyGenerationStatus()

   def mountsCompleteHook():
      global topologyGenerationWatcher
      topologyGenerationWatcher = TopologyGenerationWatcher(
         topoDir, topologyGenerationStatus )

   mg.close( mountsCompleteHook )

   # BUG48760 workaround. Objects in status keep pointers to those in
   # config. Tacc attrlog has problem dealing with it. See comments
   # attached to the bug for detailed analysis.
   context.deferredMount( 'hardware/archer/xcvr/status', 'Tac::Dir', 'ri' )
