#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Smash
import GrpcCounterLib
import Tracing

trace2 = Tracing.trace2
FapId = Tac.Type( 'FlexCounters::FapId' )
IpFlexCountersIndex = Tac.Type( 'FlexCounters::IpFlexCountersIndex' )

def getPktCounter( idx, counterTable ):
   runningCtr = counterTable.counter.get( idx )
   currentCtrPkts = runningCtr.pkts if runningCtr else 0
   trace2( '   %d pkts' % currentCtrPkts )
   return currentCtrPkts

def getByteCounter( idx, counterTable ):
   runningCtr = counterTable.counter.get( idx )
   currentCtrOctets = runningCtr.octets if runningCtr else 0
   trace2( '   %d octets' % currentCtrOctets )
   return currentCtrOctets

class Ipv4v6CounterProvider( GrpcCounterLib.DataProvider ):

   COUNTER_TYPE = "V4V6_COUNTER"
   COMMAND_OPT = "v4v6"

   def __init__( self, collector ):
      GrpcCounterLib.DataProvider.__init__( self, collector )
      self.rxIpCounterTable = None
      self.txIpCounterTable = None
      self.flexCounterFeatureDir = None
      self.rxL3IpCounterTable = None
      self.txL3IpCounterTable = None
      self.subIntfStatusDir = None

   def doInit( self, mg, shmemEm ):
      smi = Smash.mountInfo( 'reader' )
      mountPath = \
         'flexCounters/counterTable/Ipv4v6Ingress/%u' % ( FapId.allFapsId )
      self.rxIpCounterTable = shmemEm.doMount( mountPath,
                                               "FlexCounters::IpCounterTable", smi )
      mountPath = \
         'flexCounters/counterTable/Ipv4v6Egress/%u' % ( FapId.allFapsId )
      self.txIpCounterTable = shmemEm.doMount( mountPath,
                                               "FlexCounters::IpCounterTable", smi )
      self.flexCounterFeatureDir = mg.mount( 'flexCounter/featureStatusDir',
         "Tac::Dir", 'ri' )
      mountPath = \
         'flexCounters/counterTable/L3Ipv4v6Ingress/%u' % ( FapId.allFapsId )
      self.rxL3IpCounterTable = shmemEm.doMount( mountPath,
                                                 "FlexCounters::IpCounterTable",
                                                 smi )
      mountPath = \
         'flexCounters/counterTable/L3Ipv4v6Egress/%u' % ( FapId.allFapsId )
      self.txL3IpCounterTable = shmemEm.doMount( mountPath,
                                                 "FlexCounters::IpCounterTable",
                                                 smi )
      self.subIntfStatusDir = mg.mount( 'interface/status/subintf',
         "Interface::SubIntfStatusDir", 'r' )

   def fillCounter( self, intfName, intfCounters, rxTable, txTable ):
      v4index = IpFlexCountersIndex.v4IntfCounterKey( intfName )
      v6index = IpFlexCountersIndex.v6IntfCounterKey( intfName )
      # ingress counters
      intfCounters[ 'ipv4InPkts' ] = getPktCounter( v4index, rxTable )
      intfCounters[ 'ipv6InPkts' ] = getPktCounter( v6index, rxTable )
      intfCounters[ 'ipv4InOctets' ] = getByteCounter( v4index, rxTable )
      intfCounters[ 'ipv6InOctets' ] = getByteCounter( v6index, rxTable )
      # egress counters
      intfCounters[ 'ipv4OutPkts' ] = getPktCounter( v4index, txTable )
      intfCounters[ 'ipv6OutPkts' ] = getPktCounter( v6index, txTable )
      intfCounters[ 'ipv4OutOctets' ] = getByteCounter( v4index, txTable )
      intfCounters[ 'ipv6OutOctets' ] = getByteCounter( v6index, txTable )

   def getCounters( self ):
      ipCounters = {}
      trace2( 'Ipv4v6Counters getCounters' )
      allEthIntfCounterDir = self.collector.allEthIntfCounterDir()
      ingl3Ipv4v6Feature = self.flexCounterFeatureDir.get( 'L3Ipv4v6Ingress', None )
      ingl3Ipv4v6FeatureEnabled = ingl3Ipv4v6Feature and ingl3Ipv4v6Feature.state
      rxCounterTable = self.rxL3IpCounterTable if ingl3Ipv4v6FeatureEnabled else \
                       self.rxIpCounterTable
      egrl3Ipv4v6Feature = self.flexCounterFeatureDir.get( 'L3Ipv4v6Egress', None )
      egrl3Ipv4v6FeatureEnabled = egrl3Ipv4v6Feature and egrl3Ipv4v6Feature.state
      txCounterTable = self.txL3IpCounterTable if egrl3Ipv4v6FeatureEnabled else \
                       self.txIpCounterTable

      for intfName in allEthIntfCounterDir.intfCounterDir.keys():
         if intfName.startswith( 'Ethernet' ):
            trace2( 'Get interface counters for %s' % intfName )
            intfCounters = {}
            self.fillCounter( intfName, intfCounters, rxCounterTable,
                              txCounterTable )
            ipCounters[ intfName ] = intfCounters
      for intfName in self.subIntfStatusDir.intfStatus.keys():
         trace2( 'Get subinterface counters for %s' % intfName )
         subIntfCounters = {}
         self.fillCounter( intfName, subIntfCounters, rxCounterTable,
                           txCounterTable )
         ipCounters[ intfName ] = subIntfCounters

      counters = { "interfaces" : ipCounters }
      return counters

class LegacyIpv4v6CounterProvider( Ipv4v6CounterProvider ):
   """This is hidden for backward compatibility"""
   COUNTER_TYPE = "V4V6_COUNTERS"
   COMMAND_OPT = "v4v6-counters"
   HIDDEN_COMMAND_OPT = True

def Plugin( context ):
   context.register( Ipv4v6CounterProvider )
   context.register( LegacyIpv4v6CounterProvider )

