#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import GrpcCounterLib
import Tac
import Tracing

trace2 = Tracing.trace2

class QueueProvider( GrpcCounterLib.DataProvider ):

   COUNTER_TYPE = "EGRESS_QUEUE_COUNTER"
   COMMAND_OPT = "egress-queue"

   def __init__( self, collector ):
      GrpcCounterLib.DataProvider.__init__( self, collector )
      self.queueWriterConfigDir = None
      self.queueCounterAccessor = None
      self.queueReaderSm = None

   def doInit( self, mg, shmemEm ):
      self.queueWriterConfigDir = mg.mount(
         Cell.path( "interface/queueCounter/writerConfigDir" ),
         "Tac::Dir", "ri" )

   def initDone( self ):
      self.queueCounterAccessor = Tac.newInstance(
         'Interface::QueueCounter::CounterAccessor' )
      mountHelper = Tac.newInstance(
         'Interface::QueueCounter::SmashMountHelper',
         self.collector.entityManager().cEntityManager() )
      self.queueReaderSm = Tac.newInstance(
         'Interface::QueueCounter::WriterConfigDirSm',
         self.queueWriterConfigDir, self.queueCounterAccessor, mountHelper )

   def getQueueStatList( self, qStat, attrName ):
      # Return a list of values corresponding to the given attrName in qStat
      ret = [ 0 ] * len( qStat )
      for q, val in qStat.items():
         ret[ q ] = getattr( val, attrName )
      return ret

   def fillQueueStat( self, qStat ):
      return {
         "droppedBytes": self.getQueueStatList( qStat, 'bytesDropped' ),
         "droppedPackets": self.getQueueStatList( qStat, 'pktsDropped' ),
         "bytes": self.getQueueStatList( qStat, 'bytes' ),
      }

   def getCounters( self ):
      intfC = {}
      allEthIntfCounterDir = self.collector.allEthIntfCounterDir()
      for intf in allEthIntfCounterDir.intfCounterDir:
         intfQueueCounter = self.queueCounterAccessor.counter( intf )
         if not intfQueueCounter:
            continue

         numUcastQueues = self.queueCounterAccessor.numUnicastQueues( intf )
         trace2( 'Get egress queue drops for: %s. %s ucast queues' %
                 ( intf, numUcastQueues ) )

         qStat = {}
         for q in range( numUcastQueues ):
            qStat[ q ] = intfQueueCounter.intfQueueStat[ q ]

         intfC[ "%s" % intf ] = {
            "ucastQueues": self.fillQueueStat( qStat )
         }

      counter = {
         # This differs from the CLI output.
         "egressQueueCounters" : {
            "interfaces" : intfC
            }
         }
      return counter

def Plugin( context ):
   context.register( QueueProvider )
