#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import GrpcCounterLib
import Tac
import Tracing

trace2 = Tracing.trace2

class WredQueueProvider( GrpcCounterLib.DataProvider ):

   COUNTER_TYPE = "WRED_QUEUE_COUNTER"
   COMMAND_OPT = "wred-queue"

   def __init__( self, collector ):
      GrpcCounterLib.DataProvider.__init__( self, collector )
      self.configDir = None
      self.counterAccessor = None
      self.configSm = None

   def doInit( self, mg, shmemEm ):
      self.configDir = mg.mount( "qos/hardware/wredDropCounter", "Tac::Dir", "ri" )

   def initDone( self ):
      self.counterAccessor = Tac.newInstance( 'Qos::WredCounterAccessor' )
      mountHelper = Tac.newInstance(
         'Qos::SmashMountHelper',
         self.collector.entityManager().cEntityManager() )
      self.configSm = Tac.newInstance(
         'Qos::WredWriterConfigDirSm',
         self.configDir, self.counterAccessor, mountHelper )

   def getQueueStatList( self, qStat, counterName ):
      # Return a list of values corresponding to the given attrName in qStat
      ret = [ 0 ] * len( qStat )
      for q, val in qStat.items():
         ret[ q ] = getattr( val, counterName )
      return ret

   def fillQueueStat( self, qStat ):
      return {
         "droppedPackets": self.getQueueStatList( qStat, 'pktsDropped' ),
      }

   def getCounters( self ):
      intfC = {}
      allEthIntfCounterDir = self.collector.allEthIntfCounterDir()
      for intf in allEthIntfCounterDir.intfCounterDir:
         wredCounter = self.counterAccessor.counter( intf )
         if not wredCounter:
            continue

         numUcastQueues = self.counterAccessor.numUnicastQueues( intf )
         trace2( 'Get wred queue drops for: %s. %s ucast queues' %
                 ( intf, numUcastQueues ) )

         qStat = {}
         for q in range( numUcastQueues ):
            qStat[ q ] = wredCounter.intfWredQueueStat[ q ]

         intfC[ "%s" % intf ] = {
            "ucastQueues": self.fillQueueStat( qStat )
         }

      counter = {
         # This differs from the CLI output.
         "wredQueueCounters": {
            "interfaces": intfC
            }
         }
      return counter

def Plugin( context ):
   context.register( WredQueueProvider )
