# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import HttpService
from HttpServiceConstants import ServerConstants
import CapiConstants

class Capi( HttpService.HttpServiceConfig ):
   def externalServerConfig( self, vrf ):
      # pylint: disable-next=consider-using-f-string
      capiCommon = """
          # so command-api can stream responses
          uwsgi_buffering off;

          # so command-api receive larger requests
          client_max_body_size 30m;

          include uwsgi_params;
          uwsgi_param  SSL_CLIENT_VERIFY  $ssl_client_verify;
          uwsgi_param  SSL_CLIENT_S_DN    $ssl_client_s_dn_legacy;
          uwsgi_param  SSL_CLIENT_SPIFFE  $ssl_client_spiffe;
          uwsgi_param  SERVER_ADDR        $server_addr;
          uwsgi_param  VRF_NAME           {vrfName};
          uwsgi_read_timeout 60m;
          uwsgi_send_timeout 60m;
          uwsgi_pass unix://{uwsgiSocket};
          uwsgi_intercept_errors on;
          error_page 444 @drop;
      """.format( vrfName=vrf,
                  uwsgiSocket=CapiConstants.UWSGI_SOCKET )

      return """
       location = /command-api-websocket {{
          {capiCommon}
       }}

       location ~* ^/(command-api|login|logout) {{
          set $cors '';
{corsServers}

          # /command-api doesn't support GET
          if ( $request_method = 'GET' ) {{
             return 405;
          }}

          # OPTIONS indicates a CORS pre-flight request
          if ( $request_method = 'OPTIONS' ) {{
             set $cors '${{cors}}options';
          }}

          if ( $request_method = 'POST' ) {{
             set $cors '${{cors}}post';
          }}

          if ( $cors = 'truepost' ) {{
             add_header 'Access-Control-Allow-Origin' '$http_origin';
             add_header 'Access-Control-Allow-Credentials' 'true';
             add_header 'Access-Control-Expose-Headers' '{corsExposeHeaders}';
             {securityHeaders}
          }}

          if ( $cors = 'trueoptions' ) {{
             add_header 'Access-Control-Allow-Origin' '$http_origin';
             add_header 'Access-Control-Allow-Credentials' 'true';
             add_header 'Access-Control-Max-Age' {corsMaxAge};
             add_header 'Access-Control-Allow-Methods' '{allowMethods}';
             add_header 'Access-Control-Allow-Headers' '{allowHeaders}';
             add_header 'Content-Length' 0;
             add_header 'Content-Type' 'text/plain charset=UTF-8';
             {securityHeaders}
             return 204;
          }}

          {capiCommon}
       }}
      """.format( corsServers=self._getAllowedCorsOrigins(),
                  corsExposeHeaders=','.join( ServerConstants.CORS_EXPOSE_HEADERS ),
                  corsMaxAge=ServerConstants.CORS_PREFLIGHT_CACHE_TIME,
                  allowMethods=','.join( CapiConstants.ALLOWED_HTTP_METHODS ),
                  allowHeaders=','.join( ServerConstants.CORS_ALLOW_HEADERS ),
                  securityHeaders=ServerConstants.nginxSecurityHeaders( 13 ),
                  capiCommon=capiCommon )

   def internalServerConfig( self, vrf ):
      # pylint: disable-next=consider-using-f-string
      return """
       location /command-api {{
          # so command-api receive larger requests
          client_max_body_size 30m;

          include uwsgi_params;
          uwsgi_param  SERVER_ADDR        $server_addr;
          uwsgi_read_timeout 60m;
          uwsgi_send_timeout 60m;
          uwsgi_param DISABLE_AAA 1;
          uwsgi_param  VRF_NAME           {vrfName};
          uwsgi_pass unix://{uwsgiSocket};
       }}
      """.format( vrfName=vrf, uwsgiSocket=CapiConstants.UWSGI_SOCKET )

   def unixDomainSocketServerConfig( self ):
      # pylint: disable-next=consider-using-f-string
      return """
    server {{
       listen unix:{unixSocket};

       # so command-api receive larger requests
       client_max_body_size 30m;

       location / {{
          include uwsgi_params;
          uwsgi_read_timeout 60m;
          uwsgi_send_timeout 60m;
          uwsgi_param DISABLE_AAA 1;
          uwsgi_pass unix://{uwsgiSocket};
       }}

       server_name localhost;
    }}""".format( unixSocket=self.getUnixDomainSocket(),
                  uwsgiSocket=CapiConstants.UWSGI_SOCKET )

   def unixDomainSocketServerEnabled( self ):
      # uds server is only enabled when Capi is enabled
      return ( self.serverConfig and
               CapiConstants.SERVICE_NAME in self.serverConfig.service and
               self.serverConfig.service[ CapiConstants.SERVICE_NAME ].enabled )

   def getUnixDomainSocket( self ):
      return CapiConstants.SOCKET_ENDPOINT

   def _getAllowedCorsOrigins( self ):
      result = ""
      if self.serverConfig:
         for origin in self.serverConfig.corsAllowedOrigins:
            value = ( origin
                      if origin != ServerConstants.ALLOW_ALL_CROSS_ORIGIN_DOMAINS
                      else ".*" )
            # pylint: disable-next=consider-using-f-string
            result += ( "         if ($http_origin ~* (%s)) {\n"
                        "            set $cors 'true';\n"
                        "         }\n" % value )
      return result

def Plugin( ctx ):
   ctx.registerService( Capi( ctx, CapiConstants.SERVICE_NAME ) )
