#!/usr/bin/env arista-python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import math
import re

def scaleValueSi( x ):
   '''Scale a float value into a human-readable value with a prefix in
   the SI system (k, M, G etc.) that can be applied to a unit
   (e.g. 'bps' -> 'Mbps')
   
   If possible (given the subset of SI prefixes that we support) the
   value will be scaled to a value >= 1 and < 1000, returned in a
   tuple along with the necessary SI prefix. If not possible, the
   value will be transformed to use the prefix that will get us
   closest to that range, preferring values that are greater than
   1000 to values less than 1.

   At the moment this supports only the prefixes 'k', 'M', 'G' and
   'T', though other prefixes may be supported in the future. This
   function will never support the SI prefixes that do not represent
   integer powers of 1000 (hecto-, deca-, centi- etc.)

   The return value will be a tuple (float, string). If no prefix is
   necessary to represent the value in an appropriate range, an empty
   string will be returned as the prefix.
   '''

   if x < 1e3:
      return (x, '')
   elif x < 1e6:
      return (x / float(1e3), 'k')
   elif x < 1e9:
      return (x / float(1e6), 'M')
   elif x < 1e12:
      return (x / float(1e9), 'G')
   else:
      return (x / float(1e12), 'T')

def formatValueSigFigs( val, sigFigs ):
   '''Format a floating-point value to a string with a specified
   number of significant figures. This is a formatting operation, not
   a mathematical operation, so the return type is string. Scientific
   notation is never used.

   Formatting to a specified number of decimal places is well
   supported by the Python standard library, but there doesn't appear
   to be a standard way of rounding to significant figures, hence this
   interface.

   NB: Passing in a non-integer value for the number of significant
   figures may produce nonsensical answers, silently.'''

   if sigFigs < 1:
      raise ValueError( 
         'Cannot round to a non-positive number of significant figures' )

   if val == 0.0:
      # Our standard way of calculating doesn't work in this case
      return '%.*f' % ( sigFigs, val )

   if val < 0.0:
      return '-%s' % formatValueSigFigs( -val, sigFigs )

   # This works by shifting the decimal digits (multiplying / dividing
   # by a power of 10) such that when we do integer rounding we get
   # the correct number of significant figures. We then reverse the
   # shift to get the number back into the right range.
   magnitude = math.floor( math.log10( val ) )
   shift = math.pow( 10, magnitude - sigFigs + 1 )

   # Do the actual truncation, by applying integer rounding to a
   # temporarily-shifted value.
   val = shift * round( val / shift )

   # Now we have a float value with the right number of sig figs. But
   # Python doesn't let us format it easily, since it will add
   # unnecessary zeros after the decimal point by default with the %f
   # format specifier.
   trailingDecimals = max( 0, int( sigFigs - magnitude - 1 ) )

   return '%.*f' % ( trailingDecimals, val )


def plural( value, noun ):
   '''Pluralizes the singular string noun given the number value.

   e.g., plural( 2, "city" ) -> "2 cities"
         plural( -1, "kilometre" ) -> "-1 kilometre"
         plural( 473, "address" ) -> "473 addresses"

   Args:
     value: int, the value to use for pluralization test.
     noun: basestring, the singular form of the word
       to pluralize.

   Returns:
     basestring, the value and the possibly pluralized noun together.
   '''
   if not isinstance( noun, str ):
      raise TypeError( 'Second argument must be a string' )

   testNoun = noun.lower()
   if testNoun in ( 'datum', 'data' ):
      plNoun = 'data'
      noun = 'datum'
   elif testNoun.endswith( 'ty' ):
      plNoun = noun[ :-2 ] + 'ties'
   elif testNoun.endswith( 'ers' ):  # confusing singulars; eg., 'tweezers'
      plNoun = noun
      noun = noun[ :-1 ]  # singular form doesn't have the s
   elif testNoun.endswith( 's' ):
      plNoun = noun + 'es'

   # add other exceptions as elif cases above here
   # there's very few irregular plurals we need to worry about,
   # but more fun here: http://englishplus.com/grammar/00000039.htm

   else:  # default 's' plural case
      plNoun = noun + 's'

   if value == 1 or value == -1: # pylint: disable=consider-using-in
      return "%s %s" % ( value, noun )
   else:
      return "%s %s" % ( value, plNoun )

def formatTimeInterval( timeInSeconds ):
   '''Converts a given time in seconds to a concise, readable format.
      Times less than 24h are formatted: HH:MM:SS
      Times over 24h are formatted: DD:HH
   '''
   time = int( timeInSeconds )

   if time < 0:
      sign = '-'
      time = -1 * time
   else:
      sign = ''

   seconds = time % 60
   time //= 60
   minutes = time % 60
   time //= 60
   hours = time % 24
   days = time // 24

   if days:
      return '%s%dd%02dh' % ( sign, days, hours )
   else:
      return '%s%02d:%02d:%02d' % ( sign, hours, minutes, seconds )


def parseTimeInterval( ageStr ):
   '''The inverse of formatTimeInterval
      Converts a given time from a concise, readable format to seconds
      Times less than 24h are formatted: HH:MM:SS
      Times over 24h are formatted: DD:HH
   '''
   ageRe = "(?P<sign>-?)" + \
      "((?P<days>[0-9]+)d(?P<dhours>[0-9]+)h|" + \
      "(?P<hours>[0-9]+):(?P<minutes>[0-9]+):(?P<seconds>[0-9]+))"
   m = re.search( ageRe, ageStr )
   if m:
      d = m.groupdict()
      age = 0
      toInt = lambda v: int( v ) if v is not None else 0
      age += 86400 * toInt( d[ 'days' ] )
      age += 3600 * toInt( d[ 'dhours' ] )
      age += 3600 * toInt( d[ 'hours' ] )
      age += 60 * toInt( d[ 'minutes' ] )
      age += toInt( d[ 'seconds' ] )
      if d[ 'sign' ] == '-':
         age = -age
      return age
   return 0

def formatValueSi( rate, precision=4, unit="bps", noSpaces=False ):
   '''
   Convert a value to a formatted string scaled with the appropriate Si unit.
   (for example bps, kbps, Mbps, Gbps, Tbps).
   '''
   # We don't have any use for less than 3 significant figures, so explicitly
   # reject it.  Adding support for < 3 significant figures requires more work
   # since the %g format specifier will switch to exponential representation.
   assert precision >= 3, "Unsupported precision"
   ( scaledRate, siPrefix ) = scaleValueSi( rate )
   # pylint: disable-next=chained-comparison
   if ( scaledRate >= 0.0 ) and ( scaledRate <= 1000.0 ):
      if noSpaces:
         return "%.*g%s%s" % ( precision, scaledRate, siPrefix, unit )
      return "%.*g %s%s" % ( precision, scaledRate, siPrefix, unit )

   # We are in an edge condition - either a negative rate, a very large
   # positive value, nan or inf.  Don't do any conversion, let it display
   # in the default float representation.
   if noSpaces:
      return "%g%s" % ( rate, unit )
   return "%g %s" % ( rate, unit )
