# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from abc import ABCMeta, abstractmethod, abstractproperty
from collections import namedtuple
import itertools

from L1Topology.Constants import (
   PHY_SCOPE_LINE,
   PHY_SCOPE_SYSTEM,
   CATEGORY_UNKNOWN,
   CATEGORY_SWITCH_CHIP,
   CATEGORY_EXTERNAL_PHY,
   CATEGORY_CROSSPOINT,
   CATEGORY_CPU_NIC,
   UNKNOWN_LOGICAL_PORT_POOL_ID,
)
from L1Topology.Modes import (
   G400_8_RS544,
   G200_4_RS544,
   G100_2_RS544,
   G100_4_RS528,
   G100_4_NOFEC,
   G50_1_RS544,
   G50_2_RS528,
   G50_2_FCFEC,
   G50_2_NOFEC,
   G40_2_NOFEC,
   G40_4_NOFEC,
   G25_1_RS528,
   G25_1_FCFEC,
   G25_1_NOFEC,
   G10_1_NOFEC,
   G5_1_NOFEC,
   G2P5_1_NOFEC,
   G1_1_NOFEC,
   M100_1_NOFEC,
   M100_1_NOFEC_HALF,
   M10_1_NOFEC,
   M10_1_NOFEC_HALF,
   SPEED_GBPS_400,
   SPEED_GBPS_200,
   SPEED_GBPS_100,
   SPEED_GBPS_50,
   SPEED_GBPS_40,
   SPEED_GBPS_25,
   SPEED_GBPS_20,
   SPEED_GBPS_10,
   SPEED_GBPS_5,
   SPEED_GBPS_2p5,
   SPEED_GBPS_1,
   SPEED_MBPS_100,
   SPEED_MBPS_10,
   SPEED_UNKNOWN,
   DUPLEX_FULL,
   DUPLEX_HALF,
   COMPAT_GBPS_50,
   COMPAT_GBPS_25,
   COMPAT_GBPS_10,
)

from .Errors import (
   HwL1ComponentError,
   HwL1ComponentLibraryError,
   HwL1ComponentLibraryInternalError,
)
from .Serdes import (
   MappingDesc,
   SerdesPair,
   WILDCARD,
   AUTONEG_CL28,
   AUTONEG_CL37,
   AUTONEG_CL73,
   AUTONEG_DISABLED,
   AutonegDesc,
   ForcedDesc,
)
from .Cores import (
   AbsentCore,
   BabbageCore,
   BabbageLPCore,
   Blackhawk,
   BlackhawkFabric,
   BlackhawkGen3,
   Blackhawk1G,
   CellBasedPeregrine,
   ColumbiavilleCore,
   D5,
   D5MgmtTwoPort,
   Falcon16,
   Falcon16Gen3,
   Falcon16LowSpeed,
   FortvilleCore,
   FoxvilleCore,
   GPhy,
   IntelSoCCore,
   IntelX553,
   Merlin,
   MerlinG2,
   MerlinMgmt,
   MerlinMgmtOnePort,
   MerlinMgmtTwoPort,
   MerlinQ,
   Osprey,
   Peregrine,
   PeregrineNo25G,
   SparrowCore,
   Talon,
   TalonNo50G,
   Tofino2Core,
   Tofino2Core4Lane,
   TofinoCore,
   TofinoCoreAuxLane,
)
from .Tuning import (
   Main6Taps,
   Main5Taps,
   Main3Taps,
   SPEED_10G,
   SPEED_20G,
   SPEED_25G,
   SPEED_50G,
)

from Toggles.PhyEeeToggleLib import toggleDropbear100FullSupportEnabled

LogicalPortPool = namedtuple( 'LogicalPortPool', [ 'size', 'cores' ],
                                                 defaults=( None, None ) )

# This is the base component type used by all components that are supported in
# L1Topology. There are still a few exceptions that dont use this infrastructure, but
# the majority do. For any new component, the class will need to inherit from this
# type, and register themselves with the `@registerComponent` function.
#
# Currently, there a number of inherited sub-types that will likely fit the component
# being defined better than the root; please consider using one of these types as the
# base type instead:
#                                   +===========+
#                  -----------------| Component |-------------------------
#                 /                 +===========+                         \
#                /                 /      |      \                         \
#               /               ---       |       ---                       \
#              /               /          |          \                       \
#    +==========+  +============+  +=============+  +====================+  +=====+
#    | Midplane |  | Crosspoint |  | ExternalPhy |  | CoreBasedComponent |  | App |
#    +==========+  +============+  +=============+  +====================+  +=====+
#                                                             |
#                                                          +======+
#                                                          | Asic |
#                                                          +======+
#                                                         /        \
#                                                 +==========+  +============+
#                                                 | SandAsic |  | StrataAsic |
#                                                 +==========+  +============+
#
# TODO: See BUG728602; This is pretty messy and doesn't mesh well with how the
#       XcvrSlot/Midplane types should be modeled, let alone the strange heirarchy
#       with external phys. We should investigate making this and the file
#       interactions simpler.
class Component( metaclass=ABCMeta ):
   """A base class for all components. The behaviour of the components are defined
   by the abstract methods and attributes below.
   """

   # supportedModes should be a set containing the modes that this component supports
   # being configured in.
   # By default components support no modes; None is used to signify this "null" mode
   # of the component.
   supportedModes = { None }

   def __init__( self, mode=None, coreModes=None ):
      self.mode = mode
      if not isinstance( self.supportedModes, set ):
         raise HwL1ComponentError( self, 'supportedModes must be a set.',
                                   supportedModes=self.supportedModes )
      if mode not in self.supportedModes:
         raise HwL1ComponentError( self, 'Invalid component mode.',
                                   mode=mode, supportedModes=self.supportedModes )
      if coreModes:
         raise HwL1ComponentError( self, "Core modes unsupported on basic component",
                                   mode=mode, supportedModes=self.supportedModes )

   @property
   def name( self ):
      "A convienent wrapper for class name"
      return self.__class__.__name__

   def __str__( self ):
      """
      A human-readable output for what the instance of the Component represents.

      If the component is initialized in no mode ( if supported ), we will output
      the type as only '<component name>'.
      """
      # pylint: disable-next=consider-using-f-string
      return "%s-%s" % ( self.name, self.mode ) if self.mode else self.name

   def getComponentType( self, coreId ):
      """
      Parameters
      -------
      coreId: int
      The core to get information about

      Returns
      -------
      String; a name for l1 topology state machines to use for the core on the chip.
      By convention this is the string conversion of the component.
      
      """
      return str( self )

   @abstractmethod
   def getChipCategory( self ):
      """
      Returns
      -------
      String; the PhyEee::ChipCategory this component is classified as.

      """
      raise NotImplementedError

   @abstractmethod
   def getSerdes( self, serdesId ):
      """
      Parameters
      -------
      serdesId: int
      The serdes to get information about

      Returns
      -------
      tuple( int, int ). First int is the id of the core on which the input
      serdes is. Second int is the id of the lane for that serdes.
      """
      raise NotImplementedError

   @abstractmethod
   def getCoreIds( self ):
      """
      Returns
      -------
      list( int ); The integer keys for the supported coreIds on the component.
      """
      raise NotImplementedError

   def getComponentCoreIdOffset( self ):
      """
      Returns
      -------
      Int; the maximum coreId used on the component. Since some core ids might be
           absent, this is not necessarily the same as len( coreIds ).
      """
      return max( self.getCoreIds() ) + 1

   @abstractmethod
   def getNumSerdesPerCore( self, coreId ):
      """
      Parameters
      -------
      coreId: int
      The core for which to return the number of serdes
      During conversion, a default of None is used so each function
      can be converted independently. 

      Returns
      -------
      Int; the number of serdes on the core
      """
      raise NotImplementedError

   @abstractmethod
   def laneRemapCapable( self, coreId ):
      """
      Parameters
      -------
      coreId: int
      The core to get information about

      Returns
      -------
      Bool; whether the component can handle lane remaps on that core.
      """
      raise NotImplementedError

   @abstractmethod
   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      """
      Parameters
      -------
      coreId: int
      The core to get information about

      phyScope: PhyEee::PhyScope
      The side of the core to get information about

      isTx: bool
      True for TX Serdes and False for RX Serdes

      Returns
      -------
      Bool; whether the component can handle polarity remaps on that core.
      """
      raise NotImplementedError

   def getPhysicalSerdesMappings( self, coreId ):
      raise NotImplementedError

   def getLogicalSerdesMappings( self, coreId ):
      """
      Retrieves the logical SerDes mappings for a specific core.

      Returns
      -------
      An iterable which produces items of type: Serdes.MappingDesc
      The Serdes.MappingDesc iteslf has the format:
         ( Hardware::Phy::SerdesGroupMode: System Group Mode,
           list( int ): System Lanes,
           Hardware::Phy::SerdesGroupMode: Line Group Mode,
           list( int ): Line lanes )

      Group Modes are defined in the SerDes module.

      Note
      ----
      The Group Mode and the number of lanes must be consistent.
      e.g. if System Group Mode is G400_8_RS544, System Lanes must be a type that
      supports indexing with 8 items.

      TODO
      ----
         Make into an abstractmethod once all components are converted to use this
         and BUG429103 is resolved.
      """
      return []

   def getTapGroups( self, coreId ):
      """
      Retrieves the supported Taps for each speed for a specific core.

      Returns
      -------
      A dict of SerdesSpeed to Taps class, where the Taps class has the addTuning
      method defined on it. See the Tuning module for more information/examples.

      TODO
      ----
         Make into an abstractmethod once all components are converted to specify
         this and BUG489453 is resolved.
      """
      return {}

   def getLogicalSerdesPairs( self, coreId ):
      """
      Retrieves the logical serdes pairs for a PMA-only chip.
      
      Returns
      ------
      An iterable that contains information for each pair of serdes on the core.
      Each entry contains type SerdesPair defined in Serdes module.

      Note
      ----
      This function is intended for modeling PMA-specific chips with one to one
      serdes mapping such as Crosspoint, chips that require additional information
      should still use getLogicalSerdesMappings. 

      TODO
      ----
         We want to eventually merge getLogicalSerdesPairs, getLogicalSerdeMappings
         and getSupportedSpeeds into getCapabilities.
      """
      return None

   def getPossibleSerdesPairs( self, coreId ):
      """
      Retrieves the possible logical serdes pairs for a PMA-only chip.

      Returns
      ------
      An iterable that contains information for each possible pair of serdes on the
      core.  Each entry contains type SerdesPair defined in Serdes module.

      Note
      ----
      This function is intended for modeling PMA-specific chips with dynamic mapping
      capabilities such as Ds280df810.
      """
      return []

   def getSupportedSpeeds( self, coreId, serdesId, phyScope):
      """
      Retrieve the supported speeds for a serdes that defined mappings using
      getLogicalSerdesPairs.

      Returns
      -------
      A list of Interface::EthSpeed that is supported by the serdes.

      Note
      ----
      This function should be defined if and only if getLogicalSerdesPairs
      is defined.
      """
      return None

   def isPmaCore( self, coreId ):
      """
      Checks if the specified core is a PMA core.

      TODO
      ----
         See BUG578462, we should restructure PMA/PCS mappings.
      """
      # pylint: disable-next=singleton-comparison
      return self.getLogicalSerdesPairs( coreId ) != None

   def getSubdomain( self, coreId=None ):
      """
      Returns the default subdomain for this component in the topology.
      Uses the EOS default subdomain if None.

      Parameters
      -------
      coreId: int
      The core to get information about, used when a core-based component
      contains multiple subdomains.
      """
      return None

   def getSupportedAutoneg( self, coreId ):
      """
      Retrieve the autoneg configurations supported by the chip.

      Returns
      ------
      A list of AutonegDesc which each have the format:
         ( Interface::EthSpeed: systemSideSpeed,
           Interface::AutonegMode: systemSideAutoneg,
           Interface::EthSpeed: lineSideSpeed,
           Interface::AutonegMode: lineSideAutoneg )
      
      Note
      ----
      PMA Components should instead define speed as serdes rate as opposed
      to full multi-lane speed. See BUG740787.
      """
      return []

   def getSupportedForcedModes( self, coreId ):
      """
      Retrieve the cross-chip forced configurations supported by the chip.

      Returns
      ------
      A list of AutonegDesc which each have the format:
         ( Interface::EthSpeed: systemSideSpeed,
           Interface::AutonegMode: AutonegMode.anegModeDisabled,
           Interface::EthSpeed: lineSideSpeed,
           Interface::AutonegMode: AutonegMode.anegModeDisabled )
      """
      forcedModes = set()
      for m in self.getLogicalSerdesMappings( coreId ):
         sysIntfSpeed = None
         if m.sysMode and m.sysMode.intfSpeed != SPEED_UNKNOWN:
            sysIntfSpeed = m.sysMode.intfSpeed
         lineIntfSpeed = None
         if m.lineMode and m.lineMode.intfSpeed != SPEED_UNKNOWN:
            lineIntfSpeed = m.lineMode.intfSpeed
         forcedModes.add( ForcedDesc( sysIntfSpeed, lineIntfSpeed ) )
      return list( forcedModes )

   def getPllCounts( self ):
      """
      Returns
      -------
      A dict of coreId to the core's PLL count
      """
      return {}

   def getLineRateToPllRates( self ):
      """
      Returns
      -------
      A dict of coreId to a dict of the core's line rate to PLL rates.
      """
      return {}

   def getLogicalPortPoolId( self, coreId ):
      """
      Returns
      -------
      The integer that represents the ID of the logical port pool the coreId
      has access to. Return UNKNOWN_LOGICAL_PORT_POOL_ID if there is no logical port
      pools for the given coreId.
      """
      return UNKNOWN_LOGICAL_PORT_POOL_ID

   def getLogicalPortPoolSize( self, poolId ):
      """
      Returns
      -------
      The integer that represents the size of the logical port pool with ID=poolId.
      """
      return 0

   def getLogicalPortPoolInfo( self ):
      """
      Returns
      -------
      A dict of poolId to a namedtuple LogicalPortPool ( int, list( int ) ).
      The first int is the size of the logical port pool.
      The second list of integers is a list of chip-scoped coreIds.
      """
      logicalPortPools = {}
      for coreId in self.getCoreIds():
         poolId = self.getLogicalPortPoolId( coreId )

         # We either do not have logical port restrictions or we have not yet
         # implemented the required functions for the specific coreId.
         # In either case, skip the current round.
         if poolId == UNKNOWN_LOGICAL_PORT_POOL_ID:
            continue

         if poolId not in logicalPortPools:
            poolSize = self.getLogicalPortPoolSize( poolId )
            logicalPortPools[ poolId ] = LogicalPortPool( size=poolSize,
                                                          cores=[ coreId, ] )
         else:
            logicalPortPools[ poolId ].cores.append( coreId )

      return logicalPortPools

class CoreBasedComponent( Component ):
   """A base class for all core based components. The behaviour of these components
   is derived from the cores they are comprised of.
   """

   CORE_MODE = "Core"

   @abstractproperty # pylint: disable=deprecated-decorator
   def cores( self ):
      """A dict from core type to an iterable of core IDs. The core id must be unique
      per component. e.g. { Falcon16: [ 0, 3 ], Blackhawk: [ 1, 2 ] }. The
      component's core IDs must start from 0 and be contiguous.
      """
      raise NotImplementedError

   @property
   def supportedModes( self ):
      """
      Defines the supportedModes for a CoreBasedComponent to be the intersection of
      all supportedModes of the component cores, plus the special mode "Core".
      """
      coreModes = set.intersection( *( core.supportedModes for core in self.cores ) )
      return coreModes | { self.CORE_MODE }

   def __init__( self, mode=None, coreModes=None ):
      """Takes in the mode to initialize the component in. This mode is then applied
      to all the cores defined in the component. If the special mode "Core" is
      specified, however, it also takes a dict of coreId to coreMode that is used to
      initialize each core instead.
      """
      super().__init__( mode=mode )

      if mode == self.CORE_MODE and not coreModes: # pylint: disable=no-else-raise
         raise HwL1ComponentError( self,
                                   'Core mode given but have no per-core modes.',
                                   mode=mode, coreModes=coreModes )
      elif mode != self.CORE_MODE and coreModes:
         raise HwL1ComponentError( self,
                                   'Core mode not given but have per-core modes.',
                                   mode=mode, coreModes=coreModes )

      self.coreMap = {}
      for coreType, coreIds in self.cores.items():
         for coreId in coreIds:
            if coreId in self.coreMap:
               raise HwL1ComponentError( self, 'Duplicate core ID detected.',
                                         coreType=coreType, coreId=coreId,
                                         cores=self.cores )
            coreMode = mode if mode != self.CORE_MODE else coreModes.get( coreId )
            self.coreMap[ coreId ] = coreType( mode=coreMode )

      # Due to the way SerDes ID numberings are generated, core IDs are expected
      # to always start from 0 and be contiguous. In order for this restriction to be
      # relaxed, `serdesInfo` generation would have to be modified to allow the user
      # to specify a SerDes range for each core.
      sortedContiguousKeys = range( len( self.coreMap ) )  
      if any( k not in self.coreMap for k in sortedContiguousKeys ):
         raise HwL1ComponentError( self, 'Core ID definitions did not start at 0 or '
                                         'were not contiguous.',
                                         cores=self.cores,
                                         coreIds=sorted( self.coreMap.keys() ) )

      # The serdesInfo is a cached map that allows the component to easily convert
      # from a component indexed SerDes ID to the core and physical lane tuple that
      # it maps to.
      self.serdesInfo = {}
      componentSerdesId = 0
      for coreId in sortedContiguousKeys:
         core = self.coreMap[ coreId ]

         # Special case for absent cores, where we just want to offset the
         # componentSerdesId by the number of serdes that we expect the core to take
         if isinstance( core, AbsentCore ):
            componentSerdesId += core.getNumSerdes()
            continue

         for coreSerdesId in range( core.getNumSerdes() ):
            if componentSerdesId in self.serdesInfo:
               raise HwL1ComponentLibraryInternalError(
                  'Duplicate SerDes ID detected', component=self, coreId=coreId,
                  coreSerdesId=coreSerdesId, componentSerdesId=componentSerdesId,
                  cores=self.cores, serdesInfo=self.serdesInfo )

            self.serdesInfo[ componentSerdesId ] = ( coreId, coreSerdesId )
            componentSerdesId += 1

   def getSerdes( self, serdesId ):
      if serdesId not in self.serdesInfo:
         raise HwL1ComponentError( self,
            'Attempted to reference a serdes that does not exist on this component.',
            serdesId=serdesId, validSerdesIds=sorted( self.serdesInfo.keys() ) )
      return self.serdesInfo[ serdesId ]

   def getCore( self, coreId ):
      if coreId not in self.coreMap:
         raise HwL1ComponentError( self,
            'Attempted to reference a core that does not exist on this component.',
            coreId=coreId, validCoreIds=sorted( self.coreMap.keys() ) )
      return self.coreMap[ coreId ]

   def getCoreIds( self ):
      # We need to filter out cores that are defined as absent
      return [ coreId for coreId, core in self.coreMap.items()
               if not isinstance( core, AbsentCore ) ]

   def getNumSerdesPerCore( self, coreId ):
      return self.getCore( coreId ).getNumSerdes()

   def getComponentType( self, coreId ):
      return self.getCore( coreId ).getComponentType()

   def getChipCategory( self ):
      return CATEGORY_SWITCH_CHIP

   def laneRemapCapable( self, coreId ):
      return self.getCore( coreId ).laneRemapCapable()

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return self.getCore( coreId ).polarityRemapCapable( phyScope )

   def getPhysicalSerdesMappings( self, coreId ):
      return self.getCore( coreId ).getPhysicalMapping()

   def getLogicalSerdesMappings( self, coreId ):
      return self.getCore( coreId ).getLogicalSerdesMapping()

   def getTapGroups( self, coreId ):
      return self.getCore( coreId ).getTapGroups()

   def getSupportedAutoneg( self, coreId ):
      return self.getCore( coreId ).getSupportedAutoneg( coreId )

   def getPllCounts( self ):
      return { coreId: self.getCore( coreId ).getPllCount()
               for coreId in self.getCoreIds()
               if self.getCore( coreId ).getPllCount() > 0 }

   def getLineRateToPllRates( self ):
      return { coreId: self.getCore( coreId ).getLineRateToPllRateList()
               for coreId in self.getCoreIds() }

class Asic( CoreBasedComponent, metaclass=ABCMeta ):
   # Pylint is dumb and doesn't understand that this is also an abstract class
   # pylint: disable=abstract-method

   def getSubdomain( self, coreId=None ):
      # TODO: See BUG732563; This should be abstract/NotImplemented long-term once
      #       the conversion is complete
      return None

class Midplane( Component ):
   def getChipCategory( self ):
      # This makes no sense for a midplane, need to investigate restructuring
      return CATEGORY_UNKNOWN

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   @abstractmethod
   def getNumSerdesPerCore( self, coreId ):
      raise NotImplementedError

   def laneRemapCapable( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def getPhysicalSerdesMappings( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def getLogicalSerdesMappings( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

   def getTapGroups( self, coreId ):
      # This makes no sense for a midplane, need to investigate restructuring
      pass

class ExternalPhy( Component ):
   def laneRemapCapable( self, coreId ):
      return True

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return True

   def getChipCategory( self ):
      return CATEGORY_EXTERNAL_PHY

   @abstractmethod
   def getSerdes( self, serdesId ):
      raise NotImplementedError

   @abstractmethod
   def getCoreIds( self ):
      raise NotImplementedError

   @abstractmethod
   def getNumSerdesPerCore( self, coreId ):
      raise NotImplementedError

# Component Registration System
registeredComponents = {}

def registerHwL1Component( componentCls ):
   """A utility function used to register components so that the Hw L1 Topology FRU
   plugin ( as well as other external consumers ) can be made aware of them.

   Args
   ----
      componentCls:  The component class to register.

   Returns
   -------
      The componentCls.

   Raises
   ------
      HwL1ComponentLibraryError if the componentCls does not subclass Component.
   """
   name = componentCls.__name__

   if not issubclass( componentCls, Component ):
      raise HwL1ComponentLibraryError( 'Attempted to register a component that does '
                                       'not subclass from the base Component class.',
                                       offendingComponentCls=componentCls )
   if name in registeredComponents:
      raise HwL1ComponentLibraryError( 'Attempted to register multiple component '
                                       'classes with the same name.',
                                       name=name,
                                       existingComp=registeredComponents[ name ],
                                       offendingComponentCls=componentCls)

   registeredComponents[ name ] = componentCls
   return componentCls

@registerHwL1Component
class Sliver64( Midplane ):
   def getNumSerdesPerCore( self, coreId ):
      return 64

@registerHwL1Component
class Examax( Midplane ):
   def getNumSerdesPerCore( self, coreId ):
      return 18

@registerHwL1Component
class Examax64( Midplane ):
   def getNumSerdesPerCore( self, coreId ):
      return 64

@registerHwL1Component
class Examax48( Midplane ):
   def getNumSerdesPerCore( self, coreId ):
      return 24

@registerHwL1Component
class Examax36( Midplane ):
   def getNumSerdesPerCore( self, coreId ):
      return 12

class SandAsic( Asic, metaclass=ABCMeta ):
   # Pylint is dumb and doesn't understand that this is also an abstract class
   # pylint: disable=abstract-method

   fabricCoreTypes = ( BlackhawkFabric, CellBasedPeregrine )

   def getSubdomain( self, coreId=None ):
      if coreId is None:
         return "SandPhy"
      core = self.coreMap[ coreId ]
      if isinstance( core, self.fabricCoreTypes ):
         return "FabricIntf"
      else:
         return "SandPhy"

@registerHwL1Component
class Jericho3( SandAsic ):
   cores = {
      Peregrine : list( range( 0, 18 ) ),
      D5 : [ 18 ],
      CellBasedPeregrine : list( range( 19, 39 ) ),
   }

@registerHwL1Component
class Ramon3( SandAsic ):
   cores = {
      CellBasedPeregrine : list( range( 64 ) ),
   }

@registerHwL1Component
class Qumran3D( SandAsic ):
   cores = {
      Peregrine : list( range( 0, 32 ) ),
      D5 : [ 32, 33 ],
   }

@registerHwL1Component
class Qumran3A( SandAsic ):
   cores = {
      Talon : list( range( 0, 8 ) ),
      Peregrine : [ 8, 9 ],
   }

@registerHwL1Component
class Qumran3U( SandAsic ):
   cores = {
      Talon : [ 0, 1 ],
      TalonNo50G : list( range( 2, 7 ) ),
      # Q3U's serdes core 7 is absent, which would contain 8 serdes
      AbsentCore( 8 ) : [ 7 ],
      Peregrine : [ 8 ],
   }

@registerHwL1Component
class Jericho2( SandAsic ):
   cores = {
      Blackhawk : list( range( 0, 12 ) ),
   }

   def getPllCounts( self ):
      corePlls = {}
      for coreId in self.getCoreIds():
         # The second phyCore on J2 cannot do dual-PLL due to a hardware erratum
         corePlls[ coreId ] = 1 if coreId % 2 \
            else self.getCore( coreId ).getPllCount()
      return corePlls

@registerHwL1Component
class Jericho2Fabric( SandAsic ):
   cores = {
      Blackhawk : list( range( 0, 11 ) ),
      BlackhawkFabric : list( range( 11, 26 ) )
   }

   def getSubdomain( self, coreId=None ):
      if coreId == 11:
         # XXX narenberg: For the Carina L1 sim, we want the interfaces on core 11
         # to be manged by neither SandPhy nor SandFabricL1, so they get their
         # own subdomain. This is because they are front-panel ports with
         # a Fabric name that are used for testing. This will not affect any
         # active products since the Jericho2Fabric component is only used in
         # the Carina L1 sim.
         return "SpanishInquisition"
      else:
         return super().getSubdomain( coreId=coreId )

@registerHwL1Component
class Jericho2C( SandAsic ):
   cores = {
      Blackhawk : list( range( 0, 4 ) ),
      Falcon16Gen3 : list( range( 4, 28 ) ),
   }

@registerHwL1Component
class Jericho2CPlus( SandAsic ):
   cores = {
      Blackhawk : list( range( 0, 18 ) ),
   }

@registerHwL1Component
class Jericho2M( SandAsic ):
   cores = {
      Blackhawk : list( range( 0, 9 ) ),
   }

@registerHwL1Component
class Qumran2A( SandAsic ):
   cores = {
         Blackhawk : list( range( 0, 2 ) ),
         Falcon16Gen3 : list( range( 2, 11 ) )
   }

@registerHwL1Component
class Tofino( CoreBasedComponent ):
   # Tofino presently does not use L1 Topology to derive any of it's configurations.
   # Instead, it always programs polarity as being default and expects logical lanes
   # to map to the identically numbered physical lane. As such, from the L1 Topology
   # perspective, this chip is effectively non lane and non polarity remap capable.
   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   # Tofino is one chip controlling 64 serdes groups,
   # with each group controlling 4 lanes with NRZ only.
   # Last aux group is capable of 1G also (to CPU)
   cores = {
      TofinoCore : list( range( 0, 64 ) ),
      TofinoCoreAuxLane : [ 64 ],
   }

@registerHwL1Component
class Tofino2( CoreBasedComponent ):
   # Tofino2 is one chip controlling 17 serdes groups.
   # The first serdes group control 4 lanes with NRZ only.
   # The last 16 serdes groups control 8 serdes lanes capable for PAM4 and NRZ.
   cores = {
      Tofino2Core4Lane: [ 0 ],
      Tofino2Core : list( range( 1, 17 ) ),
   }

@registerHwL1Component
class IntelSoC( CoreBasedComponent ):
   # This describes an Intel SoC which have 2 qual integrated PHYs.
   # Currently we know at least two SoCs ( SnowRidge and IceLake ) use this.
   cores = {
      IntelSoCCore: [ 0, 1 ],
   }

@registerHwL1Component
class Fortville( CoreBasedComponent ):
   cores = {
       FortvilleCore: [ 0 ],
   }

@registerHwL1Component
class Columbiaville( CoreBasedComponent ):
   cores = {
       ColumbiavilleCore: [ 0 ],
   }

@registerHwL1Component
# Codename for Intel Denverton C3558 CPU
class Harrisonville( CoreBasedComponent ):
   cores = {
       IntelX553: [ 0, 1 ],
   }

@registerHwL1Component
# Codename for Intel I226 series ethernet controller
class Foxville( CoreBasedComponent ):
   cores = {
       FoxvilleCore: [ 0 ],
   }

@registerHwL1Component
class Ds280df810( ExternalPhy ):
   # The PHY DS280 is a unidirectional retimer chip:
   #            +------------------+
   #            |    Ds280df810    |
   #    Rx0 ----|---- Retimer0 --->|--- TX0
   #    Rx1 ----|---- Retimer1 --->|--- Tx1
   #    Rx2 ----|---- Retimer2 --->|--- Tx2
   #    Rx3 ----|---- Retimer3 --->|--- Tx3
   #    Rx4 ----|---- Retimer4 --->|--- Tx4
   #    Rx5 ----|---- Retimer5 --->|--- Tx5
   #    Rx6 ----|---- Retimer6 --->|--- Tx6
   #    Rx7 ----|---- Retimer7 --->|--- Tx7
   #            +------------------+
   #
   # In total, it has 8 Tx and 8 Rx SerDes, each pair of Rx/Tx SerDes is represented
   # above as a "retimer". Each pair of retimers are grouped together and can go:
   #  1. Straight across as a normal retimer would
   #  2. Cross over with the partner retimer ( e.g. Rx0->Tx1 and Rx1->Tx0 )
   #  3. Fanout the Rx of their partner retimer ( e.g. Rx0 -> Tx0, Tx1 )
   #
   # Retimer mode
   # ============
   #
   # When in retimer mode ( mode=None ), the Ds280 is modelled as having
   # bidirectional mappings between:
   #  - Tx* on the line side and Rx* on the system side
   #  - Rx* on the line side and Tx* on the system side
   #
   #  This is a workaround for a limitation of L1 Topology modeling and is fine for
   #  now. In reality, no SKU should ever define a "Tx*" or "Rx*' on more than one
   #  side. Thus, for all intents and purposes these mappings behave as
   #  unidirectional ones.
   #
   # Fanout Mode
   # ===========
   #
   # Unlike retimer mode, fanout SerDes pairs cannot be crafted generically and are
   # tied to the wiring of the SKU that they are used in ( Cottesloe ). This is
   # because the possible and static SerDes pairs are designed specifically
   # around this chip wiring scheme.
   #
   # The "static" SerDes pair act as fallbacks to always route the ASIC to the
   # Ethernet interfaces and vice versa.
   #
   #          Line     +------------------+    System
   #         serdes    |    Ds280df810    |    serdes
   #  EtX Rx   0   ----|---- Retimer0 --->|----  0    ASIC Rx
   #                   |---- Retimer1 --->|----  1    Crosspint Rx
   #  EtY Rx   2   ----|---- Retimer2 --->|----  2    ASIC Rx
   #                   |---- Retimer3 --->|----  3    Crosspint Rx
   #  EtX Tx   4   ----|<--- Retimer0 ----|----  4    ASIC Tx
   #                   |<--- Retimer1 ----|----  5    Crosspint Tx
   #  EtY Tx   6   ----|<--- Retimer2 ----|----  6    ASIC Tx
   #                   |<--- Retimer3 ----|----  7    Crosspint Tx
   #                   +------------------+
   #
   #
   #  TODO BUG739452: Convert this component to a "unidirectional component" once L1
   #                  Topology support is in place.

   # TODO: None should probably be renamed to "Retimer"
   supportedModes = { None, "Fanout" }

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   # The DS280 does not use L1 Topology to derive any of it's SerDes configurations.
   # Instead, it always programs polarity as being default as such, from the L1
   # Topology perspective, this chip is effectively non polarity remap capable.
   #
   # TODO BUG670142: Set this to true once the DS280 agent is able to handle polarity
   #                 swaps.
   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      if not self.mode:
         return { ( 0, ) : ( 0, ),
                  ( 1, ) : ( 1, ),
                  ( 2, ) : ( 2, ),
                  ( 3, ) : ( 3, ),
                  ( 4, ) : ( 4, ),
                  ( 5, ) : ( 5, ),
                  ( 6, ) : ( 6, ),
                  ( 7, ) : ( 7, ) }
      if self.mode == "Fanout":
         return {}

      raise HwL1ComponentError( self, "Invalid mode.",
                                mode=self.mode, supportedModes=self.supportedModes )

   def getLogicalSerdesPairs( self, coreId ):
      if not self.mode:
         return [
            SerdesPair( 0, PHY_SCOPE_SYSTEM, 0, PHY_SCOPE_LINE ),
            SerdesPair( 1, PHY_SCOPE_SYSTEM, 1, PHY_SCOPE_LINE ),
            SerdesPair( 2, PHY_SCOPE_SYSTEM, 2, PHY_SCOPE_LINE ),
            SerdesPair( 3, PHY_SCOPE_SYSTEM, 3, PHY_SCOPE_LINE ),
            SerdesPair( 4, PHY_SCOPE_SYSTEM, 4, PHY_SCOPE_LINE ),
            SerdesPair( 5, PHY_SCOPE_SYSTEM, 5, PHY_SCOPE_LINE ),
            SerdesPair( 6, PHY_SCOPE_SYSTEM, 6, PHY_SCOPE_LINE ),
            SerdesPair( 7, PHY_SCOPE_SYSTEM, 7, PHY_SCOPE_LINE ),
         ]
      if self.mode == "Fanout":
         return [
            SerdesPair( 0, PHY_SCOPE_LINE, 0, PHY_SCOPE_SYSTEM ),
            SerdesPair( None, None , 1, PHY_SCOPE_SYSTEM ),
            SerdesPair( 2, PHY_SCOPE_LINE, 2, PHY_SCOPE_SYSTEM ),
            SerdesPair( None, None, 3, PHY_SCOPE_SYSTEM ),
            SerdesPair( 4, PHY_SCOPE_LINE, 4, PHY_SCOPE_SYSTEM ),
            SerdesPair( None, None, 5, PHY_SCOPE_SYSTEM ),
            SerdesPair( 6, PHY_SCOPE_LINE, 6, PHY_SCOPE_SYSTEM ),
            SerdesPair( None, None, 7, PHY_SCOPE_SYSTEM ),
         ]

      raise HwL1ComponentError( self, "Invalid mode.",
                                mode=self.mode, supportedModes=self.supportedModes )

   def getPossibleSerdesPairs( self, coreId ):
      if not self.mode:
         return []
      if self.mode == "Fanout":
         return [
            SerdesPair( 0, PHY_SCOPE_LINE, 0, PHY_SCOPE_SYSTEM ),
            SerdesPair( 0, PHY_SCOPE_LINE, 1, PHY_SCOPE_SYSTEM ),
            SerdesPair( 2, PHY_SCOPE_LINE, 2, PHY_SCOPE_SYSTEM ),
            SerdesPair( 2, PHY_SCOPE_LINE, 3, PHY_SCOPE_SYSTEM ),
            SerdesPair( 4, PHY_SCOPE_LINE, 4, PHY_SCOPE_SYSTEM ),
            SerdesPair( 4, PHY_SCOPE_LINE, 5, PHY_SCOPE_SYSTEM ),
            SerdesPair( 6, PHY_SCOPE_LINE, 6, PHY_SCOPE_SYSTEM ),
            SerdesPair( 6, PHY_SCOPE_LINE, 7, PHY_SCOPE_SYSTEM ),
         ]

      raise HwL1ComponentError( self, "Invalid mode.",
                                mode=self.mode, supportedModes=self.supportedModes )

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      return [ SPEED_GBPS_1, SPEED_GBPS_10, SPEED_GBPS_25 ]

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps, SPEED_25G: Main3Taps }

   def getSupportedAutoneg( self, coreId ):
      # Ds280 does not natively support autoneg.
      return []

@registerHwL1Component
class Ds250df410( ExternalPhy ):
   # The DS250 is a unidirectional retimer chip:
   #            +------------------+
   #            |    Ds250df410    |
   #    Rx0 ----|---- Retimer0 --->|--- Tx0
   #    Rx1 ----|---- Retimer1 --->|--- Tx1
   #    Rx2 ----|---- Retimer2 --->|--- Tx2
   #    Rx3 ----|---- Retimer3 --->|--- Tx3
   #            +------------------+
   #
   # In total, it has 4 Tx and 4 Rx SerDes, a Tx SerDes from one side will pair
   # together with a Rx SerDes from the other side to form a "Retimer Channel". This
   # behaviour leads us to model the DS250 as a chip where all the 4 Tx SerDes are on
   # one "side" while all i4 Rx SerDes are on the other side.
   #
   # The DS250 is modelled as having bidirectional mappings between:
   #  - Tx* on the line side and Rx* on the system side
   #  - Rx* on the line side and Tx* on the system side
   #
   # This is a workaround for a limitation of L1 Topology modeling and is fine for
   # now. In reality, no SKU should ever define a "Tx*" or "Rx*' on more than one
   # side. Thus, for all intents and purposes these mappings behave as
   # unidirectional ones.
   #
   # TODO BUG739452: Convert this component to a "unidirectional component" once L1
   #                 Topology support is in place.

   supportedModes = { None }

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      # The real number of serdes per core is 2, but we're modeling the component
      # as 2 quads grouped together.
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   # The DS250 does not use L1 Topology to derive any of it's SerDes configurations.
   # Instead, it always programs polarity as being default as such, from the L1
   # Topology perspective, this chip is effectively non polarity remap capable.
   #
   # TODO BUG670142: Set this to true once the DS280 agent is able to handle polarity
   #                 swaps.
   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ) }

   def getLogicalSerdesPairs( self, coreId ):
      return [
         SerdesPair( 0, PHY_SCOPE_SYSTEM, 0, PHY_SCOPE_LINE ),
         SerdesPair( 1, PHY_SCOPE_SYSTEM, 1, PHY_SCOPE_LINE ),
         SerdesPair( 2, PHY_SCOPE_SYSTEM, 2, PHY_SCOPE_LINE ),
         SerdesPair( 3, PHY_SCOPE_SYSTEM, 3, PHY_SCOPE_LINE ),
      ]

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      return [ SPEED_GBPS_1, SPEED_GBPS_10, SPEED_GBPS_25 ]

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps, SPEED_25G: Main3Taps }
   
   def getSupportedAutoneg( self, coreId ):
      # Ds250 does not natively support autoneg.
      return []

@registerHwL1Component
class BlancoMux( Component ):
   #               +-------------------+
   #               |      Blanco       |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #               | |    Core 1    8| |----  8
   #               | |              9| |----  9
   #               | |             10| |----  10
   #               | |             11| |----  11
   #       4   ----| |4             4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               +-------------------+

   # This Blanco component can connect its last four system side serdes with either
   # the second group of four line side serdes (MUX) or the last four (Retimer)

   def getChipCategory( self ):
      # Blanco is not an actual component
      return CATEGORY_UNKNOWN

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 12

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ),
               ( 4, ) : ( 4, ),
               ( 5, ) : ( 5, ),
               ( 6, ) : ( 6, ),
               ( 7, ) : ( 7, ) }

   def getLogicalSerdesPairs( self, coreId ):
      return [
         SerdesPair( 0, PHY_SCOPE_SYSTEM, 0, PHY_SCOPE_LINE ),
         SerdesPair( 1, PHY_SCOPE_SYSTEM, 1, PHY_SCOPE_LINE ),
         SerdesPair( 2, PHY_SCOPE_SYSTEM, 2, PHY_SCOPE_LINE ),
         SerdesPair( 3, PHY_SCOPE_SYSTEM, 3, PHY_SCOPE_LINE ),
         SerdesPair( 4, PHY_SCOPE_SYSTEM, 4, PHY_SCOPE_LINE ),
         SerdesPair( 5, PHY_SCOPE_SYSTEM, 5, PHY_SCOPE_LINE ),
         SerdesPair( 6, PHY_SCOPE_SYSTEM, 6, PHY_SCOPE_LINE ),
         SerdesPair( 7, PHY_SCOPE_SYSTEM, 7, PHY_SCOPE_LINE ),
         # These are the serdes to the remaining lanes of the non-blanco port.
         # They aren't connected to any system side serdes by default.
         SerdesPair( None, None, 8, PHY_SCOPE_LINE ),
         SerdesPair( None, None, 9, PHY_SCOPE_LINE ),
         SerdesPair( None, None, 10, PHY_SCOPE_LINE ),
         SerdesPair( None, None, 11, PHY_SCOPE_LINE ),
      ]

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      return [ SPEED_GBPS_10, SPEED_GBPS_25, SPEED_GBPS_50 ]

@registerHwL1Component
class Babbage( CoreBasedComponent ):

   #               +-------------------+
   #               |      Babbage      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4   Core 1    4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               | +---------------+ |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12  Core 2   12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   cores = {
      BabbageCore: list( range( 0, 2 ) ),
   }

   def getChipCategory( self ):
      return CATEGORY_EXTERNAL_PHY

@registerHwL1Component
class BabbageLP( Babbage ):

   cores = {
      BabbageLPCore: list( range( 0, 2 ) ),
   }

@registerHwL1Component
class Sparrow( CoreBasedComponent ):

   #               +-------------------+
   #               |      Sparrow      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             2| |----  2
   #       1   ----| |1             3| |----  3
   #               | |    Core 0    4| |----  4
   #               | |              5| |----  5
   #               | +---------------+ |
   #               | +---------------+ |
   #       6   ----| |0             2| |----  8
   #       7   ----| |1             3| |----  9
   #               | |    Core 1    4| |----  10
   #               | |              5| |----  11
   #               | +---------------+ |
   #               | +---------------+ |
   #       12  ----| |0             2| |----  14
   #       13  ----| |1             3| |----  15
   #               | |    Core 2    4| |----  16
   #               | |              5| |----  17
   #               | +---------------+ |
   #               | +---------------+ |
   #       18  ----| |0             2| |----  20
   #       19  ----| |1             3| |----  21
   #               | |    Core 3    4| |----  22
   #               | |              5| |----  23
   #               | +---------------+ |
   #               +-------------------+
   # Serdes numbering is contiguous within every core, to match how they are
   # addressed in software.

   cores = {
      SparrowCore: list( range( 4 ) ),
   }

   def getChipCategory( self ):
      return CATEGORY_EXTERNAL_PHY

# TODO all this is wrong, but here for proof of concept
@registerHwL1Component
class B52( ExternalPhy ):

   #               +-------------------+
   #               |        B52        |
   #     serdes    | +---------------+ |    serdes
   #               | |             12| |----  12
   #               | |             13| |----  13
   #               | |             14| |----  14
   #               | |             15| |----  15
   #        0  ----| |0            16| |----  16
   #        1  ----| |1            17| |----  17
   #        2  ----| |2            18| |----  18
   #        3  ----| |3            19| |----  19
   #               | |     Die 0     | |        
   #        4  ----| |4            20| |----  20
   #        5  ----| |5            21| |----  21
   #        6  ----| |6            22| |----  22
   #        7  ----| |7            23| |----  23
   #               | |              8| |----  8
   #               | |              9| |----  9
   #               | |             10| |----  10
   #               | |             11| |----  11
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Gearbox", "Retimer" }

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId
      return ( 0, physicalLaneId ) # ( dieId, laneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getPhysicalSerdesMappings( self, coreId ):
      mapping = {}
      if self.mode == "Gearbox":
         # Map will be:
         # A0 : B12, B13
         # A1 : B14, B15
         # A2 : B16, B17
         # A3 : B18, B19
         # A4 : B20, B21
         # A5 : B22, B23
         # A6 : A8,  A9
         # A7 : A10, A11
         mapping = { ( 0, ) : ( 12, 13 ),
                     ( 1, ) : ( 14, 15 ),
                     ( 2, ) : ( 16, 17 ),
                     ( 3, ) : ( 18, 19 ),
                     ( 4, ) : ( 20, 21 ),
                     ( 5, ) : ( 22, 23 ),
                     ( 6, ) : ( 8, 9 ),
                     ( 7, ) : ( 10, 11 ) }
      elif self.mode == "Retimer":
         # Map will be:
         # A0 : B12
         # A1 : B13
         # A2 : B14
         # A3 : B15
         # A4 : B16
         # A5 : B17
         # A6 : B18
         # A7 : B19
         mapping = { ( 0, ) : ( 12, ),
                     ( 1, ) : ( 13, ),
                     ( 2, ) : ( 14, ),
                     ( 3, ) : ( 15, ),
                     ( 4, ) : ( 16, ),
                     ( 5, ) : ( 17, ),
                     ( 6, ) : ( 18, ),
                     ( 7, ) : ( 19, ) }
      else:
         raise HwL1ComponentError( self, "Invalid mode.", mode=self.mode,
                                   supportedModes=self.supportedModes )
      return mapping

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      if self.mode == "Gearbox":
         # four lane gearbox modes
         for x in range( 0, 8, 2 ):
            sysIntfs = [ x, x + 1 ]
            lineIntfs = list( range( 2 * x, 2 * x + 4 ) )
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysIntfs, G100_4_RS528, lineIntfs ),
               MappingDesc( G100_2_RS544, sysIntfs, G100_4_NOFEC, lineIntfs ),
               MappingDesc( G40_2_NOFEC, sysIntfs, G40_4_NOFEC, lineIntfs ),
            ] )
         # two lane gearbox modes
         for x in range( 0, 8 ):
            lineIntfs = [ x * 2, x * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ], G50_2_RS528, lineIntfs ),
               MappingDesc( G50_1_RS544, [ x ], G50_2_FCFEC, lineIntfs ),
               MappingDesc( G50_1_RS544, [ x ], G50_2_NOFEC, lineIntfs ),
            ] )
         # four lane modes
         for x in range( 0, 8, 4 ): 
            lineSerdes = x if x < 4 else x + 4
            mappings.extend( [
               MappingDesc( G200_4_RS544, list( range( x, x + 4 ) ), G200_4_RS544,
                            list( range( lineSerdes, lineSerdes + 4 ) ) ),
            ] ) 
         # single lane modes
         for x in range( 8 ):
            lineSerdes = x if x < 4 else x + 4
            mappings.extend( [
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_RS528, [ lineSerdes ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_FCFEC, [ lineSerdes ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_NOFEC, [ lineSerdes ] ),
               MappingDesc( G10_1_NOFEC, [ x ], G10_1_NOFEC, [ lineSerdes ] ),
            ] )
      elif self.mode == "Retimer":
         # eight lane modes
         mappings.extend( [
            MappingDesc( G400_8_RS544, list( range( 8 ) ),
                  G400_8_RS544, list( range( 8 ) ) ),
            ] )
         # four lane modes
         for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ): 
            mappings.extend( [
               MappingDesc( G200_4_RS544, x, G200_4_RS544, x ),
               MappingDesc( G100_4_NOFEC, x, G100_4_NOFEC, x ),
               MappingDesc( G100_4_NOFEC, x, G100_4_RS528, x ),
               MappingDesc( G40_4_NOFEC, x, G40_4_NOFEC, x ),
            ] )
         # two lane modes
         for x in range( 0, 8, 2 ):
            intfs = [ x, x + 1 ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, intfs, G100_2_RS544, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_RS528, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_FCFEC, intfs ),
               MappingDesc( G50_2_NOFEC, intfs, G50_2_NOFEC, intfs ),
            ] )
         # single lane modes
         for x in range( 8 ):
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ], G50_1_RS544, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_RS528, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_FCFEC, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ], G25_1_NOFEC, [ x ] ),
               MappingDesc( G10_1_NOFEC, [ x ], G10_1_NOFEC, [ x ] ),
            ] )
      else:
         raise HwL1ComponentError( self, "Invalid mode.", mode=self.mode,
                                   supportedModes=self.supportedModes )
      return mappings

   def getTapGroups( self, coreId ):
      # We need to be able to specify this per side. For Gearbox mode, 50G line side
      # tuning makes no sense
      return { SPEED_10G: Main5Taps, SPEED_20G: Main5Taps,
               SPEED_25G: Main5Taps, SPEED_50G: Main5Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( SPEED_GBPS_400, AUTONEG_DISABLED,
                      SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_200, AUTONEG_DISABLED,
                      SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_100, AUTONEG_DISABLED,
                      SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_DISABLED,
                      SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_DISABLED,
                      SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_DISABLED,
                      SPEED_GBPS_25, AUTONEG_CL73 ),
      ]

@registerHwL1Component
class B52LP( B52 ):

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( SPEED_GBPS_400, AUTONEG_DISABLED,
                      SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_200, AUTONEG_DISABLED,
                      SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_100, AUTONEG_DISABLED,
                      SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_DISABLED,
                      SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_DISABLED,
                      SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_DISABLED,
                      SPEED_GBPS_25, AUTONEG_CL73 ),
      ]

@registerHwL1Component
class Enigma( ExternalPhy ):

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      serdesCoreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( serdesCoreId, physicalLaneId )

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      # A[ 0-7 ]: B[ 0-7 ]
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ),
               ( 4, ) : ( 4, ),
               ( 5, ) : ( 5, ),
               ( 6, ) : ( 6, ),
               ( 7, ) : ( 7, ) }

   def getCoreIds( self ):
      return [ 0, 1 ]

   def getComponentType( self, coreId ):
      return "cmx42550"

   def getNumSerdesPerCore( self, coreId ):
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      # four lane modes
      intfs = list( range( 4 ) )
      mappings.extend( [
         MappingDesc( G100_4_NOFEC, intfs, G100_4_RS528, intfs ),
         MappingDesc( G100_4_NOFEC, intfs, G100_4_NOFEC, intfs ),
         MappingDesc( G40_4_NOFEC, intfs, G40_4_NOFEC, intfs ),
      ] )
      return mappings

   def getSupportedAutoneg( self, coreId ):
      # Enigma doesn't support autoneg
      return []

@registerHwL1Component
class Evora( ExternalPhy ):

   #               +-------------------+
   #               |      Evora        |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1   Core 1    1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #               | +---------------+ |
   #               | +---------------+ |
   #       4   ----| |4             4| |----  4
   #       5   ----| |5   Core 2    5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               +-------------------+

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      serdesCoreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( serdesCoreId, physicalLaneId )

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ) }

   def getCoreIds( self ):
      return [ 0, 1 ]

   def getComponentType( self, coreId ):
      return "bcm82391"

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main5Taps, SPEED_25G: Main5Taps }

   def getNumSerdesPerCore( self, coreId ):
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      # four lane modes
      intfs = list( range( 4 ) )
      mappings.extend( [
         MappingDesc( G100_4_NOFEC, intfs, G100_4_RS528, intfs ),
         MappingDesc( G100_4_NOFEC, intfs, G100_4_NOFEC, intfs ),
         MappingDesc( G40_4_NOFEC, intfs, G40_4_NOFEC, intfs ),
      ] )
      # two lane modes
      for x in range( 0, 4, 2 ):
         intfs = [ x, x + 1 ]
         mappings.extend( [
            MappingDesc( G50_2_NOFEC, intfs, G50_2_RS528, intfs ),
            MappingDesc( G50_2_NOFEC, intfs, G50_2_NOFEC, intfs ),
         ] )
      # single lane modes
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( G25_1_NOFEC, [ x ], G25_1_RS528, [ x ] ),
            MappingDesc( G25_1_NOFEC, [ x ], G25_1_FCFEC, [ x ] ),
            MappingDesc( G25_1_NOFEC, [ x ], G25_1_NOFEC, [ x ] ),
            MappingDesc( G10_1_NOFEC, [ x ], G10_1_NOFEC, [ x ] ),
            MappingDesc( G1_1_NOFEC, [ x ], G1_1_NOFEC, [ x ] ),
         ] )
      return mappings

   def getSupportedAutoneg( self, coreId ):
      # Evora doesn't support autoneg
      return []

@registerHwL1Component
class Barchetta2( ExternalPhy ):

   #               +-------------------+
   #               |     Barchetta2    |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4             4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | |     Core 0    | |
   #               | |              8| |----  8
   #               | |              9| |----  9
   #               | |             10| |----  10
   #               | |             11| |----  11
   #               | |             12| |----  12
   #               | |             13| |----  13
   #               | |             14| |----  14
   #               | |             15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Gearbox", "CrossGearbox", "Mux" }

   def getComponentType( self, coreId ):
      if self.mode == "Gearbox":
         return "bcm87728-gb"
      if self.mode == "CrossGearbox":
         return "bcm87728-cgb"
      if self.mode == "Mux":
         return "bcm87728-mux"
      raise HwL1ComponentError( self, "Invalid mode.",
                                mode=self.mode, supportedModes=self.supportedModes )

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main6Taps, SPEED_20G: Main6Taps,
               SPEED_25G: Main6Taps, SPEED_50G: Main6Taps }

   def getSerdes( self, serdesId ):
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      if coreId:
         raise HwL1ComponentError( self, "Invalid coreId.",
                                   coreId=coreId, validCoreIds=[ 0 ] )
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getNumSerdesPerCore( self, coreId ):
      return 16

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getCoreIds( self ):
      return [ 0 ]

   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      if self.mode == "Gearbox":
         # BUG663539: The modes below are placeholders to unblock Oliveville bringup.
         # Review/Update whenever the Barchetta2/PhyIsland agent is ready.

         # four lane PAM4 modes
         # also supports cross retimer modes
         mappings.extend( [
            MappingDesc( G200_4_RS544, list( range( 4 ) ),
                         G200_4_RS544, list( range( 4 ) ) ),
            MappingDesc( G200_4_RS544, list( range( 4, 8 ) ),
                         G200_4_RS544, list( range( 8, 12 ) ) ),
            MappingDesc( G200_4_RS544, list( range( 4 ) ),
                         G200_4_RS544, list( range( 4, 8 ) ) ),
            MappingDesc( G200_4_RS544, list( range( 4, 8 ) ),
                         G200_4_RS544, list( range( 12, 16 ) ) ),
         ] )
         # four lane NRZ modes
         for ( sysLanes, lineLanes ) in (
            ( ( 0, 1 ), list( range( 0, 4 ) ) ),
            ( ( 2, 3 ), list( range( 4, 8 ) ) ),
            ( ( 4, 5 ), list( range( 8, 12 ) ) ),
            ( ( 6, 7 ), list( range( 12, 16 ) ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_2_RS544, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_2_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )
         # two lane modes
         # Note: L1 team decide to disable 50g-2 No-Fec capability on Barchetta2 PHY
         # due to lack of 50G consortium PCS implementation inside its silicon
         for sysLane in range( 8 ):
            lineLanes = [ sysLane * 2, sysLane * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_RS528, lineLanes ),
            ] )
         # single lane modes
         # also supports cross retimer modes
         for ( sysLane, lineLane ) in [
            ( 0, 0 ), ( 1, 1 ), ( 2, 2 ), ( 3, 3 ),
            ( 4, 8 ), ( 5, 9 ), ( 6, 10 ), ( 7, 11 ),
            ( 0, 4 ), ( 1, 5 ), ( 2, 6 ), ( 3, 7 ),
            ( 4, 12 ), ( 5, 13 ), ( 6, 14 ), ( 7, 15 ) ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_1_RS544, [ lineLane ] ),
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )
         # two lane retimer modes
         # also supports cross retimer modes
         for sysLanes, lineLanes in [
               ( [ 0, 1 ], [ 0, 1 ] ),
               ( [ 2, 3 ], [ 2, 3 ] ),
               ( [ 4, 5 ], [ 8, 9 ] ),
               ( [ 6, 7 ], [ 10, 11 ] ),
               ( [ 0, 1 ], [ 4, 5 ] ),
               ( [ 2, 3 ], [ 6, 7 ] ),
               ( [ 4, 5 ], [ 12, 13 ] ),
               ( [ 6, 7 ], [ 14, 15 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
            ] )
      elif self.mode == "CrossGearbox":
         # PHY mode2B, four lane PAM4 retimer mode, crossed to second set of 4 serdes
         mappings.extend( [
            MappingDesc( G200_4_RS544, list( range( 4 ) ),
                         G200_4_RS544, list( range( 4, 8 ) ) ),
            MappingDesc( G200_4_RS544, list( range( 4, 8 ) ),
                         G200_4_RS544, list( range( 12, 16 ) ) ),
         ] )

         # PHY mode 3C, four lane NRZ reverse gearbox mode
         for ( sysLanes, lineLanes ) in (
            ( ( 0, 1 ), list( range( 0, 4 ) ) ),
            ( ( 2, 3 ), list( range( 4, 8 ) ) ),
            ( ( 4, 5 ), list( range( 8, 12 ) ) ),
            ( ( 6, 7 ), list( range( 12, 16 ) ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_2_RS544, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_2_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )

         # PHY mode 3A, two lane PAM4 retimer mode, crossed to second set of 4 serdes
         for sysLanes, lineLanes in [
               ( [ 0, 1 ], [ 4, 5 ] ),
               ( [ 2, 3 ], [ 6, 7 ] ),
               ( [ 4, 5 ], [ 12, 13 ] ),
               ( [ 6, 7 ], [ 14, 15 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
            ] )

         # PHY mode 4C, two lane NRZ reverse gearbox mode
         # Note: L1 team decide to disable 50g-2 No-Fec capability on Barchetta2 PHY
         # due to lack of 50G consortium PCS implementation inside its silicon
         for sysLane, lineLanes in [
               ( 0, [ 0, 1 ] ), ( 1, [ 2, 3 ] ),
               ( 2, [ 4, 5 ] ), ( 3, [ 6, 7 ] ),
               ( 4, [ 8, 9 ] ), ( 5, [ 10, 11 ] ),
               ( 6, [ 12, 13 ] ), ( 7, [ 14, 15 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_RS528, lineLanes ),
            ] )

         # PHY mode 4A / 6, single lane PAM4 / NRZ retimer mode,
         # crossed to second set of 4 serdes
         for ( sysLane, lineLane ) in [
            ( 0, 4 ), ( 1, 5 ), ( 2, 6 ), ( 3, 7 ),
            ( 4, 12 ), ( 5, 13 ), ( 6, 14 ), ( 7, 15 ) ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_1_RS544, [ lineLane ] ),
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )

      elif self.mode == "Mux":
         # BUG794950 The modes below are placeholders to unblock Brownsville2 bringup
         # More mappings should be supported.
         # Review/Update whenever the Barchetta2/PhyIsland agent is ready

         # eight lane modes
         for sysLanes, lineLanes in [
               ( [ 0, 1, 2, 3, 4, 5, 6, 7 ], [ 0, 1, 2, 3, 8, 9, 10, 11 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G400_8_RS544, sysLanes, G400_8_RS544, lineLanes ),
            ] )

         # four lane modes
         for sysLanes, lineLanes in [
               ( [ 0, 1, 2, 3 ], [ 0, 1, 2, 3 ] ),
               ( [ 4, 5, 6, 7 ], [ 8, 9, 10, 11 ] ),
               ( [ 4, 5, 6, 7 ], [ 12, 13, 14, 15 ] ), # MUX
               ]:
            mappings.extend( [
               MappingDesc( G200_4_RS544, sysLanes, G200_4_RS544, lineLanes ),
               MappingDesc( G100_4_RS528, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_4_NOFEC, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_4_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )

         # two lane mux modes
         # Note: L1 team decide to disable 50g-2 No-Fec capability on Barchetta2 PHY
         # due to lack of 50G consortium PCS implementation inside its silicon
         for sysLanes, lineLanes in [
               ( [ 0, 1 ], [ 0, 1 ] ),
               ( [ 2, 3 ], [ 2, 3 ] ),
               ( [ 4, 5 ], [ 8, 9 ] ),
               ( [ 6, 7 ], [ 10, 11 ] ),
               ( [ 4, 5 ], [ 12, 13 ] ), # Mux
               ( [ 6, 7 ], [ 14, 15 ] ), # Mux
               ]:
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
               MappingDesc( G50_2_RS528, sysLanes , G50_2_RS528, lineLanes ),
            ] )

         # single lane modes
         for ( sysLane, lineLane ) in [
            ( 0, 0 ),
            ( 1, 1 ),
            ( 2, 2 ),
            ( 3, 3 ),
            ( 4, 8 ),
            ( 5, 9 ),
            ( 6, 10 ),
            ( 7, 11 ),
            ( 4, 12 ), # MUX
            ( 5, 13 ), # MUX
            ( 6, 14 ), # MUX
            ( 7, 15 ), # MUX
            ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_1_RS544, [ lineLane ] ),
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )
      else:
         raise HwL1ComponentError( self, "Unsupported Barchetta2 mode.",
                                   mode=self.mode,
                                   supportedModes=self.supportedModes )

      return mappings

   def getPhysicalSerdesMappings( self, coreId ):
      mappings = {}
      if self.mode == "Gearbox":
         mappings = { ( 0, ) : ( 0, 1 ),
                      ( 1, ) : ( 2, 3 ),
                      ( 2, ) : ( 4, 5 ),
                      ( 3, ) : ( 6, 7 ),
                      ( 4, ) : ( 8, 9 ),
                      ( 5, ) : ( 10, 11 ),
                      ( 6, ) : ( 12, 13 ),
                      ( 7, ) : ( 14, 15 ) }
      elif self.mode == "CrossGearbox":
         mappings = { ( 0, ) : ( 4, 5 ),
                      ( 1, ) : ( 6, 7 ),
                      ( 2, ) : ( 0, 1 ),
                      ( 3, ) : ( 2, 3 ),
                      ( 4, ) : ( 12, 13 ),
                      ( 5, ) : ( 14, 15 ),
                      ( 6, ) : ( 8, 9 ),
                      ( 7, ) : ( 10, 11 ) }
      elif self.mode == "Mux":
         mappings = { ( 0, ) : ( 0, ),
                      ( 1, ) : ( 1, ),
                      ( 2, ) : ( 2, ),
                      ( 3, ) : ( 3, ),
                      ( 4, ) : ( 8, 12 ),
                      ( 5, ) : ( 9, 13 ),
                      ( 6, ) : ( 10, 14 ),
                      ( 7, ) : ( 11, 15 ) }
      else:
         raise HwL1ComponentError( self, "Unsupported Barchetta2 mode.",
                                   mode=self.mode,
                                   supportedModes=self.supportedModes )
      return mappings

   def getSupportedAutoneg( self, coreId ):
      # Barchetta2 doesn't support autoneg
      return []

class Millenio( ExternalPhy ):

   #               +-------------------+
   #               |      Millenio     |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4             4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | |     Core 0    | |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12           12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+

   supportedModes = { "Gearbox", "Mux" }

   def getComponentType( self, coreId ):
      if self.mode == "Gearbox":
         return "bcm81356-gb"
      elif self.mode == "Mux":
         return "bcm81356-mux"
      raise HwL1ComponentError( self, "Invalid mode.",
                                mode=self.mode, supportedModes=self.supportedModes )

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main5Taps, SPEED_20G: Main5Taps,
               SPEED_25G: Main5Taps, SPEED_50G: Main5Taps }

   def getSerdes( self, serdesId ):
      # Millenio has a single "core" represented by a unique SDK phy_addr
      # Note: this shouldn't be confused with physical cores which Millenio has four
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      if coreId:
         raise HwL1ComponentError( self, "Invalid coreId.",
                                   coreId=coreId, validCoreIds=[ 0 ])
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getNumSerdesPerCore( self, coreId ):
      return 16

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getCoreIds( self ):
      return [ 0 ]

   def getLogicalSerdesMappings( self, coreId ):
      # Millenio does not support lane remapping but L1 Topo will always lane remap
      # thus the logical serdes mapping depends on the specifics on how this chip
      # connects to the front panel so lane swaps can be undone
      raise NotImplementedError

   def getPhysicalSerdesMappings( self, coreId ):
      mappings = {}
      if self.mode == "Gearbox":
         mappings = { ( 0, ) : ( 0, 1 ),
                      ( 1, ) : ( 2, 3 ),
                      ( 2, ) : ( 4, 5 ),
                      ( 3, ) : ( 6, 7 ),
                      ( 8, ) : ( 8, 9 ),
                      ( 9, ) : ( 10, 11 ),
                      ( 10, ) : ( 12, 13 ),
                      ( 11, ) : ( 14, 15 ) }
      elif self.mode == "Mux":
         mappings = { ( 0, ) : ( 0, ),
                      ( 1, ) : ( 1, ),
                      ( 2, ) : ( 2, ),
                      ( 3, ) : ( 3, ),
                      ( 8, ) : ( 4, 8 ),
                      ( 9, ) : ( 5, 9 ),
                      ( 10, ) : ( 6, 10 ),
                      ( 11, ) : ( 7, 11 ) }
      else:
         raise HwL1ComponentError( self, "Unsupported Millenio mode.",
                                   mode=self.mode,
                                   supportedModes=self.supportedModes )
      return mappings

   def getSupportedAutoneg( self, coreId ):
      # Millenio doesn't support autoneg
      return []

@registerHwL1Component
class BrownsvilleMillenio( Millenio ):
   def getLogicalSerdesMappings( self, coreId ):
      mappings = []
      if self.mode == "Gearbox":
         # four lane PAM4 modes
         mappings.extend( [
            MappingDesc( G200_4_RS544, list( range( 4 ) ),
                  G200_4_RS544, list( range( 4 ) ) ),
            MappingDesc( G200_4_RS544, list( range( 4,
                  8 ) ), G200_4_RS544, list( range( 8, 12 ) ) ),
         ] )
         # four lane NRZ modes
         for ( sysLanes, lineLanes ) in (
            ( ( 0, 1 ), list( range( 0, 4 ) ) ),
            ( ( 2, 3 ), list( range( 4, 8 ) ) ),
            ( ( 4, 5 ), list( range( 8, 12 ) ) ),
            ( ( 6, 7 ), list( range( 12, 16 ) ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_2_RS544, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_2_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )
         # two lane modes
         for sysLane in range( 8 ):
            lineLanes = [ sysLane * 2, sysLane * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_RS528, lineLanes ),
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_FCFEC, lineLanes ),
               MappingDesc( G50_1_RS544, [ sysLane ], G50_2_NOFEC, lineLanes ),
            ] )
         # single lane modes
         # Millenio doesn't support lane remapping but L1 Topo will always lane remap
         # Logical lanes  (system)   5  4  7  6              1  0  3  2
         #
         # Physical lanes (system)   0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15
         #                           |  |  |  |              |  |  |  |
         #                           |  |  |  |              |  |  |  |
         # Physical lanes (line)     0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15
         #
         # Logical lanes  (line)    10 11  9  8 15 14 12 13  2  3  1  0  7  6  4  5
         #
         # Front-panel              <  Eth27  >             <  Eth25  >
         # The logical lane numbering is obtained using the topology traversal
         # algorithm described in aid/4946. To obtain the system-side-to-line-side
         # logical lane mapping, we need to map them such that the physical serdeses
         # go straight from the system side to the line side.
         # e.g. system-side-logical-lane-0 is mapped to line-side-logical-lane-3 so
         # that system-side-physical-lane-9 aligns with line-side-physical-lane-9
         for ( sysLane, lineLane ) in [
            ( 0, 3 ), ( 1, 2 ), ( 2, 0 ), ( 3, 1 ),
            ( 4, 11 ), ( 5, 10 ), ( 6, 8 ), ( 7, 9 ) ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysLane ], G50_1_RS544, [ lineLane ] ),
               MappingDesc( G25_1_RS528, [ sysLane ], G25_1_RS528, [ lineLane ] ),
               MappingDesc( G25_1_FCFEC, [ sysLane ], G25_1_FCFEC, [ lineLane ] ),
               MappingDesc( G25_1_NOFEC, [ sysLane ], G25_1_NOFEC, [ lineLane ] ),
               MappingDesc( G10_1_NOFEC, [ sysLane ], G10_1_NOFEC, [ lineLane ] ),
            ] )
      elif self.mode == "Mux":
         # Mux mode is a mode with 4 line serdes unused, 4 line serdes straight
         # through and 8 line serdes in a mux such that only the first or latter 4
         # can be active.
         # Mux mode on Millenio is unique in that there are multiple mappings for
         # groups of line side serdes to the same group of system side serdes. This
         # is a result of treating each Millenio chip as a single core and
         # "crossing" normal 8 serdes core boundaries. This also implies that
         # performing l1 topology traversal from the system side would be impossible
         # since multiple front panel interfaces can potential map to the same set
         # of system side serdes for a particular speed.
         #
         # Logical mapping (unused physical lanes get squished):
         #
         #               |  |  |  |              |  |  |  |
         #               |  |  |  |              |  |  |  |
         #               |  |  |  |              |  |  |  |
         #  15 14 13 12 11 10  9  8  7  6  5  4  3  2  1  0 physical lanes
         #  -----------------------------------------------
         #               7  6  5  4              3  2  1  0 logical lanes (syslanes)
         #               |\ |\ |\ |\             |  |  |  |
         #               | \| \| \| \            |  |  |  |
         #               |  \  \  \  \           |  |  |  |
         #               |  |\ |\ |\  \          |  |  |  |
         #               |  | \| \| \  \         |  |  |  |
         #               |  |  \  \  \  \        |  |  |  |
         #               |  |  |\ |\  \  \       |  |  |  |
         #               |  |  | \| \  \  \      |  |  |  |
         #               |  |  |  \  \  \  \     |  |  |  |
         #               |  |  |  |\  \  \  \    |  |  |  |
         #               |  |  |  | \  \  \  \   |  |  |  |
         #               |  |  |  |  \  \  \  \  |  |  |  |
         #              11 10  9  8  7  6  5  4  3  2  1  0 logical lanes (lineLanes)
         #  -----------------------------------------------
         #  15 14 13 12  3  2  0  1  4  5  7  6  8  9 11 10 physical lanes
         #               |  |  |  |  |  |  |  |  |  |  |  | (board specific)
         #               |  |  |  |  |  |  |  |  |  |  |  |
         #               |  |  |  |  |  |  |  |  |  |  |  |
         #               3  2  1  0  7  6  5  4  3  2  1  0
         #               <-QSFP28->              <-QSFP28->
         #                           <------  QSFPDD  ---->
         #                 eth34               eth33

         # eight lane modes
         for sysLanes, lineLanes in [
               ( [ 0, 1, 2, 3, 4, 5, 6, 7 ], [ 0, 1, 2, 3, 4, 5, 6, 7 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G400_8_RS544, sysLanes, G400_8_RS544, lineLanes ),
            ] )

         # four lane modes
         for sysLanes, lineLanes in [
               ( [ 0, 1, 2, 3 ], [ 0, 1, 2, 3 ] ),
               ( [ 4, 5, 6, 7 ], [ 4, 5, 6, 7 ] ), # MUX
               ( [ 4, 5, 6, 7 ], [ 8, 9, 10, 11 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G200_4_RS544, sysLanes, G200_4_RS544, lineLanes ),
               MappingDesc( G100_4_RS528, sysLanes, G100_4_RS528, lineLanes ),
               MappingDesc( G100_4_NOFEC, sysLanes, G100_4_NOFEC, lineLanes ),
               MappingDesc( G40_4_NOFEC, sysLanes, G40_4_NOFEC, lineLanes ),
            ] )

         # two lane modes
         for sysLanes, lineLanes in [
               ( [ 0, 1 ], [ 0, 1 ] ),
               ( [ 4, 5 ], [ 4, 5 ] ), # MUX
               ( [ 2, 3 ], [ 2, 3 ] ),
               ( [ 6, 7 ], [ 6, 7 ] ), # MUX
               ( [ 4, 5 ], [ 8, 9 ] ),
               ( [ 6, 7 ], [ 10, 11 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysLanes, G100_2_RS544, lineLanes ),
               MappingDesc( G50_2_RS528, sysLanes, G50_2_RS528, lineLanes ),
               MappingDesc( G50_2_NOFEC, sysLanes, G50_2_NOFEC, lineLanes ),
            ] )

         # single lane modes
         for sysLanes, lineLanes in [
               ( [ 0 ], [ 0 ] ),
               ( [ 4 ], [ 4 ] ), # MUX
               ( [ 1 ], [ 1 ] ),
               ( [ 5 ], [ 5 ] ), # MUX
               ( [ 2 ], [ 2 ] ),
               ( [ 6 ], [ 6 ] ), # MUX
               ( [ 3 ], [ 3 ] ),
               ( [ 7 ], [ 7 ] ), # MUX
               ( [ 4 ], [ 8 ] ),
               ( [ 5 ], [ 9 ] ),
               ( [ 6 ], [ 10 ] ),
               ( [ 7 ], [ 11 ] ),
               ]:
            mappings.extend( [
               MappingDesc( G50_1_RS544, sysLanes, G50_1_RS544, lineLanes ),
               MappingDesc( G25_1_RS528, sysLanes, G25_1_RS528, lineLanes ),
               MappingDesc( G25_1_FCFEC, sysLanes, G25_1_FCFEC, lineLanes ),
               MappingDesc( G25_1_NOFEC, sysLanes, G25_1_NOFEC, lineLanes ),
               MappingDesc( G10_1_NOFEC, sysLanes, G10_1_NOFEC, lineLanes ),
            ] )
      else:
         raise HwL1ComponentError( self, "Invalid mode.", mode=self.mode,
                                   supportedModes=self.supportedModes )
      return mappings

class StrataAsic( Asic, metaclass=ABCMeta ):
   # Pylint is dumb and doesn't understand that this is also an abstract class
   # pylint: disable=abstract-method

   def getSubdomain( self, coreId=None ):
      return "StrataPhy-{chipId}"

@registerHwL1Component
class T3X1( StrataAsic ):
   cores = {
      GPhy : list( range( 0, 6 ) ),
      MerlinQ : list( range( 6, 8 ) ),
      Falcon16 : [ 8 ],
      Merlin : [ 9 ],
   }

@registerHwL1Component
class T3X2( StrataAsic ):
   cores = {
      GPhy : list( range( 0, 6 ) ), # SerdesIds 0-23
      MerlinQ : list( range( 6, 8 ) ), # SerdesIds 24-31
      Falcon16 : [ 8 ], # SerdesIds 32-35
      Merlin : [ 9 ], # SerdesIds 36-39
   }

@registerHwL1Component
class T3X3( StrataAsic ):
   cores = {
      MerlinQ : list( range( 0, 3 ) ), # SerdesIds 0-11
      Merlin : list( range( 3, 7 ) ), # SerdesIds 12-27
      Falcon16 : list( range( 7, 10 ) ), # SerdesIds 28-39
   }

@registerHwL1Component
class T3X4( StrataAsic ):
   cores = {
      Falcon16 : list( range( 0, 17 ) ),
   }

@registerHwL1Component
class T3X5( StrataAsic ):
   cores = {
      Falcon16 : list( range( 0, 20 ) ),
      MerlinMgmtOnePort( 0 ) : [ 20 ],
   }

@registerHwL1Component
class T3X5L( StrataAsic ):
   cores = {
      Falcon16LowSpeed : list( range( 0, 5 ) ) + list( range( 15, 20 ) ),
      Falcon16 : list( range( 5, 15 ) ),
      MerlinMgmtOnePort( 0 ) : [ 20 ],
   }

@registerHwL1Component
class T3X7( StrataAsic ):
   cores = {
      Falcon16 : list( range( 0, 32 ) ),
      MerlinMgmtTwoPort : [ 32 ],
   }

@registerHwL1Component
class TH2( StrataAsic ):
   cores = {
      Falcon16 : list( range( 0, 64 ) ),
      MerlinMgmtTwoPort  : [ 64 ],
   }

@registerHwL1Component
class TH3( StrataAsic ):
   cores = {
      Blackhawk : list( range( 0, 32 ) ),
      MerlinMgmtTwoPort : [ 32 ],
   }

def getTrident4BlackhawkLineRateToPllRates( strataAsic ):
   t4BhSpeedCompatSetting = {
      SPEED_GBPS_50: [ COMPAT_GBPS_50 ],
      SPEED_GBPS_25: [ COMPAT_GBPS_25 ],
      SPEED_GBPS_20: [ COMPAT_GBPS_10 ],
      SPEED_GBPS_10: [ COMPAT_GBPS_10, COMPAT_GBPS_25 ] }
   coreSpeedCompatSettings = {}
   for coreId in strataAsic.getCoreIds():
      core = strataAsic.getCore( coreId )
      if core.name == "Blackhawk":
         coreSpeedCompatSettings[ coreId ] = t4BhSpeedCompatSetting
      else:
         coreSpeedCompatSettings[ coreId ] = core.getLineRateToPllRateList()
   return coreSpeedCompatSettings

@registerHwL1Component
class T4( StrataAsic ):
   cores = {
      # BUG554014: Add support for BlackhawkGen2 PHY
      Blackhawk : list( range( 0, 32 ) ),
      # BUG554021: Add/verify Merlin core for T4
      MerlinMgmt : [ 32 ],
   }

   def getLineRateToPllRates( self ):
      return getTrident4BlackhawkLineRateToPllRates( self )

@registerHwL1Component
class T4X7( StrataAsic ):
   cores = {
      # 10 Blackhawk7 Octal SerDes cores
      Blackhawk: list( range( 0, 10 ) ),
      # One Merlin7 Quad SerDes core
      MerlinMgmtTwoPort: [ 10 ],
   }

   def getLineRateToPllRates( self ):
      return getTrident4BlackhawkLineRateToPllRates( self )

@registerHwL1Component
class T4X9( StrataAsic ):
   cores = {
      # 20 Blackhawk7 Octal SerDes cores
      Blackhawk : list( range( 0, 20 ) ),
      # One Merlin7 Quad SerDes core
      MerlinMgmtTwoPort : [ 20 ],
   }

   def getLineRateToPllRates( self ):
      return getTrident4BlackhawkLineRateToPllRates( self )

@registerHwL1Component
class T4C( StrataAsic ):
   cores = {
      # 32 Blackhawk7 Octal SerDes cores
      Blackhawk: list( range( 0, 32 ) ),
      # One Merlin7 Quad SerDes core
      MerlinMgmtTwoPort: [ 32 ],
   }

   def getLineRateToPllRates( self ):
      return getTrident4BlackhawkLineRateToPllRates( self )

@registerHwL1Component
class T4CHD( StrataAsic ):
   cores = {
      # 16 Blackhawk7 Octal SerDes cores
      Blackhawk : list( range( 0, 16 ) ),
      # One Merlin7 Quad SerDes core
      MerlinMgmtTwoPort : [ 16 ],
   }

   def getLineRateToPllRates( self ):
      return getTrident4BlackhawkLineRateToPllRates( self )

@registerHwL1Component
class TH4( StrataAsic ):
   cores = {
      BlackhawkGen3 : list( range( 0, 64 ) ),
      MerlinMgmtTwoPort : [ 64 ],
   }

@registerHwL1Component
class TH4HD( StrataAsic ):
   cores = {
      BlackhawkGen3 : list( range( 0, 16 ) ) + list( range( 48, 64 ) ),
      # We have 32 "missing" Blackhawk cores on this variant of TH4, so define a
      # set of 8-lane absent cores here.
      AbsentCore( 8 ): list( range( 16, 48 ) ),
      # The Merlin core also has half the ports it usually does
      MerlinMgmtOnePort( 2 ) : [ 64 ],
   }

@registerHwL1Component
class TH4G( StrataAsic ):
   cores = {
      Osprey : list( range( 0, 32 ) ),
      MerlinMgmtTwoPort : [ 32 ],
   }

@registerHwL1Component
class TH5( StrataAsic ):
   cores = {
      PeregrineNo25G : list( range( 0, 64 ) ),
      D5MgmtTwoPort : [ 64 ],
   }

   def getLogicalPortPoolId( self, coreId ):
      if coreId not in self.cores[ PeregrineNo25G ]:
         return UNKNOWN_LOGICAL_PORT_POOL_ID
      return coreId // 2

   def getLogicalPortPoolSize( self, poolId ):
      return 10

@registerHwL1Component
class TH5T( StrataAsic ):
   cores = {
      PeregrineNo25G : list( range( 0, 16 ) ) + list( range( 48, 64 ) ),
      # We have 32 "missing" Peregrine cores on this variant of TH5, so define a
      # set of 8-lane absent cores here.
      AbsentCore( 8 ): list( range( 16, 48 ) ),
      D5MgmtTwoPort: [ 64 ],
   }

@registerHwL1Component
class Firelight( StrataAsic ):
   cores = {
      MerlinQ : list( range( 0, 3 ) ),
      Falcon16 : list( range( 3, 7 ) ),
   }

@registerHwL1Component
class Firelight2( StrataAsic ):
   cores = {
      Blackhawk1G : list( range( 0, 3 ) ),
      MerlinG2: list( range( 3, 9 ) ),
      Blackhawk : list( range( 9, 12 ) )
   }

# TODO: Currently Wh3+ L1 kept same as T3X1. Will likely changes.
@registerHwL1Component
class Wh3Plus( StrataAsic ):
   cores = {
      GPhy : list( range( 0, 6 ) ),
      Merlin : list( range( 6, 12 ) ),
   }

@registerHwL1Component
class BCM54998E( ExternalPhy ):
   # Phy that splits 2 system serdes at 10G into 8 line side serdes at 2.5G

   #  system   +-------------------+    line
   #  serdes   |     BCM54998E     |   serdes
   #           |                   |    
   #    0  ____|___________________|____  0
   #           |        |__________|____  1
   #           |        |__________|____  2
   #           |        |__________|____  3
   #           |                   |
   #    5  ____|___________________|____  4
   #           |        |__________|____  5
   #           |        |__________|____  6
   #           |        |__________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Map will be:
      # 0 : 0, 1, 2, 3
      # 5 : 4, 5, 6, 7
      return { ( 0, ) : ( 0, 1, 2, 3 ),
               ( 5, ) : ( 4, 5, 6, 7 ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Mux 4x <=2.5G Base-T signals onto a shared USXGMII signal ( 10G serdes )
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class BCM54998S( ExternalPhy ):
   # 5GBASE-T phy that passes through
   # 8 system serdes at 10G to 8 line side serdes at 5G

   #  system   +-------------------+    line
   #  serdes   |     BCM54998S     |   serdes
   #           |                   |    
   #    0  ____|___________________|____  0
   #    1  ____|___________________|____  1
   #    2  ____|___________________|____  2
   #    3  ____|___________________|____  3
   #           |                   |
   #    4  ____|___________________|____  4
   #    5  ____|___________________|____  5
   #    6  ____|___________________|____  6
   #    7  ____|___________________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ),
               ( 4, ) : ( 4, ),
               ( 5, ) : ( 5, ),
               ( 6, ) : ( 6, ),
               ( 7, ) : ( 7, ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      # TODO: We can technically also do USXGMII on these phys, but I don't see a
      #       reason we'd wire up one of these for that vs a BCM54998E
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G2P5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G2P5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G5_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G2P5_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 4 ], G1_1_NOFEC, [ 4 ] ),
         MappingDesc( G1_1_NOFEC, [ 4 ], M100_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G5_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 5 ] ),
         MappingDesc( G1_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G5_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G2P5_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 6 ], G1_1_NOFEC, [ 6 ] ),
         MappingDesc( G1_1_NOFEC, [ 6 ], M100_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G5_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G2P5_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 7 ], G1_1_NOFEC, [ 7 ] ),
         MappingDesc( G1_1_NOFEC, [ 7 ], M100_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class MraTPhy( ExternalPhy ):
   # A single lane phy that supports 10G on system side and
   # 100M, 1G, 2.5G, 5G and 10G on the line side.
   # 2.5G and 5G are not advertised to customers. 
   def getSerdes( self, serdesId ):
      return( 0, 0 )
   
   def getCoreIds( self ):
      return [ 0 ]
 
   def getNumSerdesPerCore( self, coreId ):
      return 1
 
   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ), }

   def getLogicalSerdesMappings( self, coreId ):
      # TODO: See BUG880526; Check if these should be using "channelized modes" 
      # or not.
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                       SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                       SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                       SPEED_MBPS_100, AUTONEG_CL28 ),
      ]
   
@registerHwL1Component
class BCM54998ES( BCM54998S ):
   # 2.5GBASE-T phy that passes through
   # 8 system serdes at 10G to 8 line side serdes at 2.5G

   #  system   +-------------------+    line
   #  serdes   |     BCM54998ES     |   serdes
   #           |                   |
   #    0  ____|___________________|____  0
   #    1  ____|___________________|____  1
   #    2  ____|___________________|____  2
   #    3  ____|___________________|____  3
   #           |                   |
   #    4  ____|___________________|____  4
   #    5  ____|___________________|____  5
   #    6  ____|___________________|____  6
   #    7  ____|___________________|____  7
   #           |                   |
   #           +-------------------+

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      # TODO: We can technically also do USXGMII on these phys, but I don't see a
      #       reason we'd wire up one of these for that vs a BCM54998E
      sModes = super().getLogicalSerdesMappings( coreId )
      return [ mode for mode in sModes if mode.lineMode.intfSpeed != 'speed5Gbps' ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class BCM54182( ExternalPhy ):
   #  system   +-------------------+    line
   #  serdes   |     BCM54182      |   serdes
   #           |                   |
   #    0  ____|___________________|____  0
   #           |        |__________|____  1
   #           |        |__________|____  2
   #           |        |__________|____  3
   #           |                   |
   #    1  ____|___________________|____  4
   #           |        |__________|____  5
   #           |        |__________|____  6
   #           |        |__________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Map will be:
      # 0 : 0, 1, 2, 3
      # 1 : 4, 5, 6, 7
      return { ( 0, ) : ( 0, 1, 2, 3 ),
               ( 1, ) : ( 4, 5, 6, 7 ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Mux 4x <=1G Base-T signals onto a single QSGMII signal ( 5G serdes )
      return [
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_5, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_5, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_5, AUTONEG_DISABLED,
                      SPEED_MBPS_10, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class BCM54185( ExternalPhy ):
   #  system   +-------------------+    line
   #  serdes   |     BCM54185      |   serdes
   #           |                   |
   #    0  ____|___________________|____  0
   #           |        |__________|____  1
   #           |        |__________|____  2
   #           |        |__________|____  3
   #           |                   |
   #    1  ____|___________________|____  4
   #           |        |__________|____  5
   #           |        |__________|____  6
   #           |        |__________|____  7
   #           |                   |
   #           +-------------------+

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Map will be:
      # 0 : 0, 1, 2, 3
      # 1 : 4, 5, 6, 7
      return { ( 0, ) : ( 0, 1, 2, 3 ),
               ( 1, ) : ( 4, 5, 6, 7 ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Mux 4x <= 1000BASE-X signals onto a single QSGMII signal ( 5G serdes )
      return [
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 1 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 2 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 3 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_GBPS_1, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M100_1_NOFEC_HALF, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 5 ],
                      sysChannels=[ ( 1, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 6 ],
                      sysChannels=[ ( 2, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], M10_1_NOFEC_HALF, [ 7 ],
                      sysChannels=[ ( 3, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( SPEED_GBPS_5, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL37 ),
      ]

@registerHwL1Component
class BCM54994( ExternalPhy ):
   # 5GBASE-T Phy

   def getSerdes( self, serdesId ):
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ), }

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G2P5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G2P5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class BCM84894( ExternalPhy ):
   # 10GBASE-T Quad Phy

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G2P5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G2P5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps }

@registerHwL1Component
class BCM84898( ExternalPhy ):
   # 10GBASE-T Phy

   def getSerdes( self, serdesId ):
      # All cores are the same so just use core 0
      physicalLaneId = serdesId % self.getNumSerdesPerCore( 0 )
      coreId = serdesId // self.getNumSerdesPerCore( 0 )
      return ( coreId, physicalLaneId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 8

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ),
               ( 4, ) : ( 4, ),
               ( 5, ) : ( 5, ),
               ( 6, ) : ( 6, ),
               ( 7, ) : ( 7, ) }

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of XFI with idle stuffing for >1G ( 10G Serdes ), or SGMII with
      # symbol replication for <=1G ( 1G Serdes )
      # TODO: We can technically also do USXGMII on these phys, but I don't see a
      #       reason we'd wire up one of these for that vs a BCM54899E
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], G2P5_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], G2P5_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], M100_1_NOFEC, [ 1 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], G2P5_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], M100_1_NOFEC, [ 2 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], G2P5_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], M100_1_NOFEC, [ 3 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G10_1_NOFEC, [ 4 ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G5_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 4 ], G2P5_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 4 ], G1_1_NOFEC, [ 4 ] ),
         MappingDesc( G1_1_NOFEC, [ 4 ], M100_1_NOFEC, [ 4 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G10_1_NOFEC, [ 5 ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G5_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 5 ], G2P5_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 5 ], G1_1_NOFEC, [ 5 ] ),
         MappingDesc( G1_1_NOFEC, [ 5 ], M100_1_NOFEC, [ 5 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G10_1_NOFEC, [ 6 ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G5_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 6 ], G2P5_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 6 ], G1_1_NOFEC, [ 6 ] ),
         MappingDesc( G1_1_NOFEC, [ 6 ], M100_1_NOFEC, [ 6 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G10_1_NOFEC, [ 7 ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G5_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 7 ], G2P5_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 7 ], G1_1_NOFEC, [ 7 ] ),
         MappingDesc( G1_1_NOFEC, [ 7 ], M100_1_NOFEC, [ 7 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED,
                      SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class Cortina( ExternalPhy ):

   #               +-------------------+
   #               |      Cortina      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2   Core 0    2| |----  2
   #       3   ----| |3             3| |----  3
   #               | +---------------+ |
   #               +-------------------+

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumCores( self ):
      return 1

   def getComponentType( self, coreId ):
      return "cs4317"

   def getNumSerdesPerCore( self, coreId ):
      return 4

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return phyScope == PHY_SCOPE_LINE

   def getPhysicalSerdesMappings( self, coreId ):
      # Physical mapping is straight through
      # A[ 0-3 ]: B[ 0-3 ]
      return { ( 0, ) : ( 0, ),
               ( 1, ) : ( 1, ),
               ( 2, ) : ( 2, ),
               ( 3, ) : ( 3, ) }

   def getLogicalSerdesPairs( self, coreId ):
      return [
         SerdesPair( 0, PHY_SCOPE_SYSTEM, 0, PHY_SCOPE_LINE ),
         SerdesPair( 1, PHY_SCOPE_SYSTEM, 1, PHY_SCOPE_LINE ),
         SerdesPair( 2, PHY_SCOPE_SYSTEM, 2, PHY_SCOPE_LINE ),
         SerdesPair( 3, PHY_SCOPE_SYSTEM, 3, PHY_SCOPE_LINE ),
      ]

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      supportedSpeeds = [ SPEED_GBPS_1, SPEED_GBPS_10 ]
      if toggleDropbear100FullSupportEnabled():
         supportedSpeeds.insert( 0, SPEED_MBPS_100 )
      return supportedSpeeds 



class Crosspoint( Component ):
   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def laneRemapCapable( self, coreId ):
      return True

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return True

   def getPhysicalSerdesMappings( self, coreId ):
      # No mapping
      return {}

   def getLogicalSerdesMappings( self, coreId ):
      # No mapping
      # TODO: Remove when we enable CrosspointPmaSupport toggle
      return []

   def getChipCategory( self ):
      return CATEGORY_CROSSPOINT

   def getPossibleSerdesPairs( self, coreId ):
      return WILDCARD

   @abstractmethod
   def getComponentType( self, coreId ):
      pass

   @abstractmethod
   def getNumSerdesPerCore( self, coreId ):
      pass

   def getLogicalSerdesPairs( self, coreId ):
      serdesPairs = []
      for serdes in range( self.getNumSerdesPerCore( coreId ) ):
         serdesPairs += [ SerdesPair( serdes, PHY_SCOPE_SYSTEM, None, None ),
                          SerdesPair( None, None, serdes, PHY_SCOPE_LINE ) ]
      return serdesPairs

@registerHwL1Component
class m21605( Crosspoint ):

   def getComponentType( self, coreId ):
      return "m21605"

   def getNumSerdesPerCore( self, coreId ):
      return 160
   
   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      supportedSpeeds = [ SPEED_GBPS_1, SPEED_GBPS_10 ]
      if toggleDropbear100FullSupportEnabled():
         supportedSpeeds.insert( 0, SPEED_MBPS_100 )
      return supportedSpeeds 

   def getSupportedAutoneg( self, coreId ):
      # M21605 does not natively support autoneg.
      return []

@registerHwL1Component
class m21048( Crosspoint ):

   def getComponentType( self, coreId ):
      return "m21048"

   def getNumSerdesPerCore( self, coreId ):
      return 48
   
   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      supportedSpeeds = [ SPEED_GBPS_1, SPEED_GBPS_10 ]
      if toggleDropbear100FullSupportEnabled():
         supportedSpeeds.insert( 0, SPEED_MBPS_100 )
      return supportedSpeeds 

   def getSupportedAutoneg( self, coreId ):
      # M21048 does not natively support autoneg.
      return []

@registerHwL1Component
class Marvel88E1514( ExternalPhy ):
   # 1000BASE-T Phy

   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 1

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      return { ( 0, ) : ( 0, )}

   def getLogicalSerdesMappings( self, coreId ):
      # Passthrough of SGMII with symbol replication for 10M/100M ( 1G Serdes )
      return [
         MappingDesc( G1_1_NOFEC, [ 0 ], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M100_1_NOFEC_HALF, [ 0 ],
                     sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_HALF ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M10_1_NOFEC, [ 0 ],
                      sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], M10_1_NOFEC_HALF, [ 0 ],
                     sysChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_HALF ) ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_100, AUTONEG_CL28 ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED,
                      SPEED_MBPS_10, AUTONEG_CL28 ),
      ]

@registerHwL1Component
class maxp37170( Crosspoint ):
   # mode None is the original 10G only Crosspoint chip.
   # maxp37170 supports 25G from revF onwards.
   supportedModes = { None, "25G" }

   def getComponentType( self, coreId ):
      if self.mode:
         return "maxp37170-25G"
      return "maxp37170"

   def getNumSerdesPerCore( self, coreId ):
      return 288

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      speeds = [ SPEED_GBPS_1, SPEED_GBPS_10 ]
      if self.mode:
         speeds.append( SPEED_GBPS_25 )
      return speeds

   def getTapGroups( self, coreId ):
      tapGroups = { SPEED_10G: Main3Taps }
      if self.mode:
         tapGroups |= { SPEED_25G: Main3Taps }
      return tapGroups

   def getSupportedAutoneg( self, coreId ):
      # MaxP37170 does not natively support autoneg.
      return []

@registerHwL1Component
class maxp37161( Crosspoint ):
   def getComponentType( self, coreId ):
      return "maxp37161"

   def getSupportedSpeeds( self, coreId, serdesId, phyScope ):
      return [ SPEED_GBPS_1, SPEED_GBPS_10, SPEED_GBPS_25 ]

   def getNumSerdesPerCore( self, coreId ):
      return 16

   def getTapGroups( self, coreId ):
      # TODO: Tuning parameters for a crosspoint differ I think.
      return {
         SPEED_10G: Main3Taps,
         SPEED_25G: Main3Taps,
      }
   def getSupportedAutoneg( self, coreId ):
      # MaxP37161 does not natively support autoneg.
      return []

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return isTx

class CpuNic( Component ):

   def getSerdes( self, serdesId ):
      return( 0, 0 )

   def getCoreIds( self ):
      return [ 0 ]

   def getNumSerdesPerCore( self, coreId ):
      return 1

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # No mapping
      return {}

   def getChipCategory( self ):
      return CATEGORY_CPU_NIC

   @abstractmethod
   def getLogicalSerdesMappings( self, coreId ):
      return []

@registerHwL1Component
class TenGCpuNic( CpuNic ):
   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
      ]

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps }
   
   def getSupportedAutoneg( self, coreId ):
      # TenGCpuNic does not natively support autoneg.
      return []

@registerHwL1Component
class OneGCpuNic( CpuNic ):
   def getLogicalSerdesMappings( self, coreId ):
      return [
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
      ]

   def getSupportedAutoneg( self, coreId ):
      # OneGCpuNic does not natively support autoneg.
      return []

class App( Component ):
   def getSerdes( self, serdesId ):
      return ( 0, serdesId )

   def getCoreIds( self ):
      return [ 0 ]

   def laneRemapCapable( self, coreId ):
      return False

   def polarityRemapCapable( self, coreId, phyScope, isTx ):
      return False

   def getPhysicalSerdesMappings( self, coreId ):
      # No mapping
      return {}

   def getChipCategory( self ):
      return CATEGORY_SWITCH_CHIP

   @abstractmethod
   def getNumSerdesPerCore( self, coreId ):
      pass

   @abstractmethod
   def getLogicalSerdesMappings( self, coreId ):
      pass

@registerHwL1Component
class XCKU095_2FFVB2104E( App ):

   def getNumSerdesPerCore( self, coreId ):
      return 64

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      # single lane modes
      for sgm in ( G1_1_NOFEC, G10_1_NOFEC ):
         for serdes in range( self.getNumSerdesPerCore( coreId ) ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      # four lane modes
      for sgm in ( G40_4_NOFEC, ):
         # Only enable 40G for select quads
         for serdes in itertools.chain( range( 0, 28, 4 ), range( 32, 60, 4 ) ):
            serdesList = list( range( serdes, serdes + 4 ) )
            mapping.append( MappingDesc( sgm, serdesList, None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, serdesList ) )
      return mapping
   
   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
      ]

@registerHwL1Component
class XCVU7P_2FLVB2104E( App ):

   def getNumSerdesPerCore( self, coreId ):
      return 76

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      # single lane modes
      for sgm in ( G1_1_NOFEC, G10_1_NOFEC ):
         for serdes in range( self.getNumSerdesPerCore( coreId ) ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      # four lane modes
      for sgm in ( G40_4_NOFEC, ):
         # Only enable 40G for select quads
         for serdes in itertools.chain( range( 0, 28, 4 ), range( 32, 60, 4 ) ):
            serdesList = list( range( serdes, serdes + 4 ) )
            mapping.append( MappingDesc( sgm, serdesList, None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, serdesList ) )
      return mapping
   
   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
      ]

@registerHwL1Component
class XCVU9P_3FLGB2104E( App ):

   # Dropbear EH variants have leaf fpgas that need to be handled differently
   supportedModes = { None, "Leaf" }

   def getNumSerdesPerCore( self, coreId ):
      return 76

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      # single lane modes
      for sgm in ( G1_1_NOFEC, G10_1_NOFEC ):
         for serdes in range( self.getNumSerdesPerCore( coreId ) ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      # four lane modes
      for sgm in ( G40_4_NOFEC, ):
         # Only enable 40G for select quads
         for serdes in itertools.chain( range( 0, 28, 4 ), range( 32, 68, 4 ) ):
            serdesList = list( range( serdes, serdes + 4 ) )
            mapping.append( MappingDesc( sgm, serdesList, None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, serdesList ) )
      return mapping

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
      ]

@registerHwL1Component
class XCVU9P_3FLGB2104E_25G( App ):

   @property
   def name( self ):
      return "XCVU9P_3FLGB2104E"

   def getNumSerdesPerCore( self, coreId ):
      return 76

   def getLogicalSerdesMappings( self, coreId ):
      mapping = []
      # single lane modes
      for sgm in ( G1_1_NOFEC, G10_1_NOFEC, G25_1_NOFEC ):
         for serdes in range( self.getNumSerdesPerCore( coreId ) ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      # four lane modes
      for sgm in ( G40_4_NOFEC, ):
         # Only enable 40G for select quads
         for serdes in itertools.chain( range( 0, 28, 4 ), range( 32, 68, 4 ) ):
            serdesList = list( range( serdes, serdes + 4 ) )
            mapping.append( MappingDesc( sgm, serdesList, None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, serdesList ) )
      return mapping

   def getTapGroups( self, coreId ):
      return { SPEED_10G: Main3Taps, SPEED_25G: Main3Taps }
   
   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
      ]

@registerHwL1Component
class XCVH1542_3HSEVSVA3697( App ):

   @property
   def name( self ):
      return "XCVH1542_3HSEVSVA3697"

   def getNumSerdesPerCore( self, coreId ):
      # See AMD / Xilinx documentation:
      # Versal Adaptive SoC Packaging and Pinouts Architecture Manual (AM013)
      # table "Gigabit Transceiver Channels by Device/Package"
      # says 68 GTYP serdes (52 General-purpose, 16 PCIe-only) and 20 GTM serdes.
      # This refers to the physical capabilities of the chip
      # (72 eth-capable serdes), not any specific standard in AID/11397
      return 72

   def getLogicalSerdesMappings( self, coreId ):
      # For reference see:
      # - AID12365 section 2.2.5
      # - AID11397 sheet "BVL Connection Reference"
      # - serdesId in csvgenerator models/chip/XCVH1542_VSVA3697.csv
      mapping = []
      # single lane modes
      # GTYP serdesId 0-51
      for sgm in ( G1_1_NOFEC, G10_1_NOFEC, G25_1_NOFEC ):
         for serdes in range( 51 + 1 ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )
      # GTM serdesId 52-71 can't do 1G and are higher latency
      for sgm in ( G10_1_NOFEC, G25_1_NOFEC ):
         for serdes in range( 52, 71 + 1 ):
            mapping.append( MappingDesc( sgm, [ serdes ], None, [] ) )
            mapping.append( MappingDesc( None, [], sgm, [ serdes ] ) )

      # four lane modes TBD - we may have different restrictions to VU9P
      # See BUG968666 for details
      return mapping

   def getTapGroups( self, coreId ):
      # Versal tap values are built into the fpga images for now.
      # See BUG968666 for details
      return {}

   def getSupportedAutoneg( self, coreId ):
      # Autoneg isn not yet supported by any versal apps for now
      # See BUG968666 for details
      return []

def getComponent( name ):
   componentClass = registeredComponents.get( name )
   if not componentClass:
      raise HwL1ComponentError( name, 'Invalid component class name.',
                                validNames=list( registeredComponents.values() ) )
   return componentClass
