# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Contains the definition of the PHY cores that L1 Topology supports.

TODO:
   BUG580651: We need to be able to handle core capabilities changing depending on
              the component which they are a part of. For example, Falcon can only
              run at "HG" speeds on Strata ASICs.
"""

from abc import ABCMeta, abstractmethod

from L1Topology.Constants import (
   PHY_SCOPE_LINE,
)
from L1Topology.Modes import (
   FEC_RS544,
   G800_8_RS544,
   G400_4_RS544,
   G400_8_RS544,
   G200_2_RS544,
   G200_4_RS544,
   G100_1_RS544,
   G100_2_RS544,
   G100_4_RS528,
   G100_4_NOFEC,
   G50_1_RS544,
   G50_2_RS528,
   G50_2_FCFEC,
   G50_2_NOFEC,
   G40_2_NOFEC,
   G40_4_FCFEC,
   G40_4_NOFEC,
   G25_1_RS528,
   G25_1_FCFEC,
   G25_1_NOFEC,
   G10_1_FCFEC,
   G10_1_NOFEC,
   G5_1_NOFEC,
   G2P5_1_NOFEC,
   G1_1_NOFEC,
   M100_1_NOFEC,
   M100_1_NOFEC_HALF,
   M10_1_NOFEC,
   M10_1_NOFEC_HALF,
   HG_G106_4_NOFEC,
   HG_G53_2_NOFEC,
   HG_G42_4_NOFEC,
   HG_G27_1_NOFEC,
   HG_G21_2_NOFEC,
   HG_G11_1_NOFEC,
   SPEED_GBPS_800,
   SPEED_GBPS_400,
   SPEED_GBPS_200,
   SPEED_GBPS_100,
   SPEED_GBPS_50,
   SPEED_GBPS_40,
   SPEED_GBPS_25,
   SPEED_GBPS_10,
   SPEED_GBPS_5,
   SPEED_GBPS_2p5,
   SPEED_GBPS_1,
   SPEED_MBPS_100,
   SPEED_MBPS_10,
   DUPLEX_FULL,
   DUPLEX_HALF,
   COMPAT_GBPS_100,
   COMPAT_GBPS_50,
   COMPAT_GBPS_25,
   COMPAT_GBPS_10,
)

from .Errors import (
   HwL1ComponentError,
)
from .Serdes import (
   MappingDesc,
   AUTONEG_CL28,
   AUTONEG_CL37,
   AUTONEG_CL73,
   AUTONEG_DISABLED,
   AutonegDesc,
)
from .Tuning import (
   Main3Taps,
   Main4Taps,
   Main5Taps,
   Main5Taps3Post,
   Main6Taps,
   Main6Taps3Pre,
   NoTuning,
   SPEED_10G,
   SPEED_20G,
   SPEED_25G,
   SPEED_50G,
   SPEED_100G,
   SPEED_HG_11G,
   SPEED_HG_27G,
)

from Toggles.PhyEeeToggleLib import togglePeregrineAutonegEnabled

class Core( metaclass=ABCMeta ):
   """A base class for all cores. The behaviour of the cores are defined
   by the abstract methods and attributes below.
   """

   # supportedModes should be a set containing the modes that this core supports
   # being configured in.
   # By default cores support no modes; None is used to signify this "null" mode of
   # the core.
   supportedModes = { None }

   def __init__( self, mode=None ):
      self.mode = mode
      if not isinstance( self.supportedModes, set ):
         raise HwL1ComponentError( self, 'supportedModes must be a set.',
                                   supportedModes=self.supportedModes )
      if mode not in self.supportedModes:
         raise HwL1ComponentError( self, "Invalid core mode.",
                                   mode=mode, supportedModes=self.supportedModes )

   @property
   def name( self ):
      "The name of the class to output for things like strings."
      return self.__class__.__name__

   def __str__( self ):
      """
      A human-readable output for what the instance of the Core represents.

      If the core is initialized in no mode ( if supported ), we will output
      only '<core name>'.
      """
      # pylint: disable-next=consider-using-f-string
      return "%s-%s" % ( self.name, self.mode ) if self.mode else self.name

   def getComponentType( self ):
      """
      Returns
      -------
      String; a name for l1 topology state machines to use for the core on the chip.
      By convention this is the string conversion of the component.
      
      """
      return str( self )

   @abstractmethod
   def getNumSerdes( self ):
      """
      Returns
      -------
      int; The number of serdes within the core
      """
      raise NotImplementedError

   @abstractmethod
   def laneRemapCapable( self ):
      """
      Returns
      -------
      Bool; whether the core can handle lane remaps.
      """
      raise NotImplementedError

   @abstractmethod
   def polarityRemapCapable( self, phyScope ):
      """
      Returns
      -------
      Bool; whether the given side of the core can handle polarity remaps.
      """
      raise NotImplementedError

   @abstractmethod
   def getPhysicalMapping( self ):
      """
      Parameters
      -------
      baseSerdesId: int; the base serdes of the core. All other serdes
         are numbered consecutively from here

      Returns
      -------
      Physical mapping descriptor object
      """
      raise NotImplementedError

   @abstractmethod
   def getLogicalSerdesMapping( self ):
      """
      Retrieves the logical SerDes mappings for this core.

      Returns
      -------
      An iterable which produces items of type: Serdes.MappingDesc
      The Serdes.MappingDesc iteslf has the format:
         ( Hardware::Phy::SerdesGroupMode: System Group Mode,
           list( int ): System Lanes,
           Hardware::Phy::SerdesGroupMode: Line Group Mode,
           list( int ): Line lanes )

      Group Modes are defined in the SerDes module.

      Note
      -----
      The Group Mode and the number of lanes must be consistent.
      e.g. if System Group Mode is G400_8_RS544, System Lanes must be a type that
      supports indexing with 8 items.
      """
      raise NotImplementedError

   @abstractmethod
   def getTapGroups( self ):
      """
      Retrieves the supported Taps for each speed for this core.

      Returns
      -------
      A dict of SerdesSpeed to Taps class, where the Taps class has the addTuning
      method defined on it. See the Tuning module for more information/examples.
      """
      raise NotImplementedError

   def getLogicalSerdesPairs( self, coreId ):
      """
      Retrieves the logical serdes pairs for a PMA-only chip.

      Returns
      ------
      An iterable that contains information for each pair of serdes on the core.
      Each entry contains type SerdesPair defined in Serdes module.

      Note
      ----
      This function is intended for modeling PMA-specific chips with one to one
      serdes mapping such as Crosspoint, chips that require additional information
      should still use getLogicalSerdesMappings.
      """
      return None

   def getSupportedSpeeds( self, coreId, serdesId, phyScope):
      """
      Retrieve the supported speeds for a serdes that defined mappings using
      getLogicalSerdesPairs.

      Returns
      -------
      A list of Interface::EthSpeed that is supported by the serdes.

      Note
      ----
      This function should be defined if and only if getLogicalSerdesPairs
      is defined.
      """
      return None

   def getSupportedAutoneg( self, coreId ):
      """
      Retrieve the suported autoneg configurations supported by the chip.

      Returns
      ------
      A list of AutonegDesc which each have the format:
         ( Interface::EthSpeed: systemSideSpeed,
           Interface::AutonegMode: systemSideAutoneg,
           Interface::EthSpeed: lineSideSpeed,
           Interface::AutonegMode: lineSideAutoneg )
      Note
      ----
      PMA Components should instead define speed as serdes rate as opposed
      to full multi-lane speed. See BUG740787.
      """
      return []

   def getPllCount( self ):
      """
      Returns
      -------
      int; The number of PLLs that drive the core SerDes clocks
      """
      return 0

   def getLineRateToPllRateList( self ):
      """
      Returns
      -------
      A dict of Interface::EthSpeed to a list of Interface::SpeedCompatSetting
      """
      return {}

class AbsentCore( Core ):
   """
   This core type represents a core that is not populated on the actual component in
   question. This may be the case for variants of a given chip that don't
   contain/identify some of the cores present on the full chip.

   For these depopulated components and due to how serdesIds on the component are
   identified, we need to know how large these absent cores would have been on the
   fully-populated variant.

   In the example below, we define a new component `Foo` that specifies an absent
   core that would have normally contained 4 serdes. This means that if the FooCore
   with id 1 ends with serdesId 7, then the FooCore with id 3 will start with
   serdesId 12, rather than continuing on at id 8.

   ```
   class Foo( CoreBasedComponent ):
      cores = {
         FooCore: [ 0, 1, 3 ]
         AbsentCore( 4 ): [ 2 ]
      }
   ```

   Note: Unlike other Core classes, this core must be instantiated when being defined
         on the CoreBasedComponent.
   """

   # We override the usual constructor for the core, and instead make the returned
   # object callable with the same args as the parent core type.
   # This lets us use the instance object itself as though it was a Core class.
   def __init__( self, numSerdes ):
      self.numSerdes = int( numSerdes )

   def __call__( self, mode=None ):
      super().__init__( mode=mode )
      return self

   def getNumSerdes( self ):
      return self.numSerdes

   # None of the below functions should ever be called on an absent core
   def laneRemapCapable( self ):
      raise NotImplementedError
   def polarityRemapCapable( self, phyScope ):
      raise NotImplementedError
   def getPhysicalMapping( self ):
      raise NotImplementedError
   def getLogicalSerdesMapping( self ):
      raise NotImplementedError
   def getTapGroups( self ):
      raise NotImplementedError

class AsicCore( Core ):
   # Asics do not have any mappings, return empty map
   def getPhysicalMapping( self ):
      return {}

   # By default we assume asic cores can be lane remapped
   def laneRemapCapable( self ):
      return True

   # By default we assume asic cores can be polarity remapped
   def polarityRemapCapable( self, phyScope ):
      return True

   @abstractmethod
   def getNumSerdes( self ):
      raise NotImplementedError

   @abstractmethod
   def getLogicalSerdesMapping( self ):
      raise NotImplementedError

   @abstractmethod
   def getTapGroups( self ):
      raise NotImplementedError

class Peregrine( AsicCore ):
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G100_1_RS544, [ x ] ),
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G200_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ):
         mappings.extend( [
            MappingDesc( None, [], G400_4_RS544, x ),
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, list( range( 8 ) ) ),
         MappingDesc( None, [], G800_8_RS544, list( range( 8 ) ) ),
         ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_25G: Main6Taps3Pre,
               SPEED_50G: Main6Taps3Pre,
               SPEED_100G: Main6Taps3Pre }

   def getSupportedAutoneg( self, coreId ):
      return [] # No autoneg supported yet.

   def getPllCount( self ):
      return 1

   def getLineRateToPllRateList( self ):
      return { SPEED_GBPS_100: [ COMPAT_GBPS_100 ],
               SPEED_GBPS_50: [ COMPAT_GBPS_100 ],
               SPEED_GBPS_25: [ COMPAT_GBPS_25 ],
               SPEED_GBPS_10: [ COMPAT_GBPS_25 ] }

class PeregrineNo25G( Peregrine ):
   """
   A Peregrine core which does not support 25G, as seen on TH5/BCM78900
   """

   name = 'Peregrine'

   def getLogicalSerdesMapping( self ):
      mappings = super().getLogicalSerdesMapping()
      return [ mapping for mapping in mappings
               if mapping.lineMode.intfSpeed != SPEED_GBPS_25 ]

   def getSupportedAutoneg( self, coreId ):
      if togglePeregrineAutonegEnabled():
         return [
            AutonegDesc( None, None, SPEED_GBPS_800, AUTONEG_CL73 ),
            AutonegDesc( None, None, SPEED_GBPS_400, AUTONEG_CL73 ),
            AutonegDesc( None, None, SPEED_GBPS_200, AUTONEG_CL73 ),
            AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
            AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         ]
      else:
         return []
      return []

class CellBasedPeregrine( AsicCore ):
   """
   A Peregrine core which only accept Broadcom cell based frames.

   As per BCM88860/BCM88390 DS104, cell based fabric cores do not support multi lane
   data rates.
   """

   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G100_1_RS544, [ x ] ),
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
         ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_50G: Main6Taps3Pre,
               SPEED_100G: Main6Taps3Pre }

class Blackhawk( AsicCore ):
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( G50_1_RS544, [ x ], None, [] ),
            MappingDesc( G25_1_RS528, [ x ], None, [] ),
            MappingDesc( G25_1_FCFEC, [ x ], None, [] ),
            MappingDesc( G25_1_NOFEC, [ x ], None, [] ),
            MappingDesc( G10_1_NOFEC, [ x ], None, [] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G40_2_NOFEC, [ x, x + 1 ] ),
            MappingDesc( G100_2_RS544, [ x, x + 1 ], None, [] ),
            MappingDesc( G50_2_RS528, [ x, x + 1 ], None, [] ),
            MappingDesc( G50_2_NOFEC, [ x, x + 1 ], None, [] ),
            MappingDesc( G40_2_NOFEC, [ x, x + 1 ], None, [] ),
         ] )
      # four lane modes
      for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ): 
         mappings.extend( [
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
            MappingDesc( None, [], G40_4_NOFEC, x ),
            MappingDesc( G200_4_RS544, x, None, [] ),
            MappingDesc( G100_4_RS528, x, None, [] ),
            MappingDesc( G100_4_NOFEC, x, None, [] ),
            MappingDesc( G40_4_NOFEC, x, None, []),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, list( range( 8 ) ) ),
         MappingDesc( G400_8_RS544, list( range( 8 ) ), None, []),
         ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G: Main6Taps, SPEED_20G: Main6Taps, SPEED_25G: Main6Taps,
               SPEED_50G: Main6Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_400, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_200, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_100, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_CL73, None, None ),
         # CL28
         # BUG:738023 Blackhawk does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
      ]

   def getPllCount( self ):
      return 2

   def getLineRateToPllRateList( self ):
      return { SPEED_GBPS_50: [ COMPAT_GBPS_50 ],
               SPEED_GBPS_25: [ COMPAT_GBPS_25 ],
               SPEED_GBPS_10: [ COMPAT_GBPS_10 ] }

# BlackhawkFabric is a Blackhawk core on the fabric side of an ASIC.
# It has non-standard MAC addresses.
class BlackhawkFabric( Blackhawk ):
   pass

# BlackhawkGen3 is the implementation of the Blackhawk serdes core in the Tomahawk4
# chip. It has only a maximum of 4 ports and a single PLL, while still having 8
# serdes lanes. Because this has software implications, we will differentiate it
# from the base Blackhawk type. 
class BlackhawkGen3( Blackhawk ):
   def getLogicalSerdesMapping( self ):
      # The BlackhawkGen3 core does not support FireCode FEC, but supports all the
      # other serdesMappings as the standard Blackhawk core. Thus we will filter
      # out the 25G-1 FC FEC mode for BlackhawkGen3.
      blackhawkMapping = super().getLogicalSerdesMapping()
      return [ m for m in blackhawkMapping if m.lineMode != G25_1_FCFEC ]

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
      ]

# A variant of the Blackhwak core that can run 1G/mGIG data rates
class Blackhawk1G( Blackhawk ):
   name = "Blackhawk1G"
   # XL mode for sub10G, CD mode is the default
   supportedModes = { None, "Sub10G" }

   def baseRLogicalSerdesMapping( self ):
      baseMappings = super().getLogicalSerdesMapping()
      # remove 200G and 400G as we're not supporting Y-Mux/BITW mode

      def isYmuxModeSpeed( x ):
         targetedSpeeds = [ SPEED_GBPS_200, SPEED_GBPS_400 ]
         return x.sysMode in targetedSpeeds or x.lineMode in targetedSpeeds
      return [ x for x in baseMappings if not isYmuxModeSpeed( x ) ]

   def sub10GLogicalSerdesMapping( self ):
      mappings = []

      def addBothSides( mode, serdes ):
         channel = None
         mappings.append( MappingDesc( None, [], mode, serdes ) )
         mappings.append( MappingDesc( mode, serdes, None, [] ) )
         if mode == M100_1_NOFEC:
            mode = G1_1_NOFEC
            channel = [ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ]
            mappings.append( MappingDesc( None, [], mode, serdes,
                                          lineChannels=channel ) )
            mappings.append( MappingDesc( mode, serdes, None, [],
                                          sysChannels=channel ) )
      for x in range( self.getNumSerdes() ):
         addBothSides( M100_1_NOFEC, [ x ] )
         addBothSides( G1_1_NOFEC, [ x ] )
         addBothSides( G2P5_1_NOFEC, [ x ] )
         addBothSides( G5_1_NOFEC, [ x ] )
         addBothSides( G10_1_NOFEC, [ x ] )
      return mappings

   def getLogicalSerdesMapping( self ):
      if self.mode == "Sub10G":
         return self.sub10GLogicalSerdesMapping()
      return self.baseRLogicalSerdesMapping()

class Osprey( AsicCore ):
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G100_1_RS544, [ x ] ),
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G200_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ):
         mappings.extend( [
            MappingDesc( None, [], G400_4_RS544, x ),
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
            MappingDesc( None, [], G40_4_NOFEC, x ),
         ] )
      # eight lane modes
      # BUG543066: 800g not yet supported due to TH4-100G restrictions
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, list( range( 8 ) ) ),
         ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G: Main6Taps3Pre,
               SPEED_25G: Main6Taps3Pre,
               SPEED_50G: Main6Taps3Pre,
               SPEED_100G: Main6Taps3Pre }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # CL28
         # BUG:738023 Osprey does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
      ]

class Falcon16( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], HG_G106_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], HG_G53_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], HG_G53_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_FCFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_FCFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 2, 3 ] ),
         MappingDesc( None, [], HG_G42_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], HG_G27_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], HG_G27_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], HG_G27_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], HG_G27_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 0 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 1 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 2 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 3 ] ),
         MappingDesc( None, [], HG_G21_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], HG_G21_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( HG_G106_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G100_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G100_4_RS528, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( HG_G53_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( HG_G53_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_FCFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_FCFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 2, 3 ], None, [] ),
         MappingDesc( HG_G42_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G40_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( HG_G27_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( HG_G27_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( HG_G27_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( HG_G27_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 0 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 1 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 2 ], None, [] ),
         MappingDesc( G25_1_FCFEC, [ 3 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G25_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 0 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 1 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 2 ], None, [] ),
         MappingDesc( G25_1_RS528, [ 3 ], None, [] ),
         MappingDesc( HG_G21_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( HG_G21_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main5Taps3Post,
               SPEED_25G: Main5Taps3Post,
               SPEED_HG_11G: Main5Taps3Post,
               SPEED_HG_27G: Main5Taps3Post, }

   def getSupportedAutoneg( self, coreId ):
      return [
         # Forced
         AutonegDesc( SPEED_GBPS_100, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_DISABLED ),
         # CL73
         AutonegDesc( SPEED_GBPS_100, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_CL73, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # BUG:738023 Falcon does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_2p5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class Falcon16LowSpeed( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G40_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main5Taps3Post }

   def getSupportedAutoneg( self, coreId ):
      return [
         # Forced
         AutonegDesc( SPEED_GBPS_40, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_DISABLED ),
         # CL73
         AutonegDesc( SPEED_GBPS_40, AUTONEG_CL73, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         # BUG:738023 Falcon does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_2p5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class Falcon16Gen3( Falcon16 ):
   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 2, 3 ] ),
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 0 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 1 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 2 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G100_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G100_4_RS528, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 0, 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
      ]

class Talon( AsicCore ):
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G5_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G2P5_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in range( 0, 8, 4 ):
         mappings.extend( [
            MappingDesc( None, [], G200_4_RS544, list( range( x, x + 4 ) ) ),
            MappingDesc( None, [], G100_4_RS528, list( range( x, x + 4 ) ) ),
            MappingDesc( None, [], G100_4_NOFEC, list( range( x, x + 4 ) ) ),
            MappingDesc( None, [], G40_4_NOFEC, list( range( x, x + 4 ) ) ),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, list( range( 8 ) ) )
      ] )
      return mappings

   def getTapGroups( self ):
      # TODO: BUG960718
      return {}

   def getSupportedAutoneg( self, coreId ):
      # TODO: BUG960718
      return []

class TalonNo50G( Talon ):
   # TODO: BUG960718
   # Add more functions to this class if necessary
   """
   A Talon core which only supports up to 25G instead of 50G,
   as seen on Q3U/BCM88400
   """
 
   name = 'Talon'

   def getLogicalSerdesMapping( self ):
      mappings = super().getLogicalSerdesMapping()
      # All SerdesGroupModes that use FEC_RS544 as FEC encoding are either 50G
      # or 100G per serdes lane
      return [ mapping for mapping in mappings
               if mapping.lineMode.fec != FEC_RS544 ]

class GPhy( AsicCore ):
   def getNumSerdes( self ):
      return 4

   # GPhys actually can't perform lane remaps, and it doesn't make much sense in this
   # context anyway given there are 4 serdes per actual 1 serdes modeled here since
   # each serdes is actually a BASE-T interface's set of 4 diff pairs.
   def laneRemapCapable( self ):
      return False

   def polarityRemapCapable( self, phyScope ):
      return False

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC_HALF, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC_HALF, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC_HALF, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC_HALF, [ 3 ] ),
         MappingDesc( None, [], M10_1_NOFEC_HALF, [ 0 ] ),
         MappingDesc( None, [], M10_1_NOFEC_HALF, [ 1 ] ),
         MappingDesc( None, [], M10_1_NOFEC_HALF, [ 2 ] ),
         MappingDesc( None, [], M10_1_NOFEC_HALF, [ 3 ] ),
      ]

   def getTapGroups( self ):
      # We can't tune BASE-T serdes.
      return { SPEED_10G: NoTuning }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL28
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_MBPS_100, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_MBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_10, AUTONEG_CL28 ),
      ]

class Merlin( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], HG_G21_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], HG_G21_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], HG_G11_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( HG_G21_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( HG_G21_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( HG_G11_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main4Taps,
               SPEED_HG_11G: Main4Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL37, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
         # CL28
         # BUG:738023 Merlin does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_MBPS_100, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

   def getPllCount( self ):
      return 1

   def getLineRateToPllRateList( self ):
      return { SPEED_GBPS_10: [ COMPAT_GBPS_10 ] }

class MerlinG2( AsicCore ):
   '''Merlin G2 cores differ from normal merlin cores in that they support dynamic
   port allocation, native 5G-BASER, and native 2.5G-BASEX.

   TODO: Broadcom has recommended that we use native rates instead of idle stuffing
         if possible. As such we have ommited declaration of idle stuffed rates in
         these models ( CS00012345085 ).
   '''

   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G5_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G2P5_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 3 ], None, [] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main4Taps,
               SPEED_HG_11G: Main4Taps }

class MerlinMgmt( AsicCore ):
   # The rest of the system should not differentiate between this and non-management
   # Merlin cores.
   name = "Merlin"

   # The Merlin cores used for management are not capable of performing lane swaps.
   # This capability isn't required anyway as it doesn't support multilane protocols
   # that need to know the lane ordering.
   def laneRemapCapable( self ):
      return False

   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 1 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 3 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main4Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL37, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
         # CL28
         # BUG:738023 Merlin does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_MBPS_100, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

   def getPllCount( self ):
      return 1

   def getLineRateToPllRateList( self ):
      return { SPEED_GBPS_10: [ COMPAT_GBPS_10 ] }

# TODO: See BUG649800; This type is actually modeling a core with the full 4 serdes
#       available, but the core only has 1 port that it can use. We should
#       eventually migrate this core to actually model the modes on the full 4
#       serdes, but for now we only model the IEEE modes on the lane in use.
class MerlinMgmtOnePort( MerlinMgmt ):
   # We override the usual constructor for the core, and instead make the returned
   # object callable with the same args as the parent core type.
   # This lets us use the instance object itself as though it was a Core class.
   def __init__( self, laneId ):
      self.laneId = int( laneId )

   def __call__( self, mode=None ):
      super().__init__( mode=mode )
      return self

   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G10_1_NOFEC, [ self.laneId ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ self.laneId ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ self.laneId ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ self.laneId ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ self.laneId ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ self.laneId ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ self.laneId ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ self.laneId ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ self.laneId ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ self.laneId ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ self.laneId ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ self.laneId ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

# TODO: See BUG649800; This type is actually modeling a core with the full 4 serdes
#       available, but the core only has 2 ports that it can use. We should
#       eventually migrate this core to actually model the modes on the full 4
#       serdes, but for now we only model the IEEE modes on the 0 and 2 lanes.
class MerlinMgmtTwoPort( MerlinMgmt ):
   def getLogicalSerdesMapping( self ):
      return [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 0 ], None, [] ),
         MappingDesc( M100_1_NOFEC, [ 2 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( G1_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

# MerlinCore-Q supports the same serdes rates as a normal merlin core, but can
# also break each serdes up into 4 physical ports each running at 2.5G to support
# multi-gig ports on the front-panel. We are handling this breakup at the external
# phy instead, however, as it makes the traversal easier to reason about from a
# hardware perspective (still only 1 trace between the chips, just time division
# duplexed). Thus those external phys will declare the serdesMapping between the
# 5G/10G serdes rate from the asic to the data/serdes rates on the FPP
class MerlinQ( AsicCore ):
   def getNumSerdes( self ):
      return 4

   # TODO: See BUG458080; also specifying supported data rates for now.
   # Return the serdes rates used for USXGMII and QSGMII when running in Q mode,
   # plus the usual merlin core mappings.
   def getLogicalSerdesMapping( self ):
      channelSpeeds10G = [ SPEED_GBPS_2p5, SPEED_GBPS_1,
                           SPEED_MBPS_100, SPEED_MBPS_10 ]
      channelSpeeds5G = [ SPEED_GBPS_1, SPEED_MBPS_100, SPEED_MBPS_10 ]
      lineChannels10G = []
      lineChannels5G = []

      for channelId in range( 4 ):
         for channelSpeed in channelSpeeds10G:
            lineChannels10G.append( ( channelId, channelSpeed, DUPLEX_FULL ) )
            if channelSpeed in [ SPEED_MBPS_100, SPEED_MBPS_10 ]:
               lineChannels10G.append( ( channelId, channelSpeed, DUPLEX_HALF ) )
      for channelId in range( 4 ):
         for channelSpeed in channelSpeeds5G:
            lineChannels5G.append( ( channelId, channelSpeed, DUPLEX_FULL ) )
            if channelSpeed in [ SPEED_MBPS_100, SPEED_MBPS_10 ]:
               lineChannels5G.append( ( channelId, channelSpeed, DUPLEX_HALF ) )

      return [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ], lineChannels=lineChannels10G ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ], lineChannels=lineChannels10G ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ], lineChannels=lineChannels10G ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ], lineChannels=lineChannels10G ),
         MappingDesc( None, [], G5_1_NOFEC, [ 0 ], lineChannels=lineChannels5G ),
         MappingDesc( None, [], G5_1_NOFEC, [ 1 ], lineChannels=lineChannels5G ),
         MappingDesc( None, [], G5_1_NOFEC, [ 2 ], lineChannels=lineChannels5G ),
         MappingDesc( None, [], G5_1_NOFEC, [ 3 ], lineChannels=lineChannels5G ),
         # TODO: jackemb, figure out how to handle Q vs non-Q mode for these phys
         #       Until then, we just explicitely add the non-Q mode capabilities in
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main4Taps,
               SPEED_HG_11G: Main4Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL37
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL37, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
         # CL28
         # BUG:738023 MerlinQ does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_MBPS_100, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
      ]

class D5( AsicCore ):
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      # NOTE: TH5 datasheet indicates 100G/50G support, but not 40G support
      return [
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 2, 3 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_FCFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G25_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 0 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 1 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 2 ] ),
         MappingDesc( None, [], G25_1_RS528, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G2P5_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 2 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 3 ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 2 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 3 ],
                      lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G100_4_NOFEC, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G100_4_RS528, [ 0, 1, 2, 3 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 0, 1 ], None, [] ),
         MappingDesc( G50_2_NOFEC, [ 2, 3 ], None, [] ),
         MappingDesc( G50_2_RS528, [ 0, 1 ], None, [] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 0 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 1 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 2 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
         MappingDesc( G10_1_NOFEC, [ 3 ], None, [],
                      sysChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
      ]

   def getTapGroups( self ):
      return { SPEED_10G: Main5Taps3Post,
               SPEED_25G: Main5Taps3Post, }

   def getSupportedAutoneg( self, coreId ):
      return [
         # Forced
         AutonegDesc( SPEED_GBPS_100, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_10, AUTONEG_DISABLED, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_DISABLED, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_DISABLED ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_DISABLED ),
         # CL73
         AutonegDesc( SPEED_GBPS_100, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_CL73, None, None ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_CL73, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # BUG:738023 D5 does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( SPEED_GBPS_10, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_2p5, AUTONEG_CL28, None, None ),
         AutonegDesc( SPEED_GBPS_1, AUTONEG_CL28, None, None ),
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

   def getPllCount( self ):
      return 2

   def getLineRateToPllRateList( self ):
      return { SPEED_GBPS_25: [ COMPAT_GBPS_25 ],
               SPEED_GBPS_10: [ COMPAT_GBPS_10 ] }

# On platforms like Quicksilver (Kona mgmt board) we are only using 2 of the 4 serdes
# provided (0 and 2). This is modeled similar to MerlinMgmtTwoPort.
class D5MgmtTwoPort( D5 ):
   # The rest of the system should not differentiate between this and non-management
   # D5 cores.
   name = "D5"

   def laneRemapCapable( self ):
      return False

   def getLogicalSerdesMapping( self ):
      # Remove mappings that use lane 1/3
      unusedLanes = { 1, 3 }
      return [ m for m in super().getLogicalSerdesMapping()
               if not unusedLanes.intersection( m.sysSerdes + m.lineSerdes ) ]

   def getSupportedAutoneg( self, coreId ):
      # Remove multilane speeds (100G/50G)
      unusedSpeeds = [ SPEED_GBPS_100, SPEED_GBPS_50 ]
      return [ a for a in super().getSupportedAutoneg( coreId )
               if ( a.systemSideSpeed not in unusedSpeeds and
                    a.lineSideSpeed not in unusedSpeeds ) ]

class BabbageCore( Core ):
   #               +-------------------+
   #               |      Babbage      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             0| |----  0
   #       1   ----| |1             1| |----  1
   #       2   ----| |2             2| |----  2
   #       3   ----| |3             3| |----  3
   #       4   ----| |4   Core 1    4| |----  4
   #       5   ----| |5             5| |----  5
   #       6   ----| |6             6| |----  6
   #       7   ----| |7             7| |----  7
   #               | +---------------+ |
   #               | +---------------+ |
   #       8   ----| |8             8| |----  8
   #       9   ----| |9             9| |----  9
   #       10  ----| |10           10| |----  10
   #       11  ----| |11           11| |----  11
   #       12  ----| |12  Core 2   12| |----  12
   #       13  ----| |13           13| |----  13
   #       14  ----| |14           14| |----  14
   #       15  ----| |15           15| |----  15
   #               | +---------------+ |
   #               +-------------------+
   name = "Babbage" # So getComponentType() uses "Babbage" rather than "BabbageCore".
   supportedModes = { "Gearbox", "CrossGearbox", "Retimer" }

   def getNumSerdes( self ):
      return 8

   def laneRemapCapable( self ):
      return False

   def polarityRemapCapable( self, phyScope ):
      return phyScope == PHY_SCOPE_LINE

   def getPhysicalMapping( self ):
      # The format for the mapping is a dictionary of tuples mapping
      # the system serdes to the line serdes. In most cases there will
      # be only one system serdes
      # { ( system serdes, ... ) : ( line serdes, ... ), ... }
      mapping = None
      if self.mode in [ "Gearbox", "CrossGearbox" ]:
         # Map will be:
         # A0 : B0, B1
         # A1 : B2, B3
         # A2 : B4, B5
         # A3 : B6, B7
         mapping = { ( 0, ) : ( 0, 1 ),
                     ( 1, ) : ( 2, 3 ),
                     ( 2, ) : ( 4, 5 ),
                     ( 3, ) : ( 6, 7 ) }
      elif self.mode == "Retimer":
         # Map will be:
         # A[ 0-7 ]: B[ 0-7 ]
         mapping = { ( x, ) : ( x, ) for x in range( 8 ) }
      else:
         # pylint: disable-next=consider-using-f-string
         assert False, "invalid mode %s" % self.mode
      return mapping

   def getLogicalSerdesMapping( self ):
      mappings = []
      if self.mode == "Gearbox":
         # four lane retimer modes
         serdes = list( range( 0, 4 ) )
         mappings.extend( [
            MappingDesc( G200_4_RS544, serdes, G200_4_RS544, serdes ),
         ] )
         # four lane gearbox modes
         for ( sysSerdes, lineSerdes ) in (
            ( ( 0, 1 ), list( range( 0, 4 ) ) ),
            ( ( 2, 3 ), list( range( 4, 8 ) ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysSerdes, G100_4_RS528, lineSerdes ),
               MappingDesc( G100_2_RS544, sysSerdes, G100_4_NOFEC, lineSerdes ),
               MappingDesc( G40_2_NOFEC, sysSerdes, G40_4_NOFEC, lineSerdes ),
            ] )
         # two lane retimer modes
         for lineSerdes in [ ( 0, 1 ), ( 2, 3 ) ]:
            sysSerdes = lineSerdes
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysSerdes, G100_2_RS544, lineSerdes ),
            ] )
         # two lane gearbox modes
         for sysSerdes in range( 0, 4 ):
            lineSerdes = [ sysSerdes * 2, sysSerdes * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_RS528, lineSerdes ),
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_FCFEC, lineSerdes ),
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_NOFEC, lineSerdes ),
            ] )
         # single lane retimer modes
         for x in range( 8 ):
            sysSerdes = x if x < 4 else x - 4
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysSerdes ] , G50_1_RS544, [ x ] ),
               MappingDesc( G25_1_RS528, [ sysSerdes ] , G25_1_RS528, [ x ] ),
               MappingDesc( G25_1_FCFEC, [ sysSerdes ] , G25_1_FCFEC, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ sysSerdes ] , G25_1_NOFEC, [ x ] ),
               MappingDesc( G10_1_NOFEC, [ sysSerdes ] , G10_1_NOFEC, [ x ] ),
            ] )
      elif self.mode == "CrossGearbox":
         # On a cross gearbox babbage the single lane modes start from the second set
         # of 4 lanes on the line side and go to the first set of 4 lanes on the
         # system side

         # four lane retimer modes
         mappings.extend( [
            MappingDesc( G200_4_RS544, list( range( 0, 4 ) ),
                         G200_4_RS544, list( range( 4, 8 ) ) ),
         ] )
         # four lane gearbox modes
         for ( sysSerdes, lineSerdes ) in (
            ( ( 0, 1 ), list( range( 0, 4 ) ) ),
            ( ( 2, 3 ), list( range( 4, 8 ) ) ), ):
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysSerdes, G100_4_RS528, lineSerdes ),
               MappingDesc( G100_2_RS544, sysSerdes, G100_4_NOFEC, lineSerdes ),
               MappingDesc( G40_2_NOFEC, sysSerdes, G40_4_NOFEC, lineSerdes ),
            ] )
         # two lane retimer modes
         for lineSerdes in [ ( 4, 5 ), ( 6, 7 ) ]:
            sysSerdes = [ x % 4 for x in lineSerdes ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, sysSerdes, G100_2_RS544, lineSerdes ),
            ] )
         # two lane gearbox modes
         for sysSerdes in range( 0, 4 ):
            lineSerdes = [ sysSerdes * 2, sysSerdes * 2 + 1 ]
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_RS528, lineSerdes ),
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_FCFEC, lineSerdes ),
               MappingDesc( G50_1_RS544, [ sysSerdes ], G50_2_NOFEC, lineSerdes ),
            ] )
         # single lane retimer modes
         for x in range( 4 ):
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ] , G50_1_RS544, [ x + 4 ] ),
               MappingDesc( G25_1_RS528, [ x ] , G25_1_RS528, [ x + 4 ] ),
               MappingDesc( G25_1_FCFEC, [ x ] , G25_1_FCFEC, [ x + 4 ] ),
               MappingDesc( G25_1_NOFEC, [ x ] , G25_1_NOFEC, [ x + 4 ] ),
               MappingDesc( G10_1_NOFEC, [ x ] , G10_1_NOFEC, [ x + 4 ] ),
            ] )
      elif self.mode == "Retimer":
         # eight lane modes
         mappings.extend( [
            MappingDesc( G400_8_RS544, list( range( 8 ) ),
                         G400_8_RS544, list( range( 8 ) ) ),
         ] )
         # four lane modes
         for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ):
            mappings.extend( [
               MappingDesc( G200_4_RS544, x, G200_4_RS544, x ),
               MappingDesc( G100_4_RS528, x, G100_4_RS528, x ),
               MappingDesc( G100_4_NOFEC, x, G100_4_NOFEC, x ),
               MappingDesc( G40_4_NOFEC, x, G40_4_NOFEC, x ),
            ] )
         # two lane modes
         for x in range( 0, 8, 2 ):
            serdes = [ x, x + 1 ]
            mappings.extend( [
               MappingDesc( G100_2_RS544, serdes, G100_2_RS544, serdes ),
               MappingDesc( G50_2_RS528, serdes, G50_2_RS528, serdes ),
               MappingDesc( G50_2_FCFEC, serdes, G50_2_FCFEC, serdes ),
               MappingDesc( G50_2_NOFEC, serdes, G50_2_NOFEC, serdes ),
            ] )
         # single lane modes
         for x in range( 8 ):
            mappings.extend( [
               MappingDesc( G50_1_RS544, [ x ] , G50_1_RS544, [ x ] ),
               MappingDesc( G25_1_RS528, [ x ] , G25_1_RS528, [ x ] ),
               MappingDesc( G25_1_FCFEC, [ x ] , G25_1_FCFEC, [ x ] ),
               MappingDesc( G25_1_NOFEC, [ x ] , G25_1_NOFEC, [ x ] ),
               MappingDesc( G10_1_NOFEC, [ x ] , G10_1_NOFEC, [ x ] ),
            ] )
      else:
         # pylint: disable-next=consider-using-f-string
         assert False, "invalid mode %s" % self.mode
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G: Main6Taps, SPEED_20G: Main6Taps,
               SPEED_25G: Main6Taps, SPEED_50G: Main6Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( SPEED_GBPS_400, AUTONEG_DISABLED,
                      SPEED_GBPS_400, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_200, AUTONEG_DISABLED,
                      SPEED_GBPS_200, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_100, AUTONEG_DISABLED,
                      SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_50, AUTONEG_DISABLED,
                      SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_40, AUTONEG_DISABLED,
                      SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( SPEED_GBPS_25, AUTONEG_DISABLED,
                      SPEED_GBPS_25, AUTONEG_CL73 ),
      ]

class BabbageLPCore( BabbageCore ):

   # So getComponentType() uses "BabbageLP" rather than "BabbageLPCore".
   name = "BabbageLP"

class SparrowCore( Core ):
   #               +-------------------+
   #               |      Sparrow      |
   #     serdes    | +---------------+ |    serdes
   #       0   ----| |0             2| |----  2
   #       1   ----| |1             3| |----  3
   #               | |    Core 0    4| |----  4
   #               | |              5| |----  5
   #               | +---------------+ |
   #               | +---------------+ |
   #       6   ----| |0             2| |----  8
   #       7   ----| |1             3| |----  9
   #               | |    Core 1    4| |----  10
   #               | |              5| |----  11
   #               | +---------------+ |
   #               | +---------------+ |
   #       12  ----| |0             2| |----  14
   #       13  ----| |1             3| |----  15
   #               | |    Core 2    4| |----  16
   #               | |              5| |----  17
   #               | +---------------+ |
   #               | +---------------+ |
   #       18  ----| |0             2| |----  20
   #       19  ----| |1             3| |----  21
   #               | |    Core 3    4| |----  22
   #               | |              5| |----  23
   #               | +---------------+ |
   #               +-------------------+
   # Serdes numbering is contiguous within every core, to match how they are
   # addressed in software.
   name = "Sparrow"
   supportedModes = { "FourWayGearbox" }

   def getNumSerdes( self ):
      # Return total number of serdes (not just line serdes), so that the serdes IDs
      # (which are contiguous within a core) are incremented correctly for all cores.
      return 6

   def laneRemapCapable( self ):
      return False

   def polarityRemapCapable( self, phyScope ):
      return True

   def getPhysicalMapping( self ):
      # The format for the mapping is a dictionary of tuples mapping
      # the system serdes to the line serdes.
      # { ( system serdes, ... ) : ( line serdes, ... ), ... }
      if self.mode == "FourWayGearbox":
         # The system side lane 0 is not used in this mapping.
         return { ( 1, ) : ( 2, 3, 4, 5 ) }
      else:
         assert False, f"Invalid mode {self.mode}"

   def getLogicalSerdesMapping( self ):
      mappings = []
      if self.mode == "FourWayGearbox":
         # 1 system lane -> 4 line lanes gearbox mode
         sysSerdes = [ 1 ]
         lineSerdes = [ 2, 3, 4, 5 ]
         mappings.extend( [
            MappingDesc( G100_1_RS544, sysSerdes, G100_4_RS528, lineSerdes ),
            MappingDesc( G100_1_RS544, sysSerdes, G100_4_NOFEC, lineSerdes ),
         ] )
         # TODO BUG929224 Add bitmux mappings
      else:
         assert False, f"Invalid mode {self.mode}"
      return mappings

   def getTapGroups( self ):
      return { SPEED_25G: Main6Taps, SPEED_50G: Main6Taps, SPEED_100G: Main6Taps }

   def getSupportedAutoneg( self, coreId ):
      # TODO BUG929203 - Sparrow does not support autoneg just yet.
      return []

class TofinoCore( AsicCore ):
   # Tofino has 64 serdes groups, each serdes group
   # contains 4 serdes lanes capable of 25G NRZ.
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 4, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_FCFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      mappings.extend( [
         MappingDesc( None, [], G100_4_RS528, list( range( 4 ) ) ),
         MappingDesc( None, [], G100_4_NOFEC, list( range( 4 ) ) ),
         MappingDesc( None, [], G40_4_FCFEC, list( range( 4 ) ) ),
         MappingDesc( None, [], G40_4_NOFEC, list( range( 4 ) ) ),
      ] )
      return mappings

   def getTapGroups( self ):
      # Tofino uses 3 taps, but define 6 taps here to keep L1 happy,
      # as the old parser expects 6 taps. See BUG651106
      return { SPEED_10G: Main6Taps, SPEED_25G: Main6Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # CL28
         # BUG:738023 Tofino does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class TofinoCoreAuxLane( AsicCore ):
   # This for the last serdes group for Tofino, with 4 lanes and NRZ only
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      mappings = []
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 4, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_FCFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      mappings.extend( [
         MappingDesc( None, [], G100_4_RS528, list( range( 4 ) ) ),
         MappingDesc( None, [], G100_4_NOFEC, list( range( 4 ) ) ),
         MappingDesc( None, [], G40_4_FCFEC, list( range( 4 ) ) ),
         MappingDesc( None, [], G40_4_NOFEC, list( range( 4 ) ) ),
      ] )
      return mappings

   # BUG528916 Cannot do lane remap as this core also connect to 2 cpu ports
   def laneRemapCapable( self ):
      return False

   # BUG528916 Cannot do polarity remap as this core also connect to 2 cpu ports
   def polarityRemapCapable( self, phyScope ):
      return False

   def getTapGroups( self ):
      # Tofino uses 3 taps, but define 6 taps here to keep L1 happy,
      # as the old parser expects 6 taps. See BUG651106
      return { SPEED_10G: Main6Taps, SPEED_25G: Main6Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_40, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # CL28
         # BUG:738023 Tofino does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class Tofino2Core( AsicCore ):
   # Tofino2 is the chip controlling multiple serdes groups.
   # Here is the serdes group contains 8 serdes lanes capable for PAM4 and NRZ.
   # Although in HW perspective there is no concept as cores, we still define
   # one serdes group as "Tofino2Core", based on the two assumptions:
   # 1. We will never use serdes cross two serdes groups together
   # 2. We can not lane swap serdes between two serdes groups
   def getNumSerdes( self ):
      return 8

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 8 ):
         mappings.extend( [
            MappingDesc( None, [], G50_1_RS544, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] )
      # two lane modes
      for x in range( 0, 8, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G100_2_RS544, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_FCFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G40_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      for x in ( list( range( 0, 4 ) ), list( range( 4, 8 ) ) ): 
         mappings.extend( [
            MappingDesc( None, [], G200_4_RS544, x ),
            MappingDesc( None, [], G100_4_RS528, x ),
            MappingDesc( None, [], G100_4_NOFEC, x ),
            MappingDesc( None, [], G40_4_NOFEC, x ),
         ] )
      # eight lane modes
      mappings.extend( [
         MappingDesc( None, [], G400_8_RS544, list( range( 8 ) ) ),
         ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G: Main5Taps, SPEED_20G: Main5Taps,
               SPEED_25G: Main5Taps, SPEED_50G: Main5Taps }

   def getSupportedAutoneg( self, coreId ):
      # Tofino2 don't support autoneg
      return [] 

class Tofino2Core4Lane( AsicCore ):
   # This for the first serdes group for Tofino2 chip, with 4 lanes and NRZ only
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      mappings = []
      for x in range( 4 ):
         # The trace length on woodleaf SFP ports has high db loss, so we drop 25G
         # support. If futher platform requires 25G, we need to find out how to 
         # describe it.
         mappings.extend( [
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ x ] ),
         ] )
      return mappings

   # BUG528916 Cannot do lane remap as this core also connect to 2 cpu ports
   def laneRemapCapable( self ):
      return False

   # BUG528916 Cannot do polarity remap as this core also connect to 2 cpu ports
   def polarityRemapCapable( self, phyScope ):
      return False

   def getTapGroups( self ):
      return { SPEED_10G: Main5Taps }

   def getSupportedAutoneg( self, coreId ):
      # Tofino2 don't support autoneg
      return []

class IntelSoCCore( AsicCore ):
   # Intel SoC which integrated a ethernet multirate PHY with 4 lanes nrz
   def getNumSerdes( self ):
      return 4

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ x ] ),
            # BUG720921 on caravan asic phy must run autoneg on 100m-sgmii
            # where external phy only support autoneg disable. This made us
            # dropped 100M support
            #MappingDesc( None, [], M100_1_NOFEC, [ x ] ),
            #MappingDesc( None, [], G1_1_NOFEC, [ x ],
            #             lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ],
                         lineChannels=[ ( 0, SPEED_GBPS_2p5, DUPLEX_FULL ) ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ],
                         lineChannels=[ ( 0, SPEED_GBPS_5, DUPLEX_FULL ) ] ),
         ] ) 
      mappings.extend( [
         # two lane modes
         MappingDesc( None, [], G50_2_NOFEC, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_NOFEC, [ 2, 3 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 0, 1 ] ),
         MappingDesc( None, [], G50_2_RS528, [ 2, 3 ] ),
         # four lane modes
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ), 
      ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_25G: Main3Taps, SPEED_10G: Main3Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
         # BUG:738023 IntelSoc does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_2p5, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class FortvilleCore( AsicCore ):
   # Intel Fortville Family (XL710-AM1, XL710-BM1)
   # Although inside spec there are 8 physical lanes, the above HWs only support
   # single QSFP or QUAD sfp+ and the physical lanes mapping are fixed.
   # As we don't need to access physical lanes and it is not configurable,
   # just assuming it is four lanes core. 
   def getNumSerdes( self ):
      return 4

   def laneRemapCapable( self ):
      return False

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G1_1_NOFEC, [ x ] ),
         ] ) 
      mappings.extend( [
         # four lane modes
         MappingDesc( None, [], G40_4_NOFEC, [ 0, 1, 2, 3 ] ), 
      ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G: Main3Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL37 ),
         # BUG:738023 FortvilleCore does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
      ]

class ColumbiavilleCore( AsicCore ):
   # Intel columbiaville Family (E810-XXV, E810-CQDA1,...)
   # Although inside spec there are 8 physical lanes, the above HWs only support
   # single QSFP or DUAL/QUAD sfp+ and the physical lanes mapping are fixed.
   # As we don't need to access physical lanes and it is not configurable,
   # just assuming it is four lanes core. 
   def getNumSerdes( self ):
      return 4

   def laneRemapCapable( self ):
      return False

   def getLogicalSerdesMapping( self ):
      mappings = []
 
      # single lane modes
      for x in range( 4 ):
         mappings.extend( [
            MappingDesc( None, [], G25_1_FCFEC, [ x ] ),
            MappingDesc( None, [], G25_1_NOFEC, [ x ] ),
            MappingDesc( None, [], G25_1_RS528, [ x ] ),
            MappingDesc( None, [], G10_1_NOFEC, [ x ] ),
         ] ) 
      # two lane modes
      for x in range( 0, 4, 2 ):
         mappings.extend( [
            MappingDesc( None, [], G50_2_RS528, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_FCFEC, [ x, x + 1 ] ),
            MappingDesc( None, [], G50_2_NOFEC, [ x, x + 1 ] ),
         ] )
      # four lane modes
      mappings.extend( [
         MappingDesc( None, [], G100_4_RS528, [ 0, 1, 2, 3 ] ),
         MappingDesc( None, [], G100_4_NOFEC, [ 0, 1, 2, 3 ] ),
      ] )
      return mappings

   def getTapGroups( self ):
      return { SPEED_10G : Main3Taps, SPEED_25G : Main3Taps }

   def getSupportedAutoneg( self, coreId ):
      return [
         # CL73
         AutonegDesc( None, None, SPEED_GBPS_100, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_50, AUTONEG_CL73 ),
         AutonegDesc( None, None, SPEED_GBPS_25, AUTONEG_CL73 ),
         # BUG:738023 ColumbiavilleCore does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_10, AUTONEG_CL28 ),
      ]

class IntelX553( AsicCore ):
   # Dual lanes 10GbE Mac/Phy controller
   def getNumSerdes( self ):
      return 2

   def laneRemapCapable( self ):
      return False

   def polarityRemapCapable( self, phyScope ):
      return False

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      mappings.extend( [
         MappingDesc( None, [], G10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ],
                      lineChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G10_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_MBPS_100, DUPLEX_FULL ) ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 1 ],
                      lineChannels=[ ( 0, SPEED_MBPS_10, DUPLEX_FULL ) ] ),
         ] ) 
      return mappings

   def getSupportedAutoneg( self, coreId ):
      return [
         # BUG942257: X553 don't support AN37
         # BUG:738023 IntelSoc does not support clause 28 but this core has
         # BASE-T Xcvr adaptors and we need the capability declared here
         # for traversals until Xcvr adaptor support.
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_10, AUTONEG_CL28 ),
      ]

   def getTapGroups( self ):
      # No customized tuning support 
      return { SPEED_10G: NoTuning }

class FoxvilleCore( AsicCore ):
   # A single lane 2.5G BASE-T Mac/Phy controller
   def getNumSerdes( self ):
      return 1

   def getLogicalSerdesMapping( self ):
      mappings = []
      # single lane modes
      mappings.extend( [
         MappingDesc( None, [], G2P5_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], G1_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M100_1_NOFEC_HALF, [ 0 ] ),
         MappingDesc( None, [], M10_1_NOFEC, [ 0 ] ),
         MappingDesc( None, [], M10_1_NOFEC_HALF, [ 0 ] ),
         ] ) 
      return mappings

   def getSupportedAutoneg( self, coreId ):
      return [
         AutonegDesc( None, None, SPEED_GBPS_1, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_100, AUTONEG_CL28 ),
         AutonegDesc( None, None, SPEED_MBPS_10, AUTONEG_CL28 ),
      ]

   def getTapGroups( self ):
      # We can't tune BASE-T serdes.
      return { SPEED_10G: NoTuning }
