# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines HW L1 Topology Component Lib related errors. These should be
the exceptions that are raised whenever an error in the lib or in processing
components is detected.

Note:
   All errors defined in this file should inherit from HwL1ComponentLibraryError.
'''

import ArPyUtils.Types

class HwL1ComponentLibraryInternalError( ArPyUtils.Types.ArException ):
   '''Denotes an internal error in the L1 Topology Component Library.'''
   def __init__( self, message, **kwargs ):
      # pylint: disable-next=consider-using-f-string
      message = ( 'Internal Library Error: {}\n'
                  'Please contract the library maintainers!\n' ).format( message )
      super().__init__( message, **kwargs )

class HwL1ComponentLibraryError( ArPyUtils.Types.ArException ):
   '''Denotes an error in the L1 Topology Component Library whose cause can be traced
   to user input errors.
   '''
   def __init__( self, message, **kwargs ):
      # pylint: disable-next=consider-using-f-string
      message = ( '{}\nThis is most likely a result of a '
                  'logical error in the definition of the L1 Topology component. '
                  'Please check double check the FDL arguments with L1 Topology '
                  'component library definitions.\n' ).format( message )
      super().__init__( message, **kwargs )

class HwL1ComponentError( HwL1ComponentLibraryError ):
   '''Denotes a logical error in the definition or processing of components from the
   L1 Topology Component Library.
   '''
   def __init__( self, component, message, **kwargs ):
      message = f'Error in component {component} - {message}'
      super().__init__( message, **kwargs )

class HwL1ComponentCoreError( HwL1ComponentError ):
   '''Denotes a logical error in the definition or processing of a specific core from
   a component.
   '''
   def __init__( self, component, coreId, message, **kwargs ):
      message = f'core {coreId} - {message}'
      super().__init__( component, message, **kwargs )

class SerdesLogicalMappingConsistencyError( HwL1ComponentCoreError ):
   '''Denotes an error with the definition of the logical SerDes mappings for a
   component. e.g. The number of lanes and the SerDes group mode not matching up.
   '''
   def __init__( self, component, coreId, side, **kwargs ):
      # pylint: disable-next=consider-using-f-string
      message = ( 'The {} side of a SerDes mapping is not '
                  'consistent.'.format( side ) )
      super().__init__( component,
                        coreId,
                        message,
                        **kwargs )
