# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
This module defines the interface slot type library.


All interface slot types defined in this library must be exposed to the FDL using
their standardized names. For example, a 8 lane interface slot with a per lane SerDes
rate of 50Gbps would exposed as both QSFPDD and OSFP.

TODO:
   BUG763452: Presently, there is no concrete standard for either the interface slot
              enum value or the name of the class. The only restriction is that they
              are unique.
"""


from abc import ABCMeta, abstractproperty

from TypeFuture import TacLazyType

from .Errors import (
   HwL1ComponentError,
   HwL1ComponentLibraryError,
)

from L1Topology.Modes import (
   G800_8_RS544,
   G400_4_RS544,
   G400_8_RS544,
   G200_2_RS544,
   G200_4_RS544,
   G100_1_RS544,
   G100_2_RS544,
   G100_4_RS528,
   G100_4_NOFEC,
   G50_1_RS544,
   G50_2_RS528,
   G50_2_FCFEC,
   G50_2_NOFEC,
   G40_4_NOFEC,
   G25_1_RS528,
   G25_1_FCFEC,
   G25_1_NOFEC,
   HG_G21_2_NOFEC,
   G10_1_NOFEC,
   G5_1_NOFEC,
   G2P5_1_NOFEC,
   G1_1_NOFEC,
   M100_1_NOFEC,
   M100_1_NOFEC_HALF,
   M10_1_NOFEC,
   M10_1_NOFEC_HALF
)

InterfaceIntfSlotTypes = TacLazyType( "Interface::IntfSlotTypes::IntfSlotTypes" )
InterfaceRoot = TacLazyType( "Hardware::L1Topology::InterfaceRoot" )
InterfaceSlotLaneGroupMode = TacLazyType(
   "Hardware::L1Topology::InterfaceSlotLaneGroupMode" )

_rj45 = InterfaceIntfSlotTypes.rj45
_single1 = InterfaceIntfSlotTypes.single1
_single10 = InterfaceIntfSlotTypes.single10
_dual100 = InterfaceIntfSlotTypes.dual100
_single25 = InterfaceIntfSlotTypes.single25
_single50 = InterfaceIntfSlotTypes.single50
_quad40 = InterfaceIntfSlotTypes.quad40
_quad100 = InterfaceIntfSlotTypes.quad100
_quad200 = InterfaceIntfSlotTypes.quad200
_quad400 = InterfaceIntfSlotTypes.quad400
_octal400 = InterfaceIntfSlotTypes.octal400
_octal800 = InterfaceIntfSlotTypes.octal800

# Special IntfSlot Types
_agileQuad40 = InterfaceIntfSlotTypes.agileQuad40

class IntfSlotType( metaclass=ABCMeta ):
   """
   A base class for all interface slot types.

   Below is an example of a QSFP interface slot in L1Topology and the
   information this library provides.

   +-------------------------------------+
   |         Interface slot QSFP40       |
   +-------------------------------------+
   |                                     |
   | +---------------------------------+ |
   | |        interface root  0        | |
   | +---------------------------------+ |
   | | M100_1_NOFEC : [ 0 ]            | |
   | | M100_1_NOFEC_HALF : [ 0 ]       | |
   | | G1_1_NOFEC : [ 0 ]              | |
   | | G10_1_NOFEC : [ 0 ]             | |
   | | G25_1_NOFEC : [ 0 ]             | |
   | | G25_1_FCFEC : [ 0 ]             | |
   | | G25_1_RS528 : [ 0 ]             | |
   | | G50_2_NOFEC : [ 0, 1 ]          | |
   | | G50_2_FCFEC : [ 0, 1 ]          | |
   | | G50_2_RS528 : [ 0, 1 ]          | |
   | | G40_4_NOFEC : [ 0, 1, 2, 3 ]    | |
   | +---------------------------------+ |
   |                                     |
   | +---------------------------------+ |
   | |        interface root  1        | |
   | +---------------------------------+ |
   | | M100_1_NOFEC : [ 1 ]            | |
   | | M100_1_NOFEC_HALF : [ 1 ]       | |
   | | G1_1_NOFEC : [ 1 ]              | |
   | | G10_1_NOFEC : [ 1 ]             | |
   | | G25_1_NOFEC : [ 1 ]             | |
   | | G25_1_FCFEC : [ 1 ]             | |
   | | G25_1_RS528 : [ 1 ]             | |
   | +---------------------------------+ |
   |                                     |
   | +---------------------------------+ |
   | |        interface root  2        | |
   | +---------------------------------+ |
   | | M100_1_NOFEC : [ 2 ]            | |
   | | M100_1_NOFEC_HALF : [ 2 ]       | |
   | | G1_1_NOFEC : [ 2 ]              | |
   | | G10_1_NOFEC : [ 2 ]             | |
   | | G25_1_NOFEC : [ 2 ]             | |
   | | G25_1_FCFEC : [ 2 ]             | |
   | | G25_1_RS528 : [ 2 ]             | |
   | | G50_2_NOFEC : [ 2, 3 ]          | |
   | | G50_2_FCFEC : [ 2, 3 ]          | |
   | | G50_2_RS528 : [ 2, 3 ]          | |
   | +---------------------------------+ |
   |                                     |
   | +---------------------------------+ |
   | |        interface root 3         | |
   | +---------------------------------+ |
   | | M100_1_NOFEC : [ 3 ]            | |
   | | M100_1_NOFEC_HALF : [ 3 ]       | |
   | | G1_1_NOFEC : [ 3 ]              | |
   | | G10_1_NOFEC : [ 3 ]             | |
   | | G25_1_NOFEC : [ 3 ]             | |
   | | G25_1_FCFEC : [ 3 ]             | |
   | | G25_1_RS528 : [ 3 ]             | |
   | +---------------------------------+ |
   |                                     |
   +-------------------------------------+
   """
   @property
   def name( self ):
      """A convenient wrapper for class name"""
      return self.__class__.__name__

   def __str__( self ):
      """A human readable output for the interface slot type"""
      return self.name

   def getIntfSlotType( self ):
      return str( self )

   def generateIntfSlotTypeModel( self, intfSlotTypeDir ):
      intfSlotType = intfSlotTypeDir.newInterfaceSlotType(
         self.intfSlotTypeEnum )

      # Populate interface slot lanes for hw model
      intfSlotType.interfaceSlotLanes = self.getAllInterfaceSlotLanes()

      # Populate interface roots and interface group modes for hw model
      allGroupModes = self.getAllInterfaceGroupModes()
      groupModeId = 0
      for rootId, groupMode in allGroupModes.items():
         interfaceRoot = InterfaceRoot( rootId )

         for sgm, lanesConsumed in groupMode.items():
            # Assign groupModeId to interface group mode
            interfaceGroupMode = InterfaceSlotLaneGroupMode( groupModeId, sgm )
            for laneConsumed in lanesConsumed:
               interfaceGroupMode.lanesConsumed[ laneConsumed ] = True

            intfSlotType.addTotalSupportedModes( interfaceGroupMode )
            interfaceRoot.supportedModes.add( groupModeId )
            groupModeId += 1

         intfSlotType.addInterfaceRoots( interfaceRoot )

      return intfSlotType

   def getAllInterfaceSlotLanes( self ):
      """
      Returns
      -------
      Int : Total number of interface slot lane for a particular interface slot
      type.
      """
      return self.interfaceSlotLanes

   def getAllInterfaceGroupModes( self ):
      """
      Returns
      -------
      A dict from intfRootId to another dict mapping the static
      SerdesGroupModes that the interface slot type supports to the set of
      lanes that it consumes.
      """
      return self.interfaceSlotRootToLaneMappings

   @abstractproperty # pylint: disable=deprecated-decorator
   def interfaceSlotRootToLaneMappings( self ):
      """interfaceSlotRootToLaneMappings is a dict from intfRootId to another
      dict mapping the static SerdesGroupModes that the interface slot type
      supports to the set of lanes that it consumes."""
   
   @abstractproperty # pylint: disable=deprecated-decorator
   def interfaceSlotLanes( self ):
      """An int that describes the total number of lanes for the particular
      interface slot type."""

   @abstractproperty # pylint: disable=deprecated-decorator
   def intfSlotTypeEnum( self ):
      """The Interface::IntfSlotTypes::IntfSlotTypes that corresponds to this
      particular interface slot type."""

# Interface Slot Type Registration System
registeredIntfSlotTypes = {}

def registerIntfSlotType( slotTypeName=None ):
   def registerFunc( intfSlotTypeCls ):
      """A utility function used to register interface slot types so that the Hw
      L1 Topology FRU plugin ( as well as other external consumers ) can be made
      aware of them.
      
      Args
      ----
         slotName: Optional arg to customise the name of the slot type.
         intfSlotTypeCls:  The interface slot type class to register.

      Returns
      -------
         The intfSlotTypeCls.

      Raises
      ------
         HwL1ComponentLibraryError if the intfSlotTypeCls does not subclass
         IntfSlotType.
      """

      name = intfSlotTypeCls.__name__
      if slotTypeName:
         name = slotTypeName
      name = name.lower()

      if not issubclass( intfSlotTypeCls, IntfSlotType ):
         raise HwL1ComponentLibraryError(
            'Attempted to register an intfSlotType that does '
            'not subclass from the base IntfSlotType class.',
            offendingComponentCls=intfSlotTypeCls )
      if name in registeredIntfSlotTypes:
         raise HwL1ComponentLibraryError(
            'Attempted to register multiple intfSlotType '
            'classes with the same name.',
            name=name,
            existingComp=registeredIntfSlotTypes[ name ],
            offendingComponentCls=intfSlotTypeCls )

      # Verify that all interface slot lanes are used
      seenIntfSlotLanes = set()
      for groupModes in intfSlotTypeCls.interfaceSlotRootToLaneMappings.values():
         for laneIds in groupModes.values(): 
            seenIntfSlotLanes = seenIntfSlotLanes.union( laneIds )
      allIntfSlotLanes = set( range( 0, intfSlotTypeCls.interfaceSlotLanes ) )
      if seenIntfSlotLanes != allIntfSlotLanes:
         raise HwL1ComponentLibraryError(
            'Attempted to register an intfSlotType where not all '
            'interface slot lanes have been mapped.',
            offendingComponentCls=intfSlotTypeCls )
      
      # Verify that SGMs and the set lanes it consumes is not duplicated between
      # interface roots
      seenGroupModes = set()
      for groupModes in intfSlotTypeCls.interfaceSlotRootToLaneMappings.values():
         for sgm, laneIds in groupModes.items():
            intfGroupMode = ( sgm, frozenset( laneIds ) )
            if intfGroupMode not in seenGroupModes:
               seenGroupModes.add( intfGroupMode )
            else:
               raise HwL1ComponentLibraryError(
                  'Attempted to register an intfSlotType where a SerdesGroupMode '
                  'and corresponding lanes consumed has been duplicated in the '
                  'interfaceSlotRootToLaneMapping.',
                  offendingComponentCls=intfSlotTypeCls )

      registeredIntfSlotTypes[ name ] = intfSlotTypeCls

      return intfSlotTypeCls
   
   return registerFunc

@registerIntfSlotType()
class Rj45( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : { 
         M100_1_NOFEC : { 0 },
         M100_1_NOFEC_HALF : { 0 },
         M10_1_NOFEC : { 0 },
         M10_1_NOFEC_HALF : { 0 },
         G1_1_NOFEC : { 0 },
         G2P5_1_NOFEC : { 0 },
         G5_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
      },
   }

   interfaceSlotLanes = 1
   intfSlotTypeEnum = _rj45

@registerIntfSlotType()
@registerIntfSlotType( "Sfp10" )
@registerIntfSlotType( "Sfp+" )
@registerIntfSlotType( "SfpPlus" )
class Single10( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         M100_1_NOFEC : { 0 },
         M100_1_NOFEC_HALF : { 0 },
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
      },
   }

   interfaceSlotLanes = 1
   intfSlotTypeEnum = _single10

@registerIntfSlotType()
class Single1( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         M100_1_NOFEC : { 0 },
         M100_1_NOFEC_HALF : { 0 },
         G1_1_NOFEC : { 0 },
      },
   }

   interfaceSlotLanes = 1
   intfSlotTypeEnum = _single1

@registerIntfSlotType()
@registerIntfSlotType( "Sfp25" )
@registerIntfSlotType( "Sfp28" )
class Single25( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
      },
   }

   interfaceSlotLanes = 1
   intfSlotTypeEnum = _single25

@registerIntfSlotType()
@registerIntfSlotType( "Sfp50" )
@registerIntfSlotType( "Sfp56" )
class Single50( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G50_1_RS544 : { 0 },
      },
   }

   interfaceSlotLanes = 1
   intfSlotTypeEnum = _single50

@registerIntfSlotType()
@registerIntfSlotType( "sfpDd" )
@registerIntfSlotType( "dsfp" )
class Dual100( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G50_1_RS544 : { 0 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_2_RS544 : { 0, 1 },
      },
      1 : {
         G1_1_NOFEC : { 1 },
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
         G50_1_RS544 : { 1 },
      },
   }

   interfaceSlotLanes = 2
   intfSlotTypeEnum = _dual100

@registerIntfSlotType()
@registerIntfSlotType( 'Qsfp28' )
@registerIntfSlotType( 'Qsfp100' )
class Quad100( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_4_NOFEC : { 0, 1, 2, 3 },
         G100_4_RS528 : { 0, 1, 2, 3 },
      },
      1 : {
         G1_1_NOFEC : { 1 },
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
      },
      2 : {
         G1_1_NOFEC : { 2 },
         G10_1_NOFEC : { 2 },
         G25_1_NOFEC : { 2 },
         G25_1_FCFEC : { 2 },
         G25_1_RS528 : { 2 },
         G50_2_NOFEC : { 2, 3 },
         G50_2_FCFEC : { 2, 3 },
         G50_2_RS528 : { 2, 3 },
      },
      3 : {
         G1_1_NOFEC : { 3 },
         G10_1_NOFEC : { 3 },
         G25_1_NOFEC : { 3 },
         G25_1_FCFEC : { 3 },
         G25_1_RS528 : { 3 },
      },
   }

   interfaceSlotLanes = 4
   intfSlotTypeEnum = _quad100

@registerIntfSlotType()
class Quad40( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         HG_G21_2_NOFEC : { 0, 1 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
      },
      1 : {
         G1_1_NOFEC : { 1 },
         G10_1_NOFEC : { 1 },
      },
      2 : {
         G1_1_NOFEC : { 2 },
         G10_1_NOFEC : { 2 },
         HG_G21_2_NOFEC : { 2, 3 },
      },
      3 : {
         G1_1_NOFEC : { 3 },
         G10_1_NOFEC : { 3 },
      },
   }

   interfaceSlotLanes = 4
   intfSlotTypeEnum = _quad40

@registerIntfSlotType()
@registerIntfSlotType( 'Cfp2-4' )
@registerIntfSlotType( 'Qsfp200' )
@registerIntfSlotType( 'Qsfp56' )
class Quad200( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
         G50_1_RS544 : { 0 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_2_RS544 : { 0, 1 },
         G100_4_NOFEC : { 0, 1, 2, 3 },
         G100_4_RS528 : { 0, 1, 2, 3 },
         G200_4_RS544 : { 0, 1, 2, 3 },
      },
      1 : {
         G1_1_NOFEC : { 1 },
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
         G50_1_RS544 : { 1 },
      },
      2 : {
         G1_1_NOFEC : { 2 },
         G10_1_NOFEC : { 2 },
         G25_1_NOFEC : { 2 },
         G25_1_FCFEC : { 2 },
         G25_1_RS528 : { 2 },
         G50_1_RS544 : { 2 },
         G50_2_NOFEC : { 2, 3 },
         G50_2_FCFEC : { 2, 3 },
         G50_2_RS528 : { 2, 3 },
         G100_2_RS544 : { 2, 3 },
      },
      3 : {
         G1_1_NOFEC : { 3 },
         G10_1_NOFEC : { 3 },
         G25_1_NOFEC : { 3 },
         G25_1_FCFEC : { 3 },
         G25_1_RS528 : { 3 },
         G50_1_RS544 : { 3 },
      },
   }

   interfaceSlotLanes = 4
   intfSlotTypeEnum = _quad200

@registerIntfSlotType()
class Quad400( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G1_1_NOFEC : { 0 },
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
         G50_1_RS544 : { 0 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_1_RS544 : { 0 },
         G100_2_RS544 : { 0, 1 },
         G100_4_NOFEC : { 0, 1, 2, 3 },
         G100_4_RS528 : { 0, 1, 2, 3 },
         G200_2_RS544 : { 0, 1 },
         G200_4_RS544 : { 0, 1, 2, 3 },
         G400_4_RS544 : { 0, 1, 2, 3 },
      },
      1 : {
         G1_1_NOFEC : { 1 },
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
         G50_1_RS544 : { 1 },
         G100_1_RS544 : { 1 },
      },
      2 : {
         G1_1_NOFEC : { 2 },
         G10_1_NOFEC : { 2 },
         G25_1_NOFEC : { 2 },
         G25_1_FCFEC : { 2 },
         G25_1_RS528 : { 2 },
         G50_1_RS544 : { 2 },
         G50_2_NOFEC : { 2, 3 },
         G50_2_FCFEC : { 2, 3 },
         G50_2_RS528 : { 2, 3 },
         G100_1_RS544 : { 2 },
         G100_2_RS544 : { 2, 3 },
         G200_2_RS544 : { 2, 3 },
      },
      3 : {
         G1_1_NOFEC : { 3 },
         G10_1_NOFEC : { 3 },
         G25_1_NOFEC : { 3 },
         G25_1_FCFEC : { 3 },
         G25_1_RS528 : { 3 },
         G50_1_RS544 : { 3 },
         G100_1_RS544 : { 3 },
      },
   }

   interfaceSlotLanes = 4
   intfSlotTypeEnum = _quad400

@registerIntfSlotType()
@registerIntfSlotType( "Cfp2-8" )
# Since QsfpDd and Osfp have the same capabilities there is no need to
# differentiate between the two slot types.
@registerIntfSlotType( "QsfpDd" )
@registerIntfSlotType( "Osfp" )
@registerIntfSlotType( "QsfpDd400" )
@registerIntfSlotType( "Osfp400" )
# TODO BUG737604: These fake interface slot types should be removed as part of the
#                 full interfaces V2 migration effort.
@registerIntfSlotType( "QSFPDD6Ln" )
@registerIntfSlotType( "QSFPDD4Lane" )
@registerIntfSlotType( "OSFP6Ln" )
@registerIntfSlotType( "QSFPDDPr" )
class Octal400( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
         G50_1_RS544 : { 0 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_2_RS544 : { 0, 1 },
         G100_4_NOFEC : { 0, 1, 2, 3 },
         G100_4_RS528 : { 0, 1, 2, 3 },
         G200_4_RS544 : { 0, 1, 2, 3 },
         G400_8_RS544 : { 0, 1, 2, 3, 4, 5, 6, 7 },
      },
      1 : { 
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
         G50_1_RS544 : { 1 },
      },
      2 : {
         G10_1_NOFEC : { 2 },
         G25_1_NOFEC : { 2 },
         G25_1_FCFEC : { 2 },
         G25_1_RS528 : { 2 },
         G50_1_RS544 : { 2 },
         G50_2_NOFEC : { 2, 3 },
         G50_2_FCFEC : { 2, 3 },
         G50_2_RS528 : { 2, 3 },
         G100_2_RS544 : { 2, 3 },
      },
      3 : {
         G10_1_NOFEC : { 3 },
         G25_1_NOFEC : { 3 },
         G25_1_FCFEC : { 3 },
         G25_1_RS528 : { 3 },
         G50_1_RS544 : { 3 },
      },
      4 : {
         G10_1_NOFEC : { 4 },
         G25_1_NOFEC : { 4 },
         G25_1_FCFEC : { 4 },
         G25_1_RS528 : { 4 },
         G40_4_NOFEC : { 4, 5, 6, 7 },
         G50_1_RS544 : { 4 },
         G50_2_NOFEC : { 4, 5 },
         G50_2_FCFEC : { 4, 5 },
         G50_2_RS528 : { 4, 5 },
         G100_2_RS544 : { 4, 5 },
         G100_4_NOFEC : { 4, 5, 6, 7 },
         G100_4_RS528 : { 4, 5, 6, 7 },
         G200_4_RS544 : { 4, 5, 6, 7 },
      },
      5 : {
         G10_1_NOFEC : { 5 }, 
         G25_1_NOFEC : { 5 },
         G25_1_FCFEC : { 5 },
         G25_1_RS528 : { 5 },
         G50_1_RS544 : { 5 },
      },
      6 : {
         G10_1_NOFEC : { 6 },
         G25_1_NOFEC : { 6 },
         G25_1_FCFEC : { 6 },
         G25_1_RS528 : { 6 },
         G50_1_RS544 : { 6 },
         G50_2_NOFEC : { 6, 7 },
         G50_2_FCFEC : { 6, 7 },
         G50_2_RS528 : { 6, 7 },
         G100_2_RS544 : { 6, 7 },
      },
      7 : {
         G10_1_NOFEC : { 7 },
         G25_1_NOFEC : { 7 },
         G25_1_FCFEC : { 7 },
         G25_1_RS528 : { 7 },
         G50_1_RS544 : { 7 },
      },
   }

   interfaceSlotLanes = 8
   intfSlotTypeEnum = _octal400

@registerIntfSlotType()
@registerIntfSlotType( "QsfpDd800" )
@registerIntfSlotType( "Osfp800" )
class Octal800( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         G10_1_NOFEC : { 0 },
         G25_1_NOFEC : { 0 },
         G25_1_FCFEC : { 0 },
         G25_1_RS528 : { 0 },
         G40_4_NOFEC : { 0, 1, 2, 3 },
         G50_1_RS544 : { 0 },
         G50_2_NOFEC : { 0, 1 },
         G50_2_FCFEC : { 0, 1 },
         G50_2_RS528 : { 0, 1 },
         G100_1_RS544 : { 0 },
         G100_2_RS544 : { 0, 1 },
         G100_4_NOFEC : { 0, 1, 2, 3 },
         G100_4_RS528 : { 0, 1, 2, 3 },
         G200_2_RS544 : { 0, 1 },
         G200_4_RS544 : { 0, 1, 2, 3 },
         G400_4_RS544 : { 0, 1, 2, 3 },
         G400_8_RS544 : { 0, 1, 2, 3, 4, 5, 6, 7 },
         G800_8_RS544 : { 0, 1, 2, 3, 4, 5, 6, 7 },
      },
      1 : { 
         G10_1_NOFEC : { 1 },
         G25_1_NOFEC : { 1 },
         G25_1_FCFEC : { 1 },
         G25_1_RS528 : { 1 },
         G50_1_RS544 : { 1 },
         G100_1_RS544 : { 1 },
      },
      2 : {
         G10_1_NOFEC : { 2 },
         G25_1_NOFEC : { 2 },
         G25_1_FCFEC : { 2 },
         G25_1_RS528 : { 2 },
         G50_1_RS544 : { 2 },
         G50_2_NOFEC : { 2, 3 },
         G50_2_FCFEC : { 2, 3 },
         G50_2_RS528 : { 2, 3 },
         G100_1_RS544 : { 2 },
         G100_2_RS544 : { 2, 3 },
         G200_2_RS544 : { 2, 3 },
      },
      3 : {
         G10_1_NOFEC : { 3 },
         G25_1_NOFEC : { 3 },
         G25_1_FCFEC : { 3 },
         G25_1_RS528 : { 3 },
         G50_1_RS544 : { 3 },
         G100_1_RS544 : { 3 },
      },
      4 : {
         G10_1_NOFEC : { 4 },
         G25_1_NOFEC : { 4 },
         G25_1_FCFEC : { 4 },
         G25_1_RS528 : { 4 },
         G40_4_NOFEC : { 4, 5, 6, 7 },
         G50_1_RS544 : { 4 },
         G50_2_NOFEC : { 4, 5 },
         G50_2_FCFEC : { 4, 5 },
         G50_2_RS528 : { 4, 5 },
         G100_1_RS544 : { 4 },
         G100_2_RS544 : { 4, 5 },
         G100_4_NOFEC : { 4, 5, 6, 7 },
         G100_4_RS528 : { 4, 5, 6, 7 },
         G200_2_RS544 : { 4, 5 },
         G200_4_RS544 : { 4, 5, 6, 7 },
         G400_4_RS544 : { 4, 5, 6, 7 },
      },
      5 : {
         G10_1_NOFEC : { 5 }, 
         G25_1_NOFEC : { 5 },
         G25_1_FCFEC : { 5 },
         G25_1_RS528 : { 5 },
         G50_1_RS544 : { 5 },
         G100_1_RS544 : { 5 },
      },
      6 : {
         G10_1_NOFEC : { 6 },
         G25_1_NOFEC : { 6 },
         G25_1_FCFEC : { 6 },
         G25_1_RS528 : { 6 },
         G50_1_RS544 : { 6 },
         G50_2_NOFEC : { 6, 7 },
         G50_2_FCFEC : { 6, 7 },
         G50_2_RS528 : { 6, 7 },
         G100_1_RS544 : { 6 },
         G100_2_RS544 : { 6, 7 },
         G200_2_RS544 : { 6, 7 },
      },
      7 : {
         G10_1_NOFEC : { 7 },
         G25_1_NOFEC : { 7 },
         G25_1_FCFEC : { 7 },
         G25_1_RS528 : { 7 },
         G50_1_RS544 : { 7 },
         G100_1_RS544 : { 7 },
      },
   }

   interfaceSlotLanes = 8
   intfSlotTypeEnum = _octal800

@registerIntfSlotType()
class AgileQuad40( IntfSlotType ):
   interfaceSlotRootToLaneMappings = {
      0 : {
         M100_1_NOFEC: { 0 },
         M100_1_NOFEC_HALF: { 0 },
         G1_1_NOFEC: { 0 },
         G10_1_NOFEC: { 0 },
         G25_1_NOFEC: { 0 },
         G25_1_FCFEC: { 0 },
         G25_1_RS528: { 0 },
         G40_4_NOFEC: { 0, 1, 2, 3 },
      },
   }

   interfaceSlotLanes = 4
   intfSlotTypeEnum = _agileQuad40

def getIntfSlotType( name ):
   intfSlotTypeClass = registeredIntfSlotTypes.get( name.lower() )
   if not intfSlotTypeClass:
      raise HwL1ComponentError(
         name, 'Invalid intfSlotType class name.',
         validNames=list( registeredIntfSlotTypes.values() ) )
   return intfSlotTypeClass
