# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines all types, enums, etc... related with defining logical SerDes
mappings on HW L1 components.
'''

from collections import namedtuple

import Tac

from L1Topology.Constants import (
   UNSUPPORTED_CHANNEL_ID,
)
from L1Topology.Modes import (
   SPEED_UNKNOWN,
   DUPLEX_UNKNOWN,
)

WILDCARD = None

AutonegMode = Tac.Type( 'Interface::AutonegMode' )
AUTONEG_CL28 = AutonegMode.anegModeClause28
AUTONEG_CL37 = AutonegMode.anegModeClause37
AUTONEG_CL73 = AutonegMode.anegModeClause73
AUTONEG_SGMII = AutonegMode.anegModeSgmii
AUTONEG_DISABLED = AutonegMode.anegModeDisabled

AutonegDesc = namedtuple( 'AutonegDesc', [ 'systemSideSpeed',
                                           'systemSideAutoneg',
                                           'lineSideSpeed',
                                           'lineSideAutoneg' ] )

def ForcedDesc( systemSideSpeed, lineSideSpeed ):
   assert systemSideSpeed is not None or lineSideSpeed is not None, \
      "Both systemSideSpeed and lineSideSpeed cannot be None"
   if systemSideSpeed is not None and lineSideSpeed is not None:
      return AutonegDesc( systemSideSpeed, AUTONEG_DISABLED,
                          lineSideSpeed, AUTONEG_DISABLED )
   elif systemSideSpeed is None:
      return AutonegDesc( None, None, lineSideSpeed, AUTONEG_DISABLED )
   else:
      return AutonegDesc( systemSideSpeed, AUTONEG_DISABLED, None, None )

class MappingDesc:
   """
   A MappingDesc is used to describe the serdes and modes that a chip has,
   and how they map to one another across the chip.

   Attributes:
      sysMode ( Hardware::Phy::SerdesGroupMode ): Serdes mode of system side serdes.
      sysSerdes ( list( int ) ): Serdes Ids of the system side serdes.
      lineMode ( Hardware::Phy::SerdesGroupMode ): Serdes mode of line side serdes.
      lineSerdes ( list( int ) ): Serdes Ids of the line side serdes.
      sysChannels ( list( int, Interface::EthSpeed ) ):
         List of channels and channel speeds supported by the system side serdes.
      lineChannels ( list( int, Interface::EthSpeed ) ):
         List of channels and channel speeds supported by the line side serdes.
   """

   def __init__ ( self, sysMode, sysSerdes, lineMode, lineSerdes,
                  sysChannels=None, lineChannels=None ):
      """ 
      The constructor for MappingDesc class.
      """
      self.sysMode = sysMode
      self.sysSerdes = sysSerdes
      self.lineMode = lineMode
      self.lineSerdes = lineSerdes
      if not sysChannels:
         self.sysChannels = [ ( UNSUPPORTED_CHANNEL_ID,
                                SPEED_UNKNOWN, DUPLEX_UNKNOWN ) ]
      else:
         self.sysChannels = sysChannels
         assert all( len( x ) == 3 for x in self.sysChannels )

      if not lineChannels:
         self.lineChannels = [ ( UNSUPPORTED_CHANNEL_ID,
                                 SPEED_UNKNOWN, DUPLEX_UNKNOWN ) ]
      else:
         self.lineChannels = lineChannels
         assert all( len( x ) == 3 for x in self.lineChannels )

# Serdes Pair keeps information for how serdes are connected logically on a chip.
# Note: Serdes Pair is used by chips with only PMA layers, since PCS chips care about
# Fec, Channels, they will use MappingDesc
# Parameters:
#    firstSerdesId ( int ): Logical Id of first serdes.
#    firstSerdesSide ( PhyEee::PhyScope ): Chip side of first serdes.
#    secondSerdesId ( int ): Logical Id of second serdes.
#    secondSerdesSide ( PhyEee::PhyScope ): Chip side of second serdes.
SerdesPair = namedtuple( 'SerdesPair', [ 'firstSerdesId', 'firstSerdesSide',
                                         'secondSerdesId', 'secondSerdesSide' ] )
