#!/usr/bin/env python3
# Copyright (c) 2008-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac, Agent, Smash, Tracing, Plugins, weakref, QuickTrace, os
import Cell
import Shark
from Arnet.NsLib import DEFAULT_NS
from Toggles.IgmpSnoopingToggleLib import toggleIgmpSnoopingL2SubinterfacesEnabled
from Toggles.StageMgrToggleLib import (
   toggleStrataIgmpSnoopingAsuEnabled,
   toggleEvpnMulticastSSOEnabled
)
from StageHelper import PyStagesHelper
from StageMgr import defaultStageInstance

__defaultTraceHandle__ = Tracing.Handle( "IgmpSnooping" )
t0 = Tracing.trace0
t3 = Tracing.trace3
t8 = Tracing.trace8
t6 = Tracing.trace6

Tac.activityManager.useEpoll = True
Client = Tac.Type( "Irb::Multicast::App::Client" )
MembershipJoinStatus = Tac.Type( "Irb::Multicast::Gmp::MembershipJoinStatus" )
MembershipJoinStatusCli = Tac.Type( "Irb::Multicast::Gmp::MembershipJoinStatusCli" )
MembershipRenewStatus = Tac.Type( "Irb::Multicast::Gmp::MembershipRenewStatus" )

IGMP_SNOOPING_ASU_L2RECONCILE_TIMEOUT = 250
IGMP_SNOOPING_SSO_TIMEOUT = 60

class IgmpSnooping ( Agent.Agent ):
   """The IGMP Snooping Agent. This agent implements the IGMP Snooping protocol
   as well as the shim layer that manages the interactions between IGMP
   snooping and the rest of the system.  It reacts to vlan state, topolib state,
   igmp snooping configs, and igmp packets that float by.  It updates IGMP
   Snooping status objects.  The agent is split into two pieces, the shim/glue,
   and the core, although both reside in the same agent."""

   def __init__( self, entityManager, _agentName=None, blocking=False ):
      t0( "Starting IgmpSnooping Agent" )
      self.block = blocking
      self.snoopConfig = None
      self.swForceConfig = None
      self.snoopCounterConfig = None
      self.snoopStatus = None
      self.advSnoopStatus = None
      self.coreConfig = None
      self.coreStatus = None
      self.brConfig = None
      self.brTopoConfig = None
      self.brTopoStatus = None
      self.querierConfigDir = None
      self.querierStatusDir = None
      self.profileConfig = None
      self.counterDir = None
      self.counterCheckpointDir = None
      self.hwConfig = None
      self.proxyConfig = None
      self.proxyIgmpHostDir = None
      self.eventMonConfig = None
      self.eventMonStatus = None
      self.kniStatus = None
      self.root_ = None
      self.pluginContext_ = None
      self.redundancyStatusReactor_ = None
      self.standbyModeReactor_ = None
      self.allIntfStatusDir = None
      self.allIntfStatusLocalDir = None
      self.ethIntfStatusDir = None
      self.intfEncapServiceIntfColl = None
      self.serviceIntfFwdInfoColl = None
      self.subIntfStatusDir = None
      self.intfConfigDir = None
      self.bridgingInput = None
      self.membershipJoinStatusMountReferences = []
      self.membershipJoinStatusColl = None
      self.cliJoinStatusSysdb = None
      self.cliJoinStatusShark = None
      self.mergedJoinStatus = None
      self.renewStatus = None
      self.evpnRenewStatus = None
      self.evpnStatus = None
      self.evpnProxyStatus = None
      self.dynamicVlanStatus = None
      self.mlagStatus = None
      self.mlagProtoStatus = None
      self.shmemMg = None
      self.shmemMgReactor_ = None
      self.cleanupShmemMgTimer_ = None
      self.vxlanSnoopingConfig = None

      self.bootConfig_ = None
      self.bootStatus_ = None
      self.asuBootProgressDir_ = None
      self.asuBootStageCompletionStatus_ = None
      self.asuStatus_ = None
      self.asuCompletionDetector_ = None
      self.bootStagesHelper_ = None

      self.asuShutdownStagesHelper_ = None
      self.asuShutdownStatus_ = None
      self.asuShutdownConfig_ = None
      self.asuShutdownProgressDir_ = None
      self.asuShutdownCompletionStatus_ = None

      self.snoopAsuTimeout_ = IGMP_SNOOPING_ASU_L2RECONCILE_TIMEOUT
      self.snoopReconcileAsuTimer_ = Tac.ClockNotifiee(
         self.handleAsuSnoopReconcile )
      self.snoopReconcileAsuTimer_.timeMin = Tac.endOfTime

      self.ssoConfig_ = None
      self.ssoStatus_ = None
      self.ssoProgressDir_ = None
      self.ssoStageCompletionStatus_ = None
      self.ssoStagesHelper_ = None
      self.snoopSsoTimeout_ = IGMP_SNOOPING_SSO_TIMEOUT
      self.snoopReconcileSsoTimer_ = Tac.ClockNotifiee(
         self.handleSsoSnoopReconcile )
      self.snoopReconcileSsoTimer_.timeMin = Tac.endOfTime
      self.ignoredStage = set()

      Agent.Agent.__init__( self, entityManager, _agentName )
      qtfile = "{}{}.qt".format( self.agentName, "-%d" if "QUICKTRACEDIR"
                                 not in os.environ else "" )
      QuickTrace.initialize( qtfile, "32,32,32,32,32,32,32,32,32,32" )

   def doInit( self, entityManager ):
      mountGroup = entityManager.mountGroup()
      self.snoopConfig = mountGroup.mount( 'bridging/igmpsnooping/config',
                                      'Bridging::IgmpSnooping::Config', 'r' )
      self.swForceConfig = mountGroup.mount( 'bridging/igmpsnooping/swForceConfig',
                                      'Bridging::IgmpSnooping::SwForceConfig', 'r' )
      self.snoopCounterConfig = mountGroup.mount(
         'bridging/igmpsnooping/counterConfig',
         'Bridging::IgmpSnooping::CounterConfig', 'r' )
      self.snoopStatus = mountGroup.mount( 'bridging/igmpsnooping/forwarding/status',
                                     'Bridging::IgmpSnooping::Status', 'w' )
      self.advSnoopStatus = mountGroup.mount(
                                          'bridging/igmpsnooping/advertised/status',
                                          'Bridging::IgmpSnooping::SyncStatus', 'w' )
      self.coreConfig = mountGroup.mount(
                                       'bridging/igmpsnooping/protocol/config',
                                       'Bridging::IgmpSnooping::IgmpProtocolConfig',
                                       'w' )
      self.coreStatus = mountGroup.mount(
                                       'bridging/igmpsnooping/protocol/status',
                                       'Bridging::IgmpSnooping::IgmpProtocolStatus',
                                       'w' )
      self.brConfig = mountGroup.mount( 'bridging/config', 'Bridging::Config', 'r' )
      self.brTopoConfig = mountGroup.mount( 'bridging/topology/config',
                                       'Bridging::Topology::Config', 'r' )
      self.querierConfigDir = mountGroup.mount( \
                                           'bridging/igmpsnooping/querier/config',
                                           'Igmp::QuerierConfigDir', 'w' )
      self.querierStatusDir = mountGroup.mount( \
                                           'bridging/igmpsnooping/querier/status',
                                           'Igmp::QuerierStatusDir', 'w' )
      self.profileConfig = mountGroup.mount( 'igmpprofile/profileconfig',
                                        'IgmpProfile::ProfileConfig', 'r' )
      self.counterDir = mountGroup.mount( 'bridging/igmpsnooping/counterDir',
                                     'Bridging::IgmpSnooping::IgmpCounterDir', 'w' )
      self.counterCheckpointDir = mountGroup.mount(
                                  'bridging/igmpsnooping/counterCheckpointDir',
                                  'Bridging::IgmpSnooping::IgmpCounterDir', 'r' )
      self.hwConfig = mountGroup.mount( 'bridging/igmpsnooping/hardware/config',
                                   'Bridging::IgmpSnooping::HwConfig', 'r' )

      self.proxyConfig = mountGroup.mount( 'bridging/igmpsnooping/proxy/config',
                                    'Igmp::Snooping::Proxy::Config', 'r' )
      self.proxyIgmpHostDir = mountGroup.mount(
                                    'bridging/igmpsnooping/proxy/igmphost/config',
                                    'Igmp::IgmpHostDir', 'w' )
      self.bridgingInput = mountGroup.mount( 'bridging/igmpsnooping/input',
                                             'Tac::Dir', 'w' )

      self.eventMonConfig = mountGroup.mount( 'eventMon/config',
                                 'EventMon::Config', 'r' )

      self.eventMonStatus = mountGroup.mount( 'eventMon/igmpsnooping/status',
                                 'EventMon::TableStatus', 'w' )
      self.vxlanSnoopingConfig = mountGroup.mount(
         'bridging/igmpsnooping/vxlan/config',
         'Bridging::Gmp::VxlanGmpSnoopingConfig',
         'r' )

      if toggleStrataIgmpSnoopingAsuEnabled():
         # StageHelper for ASU Boot progress
         t0( "Setting up ASU Mounts" )
         self.bootConfig_ = mountGroup.mount( Cell.path(
            'stageInput/boot/IgmpSnooping' ), 'Stage::AgentConfig', 'r' )
         self.bootStatus_ = mountGroup.mount( Cell.path(
            'stageAgentStatus/boot/IgmpSnooping' ), 'Stage::AgentStatus', 'wcf' )
         self.asuBootProgressDir_ = mountGroup.mount( Cell.path(
            'stage/boot/progress' ), 'Stage::ProgressDir', 'r' )
         self.asuBootStageCompletionStatus_ = mountGroup.mount( Cell.path(
            'stage/boot/completionstatus' ), 'Stage::CompletionStatusDir', 'r' )
         self.asuStatus_ = mountGroup.mount( 'asu/hardware/status', 'Asu::AsuStatus',
                                             'r' )
         # StageHelper for ASU Shutdown progress
         self.asuShutdownConfig_ = mountGroup.mount( Cell.path(
            'stageInput/shutdown/IgmpSnooping' ), 'Stage::AgentConfig', 'r' )
         self.asuShutdownStatus_ = mountGroup.mount( Cell.path(
            'stageAgentStatus/shutdown/IgmpSnooping' ), 'Stage::AgentStatus', 'wcf' )
         self.asuShutdownProgressDir_ = mountGroup.mount( Cell.path(
            'stage/shutdown/progress' ), 'Stage::ProgressDir', 'r' )
         self.asuShutdownCompletionStatus_ = mountGroup.mount( Cell.path(
            'stage/shutdown/completionstatus' ), 'Stage::CompletionStatusDir', 'r' )
         t0( "Done setting up ASU Mounts" )

      if toggleEvpnMulticastSSOEnabled():
         # StageHelper for SSO progress
         t0( "Setting up SSO Mounts" )
         self.ssoConfig_ = mountGroup.mount( Cell.path(
            'stageInput/switchover/IgmpSnooping' ), 'Stage::AgentConfig', 'r' )
         self.ssoStatus_ = mountGroup.mount( Cell.path(
            'stageAgentStatus/switchover/IgmpSnooping' ), 'Stage::AgentStatus',
                                             'wcf' )
         self.ssoProgressDir_ = mountGroup.mount( Cell.path(
            'stage/switchover/progress' ), 'Stage::ProgressDir', 'r' )
         self.ssoStageCompletionStatus_ = mountGroup.mount( Cell.path(
            'stage/switchover/completionstatus' ), 'Stage::CompletionStatusDir',
            'r' )

      shmemMg = self.shmemEntityManager.getMountGroup()
      self.shmemMg = shmemMg
      self.shmemMg.notifySync = True

      self.kniStatus = shmemMg.doMount( "kni/ns/%s/status" % DEFAULT_NS,
                                        "KernelNetInfo::Status",
                                        Smash.mountInfo( 'keyshadow' ) )

      self.membershipJoinStatusColl = Tac.newInstance(
         "Bridging::IgmpSnooping::JoinStatusColl" )
      writerMembershipJoinClients = [
         Client.igmpsnooping,
         Client.igmpsnoopingStatic,
         Client.mlag,
      ]

      for client in writerMembershipJoinClients:
         joinStatus = shmemMg.doMount(
         MembershipJoinStatus.mountPath( 'ipv4', client ),
         'Irb::Multicast::Gmp::MembershipJoinStatus',
         Shark.mountInfo( 'writer' ) )
         self.membershipJoinStatusMountReferences.append( joinStatus )

      readerMembershipJoinClients = [
         Client.evpn,
         Client.evpnRemoteDomain,
         Client.mcs,
      ]
      for client in readerMembershipJoinClients:
         joinStatus = shmemMg.doMount(
         MembershipJoinStatus.mountPath( 'ipv4', client ),
         'Irb::Multicast::Gmp::MembershipJoinStatus',
         Shark.mountInfo( 'shadow' ) )
         self.membershipJoinStatusMountReferences.append( joinStatus )

      self.cliJoinStatusSysdb = mountGroup.mount(
            MembershipJoinStatusCli.mountPath( 'ipv4', Client.cli ),
            'Irb::Multicast::Gmp::MembershipJoinStatusCli', 'r' )

      # Create a instance for CLI,
      # and convert MembershipJoinStatusCli => MembershipJoinStatus
      # for easy use in JoinStateImportSm
      self.cliJoinStatusShark = Tac.newInstance(
            'Irb::Multicast::Gmp::MembershipJoinStatus', 'cli' )

      self.mergedJoinStatus = Tac.newInstance(
            'Irb::Multicast::Gmp::MembershipJoinStatus', 'igmpsnoopingMerged' )

      self.renewStatus = shmemMg.doMount(
         MembershipRenewStatus.mountPath( 'ipv4', Client.igmpsnooping ),
         'Irb::Multicast::Gmp::MembershipRenewStatus',
         Shark.mountInfo( 'writer' ) )

      self.evpnRenewStatus = shmemMg.doMount(
         MembershipRenewStatus.mountPath( 'ipv4', Client.evpn ),
         'Irb::Multicast::Gmp::MembershipRenewStatus',
         Shark.mountInfo( 'shadow' ) )

      self.evpnStatus = mountGroup.mount( 'evpn/status', 'Evpn::EvpnStatus', 'r' )
      IgmpProxyStatus = Tac.Type( "Irb::Multicast::IgmpProxyStatus" )
      self.evpnProxyStatus = shmemMg.doMount(
         IgmpProxyStatus.afMountPath( 'ipv4' ),
         "Irb::Multicast::IgmpProxyStatus",
         Smash.mountInfo( 'keyshadow' )
      )
      self.dynamicVlanStatus = mountGroup.mount(
         'bridging/input/dynvlan/vlan/evpn',
         'Bridging::Input::VlanIdSet',
         'r' )
      self.mlagStatus = mountGroup.mount( 'mlag/status', 'Mlag::Status', 'r' )
      self.mlagProtoStatus = mountGroup.mount( 'mlag/proto', 'Mlag::ProtoStatus',
                                               'r' )
      if toggleIgmpSnoopingL2SubinterfacesEnabled():
         self.intfEncapServiceIntfColl = mountGroup.mount(
            'interface/encap/serviceIntf',
            'Ebra::IntfEncapServiceIntfColl',
            'r' )
         self.serviceIntfFwdInfoColl = mountGroup.mount(
            'interface/encap/fwdInfo',
            'Ebra::ServiceIntfFwdInfoColl',
            'r' )
         self.subIntfStatusDir = mountGroup.mount(
            'interface/status/subintf',
            'Interface::SubIntfStatusDir',
            'r' )

      self.createLocalEntity( "TopoStatus",
                              "Bridging::Topology::Status",
                              "bridging/topology/status" )
      self.createLocalEntity( "AllIntfStatusDir",
                              "Interface::AllIntfStatusDir",
                              "interface/status/all" )
      self.createLocalEntity( "AllIntfStatusLocalDir",
                              "Interface::AllIntfStatusLocalDir",
                              Cell.path( 'interface/status/local' ) )
      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )
      self.createLocalEntity( "EthIntfConfigDir",
                              "Interface::EthIntfConfigDir",
                              "interface/config/eth/intf" )

      self.localEntitiesCreatedIs( True )

      local = entityManager.root().parent
      t0( "Started IgmpSnoopingAgent." )
      self.root_ = local.newEntity( 'IgmpSnoopingAgent::Root',
                                    'IgmpSnoopingAgentRoot' )
      self.root_.agent = None

      self.root_.standbyMode = True
      self.pluginContext_ = None
      self.redundancyStatusReactor_ = None

      self.standbyModeReactor_ = None
      self.shmemMgReactor_ = None

      def _finish():
         t0( "Mounts complete. Wait for SharedMem mounts to be stable" )
         self.shmemMgReactor_ = ShmemMgReactor_( self.shmemMg, self )

      shmemMg.doClose()
      if self.block:
         mountGroup.close( blocking=True )
         _finish()
      else:
         mountGroup.close( _finish )

   def cleanupShmemMgReactor( self ):
      t0( "cleanupShmemMgReactor" )
      self.shmemMgReactor_ = None
      self.shmemMg = None

   def sharedMemMountGroupStable( self ):
      t0( "sharedMemMountGroupStable: ready=%s, stable=%s" %
          ( self.shmemMg.ready, self.shmemMg.stable ) )

      self.brTopoStatus = \
                  self.entityManager.getLocalEntity( "bridging/topology/status" )
      self.allIntfStatusDir = \
                  self.entityManager.getLocalEntity( "interface/status/all" )
      self.allIntfStatusLocalDir = \
      self.entityManager.getLocalEntity( Cell.path( 'interface/status/local' ) )
      self.ethIntfStatusDir = \
                  self.entityManager.getLocalEntity( "interface/status/eth/intf" )
      self.intfConfigDir = \
                  self.entityManager.getLocalEntity( "interface/config/eth/intf" )

      for joinStatus in self.membershipJoinStatusMountReferences:
         self.membershipJoinStatusColl.status.addMember( joinStatus )
      self.membershipJoinStatusColl.status.addMember( self.cliJoinStatusShark )
      self.membershipJoinStatusColl.statusCli = self.cliJoinStatusSysdb

      if toggleStrataIgmpSnoopingAsuEnabled():
         self.asuCompletionDetector_ = \
         Tac.newInstance( "Stage::AsuCompletionDetector",
                          self.asuStatus_,
                          self.asuBootStageCompletionStatus_,
                          self.redundancyStatus() )
         t0( "sharedMemMountGroupStable: Waiting for Stages " )
         self.bootStagesHelper_ = PyStagesHelper(
            "boot", None,
            self.asuBootProgressDir_,
            self.bootConfig_, self,
            completionStatus=self.asuBootStageCompletionStatus_ )
         self.bootStagesHelper_.waitForStage( 'NetworkAgent' )
         self.bootStagesHelper_.waitForStage( 'L2SnoopReconcile' )

         self.asuShutdownStagesHelper_ = PyStagesHelper(
            "shutdownStageClass", None, self.asuShutdownProgressDir_,
            self.asuShutdownConfig_, self, agentStatus=self.asuShutdownStatus_ )
         self.asuShutdownStagesHelper_.waitForStage( 'NetworkAgentShutdown' )
         t0( "sharedMemMountGroupStable: Waiting for shutdown Stages " )

      if toggleEvpnMulticastSSOEnabled():
         t0( "sharedMemMountGroupStable: Waiting for Stages " )
         self.ssoStagesHelper_ = PyStagesHelper(
            "switchover", None, self.ssoProgressDir_, self.ssoConfig_, self,
            completionStatus=self.ssoStageCompletionStatus_,
            agentStatus=self.ssoStatus_ )
         self.ssoStagesHelper_.waitForStage( 'IgmpSnoopingReconcile' )
         self.ssoStagesHelper_.waitForStage( 'IgmpSnoopingEvpnSmet' )

      self.redundancyStatusReactor_ = RedundancyStatusReactor(
                  self.redundancyStatus(), self )
      self.standbyModeReactor_ = StandbyModeReactor( self )
      self.standbyModeReactor_.handleStandbyMode()

      self.cleanupShmemMgTimer_ = Tac.ClockNotifiee( self.cleanupShmemMgReactor )
      self.cleanupShmemMgTimer_.timeMin = Tac.now() + 1


   def doStage( self, stageClass, stage ):
      """
      Callback function invoked by the PyStagesHelpers.

      The waitForStage( stage ) registers this callback for the specified stage.
      On normal boot, the stage graph does not include any of these stages, so the
      callbacks are immediate.
      """

      t0( "doStage callback, stage: %s, stageClass %s" % ( stage, stageClass ) )
      if toggleEvpnMulticastSSOEnabled() and stageClass == 'switchover':
         assert stage in [ "IgmpSnoopingReconcile", "IgmpSnoopingEvpnSmet" ]

         agentStatusKey = Tac.Value( "Stage::AgentStatusKey", self.agentName,
                                     stage, defaultStageInstance )

         if self.ssoStatus_.complete.get( agentStatusKey ):
            self.ignoredStage.add( stage )
            if self.root_.agent:
               t0( "Stage already completed. Marking forwarding ssoEvpnSmetReady" )
               self.setForwardingConfigSsoEvpnSmetReady()
            return

         if stage == "IgmpSnoopingReconcile":
            self.snoopReconcileSsoTimer_.timeMin = Tac.now() + self.snoopSsoTimeout_
         else:
            t0( "Marking IgmpSnoopingEvpnSmet stage as complete" )
            self.ssoStagesHelper_.doStageComplete( "IgmpSnoopingEvpnSmet" )
            t0( "IgmpSnoopingEvpnSmet doStage Complete" )
            if self.root_.agent:
               t0( "Marking forwarding ssoEvpnSmetReady" )
               self.setForwardingConfigSsoEvpnSmetReady()
      else:
         agentStatusKey = Tac.Value( "Stage::AgentStatusKey", self.agentName,
                                     stage, defaultStageInstance )
         if stageClass == 'boot':
            if stage == "NetworkAgent":
               self.bootStatus_.complete[ agentStatusKey ] = True
            elif stage == "L2SnoopReconcile":
            # IgmpSnoopingAgent was launched when NetworkAgent stage was published
            # All mounts are done. For stage L2SnoopReconcile, lets just start a
            # timer for 250 seconds. This timer is a best effort to wait for Strata
            # L2/L3 to converge.
               self.snoopReconcileAsuTimer_.timeMin = Tac.now() + \
                  self.snoopAsuTimeout_
         elif stageClass == 'shutdownStageClass':
            if stage == 'NetworkAgentShutdown':
               self.root_.agent.glueSm.asuShutdown = True
               self.asuShutdownStatus_.complete[ agentStatusKey ] = True
         else :
            assert stageClass 

         t0( "doStage Complete, stage: %s, stageClass %s" % ( stage,
                                                              stageClass ) )

   def handleAsuSnoopReconcile( self ):
      t0( "Marking L2SnoopReconcile stage as complete" )
      agentStatusKey = Tac.Value( "Stage::AgentStatusKey", self.agentName,
                                  "L2SnoopReconcile", defaultStageInstance )
      self.bootStatus_.complete[ agentStatusKey ] = True
      t0( "L2SnoopReconcile: doStage Complete" )

   def handleSsoSnoopReconcile( self ):
      t0( "Marking IgmpSnoopingReconcile stage as complete" )
      self.ssoStagesHelper_.doStageComplete( "IgmpSnoopingReconcile" )
      t0( "IgmpSnoopingReconcile: doStage Complete" )

   def launchAgentCreator( self ):
      t0( "launchAgentCreator" )

      t0( "dirs: ", self.snoopConfig, self.snoopStatus, self.coreConfig,
                    self.coreStatus, self.brConfig, self.brTopoStatus,
                    self.querierConfigDir, self.querierStatusDir )
      self.root_.agent = ( self.entityManager.sysname(),
                           self.snoopConfig, self.swForceConfig,
                           self.snoopCounterConfig, self.snoopStatus,
                           self.advSnoopStatus, self.coreConfig, self.coreStatus,
                           self.brConfig, self.allIntfStatusDir,
                           self.allIntfStatusLocalDir, self.kniStatus,
                           self.ethIntfStatusDir, self.intfEncapServiceIntfColl,
                           self.serviceIntfFwdInfoColl, self.subIntfStatusDir,
                           self.brTopoConfig, self.brTopoStatus,
                           self.querierConfigDir, self.querierStatusDir,
                           self.profileConfig, self.counterDir, self.hwConfig,
                           self.proxyConfig, self.proxyIgmpHostDir,
                           self.agentCmdRequestDirSm(), self.eventMonConfig,
                           self.eventMonStatus, self.redundancyStatus(),
                           self.membershipJoinStatusColl, self.mergedJoinStatus,
                           self.renewStatus, self.evpnRenewStatus, self.evpnStatus,
                           self.evpnProxyStatus, self.dynamicVlanStatus,
                           None,
                           self.mlagStatus, self.mlagProtoStatus,
                           self.vxlanSnoopingConfig,
                           self.agentScheduler(),
                           self.entityManager.cEntityManager() )

      t0( "Loading IgmpSnoopingPlugins" )
      self.pluginContext_ = IgmpSnoopingPluginContext( self )
      Plugins.loadPlugins( 'IgmpSnoopingPlugin', self.pluginContext_ )

      if self.ssoEvpnSmetReadyStageComplete():
         t0( "Marking ssoEvpnSmetReady in launchAgentCreator" )
         self.setForwardingConfigSsoEvpnSmetReady()

   def setForwardingConfigSsoEvpnSmetReady( self ):
      forwardingStateSmConfig = self.root_.agent.forwardingStateSmConfig
      forwardingStateSmConfig.ssoEvpnSmetReady = True

   def ssoEvpnSmetReadyStageComplete( self ):
      if not toggleEvpnMulticastSSOEnabled():
         return True

      if self.redundancyStatus().protocol != "sso":
         t0( "Not sso: ssoEvpnSmetReady" )
         return True
      else:
         agentStatusKey = Tac.Value( "Stage::AgentStatusKey", self.agentName,
                                     "IgmpSnoopingEvpnSmet", defaultStageInstance )
         if self.ssoStatus_.complete.get( agentStatusKey ):
            t0( "sso and ssoEvpnSmetReady" )
            return True

      t0( "sso. ssoEvpnSmet is not ready" )
      return False

   def getDefaultTrace( self ):
      return 'Shark*/01,Smash*/01'

   def __del__( self ):
      t0( "__del__" )
      if self.root_:
         self.root_.agent = None
         parent = self.root_.parent
         if parent:
            parent.deleteEntity( self.root_.name )
         self.root_ = None
      for base in self.__class__.__bases__:
         if hasattr( base, "__del__" ):
            base.__del__( self )

class ShmemMgReactor_( Tac.Notifiee ):
   notifierTypeName = 'TacSharedMem::MountGroup'

   def __init__( self, shmemMg, agent ):
      t0( "__init__ShmemMgReactor" )
      Tac.Notifiee.__init__( self, shmemMg )
      self.shmemMg_ = shmemMg
      self.agent_ = weakref.ref( agent )

      self.handleSharedMemMountGroupStable()

   @Tac.handler( 'stable' )
   def handleSharedMemMountGroupStable( self ):
      t0( "handleSharedMemMountGroupStable: ready=%s, stable=%s" %
          ( self.notifier_.ready, self.notifier_.stable ) )

      agent = self.agent_()
      if agent and self.notifier_.stable:
         # If we have a redundancyStatusReactor_ reactor, then
         # we have already completed this step once.
         if agent.redundancyStatusReactor_ is None:
            agent.sharedMemMountGroupStable()

   def __del__( self ):
      t0( "__del__ ShmemMgReactor" )


class RedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redundancyStatus, agent ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.redundancyStatus_ = redundancyStatus
      self.agent_ = weakref.ref( agent )
      self.sysdbWritable_ = 'SysdbWritable'

      self.handleRedundancyStatus()

   @Tac.handler( 'protocol' )
   @Tac.handler( 'mode' )
   def handleRedundancyStatus( self ):
      t0( "handleRedundancyStatus" )
      agent = self.agent_()

      if self.redundancyStatus_.protocol == "sso" and \
         not self.redundancyStatus_.mode == "active":
         agent.root_.standbyMode = True
         self.handleSwitchoverStage( self.sysdbWritable_ )
      else:
         agent.root_.standbyMode = False

   @Tac.handler( 'switchoverStage' )
   def handleSwitchoverStage( self, stage ):
      t0( "handleSwitchoverStage: protocol=%s, switchoverStage=%s" %
          ( self.notifier_.protocol, stage ) )

      if ( self.sysdbWritable_ in self.redundancyStatus_.switchoverStage and
           self.redundancyStatus_.switchoverStage[ self.sysdbWritable_ ] ):
         self.agent_().root_.standbyMode = False

class StandbyModeReactor( Tac.Notifiee ):
   notifierTypeName = "IgmpSnoopingAgent::Root"

   def __init__( self, agent ):
      self.agent_ = weakref.ref( agent )
      self.root_ = self.agent_().root_
      Tac.Notifiee.__init__( self, self.root_ )

   @Tac.handler( "standbyMode" )
   def handleStandbyMode( self ):
      if self.root_.agent or self.root_.standbyMode:
         return

      t8( 'handleStandbyMode' )
      agent = self.agent_()
      agent.launchAgentCreator()

class IgmpSnoopingPluginContext:
   def __init__( self, agent ):
      self.topAgent_ = weakref.proxy( agent )

   def agent( self ):
      return self.topAgent_.root_.agent

   def entityManager( self ):
      return self.topAgent_.entityManager

   def floodControlIs( self, fc ):
      t0( "New FloodControl:", fc )
      self.agent().glueSm.floodControlManager.floodControl.addMember( fc )

   def floodControlDel( self, fcName ):
      t0( "Deleting FloodControl:", fcName )
      del self.agent().glueSm.floodControlManager.floodControl[ fcName ]

   def intfStateControlIs( self, sc ):
      t0( "New IntfStateControl:", sc )
      self.agent().glueSm.intfStateControl.intfStateControl.addMember( sc )

   def intfStateControlDel( self, scName ):
      t0( "Deleting IntfStateControl:", scName )
      del self.agent().glueSm.intfStateControl.intfStateControl[ scName ]

   def peerStatusIs( self, ps ):
      t0( "Add peer snooping status :" )
      self.agent().glueSm.peerStatus = ps

   def peerStatusDel( self ):
      t0( "Delete peer snooping status:" )
      self.agent().glueSm.peerStatus = None

   def snoopingSyncEnable( self ):
      t0( "Enable snooping sync:" )
      self.agent().glueSm.snoopingSync = True

   def snoopingSyncDisable( self ):
      t0( "Disable snooping sync:" )
      self.agent().glueSm.snoopingSync = False

   def mlagActiveIs( self, active, mi ):
      if active:
         t0( "Mlag is active" )
         if not self.agent().mlagPeerIntfDownSm:
            t0( "Starting mlagPeerIntfDownSm" )
            smArgs = ( mi.mlagStatus,
                       self.agent().protocolStatus,
                       self.agent().softwareForwardStatus,
                       self.agent().topoLibVlanState )
            self.agent().mlagPeerIntfDownSm = smArgs
      else:
         t0( "Mlag is inactive" )
         t0( "Deleting mlagPeerIntfDownSm" )
         self.agent().mlagPeerIntfDownSm = None

   def mlagInfoIs( self, mi ):
      t0( "New Mlag Information:", mi )
      self.agent().glueSm.mlagInfo = mi
      self.agent().agentSyncConfig.mlagInfo = mi

   def mlagInfoDel( self ):
      t0( "Deleting mlag information" )
      self.agent().glueSm.mlagInfo = None
      self.agent().agentSyncConfig.mlagInfo = None

   def handleDisabledIntf( self, intf ):
      t0( "handleDisabledIntf from plugin ctx:", intf )
      for glueVlanSm in self.agent().glueSm.vlanSm.values():
         coreSm = glueVlanSm.coreSm
         if coreSm:
            coreSm.handleDisabledIntf( intf, True )

   def getAgentSyncStatus( self ):
      return self.agent().agentSyncStatus

   def agentSyncEnabledIs( self, enabled ):
      t0( "Enabling Direct agent to agent sync" )
      self.agent().agentSyncConfig.setupDirectSync = enabled
      if enabled:
         mlagInfo = self.agent().agentSyncConfig.mlagInfo
         assert mlagInfo
         agentSyncConfig = self.agent().agentSyncConfig
         agentSyncStatus = self.agent().agentSyncStatus
         if mlagInfo.getMlagLocalNetNs() != agentSyncConfig.netNs:
            t0( 'Clearing agentSycnSm due to namespace change from <%s> to <%s>'
                % ( mlagInfo.getMlagLocalNetNs(), agentSyncConfig.netNs ) )
            self.agent().agentSyncSm = None
            self.agent().mlagLeaveStateExportSm = None
         forwardingStateSmConfig = self.agent().forwardingStateSmConfig
         t0( 'Instantiate AgentSyncSm' )
         smArgs = ( agentSyncConfig, agentSyncStatus, forwardingStateSmConfig,
               self.agent().agentname )
         self.agent().agentSyncSm = smArgs

         t0( 'Instantiate MlagLeaveStateExportSm' )
         smArgs = ( self.agent().msgStatus, mlagInfo.mlagStatus,
                    self.agent().agentSyncStatus.localP2pLeaveStatus,
                    self.agent().scheduler )
         self.agent().mlagLeaveStateExportSm = smArgs
      else:
         self.agent().agentSyncSm = None
         self.agent().mlagLeaveStateExportSm = None

   def vxlanInfoIs( self, vx ):
      t0( "Vxlan Information:", vx )
      self.agent().glueSm.vxlanInfo = vx

   def vxlanInfoDel( self ):
      t0( "Deleting mlag information" )
      self.agent().glueSm.vxlanInfo = None

