#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
@ Summary:
Variables used for building EOS images
Previously defined in EosImage/Makefile.am

@ Vars List:
- BASE PACKAGES
  Base packages to install into EOS. These packages aim to aggregate dependencies of
  RPMs that should be installed in EOS for platform-independent feature support.

- PRODUCT PACKAGES
  Product packages ( platform-dependent )
  These RPMs come from various SKU Family Packages and are aggregators of
  RPMs needed to support various SKUs belonging to that SKU Family.

- ASIC PACKAGES
  Packages containing support for ASICs and related features

- BAD PACKAGES
  Bad packages is a list of packages whose immediate dependencies must not
  appear in the image. Note that it is added to in the SHIPPING case.

- BLOCKED PACKAGES
  A list of package name regex that must never appear in the image

- ALLOWED PACKAGES
  A list of package name regex that we allow in the SWI, used to
  override our badpackage and blocked checks.

- BADPLUGINS
  A list of plugin types, where there must be no
  %Plugin directory in /usr/lib*/python*/site-packages (or it must be empty)

- YUM REPOS
  A default list of repos to be enabled or disabled while using the Yum command
  to build images.
'''

import platform
from enum import Enum, unique

ArmPlatform = False
IntelPlatform = False
# pylint: disable-msg=R1703
if platform.machine() == 'aarch64': # A4NOCHECK
   ArmPlatform = True
else:
   IntelPlatform = True

# BASE PACKAGES
basePkgs = [ 'Eos' ]

# PRODUCT PACKAGES
# RPM Lists for optimized SWIM
if IntelPlatform:
   skusDefault = [
      'Baremetal-EosImage',
      'BfnFixed-EosImage',
      'CaravanSku-EosImage',
      'DenaliModular-EosImage',
      'Glacier-Strata-EosImage',
      'Glacier-MontereyC-EosImage',
      'Glacier-Sand-EosImage',
      'Hollywood-EosImage',
      'Jericho2Fixed-EosImage',
      'Jericho2PlusFixed-EosImage',
      'Jericho3Fixed-EosImage',
      'Tomahawk2Fixed-EosImage',
      'Tomahawk3Fixed-EosImage',
      'Tomahawk4Fixed-EosImage',
      'Tomahawk5Fixed-EosImage',
      'Trident4Fixed-EosImage',
      'Tundra-EosImage',
      'VEosSku-EosImage',
   ]

   # 64-bit specific packages
   if platform.machine() == 'x86_64': # A4NOCHECK
      skusDefault += [
         'MakoBfnFixed-EosImage',
         'MakoCortinaFixed-EosImage',
         'MakoFixed-EosImage',
         'MakoJericho2Fixed-EosImage',
      ]

   skus4gbShared = [
      'Alameda-EosImage',
      'Bodega-EosImage',
      'PtReyes-EosImage',
   ]

   skus4gbSand = [
      'JerichoPlusFixed-EosImage',
      'Yosemite-EosImage',
   ]

   skus4gbFlashStrata = [
      'Mendocino-EosImage',
      'Sequoia-EosImage',
   ]

   skus4gbRamStrata = [
      'SaltLakeFixed-EosImage',
      'Prairie-EosImage',
   ]

   strata4gb = [
      'StrataEosImage-common',
      # T2, T2+, H4, TH, TH+
      'StrataSdkShimImpl-v1',
      # FL, T3, T3X5, T3X4, T3X3, T3X2, T3X1
      # 4GB flash modulars and 4GB RAM fixed systems are using T3
      'StrataSdkShimImpl-v6',
   ]

   strata = [
      'StrataEosImage-full',
      'StrataSdkShimImpl-Full',
   ]

   sand = [
      'SandEosImage-full',
   ]

   bfnDefault = [
      'BfnEosImage-full',
   ]

if ArmPlatform:
   skusDefault = [
      'Baremetal-EosImage',
      'VEosSku-EosImage',
   ]

   skus4gbShared = [
   ]

   skus4gbSand = [
   ]

   skus4gbFlashStrata = [
   ]

   skus4gbRamStrata = [
      'Wolfhound3Plus-EosImage',
   ]

   strata4gb = [
   ]

   strata = [
   ]

   sand = [
   ]

   bfnDefault = [
   ]

# If this image is being shipped to customers, then we exclude all support for
# products that have not been released yet ( e.g. we exclude Yosemite-future )
# For internal builds, we include support for all products into the swi.

# BAD PACKAGES
badPkgs = [ 'Eos-devel', ]

# When adding RPMs to EOS, please ensure the RPMs
# are added to all necessary locations listed in
# AID/8161 - Getting RPMs installed into EOS
# Map of packages with future to their future
if IntelPlatform:
   pkgToFutureMap = {
      'basePkgs': {
         'Eos': 'Eos-future'
      },
      'productPkgs': {
         'Alameda-EosImage': 'Alameda-future',
         'BfnFixed-EosImage': 'BfnFixed-future',
         'Bodega-EosImage': 'Bodega-future',
         'CaravanSku-EosImage': 'CaravanSku-future',
         'DenaliModular-EosImage': 'DenaliModular-future',
         'Glacier-Strata-EosImage': 'Glacier-Strata-future',
         'Glacier-MontereyC-EosImage': 'Glacier-Strata-future',
         'Glacier-Sand-EosImage': 'Glacier-Sand-future',
         'JerichoPlusFixed-EosImage': 'JerichoPlusFixed-future',
         'Jericho2Fixed-EosImage': 'Jericho2Fixed-future',
         'Jericho2PlusFixed-EosImage': 'Jericho2PlusFixed-future',
         'Jericho3Fixed-EosImage': 'Jericho3Fixed-future',
         'MakoBfnFixed-EosImage': 'MakoBfnFixed-future',
         'MakoCortinaFixed-EosImage': 'MakoCortinaFixed-future',
         'MakoFixed-EosImage': 'MakoFixed-future',
         'MakoJericho2Fixed-EosImage': 'MakoJericho2Fixed-future',
         'Prairie-EosImage': 'Prairie-future',
         'PtReyes-EosImage': 'PtReyes-future',
         'SaltLakeFixed-EosImage': 'SaltLakeFixed-future',
         'Sequoia-EosImage': 'Sequoia-future',
         'Tomahawk2Fixed-EosImage': 'Tomahawk2Fixed-future',
         'Tomahawk3Fixed-EosImage': 'Tomahawk3Fixed-future',
         'Tomahawk4Fixed-EosImage': 'Tomahawk4Fixed-future',
         'Tomahawk5Fixed-EosImage': 'Tomahawk5Fixed-future',
         'Trident4Fixed-EosImage': 'Trident4Fixed-future',
         'Tundra-EosImage': 'Tundra-future',
         'Yosemite-EosImage': 'Yosemite-future',
         'VEosSku-EosImage': 'VEosSku-future',
      },
      'asicPkgs': {
         # RPMs aggregated under <ASIC>EosImage-future are staged/tested
         # in eos-trunk, and eventualy moved under <ASIC>EosImage-common
         # once they're ready to release.
         'BfnEosImage-common': 'BfnEosImage-future',
         'SandEosImage-common': 'SandEosImage-future',
         'StrataEosImage-common': 'StrataEosImage-future',
      },
      'miscPkgs': {
         'StrataSdkShimImpl-Full': 'StrataSdkShimImpl-future'
      }
   }
if ArmPlatform:
   pkgToFutureMap = {
      'basePkgs': {
         'Eos': 'Eos-future'
      },
      'productPkgs': {
         'Wolfhound3Plus-EosImage': 'Wolfhound3Plus-future',
      },
      'asicPkgs': {
      },
      'miscPkgs': {
      }
   }

if IntelPlatform:
   oliveFirmware = [
      "BfnFixed-olive-firmware",
      "CpuComplex-olive-firmware",
      "DenaliModular-olive-firmware",
      "Jericho2Fixed-olive-firmware",
      "Jericho2PlusFixed-olive-firmware",
      "Trident4Fixed-olive-firmware",
   ]

if ArmPlatform:
   # we're planning to build some arm systems
   oliveFirmware = []

# NOTE/WARNING:
# If adding to these lists, please also add
# the dpe RPM to EosImage-imagebuild's dependencies
# in EosImage/PkgSpec.py
if IntelPlatform:
   dpeFeatureRpms = [
      # "Feature" DPE RPMs
      "Ipsec-dpe",
      "Macsec-dpe",
      "Tunnelsec-dpe",
   ]
# Exclude Tunnelsec-dpe for now until we know whether it's needed
# for non-Sand. Installing Tunnelsec-dpe pulls in extra
# Tunnelsec RPMs which aren't normally pulled into the image through
# Eos RPM deps, which indicates the Tunnelsec RPMs are not PI.
if ArmPlatform:
   dpeFeatureRpms = [
      # "Feature" DPE RPMs
      "Ipsec-dpe",
      "Macsec-dpe",
   ]

# This map is used in the SWIM DPE Optimization installation step
# to determine which DPE FPGA RPMs should be installed in each
# optimization.
if IntelPlatform:
   skuToDpeFpgaRpms = {
      # SKU Specific DPE FPGA RPMs
      "BfnFixed-firmware": "BfnFixed-dpe",
      "DenaliModular-firmware": "DenaliModular-dpe",
      "Glacier-firmware": "CpuComplex-dpe",
      "Jericho2Fixed-firmware": "Jericho2Fixed-dpe",
      "Jericho2PlusFixed-firmware": "Jericho2PlusFixed-dpe",
      "Jericho3Fixed-firmware": "Jericho3Fixed-dpe",
      "SaltLakeFixed-firmware": "SaltLakeFixed-dpe",
      "Trident4Fixed-firmware": "Trident4Fixed-dpe",
   }
if ArmPlatform:
   skuToDpeFpgaRpms = {}

dpeFlavorRpms = dpeFeatureRpms + list( skuToDpeFpgaRpms.values() )

def getPkgsWithFuture():
   pkgs = []
   for v in pkgToFutureMap.values():
      pkgs.extend( v )
   return pkgs

def getAllFuturePkgs():
   pkgs = []
   for v in pkgToFutureMap.values():
      pkgs.extend( list( v.values() ) )
   return pkgs

def getFuturePkgs( pkgWithFuture=None ):
   if pkgWithFuture:
      for v in pkgToFutureMap.values():
         if pkgWithFuture in v:
            return v[ pkgWithFuture ]
      return []
   else:
      return getAllFuturePkgs()

def getBadPkgs( shipping ):
   if shipping:
      return badPkgs + getAllFuturePkgs()
   else:
      return badPkgs

# BLOCKED PACKAGES
# pylint: disable-msg=W1401
blockedPkgsCommon = [
   ".*-devel",
   ".*-no-swi",
   ".*-pysrc",
   ".*sim",
   ".*test.*",
   ".*Dut",
   "Aboot.*",
   "Art",
   "Artools",
   "Dos.*-src",
   "GoExternal",
   "Hello",
   "World",
   "ProjectReport.*",
   "gotacc",
   "goworkspace",
   r"libstdc\+\+8",
   "tacc",
   "tacgdb", # contains tacgdb.py with intimate knowledge of core types
   ".*-yangpathfiles",
   "CpuFabricKernelShim",
]

blockedPkgs = [
   ".*[Ee]tba.*",
   ".*Test.*",
] + blockedPkgsCommon

blockPkgsCeoslab = [
] + blockedPkgsCommon

blockPkgsVeoslab = [
] + blockedPkgsCommon

blockedPkgsByFlavor = {
   "cEOSLab": blockPkgsCeoslab,
   "vEOSLab": blockPkgsVeoslab,
}

def getBlockedPkgs( shipping, flavor=None ):
   return blockedPkgsByFlavor.get( flavor, blockedPkgs )

# ALLOWED PACKAGES
allowedPkgs = [
   # Only allow specific aboot RPMs to go into the SWI
   'Aboot-utils',
   'AbootEos-cli',
   'AbootEos-lib',
   'AbootEos-swi',
   'AbootEos-toggles',
   'AbootGpl-swi',
   # Allow this as it's needed by yum in CentOS
   'yum-plugin-fastestmirror',
   'mtest',
   'memtester',
]

def getAllowedPkgs():
   return allowedPkgs

# BADPLUGINS
badPluginsCommon = [
   'ArosTest',
   'ArosTestUi',
   'CliSaveTest',
   'EosTortureTest',
   'Riota',
   'Scapy',
   'SilvercreekTest',
   'Simulation',
   'Skegness',
   'TestRiota',
]

badPlugins = [
   'EbraTestBridge',
] + badPluginsCommon

badPluginsCeoslab = [
] + badPluginsCommon

badPluginsVeoslab = [
] + badPluginsCommon

badPluginsByFlavor = {
   "cEOSLab": badPluginsCeoslab,
   "vEOSLab": badPluginsVeoslab,
}

def getBadPlugins( flavor=None ):
   return badPluginsByFlavor.get( flavor, badPlugins )

def getDefaultYumRepos( mock=False, fmtArgs=False ):
   enableRepos = []
   disableRepos = []

   if not mock:
      enableRepos += [
         'local',
      ]
   else: # MOCK
      enableRepos += [
         'Abuild*',
      ]
      disableRepos += [
         'local',
         'prebuilt',
      ]

   if fmtArgs:
      enableRepos = [ '--enablerepo=' + repo for repo in enableRepos ]
      disableRepos = [ '--disablerepo=' + repo for repo in disableRepos ]

   return enableRepos, disableRepos

@unique
class SwimFlavor( Enum ):
   DEFAULT = 'DEFAULT'
   DPE = 'DPE'

ValidSwimFlavors = [ flavor.value for flavor in SwimFlavor ]

# list of Swim Optimizations that can be referenced by class variable names
class SwimOptimizations:
   Default = "Default"
   Strata4GB = "Strata-4GB"
   Sand4GB = "Sand-4GB"
   Dpe = "Default-DPE"
   Strata4GBDPE = "Strata-4GB-DPE"
   Sand4GBDPE = "Sand-4GB-DPE"

# List of optimizations with DPE counterparts, used
# to create DPE squash dirs during installswimfs()
# This list will become more vital if we add optimizations to SWIM
# that shouldn't have DPE counterparts such as vEOS-lab
optimizationsWithDPE = [
   SwimOptimizations.Default,
   SwimOptimizations.Strata4GB,
   SwimOptimizations.Sand4GB,
]

swimFlavorToOptimizations = {
   SwimFlavor.DEFAULT.value: [
      SwimOptimizations.Default,
      SwimOptimizations.Strata4GB,
   ] if ArmPlatform else [
      SwimOptimizations.Default,
      SwimOptimizations.Strata4GB,
      SwimOptimizations.Sand4GB,
   ],
   SwimFlavor.DPE.value: [
      SwimOptimizations.Dpe,
      SwimOptimizations.Strata4GBDPE,
   ] if ArmPlatform else [
      SwimOptimizations.Dpe,
      SwimOptimizations.Strata4GBDPE,
      SwimOptimizations.Sand4GBDPE,
   ],
}

swimFlavorToFullOptimization = {
   SwimFlavor.DEFAULT.value: SwimOptimizations.Default,
   SwimFlavor.DPE.value: SwimOptimizations.Dpe
}
