#!/usr/bin/env arista-python
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import select
import TacSigint

# This utility works around Python3's problem of auto-restarting blocking syscalls
# interrupted by signals in non-main threads. As a result, CLI threads in ConfigAgent
# never gets interrupted during a blocking call. Instead of forwarding frontend
# signal to backend thread, we write to a pipe that wakes up the thread. This
# requires the thread to call poll() on the pipe. This library provides the necessary
# APIs for this purpose.
#
# Usage:
#
# p = InterruptiblePoller.getPoller()
# events = p.poll()
#
# If interrupted, a KeyboardInterrupt will be raised. The poll() call
# internally will enable TacSigint.immediateMode().

__pollerProvider__ = None

def setPollerProvider( provider ):
   global __pollerProvider__
   assert not ( __pollerProvider__ and __pollerProvider__ != provider )
   __pollerProvider__ = provider

class InterruptiblePoller:
   # Used as select poll object, but interruptible.

   def __init__( self, sigfd ):
      self.poller_ = select.poll()
      self.sigfd_ = sigfd # fd for signal interrupt
      if sigfd is not None:
         self.poller_.register( sigfd, select.POLLIN )

   def poll( self, timeout=None ):
      # do the poll, but raise KeyboardInterrupt if we are actually interrupted
      # check if we are interrupted before blocking
      with TacSigint.immediateMode():
         TacSigint.check()
         eventList = self.poller_.poll( timeout )
         TacSigint.check()
         return eventList

   # APIs to pass to poller
   def register( self, *args ):
      return self.poller_.register( *args )

   def modify( self, *args ):
      return self.poller_.modify( *args )

   def unregister( self, *args ):
      return self.poller_.unregister( *args )

def getPoller():
   # Get an InterruptiblePoller object. If we don't have a provider, we still
   # re-enable SIGINT and depend on it to interrupt the poll() call (works
   # in main thread).
   return __pollerProvider__() if __pollerProvider__ else InterruptiblePoller( None )
