# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
This module contains templates for the interface interactions CLI command
(AID4425, AID7499). These templates are imported by the CLI to product the output
of 'show interfaces interactions'. They are also used in tests that use/mock
interactions CLI output in some way.
'''

from abc import ABCMeta, abstractmethod
from collections import namedtuple
import natsort

import CliPlugin.AlePhyIntfInteractionsModel as IxnModel
import EthIntfLib
import Tracing
from TypeFuture import TacLazyType

t1 = Tracing.trace1

################################################################################
# Tac Type declarations

EthLinkModeSet = TacLazyType( "Interface::EthLinkModeSet" )
LogicalPortPoolDescDir = TacLazyType( "AlePhy::LogicalPortPoolDescDir" )
EthSpeedType = TacLazyType( "Interface::EthSpeed" )

################################################################################
# Global declarations

# Speed tokens
s800g8 = "800G-8"
s400g8 = "400G-8"
s400g4 = "400G-4"
s200g4 = "200G-4"
s200g2 = "200G-2"
s100g4 = "100G-4"
s100g2 = "100G-2"
s100g1 = "100G-1"
s50g2 = "50G-2"
s50g1 = "50G-1"
s40g = "40G"
s25g = "25G"
s10g = "10G"
s1g = "1G"
s100m = "100M"

# speed-token to EthSpeed dict
toEthSpeed = {
   s800g8: EthSpeedType.speed800Gbps,
   s400g8: EthSpeedType.speed400Gbps,
   s400g4: EthSpeedType.speed400Gbps,
   s200g4: EthSpeedType.speed200Gbps,
   s200g2: EthSpeedType.speed200Gbps,
   s100g4: EthSpeedType.speed100Gbps,
   s100g2: EthSpeedType.speed100Gbps,
   s100g1: EthSpeedType.speed100Gbps,
   s50g2: EthSpeedType.speed50Gbps,
   s50g1: EthSpeedType.speed50Gbps,
   s40g: EthSpeedType.speed40Gbps,
   s25g: EthSpeedType.speed25Gbps,
   s10g: EthSpeedType.speed10Gbps,
   s1g: EthSpeedType.speed1Gbps,
   s100m: EthSpeedType.speed100Mbps,
}

# Speed-group rate tokens
sg100g = "100g"
sg50g = "50g"
sg25g = "25g"
sg10g = "10g"

################################################################################
# CAPI Model Helper
################################################################################

class IntfInteractionsModelHelper:
   """
   Provides interface to populate the interactions CAPI model. Primary goal is
   to reduce repetitious model instantiation code.

   Parameters
   ----------
   modes: EthLinkModeSet
     In some cases, models will need to filter out modes listed in the template
     based on the capabilities of the system (see `compatibleRateList`). `modes`
     should describe the union of capabilities of the template's member interfaces.

   model: IxnModel.InterfacesInteractions
      Sometimes we would like to update an existing model instead creating a new one.

   Note
   ----
   More of a note-to-self. Perhaps these belong as helper functions of the
   CAPI model itself. CAPI models providing mutators for attributes is a
   thing. Perhaps evaluate as a side-project...

   Also, to get certain renders to combine intfIds correctly, need to do
   something like the following during a test run to avoid importing a whole
   bunch of IntfRangePlugin and EthIntf stuff.

   # make render work right
   def dummyCombine( intfs, optarg=False ):
      return Utils.combineInterfaces( intfs )[ 0 ].replace( "Et", "Ethernet" )
   CliPlugin.AlePhyIntfInteractionsModel._combineIntfNames = dummyCombine

   Reason being that the combining method used relies on certain Sysdb structures
   to be in place to validate intfIds. This method substitution will allow
   tests to run mostly the same without building out all that Sysdb state.
   """
   def __init__( self, modes=None, model=None ):
      self.model = model or IxnModel.InterfacesInteractions()
      self.filterModes = modes or EthLinkModeSet.allIeeeCapsSet

   def compatibleRateList( self, rates ):
      """
      There are cases where when we populate a model with a list of rates, we
      should limit the rates to those supported by the reference's supported
      link modes. Consider the following contrived example:

      Ethernet1/1:
        For speed 100G:
           Ethernet1/2 becomes inactive
        For speed 50G:
           No interactions with other interfaces
        For speed 40G:
           No interactions with other interfaces
      Ethernet1/2:
        For speed 50G:
           Ethernet1/1 must be configured to 50G/40G
        For speed 40G:
           Ethernet1/1 must be configured to 50G/40G

      In this example, Et1/1 and Et1/2 make up a serdes-domain. The capabilities
      of the domain are 100G, 50G, and 40G. If we imagine that there is another
      flavor of this domain that can only do 100G and 40G, we would want the output
      to change to:

      Ethernet1/1:
        For speed 100G:
           Ethernet1/2 becomes inactive
        For speed 40G:
           No interactions with other interfaces
      Ethernet1/2:
        For speed 40G:
           Ethernet1/1 must be configured to 40G  <-- no 50G listed

      In other words, the rates of Et1/1 that are compatible with Et1/2 must be
      filtered from the template's superset based on the capabilities of the domain.
      This helper function provides that filtering.

      Parameters
      ----------
      rates: list of strings, values of EthIntfLib.speedLanestoSpeedLanesStr
         This function has the fun job of filtering a list of strings based on
         the contents of an EthLinkModeSet (lots of conversions incoming)
      """
      groupRates = EthIntfLib.linkModeSetToSpeedLaneList( self.filterModes )
      # The membership check will handle cases where a speed-lane combination
      # is not populated in speedLanestoSpeedLanesStr. I initially thought this
      # was overly cautious, but at the time of writing, it did not contain a
      # mapping for ( speed2p5Gbps, laneCount1 ).
      groupRateStrs = [ EthIntfLib.speedLanestoSpeedLanesStr[ rate ]
                        for rate in groupRates
                        if rate in EthIntfLib.speedLanestoSpeedLanesStr ]
      filteredRates = [ rate for rate in rates if rate in groupRateStrs ]
      return filteredRates

   def intfData( self, intf ):
      """
      Accessor for interactions data associated with intfId. If data does
      not exist, new object will be created.

      Parameters
      ----------
      intf: string representation of Arnet::IntfId
      """
      intfData = self.model.interfaces.setdefault( intf,
                  IxnModel.InterfacesInteractionsDataBySpeed() )
      return intfData

   def addIntfMode( self, intf, speed ):
      """
      Creates speed entry for a given interface.

      Parameters
      ----------
      intf: string representation of Arnet::IntfId
      speed: string representation of speed; expected values are from
         EthIntfLib.speedLanestoSpeedLanesStr

      Notes
      -----
      Not quite sure why I decided 'intfData' should be a create-if-not-exist and
      this method should be strictly create-on-request. I think it works well, but
      I'm not sure how to justify it.
      """
      intfData = self.intfData( intf )
      intfData.speeds[ speed ] = IxnModel.InterfacesInteractionsData()

   def inactiveIxn( self, parentIntf, speed, inactives ):
      # If for whatever reason 'inactives' is empty, return without constructing
      # an empty model
      if inactives:
         intfData = self.intfData( parentIntf )
         inactivesModel = intfData.speeds[ speed ].inactiveInterfaces
         if not inactivesModel:
            inactivesModel = IxnModel.InactiveInterfacesInteraction()
            intfData.speeds[ speed ].inactiveInterfaces = inactivesModel
         inactivesModel.interfaces.extend( inactives )
      else:
         t1( "inactiveIxn called with empty inactives; args:",
             self, parentIntf, speed, inactives )

   def speedLimitation( self, parentIntf, speed, subLimitations ):
      """
      Notes
      -----
      This interaction type doesn't lend itself to being called for the same
      parentIntf-speed combination more than once. I haven't had to yet, but
      it is something to look out for in the future.
      """
      intfData = self.intfData( parentIntf )
      limitationModel = intfData.speeds[ speed ].speedLimitedInterfaces
      if not limitationModel:
         limitationModel = IxnModel.SpeedLimitedInterfacesInteraction()
         intfData.speeds[ speed ].speedLimitedInterfaces = limitationModel
      for subIntf, speeds in subLimitations.items():
         compatibleSpeeds = self.compatibleRateList( speeds )
         speedStr = "/".join( compatibleSpeeds )
         limitationModel.interfaces[ subIntf ] = speedStr

   def speedRequirement( self, subIntf, speed, parentRequirements ):
      """
      Note
      ----
      This may need some finagling to work with both
      'compatibleParentInterfaceConfigurations' and its outdated alternative
      'primaryInterfaceConfigurationRequirement'
      """
      lineShortener = self.intfData( subIntf ).speeds[ speed ]  # too snarky?
      parentRequirement = lineShortener.compatibleParentInterfaceConfigurations
      if not parentRequirement:
         parentRequirement = IxnModel.CompatibleParentInterfaceConfigurations()
         lineShortener.compatibleParentInterfaceConfigurations = parentRequirement
      for parentIntf, configs in parentRequirements.items():
         configListModel = IxnModel.SpeedConfigurationList()
         compatibleConfigs = self.compatibleRateList( configs )
         for config in compatibleConfigs:
            configModel = IxnModel.SpeedConfiguration()
            configModel.populateFromStr( config )
            configListModel.configurations.append( configModel )
         parentRequirement.interfaces[ parentIntf ] = configListModel

   def primaryIntfSpeedRequirement( self, intf, primary, speed ):
      """
      Helps populate the 'primaryInterfaceConfigurationRequirement' member
      of the interactions model. This is more restrictive that the
      'compatibleParentInterfaceConfigurations' attribute since it assumes
      a single parent intf, and it assumes that the parent intf and the
      subordinate intf are restricted to the same speed.

      Note
      ----
      For newer types of interfaces, you likely want to use the
      'speedRequirement' function above instead. Long-story short, the
      corresponding attribute in the model could not accommodate the feedback
      given by reviewers of the output for newer systems. However, the attribute
      has to stick around of CAPI compatibility reasons, so we still use it for
      the port types that released with it.
      """
      lineShortener = self.intfData( intf ).speeds[ speed ]
      parentRequirement = lineShortener.primaryInterfaceConfigurationRequirement
      if not parentRequirement:
         parentRequirement = IxnModel.PrimaryInterfaceConfigurationRequirement()
         lineShortener.primaryInterfaceConfigurationRequirement = parentRequirement
      parentRequirement.interface = primary
      parentRequirement.requiredConfiguration = speed

   def speedGroupRequirement( self, intf, speed, speedGroupName, setting ):
      intfData = self.intfData( intf )
      speedGroupModel = intfData.speeds[ speed ].speedGroupRequirement
      if not speedGroupModel:
         speedGroupModel = IxnModel.SpeedGroupRequirement()
         intfData.speeds[ speed ].speedGroupRequirement = speedGroupModel
      speedGroupModel.speedGroup = str( speedGroupName )
      speedGroupModel.serdes = setting

   def hardwarePortGroupRequirement( self, intf, speed, portGroupName, portMode ):
      intfData = self.intfData( intf )
      portGroupModel = intfData.speeds[ speed ].hardwarePortGroupRequirement
      if not portGroupModel:
         portGroupModel = IxnModel.HardwarePortGroupRequirement()
         intfData.speeds[ speed ].hardwarePortGroupRequirement = portGroupModel
      portGroupModel.portGroup = portGroupName
      portGroupModel.portMode = portMode

   def logicalPortDescription( self, intf, poolId, memberIntfs, maxPorts ):
      intfData = self.intfData( intf )
      logicalPortModel = intfData.interfaceHardwareResourcePool
      if not logicalPortModel:
         logicalPortModel = IxnModel.InterfaceHardwareResourcePool()
         intfData.interfaceHardwareResourcePool = logicalPortModel
      logicalPortModel.poolId = int(
         LogicalPortPoolDescDir.poolIdToPoolName( poolId ) )
      logicalPortModel.memberInterfaces = memberIntfs
      logicalPortModel.capacity = maxPorts

   def blackhawkAutoFecRequirement( self, intf, primaryIntf ):
      intfData = self.intfData( intf )
      s100g4Data = intfData.speeds.get( s100g4 )
      # intfData may not be ready yet.
      if not s100g4Data:
         return
      anReq = s100g4Data.autonegRequirement
      if not anReq:
         anReq = IxnModel.AutonegFecRequirement()
         intfData.speeds[ s100g4 ].autonegRequirement = anReq

      anReq.primaryIntf = primaryIntf

   def inactiveReason( self, intf, reason ):
      intfData = self.intfData( intf )
      intfData.inactiveReason = reason

################################################################################
# Reference/Template objects
################################################################################

# Unlike for speed-groups, logical port interactions need more than just a name
# to populate the IntfInteractionReference object.
LogicalPortInteractionDesc = namedtuple( "LogicalPortInteractionDesc",
                                         [ "poolId", "memberIntfs", "maxPorts" ] )

class SerdesDomain:
   """
   SerdesDomain objects store a list of interfaces, which are used to populate an
   IntfInteractionReference
   """
   def __init__( self, members ):
      self.memberIntfs = members

   def notNoneIntfs( self ):
      return natsort.natsorted( [ i for i in self.memberIntfs.values() if i ] )

class IntfInteractionReference:
   """
   Base-class for modularly generated intf interactions CLI model references
   for use in tests. Interactions modeled are for a particular resource group
   type. Each group type should be implemented in a base class.

   Common Members
   --------------
   groupLinkModes: Interface::EthLinkModeSet
      Speeds that interfaces in this group are capable of. Note that these are
      the capabilities of the group, not individual interfaces. The interfaces
      used to service a given speed are implied based on group type. For
      example, if a Babbage group lists 50G-2 as a capability, it is assumed
      that the configuration is serviced via primary/1+3 and secondary/1+3. By
      default, all capabilities are assumed servicable.

   speedGroupName: str or None
      If set, reference will use this speed-group name to populate fields for
      the relevant link modes. 

   logicalPortPoolDesc : LogicalPortInteractionDesc or None
      If set, reference will use information to populate logical port fields.
      If not set, the product is assumed to not have logical port limitations.

   hardwarePortGroup: str or None
      If set, reference will use this port-group name to populate fields for
      the relevant port-group modes.

   hardwarePortModes: list of str or None
      Hardware port-group modes of a port-group

   Note
   ----
   For speed-group references: each template will hard-code the speed-group
      settings for a link-mode (e.g. 400G-8 will be assumed to require the
      50g serdes rate from the appropriate speed-group). If future products
      require templates with customizable speed-group parameters, we can
      extend the architecture to support it relatively painlessly.
   """
   def __init__( self ):
      self.groupLinkModes = EthLinkModeSet.allIeeeCapsSet
      self.speedGroupName = None
      self.logicalPortPoolDesc = None
      self.hardwarePortGroup = None
      self.hardwarePortModes = None

   @staticmethod
   def _notNoneIntfs( intfs ):
      """ Returns the list of intfs which are not None from the input. With
      port-renumbered cards, there may be depopulated SerdesDomains that we need
      to generate models for. This function is going to be heavily used to
      generate output for those cases."""
      return [ i for i in intfs if i ]

   def model( self ):
      """
      Should be overwritten by child classes
      """
      return IxnModel.InterfacesInteractions()

   def json( self ):
      return self.model().toDict()

   def render( self ):
      # BUG479185: should capture STDOUT here, probably, to return string directly.
      return self.model().render()

################################################################################
# Interaction Template Definitions
################################################################################

class Tomahawk450GQsfpDd( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for the interfaces of a
   TH450G Fanshell QSFPDD slot and Pebble w/ and w/o Blanco interactions.

   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided octal port

      Parameters
      ----------
      serdesDomain: map of { "laneX" : "EthernetY/X" }
      """
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]
      # lane5 and lane7 is not None but they only work when blanco is inserted
      self.lane5 = serdesDomain.memberIntfs[ "lane5" ]
      self.lane7 = serdesDomain.memberIntfs[ "lane7" ]
      # lane6 and lane8 are default to be None
      self.lane6 = serdesDomain.memberIntfs.get( "lane6" )
      self.lane8 = serdesDomain.memberIntfs.get( "lane8" )

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3,
                   self.lane4, self.lane5, self.lane7 ]
      # p = primary, s = secondary
      p1, p2, p3, p4, p5, p7 = allIntfs
      # secondary1 and secondary3 will be None if blanco is inserted
      s1, s3 = None, None
      if self.logicalPortPoolDesc:
         # If set, self.logicalPortPoolDesc can be unpacked
         # pylint: disable=unbalanced-tuple-unpacking
         s1, s3 = self.getSecondaryIntfs( allIntfs, 
                                          self.logicalPortPoolDesc.memberIntfs )

      compatibleParentConfigs = {
         # These entries are in the format
         #    subIntf: { parent1: ( compatible speeds ),
         #               parent2: ( compatible speeds ),
         #               ... }
         p2: { p1: ( s50g1, s25g, s10g ),
               p3: ( s50g1, s25g, s10g ) },
         p3: { p1: ( s100g2, s50g1, s50g2, s25g, s10g ) },
         p4: { p1: ( s50g1, s25g, s10g ),
               p3: ( s50g1, s25g, s10g ) },
         p5: { p1: ( s200g4, s100g2, s100g4, s50g2, s40g, s50g1, s25g, s10g ),
               p3: ( s100g2, s50g1, s50g2, s25g, s10g ) },
         p7: { p1: ( s200g4, s100g2, s100g4, s50g1, s50g2, s40g, s25g, s10g ),
               p3: ( s100g2, s50g2 ),
               p5: ( s100g2, s50g2, s50g1, s25g, s10g ) },
      }
      
      if s1 and s3:
         # add p5, p7 entries and set p5, p7 to None
         # Only enter here when it's a Pebble w/o Blanco
         helper.inactiveReason( p5, f"QDD-BP-200 not installed on {s1}" )
         helper.inactiveReason( p7, f"QDD-BP-200 not installed on {s1}" )
         p5 = None
         p7 = None
      if self.groupLinkModes.mode400GbpsFull8Lane:
         # Should only enter here when it's Fanshell or Pebble w/ Blanco, 
         # since 400G-capability is tied to Blanco module being installed
         helper.addIntfMode( p1, s400g8 )
         # 400G-8 interactions:
         # * Applied on primary1; all others go inactive
         # * Only when blanco is inserted for Pebble
         if self.speedGroupName:
            helper.speedGroupRequirement( p1, s400g8, self.speedGroupName, sg50g )
         helper.inactiveIxn( p1, s400g8, ( p2, p3, p4, p5, p7 ) )
      
      # Four-lane config interactions (200G-4, 100G-4, 40G-4)
      # * Applied on Primary/1; Primary/2-4 becomes inactive
      # * Applied on Primary/5 if blanco is inserted. Primary/7 becomes inactive
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode200GbpsFull4Lane:
         supportedFourLaneSpeeds.append( s200g4 )
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      
      for speed in supportedFourLaneSpeeds:
         for intf in self._notNoneIntfs( ( p1, p5 ) ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s200g4: sg50g,
                          s100g4: sg25g,
                          s40g: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed,
                                             self.speedGroupName, sgRate )
         helper.inactiveIxn( p1, speed, ( p2, p3, p4 ) )
         if p5 and p7:
            helper.inactiveIxn( p5, speed, ( p7, ) )
            helper.speedRequirement( p5, speed, compatibleParentConfigs[ p5 ] )

      # Two-lane config interactions (100G-2, 50G-2)
      # * Applied on Primary/1; Primary/2 becomes inactive
      # * Applied on Primary/3; Primary/4 becomes inactive
      # * Applied on Primary/5 if blanco is inserted
      # * Applied on Primary/7 if blanco is inserted
      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )
      for speed in supportedTwoLaneSpeeds:
         for intf in self._notNoneIntfs( ( p1, p3, p5, p7 ) ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s100g2: sg50g,
                          s50g2: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed,
                                             self.speedGroupName, sgRate )
         helper.inactiveIxn( p1, speed, ( p2, ) )
         helper.inactiveIxn( p3, speed, ( p4, ) )
         for intf in self._notNoneIntfs( ( p3, p5, p7 ) ):
            helper.speedRequirement( intf, speed, compatibleParentConfigs[ intf ] )

      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in self._notNoneIntfs( ( p1, p2, p3,
                                           p4, p5, p7 ) ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s50g1: sg50g,
                          s25g: sg25g,
                          s10g: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         for intf in self._notNoneIntfs( ( p2, p3, p4, p5, p7 ) ):
            helper.speedRequirement( intf, speed, compatibleParentConfigs[ intf ] )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

   def getSecondaryIntfs( self, portIntfs, lppMemberIntfs ):
      sIntfs = [ intf for intf in lppMemberIntfs if intf not in portIntfs ]
      return sorted( sIntfs ) if sIntfs else ( None, None )

class Tomahawk450GQsfpPlusBase( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for primary & secondary interfaces of a
   BlackhawkGen3 QSFP pair. This is a specific case as the serdesDomain does not list
   the lanes as 'primary' or 'secondary' and is determined based on available lanes.
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided primary-secondary pair

      Parameters
      ----------
      serdesDomain: map of { "laneX" : "EthernetY/X" }
      Lane2 and 4 does not exist for secondary ports.
      """
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      # if lane2 and lane4 are undefined; it means this is the secondary port.
      self.lane2 = serdesDomain.memberIntfs.get( "lane2" )
      self.lane4 = serdesDomain.memberIntfs.get( "lane4" )

   def model( self ):
      """
      Create models for all member interfaces based on provided options

      Note
      ----
      Utilizes boolean on init to check if this is the primary or the secondary
      interface of this pair.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # Should have logicalPortPoolDesc given primary-secondary pair.
      assert self.logicalPortPoolDesc

      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode200GbpsFull4Lane:
         supportedFourLaneSpeeds.append( s200g4 )
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )

      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )

      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )

      # Currently unused by Gardena4Th4 but added for instant support should we need
      # speedgroups in the future.
      speedGroupRates = { s200g4: sg50g,
                          s100g4: sg25g,
                          s100g2: sg50g,
                          s50g2: sg25g,
                          s50g1: sg50g,
                          s40g: sg25g,
                          s25g: sg25g,
                          s10g: sg25g }

      self.createTemplate( helper, supportedFourLaneSpeeds, supportedTwoLaneSpeeds,
                           supportedOneLaneSpeeds, speedGroupRates )

      return helper.model

   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedTwoLaneSpeeds,
                        supportedOneLaneSpeeds, speedGroupRates ):
      """
      Placeholder function to be overridden by child functions.
      """
      return

   def setupLogicalPools( self, helper, allIntfs ):
      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )
      return helper.model

   def setUpSpeeds( self, helper, intfsToAdd, speed, speedGroupRates ):
      """
      Helper function to add interface modes and setup speed requirements if there
      are speedGroups.
      """
      for intf in intfsToAdd:
         helper.addIntfMode( intf, speed )
         if self.speedGroupName:
            helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                          speedGroupRates )

   def getSecondaryIntfs( self, portIntfs, lppMemberIntfs ):
      """
      Returns an ordered list of interfaces that are part of the same logical port
      pool but not the current list of interfaces.
      """
      sIntfs = [ intf for intf in lppMemberIntfs if intf not in portIntfs ]
      return sorted( sIntfs ) if sIntfs else ( None, None )

class Tomahawk450GQsfpPlusPrimary( Tomahawk450GQsfpPlusBase ):
   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedTwoLaneSpeeds,
                       supportedOneLaneSpeeds, speedGroupRates ):
      """
      Handler to create a primary port template for the BlackhawkGen3 slot pair.
      """
      # p = primary, s = secondary
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      p1, p2, p3, p4 = allIntfs
      # If set, self.logicalPortPoolDesc can be unpacked
      # pylint: disable=unbalanced-tuple-unpacking

      # Four-lane config interactions (200G-4, 100G-4, 40G-4)
      # * Applied on Primary/1; Primary/2-4 becomes inactive
      for speed in supportedFourLaneSpeeds:
         self.setUpSpeeds( helper, [ p1 ], speed, speedGroupRates[ speed ] )
         helper.inactiveIxn( p1, speed, ( p2, p3, p4 ) )

      # Two-lane config interactions (100G-2, 50G-2)
      # * Applied on Primary/1; Primary/2 becomes inactive
      # * Applied on Primary/3; Primary/4 becomes inactive
      for speed in supportedTwoLaneSpeeds:
         self.setUpSpeeds( helper, [ p1, p3 ], speed, speedGroupRates[ speed ] )
         helper.inactiveIxn( p1, speed, ( p2, ) )
         helper.inactiveIxn( p3, speed, ( p4, ) )
         helper.speedRequirement( p3, speed,
                                 { p1: ( s100g2, s50g1, s50g2, s25g, s10g ) } )

      # One-Lane config interactions (50G-1, 25G-1, 10G-1)
      # * Applied on Primary/1;
      #      Primary/2 and Primary/4 becomes active.
      #      Secondary/1 and Secondary/3 becomes inactive.
      oneLaneCompatConfigs = {
         # These entries are in the format
         #    subIntf: { parent1: ( compatible speeds ),
         #               parent2: ( compatible speeds ),
         #               ... }
         p2: { p1: [ s50g1, s25g, s10g ],
               p3: [ s50g1, s25g, s10g ] },
         p3: { p1: [ s50g1, s25g, s10g ] },
         p4: { p1: [ s50g1, s25g, s10g ],
               p3: [ s50g1, s25g, s10g ] },
      }

      for speed in supportedOneLaneSpeeds:
         self.setUpSpeeds( helper, [ p1, p2, p3, p4 ], speed,
                           speedGroupRates[ speed ] )
         for intf in self._notNoneIntfs( ( p2, p3, p4 ) ):
            helper.speedRequirement( intf, speed, oneLaneCompatConfigs[ intf ] )

      self.setupLogicalPools( helper, allIntfs )

class Tomahawk450GQsfpPlusSecondary( Tomahawk450GQsfpPlusBase ):
   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedTwoLaneSpeeds,
                       supportedOneLaneSpeeds, speedGroupRates ):
      """
      Handler to create a secondary port template for the BlackhawkGen3 slot pair.
      """
      # p = primary, s = secondary
      # secondary for the slot pair is the 'primary'
      allIntfs = [ self.lane1, self.lane3 ]
      s1, s3 = allIntfs

      # Four-lane config interactions (200G-4, 100G-4, 40G-4)
      # * Applied on Secondary/1; Secondary/3 becomes inactive
      for speed in supportedFourLaneSpeeds:
         self.setUpSpeeds( helper, [ s1 ], speed, speedGroupRates[ speed ] )
         helper.inactiveIxn( s1, speed, ( s3, ) )

      # Two-lane config interactions (100G-2, 50G-2)
      # * Applied on Secondary/1;
      # * Applied on Secondary/3;
      for speed in supportedTwoLaneSpeeds:
         self.setUpSpeeds( helper, [ s1, s3 ], speed, speedGroupRates[ speed ] )

      # One-Lane config interactions (50G-1, 25G-1, 10G-1)
      # * Applies to Secondary/1 and Secondary/3
      for speed in supportedOneLaneSpeeds:
         self.setUpSpeeds( helper, [ s1, s3 ], speed, speedGroupRates[ speed ] )
         helper.speedRequirement( s3, speed, { s1: [ s50g2, s25g, s10g ] } )

      self.setupLogicalPools( helper, allIntfs )

class Tomahawk450GBarchetta2Base( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for the interfaces of a
   TH450G connected to a Barchetta2, connected to 4 QSFP slots.

   Note
   ----
   This is a Gardena4Th4 specific case due to the physical port limitation of the
   Th4-50G and partial remap implementation.
   There are 4 ports involved with this template with very specific interactions.
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided a quad set of QSFP ports.
      The first two ports (first primary and secondary) has all the lanes show up,
      while the rest of the ports is per lane.
      There is an additional boolean to check which of the quad port we are currently
      setting intf interactions for.

      Parameters
      ----------
      serdesDomain: map of { "laneX" : "EthernetY/X" }
      """
      IntfInteractionReference.__init__( self )

      if serdesDomain.memberIntfs.get( "primary1" ):
         self.lane1 = serdesDomain.memberIntfs[ "primary1" ]
         self.lane2 = serdesDomain.memberIntfs[ "primary2" ]
         self.lane3 = serdesDomain.memberIntfs[ "primary3" ]
         self.lane4 = serdesDomain.memberIntfs[ "primary4" ]
         self.lane5 = serdesDomain.memberIntfs[ "secondary1" ]
         self.lane7 = serdesDomain.memberIntfs[ "secondary3" ]
         # Adding a boolean to do a simple check if this is the primary or secondary
         # port
         self.currFirstPorts = True
      else:
         self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
         self.lane3 = serdesDomain.memberIntfs.get( "lane3" )
         self.lane2 = None
         self.lane4 = None
         self.lane5 = None
         self.lane7 = None

   def model( self ):
      """
      This model will create all common data between all 4 different ports, and then
      depending on which port we're creating (decided during init), will determine
      how to create the interactions model.

      Note
      ----
      Due to the complexity of the interactions, we must stress that for each
      interface, we do not 'recurse more than 1 level of interactions'.
      For example, Et1/1 and Et2/1's statements are true in isolation and we do not
      explain the interactions between the two statements.
      For a specific speed on Et1/1, "it must have serdes 25g" but that may conflict
      with a different speed on Et2/1 which "must have serdes 50g".
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      assert self.logicalPortPoolDesc

      # Four-lane config interactions (100G-4, 40G-4)
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )

      # Two-lane config interactions (50G-2)
      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )

      # One-Lane config interactions (25G-1, 10G-1)
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )

      # speeds and their speed group serdes rates.
      speedGroupRates = { s100g4: sg50g,
                          s50g2: sg50g,
                          s40g: sg10g,
                          s25g: sg25g,
                          s10g: sg10g }

      self.createTemplate( helper, supportedFourLaneSpeeds, supportedOneLaneSpeeds,
                           speedGroupRates )

      return helper.model

   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedOneLaneSpeeds,
                       speedGroupRates ):
      pass

   def setupLogicalPools( self, helper, allIntfs ):
      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )
      return helper.model

   def setUpSpeeds( self, helper, intfsToAdd, speed, speedGroupRates ):
      """
      Helper function to add interface modes and setup speed requirements if there
      are speedGroups.
      """
      for intf in intfsToAdd:
         helper.addIntfMode( intf, speed )
         if self.speedGroupName:
            helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                          speedGroupRates )

   def getSecondaryIntfs( self, portIntfs, lppMemberIntfs ):
      sIntfs = [ intf for intf in lppMemberIntfs if intf not in portIntfs ]
      return sorted( sIntfs ) if sIntfs else ( None, None )

class Tomahawk450GBarchetta2FirstPorts( Tomahawk450GBarchetta2Base ):
   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedOneLaneSpeeds,
                       speedGroupRates ):
      """
      Handles the creation of the first primary and secondary port pair.
      """
      fp1, fp2, fp3, fp4 = None, None, None, None
      fs1, fs3 = None, None
      sp3 = None
      ss1 = None
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4,
                   self.lane5, self.lane7 ]
      fp1, fp2, fp3, fp4, fs1, fs3 = allIntfs
      _, sp3, ss1 = self.getSecondaryIntfs( allIntfs,
                                              self.logicalPortPoolDesc.memberIntfs )

      # For four-lane speeds
      # * Applied on Primary/1; Primary/2-4 becomes inactive
      # * Applied on Secondary/1; Secondary/2-4 becomes inactive
      for speed in supportedFourLaneSpeeds:
         self.setUpSpeeds( helper, [ fp1, fs1 ], speed, speedGroupRates[ speed ] )
         helper.inactiveIxn( fp1, speed, ( fp2, fp3, fp4 ) )
         helper.inactiveIxn( fs1, speed, ( fs3, ) )
         helper.speedRequirement( fs1, speed, { fp1: ( s100g4, s50g2, s40g ) } )

      # For two-lane speeds
      # * Applied on Primary/1; Primary/2 becomes inactive
      # * Applied on Primary/3; Primary/4 becomes inactive
      twoLaneCompatConfigs = {
         # These entries are in the format
         #    subIntf: { parent1: ( compatible speeds ),
         #               parent2: ( compatible speeds ),
         #               ... }
         fp3: { fp1: [ s50g2, s25g, s10g ] },
         fs1: { fp1: [ s100g4, s50g2, s40g ] },
         fs3: { fp1: [ s100g4, s50g2, s40g ],
                fs1: [ s50g2 ] }
      }

      self.setUpSpeeds( helper, [ fp1, fp3, fs1, fs3 ], s50g2, sg50g )
      helper.inactiveIxn( fp1, s50g2, ( fp2, ) )
      helper.inactiveIxn( fp3, s50g2, ( fp4, ) )
      for intf in [ fp3, fs1, fs3 ]:
         helper.speedRequirement( intf, s50g2, twoLaneCompatConfigs[ intf ] )

      # For single-lane speeds
      # * Applied on Primary/1; Primary/2 becomes active
      # * Applied on Primary/3; Primary/4 becomes active
      # * Secondary ports do not support single-lane speeds
      oneLaneCompatConfigs = {
         # These entries are in the format
         #    subIntf: {
         #       speed of subIntf: {
         #          parent1: ( compatible speeds ),
         #          parent2: ( compatible speeds ),
         #          ...
         #       },
         #    }
         fp2: { fp1: [ s25g, s10g ] },
         fp3: { fp1: [ s25g, s10g ] },
         fp4: { fp1: [ s25g, s10g ],
                fp3: [ s25g, s10g ] },
      }

      for speed in supportedOneLaneSpeeds:
         self.setUpSpeeds( helper, [ fp1, fp2, fp3, fp4 ], speed,
                           speedGroupRates[ speed ] )
         # Only showing configurations where all required interfaces are active.
         helper.inactiveIxn( fp1, speed, ( fs1, fs3, sp3, ss1 ) )

         for intf in [ fp2, fp3, fp4 ]:
            helper.speedRequirement( intf, speed,
                                     oneLaneCompatConfigs[ intf ] )

      self.setupLogicalPools( helper, allIntfs )

class Tomahawk450GBarchetta2SecondPrimary ( Tomahawk450GBarchetta2Base ):
   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedOneLaneSpeeds,
                       speedGroupRates ):
      """
      Handles the creation of the second primary port
      """
      # fp = first primary,
      # fs = first secondary,
      # sp = second primary,
      # ss = second secondary
      fp1 = None
      fs1 = None
      sp1, sp3 = None, None

      allIntfs = [ self.lane1, self.lane3 ]
      sp1, sp3 = allIntfs
      fp1, _, _, _, fs1, _, _ = self.getSecondaryIntfs( allIntfs,
         self.logicalPortPoolDesc.memberIntfs )

      # For four-lane speeds
      # * Applied on Primary/1; Primary/3 becomes inactive
      for speed in supportedFourLaneSpeeds:
         self.setUpSpeeds( helper, [ sp1, ], speed, speedGroupRates[ speed ] )
         helper.inactiveIxn( sp1, speed, ( sp3, ) )

      # For two-lane speeds
      self.setUpSpeeds( helper, [ sp1, sp3 ], s50g2, sg50g )
      helper.speedRequirement( sp3, s50g2, { sp1: [ s50g2 ] } )

      # For single-lane speeds
      # * Can only be applied to Primary/1 iff:
      #    - fp1 and fs1 are 40G
      #    - fp1/1, fp1/2 are single-lane speeds and fp1/3 is 50g/2 (errDisabled)
      #      * not displayed because unsure how to inform that fp1/3 is disabled*
      self.setUpSpeeds( helper, [ sp1 ], s10g, sg10g )
      helper.inactiveIxn( sp1, s10g, ( sp3, ) )
      helper.speedRequirement( sp1, s10g, { fp1: [ s100g4, s40g ],
                                            fs1: [ s100g4, s40g ] } )

      self.setupLogicalPools( helper, allIntfs )

class Tomahawk450GBarchetta2SecondSecondary( Tomahawk450GBarchetta2Base ):
   def createTemplate( self, helper, supportedFourLaneSpeeds, supportedOneLaneSpeeds,
                       speedGroupRates ):
      """
      Handles the creation of the second secondary port - only one lane.
      """
      # fp = first primary,
      # fs = first secondary,
      # sp = second primary,
      # ss = second secondary
      ss1 = None
      allIntfs = [ self.lane1 ]
      ss1 = self.lane1

      # For four-lane speeds
      for speed in supportedFourLaneSpeeds:
         self.setUpSpeeds( helper, [ ss1 ], speed, speedGroupRates[ speed ] )

      # For two-lane speeds
      self.setUpSpeeds( helper, [ ss1 ], s50g2, sg50g )

      # Doesn't support single-lane speeds (only primary does)

      self.setupLogicalPools( helper, allIntfs )

class GearboxQsfp28Pair( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for primary+secondary interfaces of a
   gearbox QSFP28 pair.

   Options (enabled by default unless specified):
     * 10G/25G retimer-mode
     * 40G/50G/100G gearbox-mode
     * Speed-group membership
     * Logical port pool membership (disabled)
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided primary-secondary pair.

      Parameters
      ----------
      primary: string representation of primary port (i.e. EthernetX, without lane)
      secondary: string representation of secondary port (again, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.primary1 = serdesDomain.memberIntfs[ "primary1" ]
      self.primary2 = serdesDomain.memberIntfs[ "primary2" ]
      self.primary3 = serdesDomain.memberIntfs[ "primary3" ]
      self.primary4 = serdesDomain.memberIntfs[ "primary4" ]
      self.secondary1 = serdesDomain.memberIntfs[ "secondary1" ]
      # For implementations that don't support 50G-2/don't use the secondary/3,
      # store None here. The None will get filtered out when populating the model.
      self.secondary3 = serdesDomain.memberIntfs.get( "secondary3" )

   def model( self ):
      """
      Create models for all member interfaces based on provided options.

      Note
      ----
      To optimize, cache models as class member as well as a dirty flag. Dirty
      flag should be set by default and in base-class's option mutators.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # Rename to save me some typing...
      p1, p2, p3, p4 = [ self.primary1, self.primary2, self.primary3, self.primary4 ]
      s1, s3 = [ self.secondary1, self.secondary3 ]
      if self.groupLinkModes.mode200GbpsFull4Lane:
         # 200G-4 interactions:
         # * Applied on M/1; M/2-4,S/1,S/3 become inactive
         helper.addIntfMode( p1, s200g4 )
         if self.speedGroupName:
            helper.speedGroupRequirement( p1, s200g4, self.speedGroupName, sg50g )
         helper.inactiveIxn( p1, s200g4,
                             self._notNoneIntfs( ( p2, p3, p4, s1, s3 ) ) )
      if self.groupLinkModes.mode100GbpsFull2Lane:
         # 100G-2 interactions:
         # * Applied on M/1; M/2,M/4,S/1,S/3 become inactive
         # * Applied on M/3; M/4 becomes inactive
         # * M/1 limits M/3 to 100G-2
         # * M/3 requires M/1 to be 100g-2
         for intf in ( p1, p3 ):
            helper.addIntfMode( intf, s100g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g2, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( p1, s100g2, self._notNoneIntfs( ( p2, p4, s1, s3 ) ) )
         helper.inactiveIxn( p3, s100g2, ( p4, ) )
         helper.speedLimitation( p1, s100g2, { p3: ( s100g2, ) } )
         helper.speedRequirement( p3, s100g2, { p1: ( s100g2, ) } )
      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4 interactions:
         # * Applied on M/1 and S/1; M/2-4 and S/3 become inactive
         # * S/1 requires M/1 to be either 100G-4/50G-2/40G
         for intf in ( p1, s1 ):
            helper.addIntfMode( intf, s100g4 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g4, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( p1, s100g4, ( p2, p3, p4 ) )
         helper.inactiveIxn( s1, s100g4, self._notNoneIntfs( ( s3, ) ) )
         helper.speedRequirement( s1, s100g4, { p1: ( s100g4, s50g2, s40g ) } )
      if self.groupLinkModes.mode50GbpsFull1Lane:
         # 50G-1 interactions:
         # * Applied on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( p1, p2, p3, p4 ):
            helper.addIntfMode( intf, s50g1 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s50g1, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( p1, s50g1, self._notNoneIntfs( ( s1, s3 ) ) )
         rates = ( s50g1, s25g, s10g, )
         helper.speedLimitation( p1, s50g1, { p3: rates } )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s50g1, { p1: rates } )
         helper.speedRequirement( p4, s50g1, { p3: rates } )
      if self.groupLinkModes.mode50GbpsFull:
         # 50G-2 interactions:
         # * Applied on M/1,3 and S/1,3; M/2,4 become inactive.
         # * M/1 limits M/3 to 50G
         # * M/3 requires M/1 to be at 50G
         # * S/1,3 require M/1 to be either 100G-4/50G-2/40G
         # * S/3 additionally requires S/1 to be 50G-2
         for intf in self._notNoneIntfs( ( p1, p3, s1, s3 ) ):
            helper.addIntfMode( intf, s50g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s50g2, self.speedGroupName,
                                             sg50g )
         helper.inactiveIxn( p1, s50g2, ( p2, p4 ) )
         helper.inactiveIxn( p3, s50g2, ( p4, ) )
         helper.speedLimitation( p1, s50g2, { p3: ( s50g2, ) } )
         helper.speedRequirement( p3, s50g2, { p1: ( s50g2, ) } )
         for intf in self._notNoneIntfs( ( s1, s3 ) ):
            helper.speedRequirement( intf, s50g2, { p1: ( s100g4, s50g2, s40g ) } )
         if s3:
            helper.speedRequirement( s3, s50g2, { s1: ( s50g2, ) } )
      if self.groupLinkModes.mode40GbpsFull:
         # 40G interactions (same as 100G-4):
         # * Applied on M/1 and S/1; M/2-4 and S/3 become inactive
         # * S/1 requires M/1 to be either 100G-4/50G-2/40G
         for intf in ( p1, s1 ):
            helper.addIntfMode( intf, s40g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s40g, self.speedGroupName, sg10g )
         helper.inactiveIxn( p1, s40g, ( p2, p3, p4 ) )
         helper.inactiveIxn( s1, s40g, self._notNoneIntfs( ( s3, ) ) )
         helper.speedRequirement( s1, s40g, { p1: ( s100g4, s50g2, s40g ) } )
      if self.groupLinkModes.mode25GbpsFull:
         # 25G interactions:
         # * Applies on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to either 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( p1, p2, p3, p4 ):
            helper.addIntfMode( intf, s25g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s25g, self.speedGroupName, sg25g )
         helper.inactiveIxn( p1, s25g, self._notNoneIntfs( ( s1, s3 ) ) )
         rates = ( s50g1, s25g, s10g )
         helper.speedLimitation( p1, s25g, { p3: rates } )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s25g, { p1: rates } )
         helper.speedRequirement( p4, s25g, { p3: rates } )
      if self.groupLinkModes.mode10GbpsFull:
         # 10G interactions (same as 25G):
         # * Applies on M/1-4; S/1,3 become inactive
         # * M/1 limits M/3 to either 50G-1/25G/10G
         # * M/2-4 require M/1 to be either 50G-1/25G/10G
         # * M/4 also requires M/3 to be either 50G-1/25G/10G
         for intf in ( p1, p2, p3, p4 ):
            helper.addIntfMode( intf, s10g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s10g, self.speedGroupName, sg10g )
         helper.inactiveIxn( p1, s10g, self._notNoneIntfs( ( s1, s3 ) ) )
         rates = ( s50g1, s25g, s10g )
         helper.speedLimitation( p1, s10g, { p3: rates } )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s10g, { p1: rates } )
         helper.speedRequirement( p4, s10g, { p3: rates } )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in self._notNoneIntfs( ( p1, p2, p3, p4, s1, s3 ) ):
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class QsfpGearboxSingleIntfSingleSlot( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for a QSFP connected to a Babbage-like
   gearbox where only the /1 interface exists. Specifically this is the case where
   there is no serdes-overlap between the two QSFP ports (i.e. no configuration
   where the primary port subsumes the secondary port).
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]

   def getSgRates( self ):
      return { s100g4: sg50g,
               s50g2: sg50g,
               s40g: sg10g,
               s25g: sg25g,
               s10g: sg10g }

   def model( self ):
      """
      The interactions here are deceptively straight-forward. At 200G, the
      primary intf must take resources from the secondary intf. For all other
      configurations, both interfaces can be active because there is no overlap
      of serdes resources.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )

      supportedModes = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedModes.append( s100g4 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedModes.append( s50g2 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedModes.append( s40g )
      if self.groupLinkModes.mode25GbpsFull:
         supportedModes.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedModes.append( s10g )

      for mode in supportedModes:
         # For all modes:
         # * Only speed-group interactions apply
         helper.addIntfMode( self.lane1, mode )
         if self.speedGroupName:
            rate = self.getSgRates()[ mode ]
            helper.speedGroupRequirement( self.lane1, mode, self.speedGroupName,
                                          rate )

      return helper.model

class DepopulatedQsfpGearboxPair( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for the interfaces of QSFP port pair
   where only the /1 interfaces for both ports exist.
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided primary-secondary pair.

      Parameters
      ----------
      primary: string representation of primary port (i.e. EthernetX, without lane)
      secondary: string representation of secondary port (again, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.primary1 = serdesDomain.memberIntfs[ "primary1" ]
      self.secondary1 = serdesDomain.memberIntfs[ "secondary1" ]

   def getSgRates( self ):
      return { s200g4: sg50g,
               s100g2: sg50g,
               s100g4: sg50g,
               s50g1: sg50g,
               s50g2: sg50g,
               s40g: sg10g,
               s25g: sg25g,
               s10g: sg10g }

   def model( self ):
      """
      The interactions here are deceptively straight-forward. At 200G, the
      primary intf must take resources from the secondary intf. For all other
      configurations, both interfaces can be active because there is no overlap
      of serdes resources.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # Rename to save me some typing...
      p1, s1 = [ self.primary1, self.secondary1 ]

      if self.groupLinkModes.mode200GbpsFull4Lane:
         # 200G-4 interactions
         # * Applied on P/1; S/1 becomes inactive
         helper.addIntfMode( p1, s200g4 )
         if self.speedGroupName:
            rate = self.getSgRates()[ s200g4 ]
            helper.speedGroupRequirement( p1, s200g4, self.speedGroupName, rate )
         helper.inactiveIxn( p1, s200g4, ( s1, ) )

      supportedSecondaryIntfModes = []
      supportedPrimaryIntfModesSans200g4 = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedPrimaryIntfModesSans200g4.append( s100g2 )
      if self.groupLinkModes.mode100GbpsFull:
         supportedPrimaryIntfModesSans200g4.append( s100g4 )
         supportedSecondaryIntfModes.append( s100g4 )
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedPrimaryIntfModesSans200g4.append( s50g1 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedPrimaryIntfModesSans200g4.append( s50g2 )
         supportedSecondaryIntfModes.append( s50g2 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedPrimaryIntfModesSans200g4.append( s40g )
         supportedSecondaryIntfModes.append( s40g )
      if self.groupLinkModes.mode25GbpsFull:
         supportedPrimaryIntfModesSans200g4.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedPrimaryIntfModesSans200g4.append( s10g )

      for mode in supportedPrimaryIntfModesSans200g4:
         # For all modes:
         # * If applicable on S/1, requires P/1 to not be 200G-4
         helper.addIntfMode( p1, mode )
         if mode in supportedSecondaryIntfModes:
            helper.addIntfMode( s1, mode )
            helper.speedRequirement( s1, mode,
                                     { p1: supportedPrimaryIntfModesSans200g4 } )
         if self.speedGroupName:
            rate = self.getSgRates()[ mode ]
            helper.speedGroupRequirement( p1, mode, self.speedGroupName, rate )
            if mode in supportedSecondaryIntfModes:
               helper.speedGroupRequirement( s1, mode, self.speedGroupName, rate )

      return helper.model

class OctalBase( IntfInteractionReference, metaclass=ABCMeta ):
   """
   Provides a general reference for single eight-lane port (e.g. OSFP, QSFP-DD).

   Options:
     * Speeds: 10G/25G/40G/50G-1/100G-1/50G-2/100G-2/
               200G-2/100G-4/200G-4/400G-4/400G-8/800G-8
     * Speed-group membership
     * Logical port pool membership (disabled by default)
   """
   def __init__( self, serdesDomain ):
      """
      Parameters
      ----------
      port: string representation of Ethernet port (i.e. EthernetX, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]
      self.lane5 = serdesDomain.memberIntfs[ "lane5" ]
      self.lane6 = serdesDomain.memberIntfs[ "lane6" ]
      self.lane7 = serdesDomain.memberIntfs[ "lane7" ]
      self.lane8 = serdesDomain.memberIntfs[ "lane8" ]

   def model( self ):
      """
      Note
      ----
      Since these interactions are primarily lane/serdes-based, many speeds will
      share the exact same interactions. For example, from an interface-activeness
      perspective, 100G-4 and 200G-4 are the same. Logic here needs to be
      revisited if that ever becomes false.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # This seems silly, but it gets me the variable names I want. eh.
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4,
                   self.lane5, self.lane6, self.lane7, self.lane8 ]
      lane1, lane2, lane3, lane4, lane5, lane6, lane7, lane8 = allIntfs
      oneLaneSpeeds = self.supportedSpeedsByLaneCount( 1 )
      twoLaneSpeeds = self.supportedSpeedsByLaneCount( 2 )
      fourLaneSpeeds = self.supportedSpeedsByLaneCount( 4 )
      eightLaneSpeeds = self.supportedSpeedsByLaneCount( 8 )
      sgRates = self.getSgRates()

      # Given that this is a very lane-based interaction group, subordinate intfs
      # can only access the serdes it needs if one of its parent intfs doesn't
      # take it. This applies to the subordinate intfs at each of its
      # configurations. I'll list them here to get them out of the way.
      compatibleParentConfigs = {
         # These entries are in the format
         #    subIntf: { parent1: ( compatible speeds ),
         #               parent2: ( compatible speeds ),
         #               ... }
         lane2: { lane1: oneLaneSpeeds },
         lane3: { lane1: twoLaneSpeeds + oneLaneSpeeds },
         lane4: { lane1: twoLaneSpeeds + oneLaneSpeeds,
                  lane3: oneLaneSpeeds },
         lane5: { lane1: fourLaneSpeeds + twoLaneSpeeds + oneLaneSpeeds },
         lane6: { lane1: fourLaneSpeeds + twoLaneSpeeds + oneLaneSpeeds,
                  lane5: oneLaneSpeeds },
         lane7: { lane1: fourLaneSpeeds + twoLaneSpeeds + oneLaneSpeeds,
                  lane5: twoLaneSpeeds + oneLaneSpeeds },
         lane8: { lane1: fourLaneSpeeds + twoLaneSpeeds + oneLaneSpeeds,
                  lane5: twoLaneSpeeds + oneLaneSpeeds,
                  lane7: oneLaneSpeeds }
      }

      # 8-lane speed interactions:
      # * Applied on lane1; all others go inactive
      for speed in eightLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1, speed, self.speedGroupName,
                                          sgRates[ speed ] )
         helper.inactiveIxn( lane1, speed, self._notNoneIntfs( ( lane2, lane3,
                                                                 lane4, lane5,
                                                                 lane6, lane7,
                                                                 lane8 ) ) )
      for speed in fourLaneSpeeds:
         # 4-lane speed interactions:
         # * Applied on lanes 1 and 5; lanes 2-4 go inactive if configured on
         #   lane 1, and lanes 6-8 do the same if configured on lane 5.
         # * Lane 5 needs Lane 1 to not use the bottom four serdes
         for intf in ( lane1, lane5 ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRates[ speed ] )
         helper.inactiveIxn( lane1, speed, self._notNoneIntfs( ( lane2,
                                                                 lane3,
                                                                 lane4 ) ) )
         helper.inactiveIxn( lane5, speed, self._notNoneIntfs( ( lane6,
                                                                 lane7,
                                                                 lane8 ) ) )
         helper.speedRequirement( lane5, speed,
                                 compatibleParentConfigs[ lane5 ] )

      for speed in twoLaneSpeeds:
         # 2-lane speed interactions:
         # * Applied on odd-numbered lanes; even-numbered lanes go inactive if
         #   preceding odd lane is configured for one of these speeds.
         # * All odd-numbered lanes must not be made inactive by one of the
         #   larger odd-numbered lanes.
         for intf in self._notNoneIntfs( ( lane1, lane3, lane5, lane7 ) ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRates[ speed ] )
         helper.inactiveIxn( lane1, speed, self._notNoneIntfs( ( lane2, ) ) )
         if lane3:
            helper.inactiveIxn( lane3, speed, self._notNoneIntfs( ( lane4, ) ) )
         helper.inactiveIxn( lane5, speed, self._notNoneIntfs( ( lane6, ) ) )
         if lane7:
            helper.inactiveIxn( lane7, speed, self._notNoneIntfs( ( lane8, ) ) )
         for intf in self._notNoneIntfs( ( lane3, lane5, lane7 ) ):
            helper.speedRequirement( intf, speed,
                                     compatibleParentConfigs[ intf ] )

      allNotNoneIntfs = self._notNoneIntfs( allIntfs )
      for speed in oneLaneSpeeds:
         # 1-lane speed interactions:
         # * All lanes greater than 1 must not be made inactive by one of the
         #   other lanes.
         for intf in allNotNoneIntfs:
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRates[ speed ] )
         for intf in allNotNoneIntfs[ 1 : ]:  # exclude lane1 in this section
            helper.speedRequirement( intf, speed,
                                     compatibleParentConfigs[ intf ] )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allNotNoneIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

   @abstractmethod
   def getSgRates( self ):
      """
      This method returns a dictionary mapping speed tokens to speed-group rate
      tokens for every supported speed
      """

   @abstractmethod
   def supportedSpeedsByLaneCount( self, numLane ):
      """
      This method returns a list of speed tokens with specified lane count
      and which are supported in groupLinkModes
      """

class OctalBlackhawk( OctalBase ):
   """
   Provides reference for single eight-lane port for Blackhawk.

   Options:
     * Speeds: 10G/25G/40G/50G-1/50G-2/100G-2/100G-4/200G-4/200G-8/400G-8
     * Speed-group membership
     * Logical port pool membership (disabled by default)
     * Blackhawk core autoneg fec restrictions (disabled by default)

   Members:
   blackhawkAutoFecRequirement : dict mapping secondary intfId to primary
      If set, reference will use information to populate blackhawk restrictions.
   """
   def __init__( self, serdesDomain ):
      OctalBase.__init__( self, serdesDomain )
      self.blackhawkAutoFecRequirement = {}

   def model( self ):
      # the returned model from parent class is not the final one
      model = super().model()
      helper = IntfInteractionsModelHelper( self.groupLinkModes, model )
      if self.blackhawkAutoFecRequirement:
         for intf, primaryIntf in self.blackhawkAutoFecRequirement.items():
            helper.blackhawkAutoFecRequirement( intf, primaryIntf )

      return helper.model

   def getSgRates( self ):
      return { s400g8: sg50g,
               s200g4: sg50g,
               s100g4: sg25g,
               s40g: sg10g,
               s100g2: sg50g,
               s50g2: sg25g,
               s50g1: sg50g,
               s25g: sg25g,
               s10g: sg10g }

   def supportedSpeedsByLaneCount( self, numLane ):
      if numLane == 1:
         speeds = ( s10g, s25g, s50g1 )
      elif numLane == 2:
         speeds = ( s50g2, s100g2 )
      elif numLane == 4:
         speeds = ( s40g, s100g4, s200g4 )
      elif numLane == 8:
         return ( s400g8, )
      else:
         raise ValueError( "only 1, 2, 4, 8 lanes are supported" )

      return [ speed for speed in speeds
               if self.groupLinkModes.hasSpeed( toEthSpeed[ speed ] ) ]

class OctalPeregrine( OctalBase ):
   def __init__( self, serdesDomain ):
      OctalBase.__init__( self, serdesDomain )

   def getSgRates( self ):
      return { s800g8: sg100g,
               s400g8: sg100g,
               s400g4: sg100g,
               s200g4: sg100g,
               s100g4: sg25g,
               s200g2: sg100g,
               s100g2: sg100g,
               s50g2: sg25g,
               s100g1: sg100g,
               s50g1: sg100g,
               s25g: sg25g,
               s10g: sg25g }

   def supportedSpeedsByLaneCount( self, numLane ):
      if numLane == 1:
         speeds = ( s10g, s25g, s50g1, s100g1 )
      elif numLane == 2:
         speeds = ( s50g2, s100g2, s200g2 )
      elif numLane == 4:
         speeds = ( s40g, s100g4, s200g4, s400g4 )
      elif numLane == 8:
         return ( s400g8, s800g8 )
      else:
         raise ValueError( "only 1, 2, 4, 8 lanes are supported" )

      return [ speed for speed in speeds
               if self.groupLinkModes.hasSpeed( toEthSpeed[ speed ] ) ]

class Octal2x400G( OctalBase ):
   """
   Provides reference for octal port capable of 100G-1 lanes but without an
   800G mac.

   Options:
     * NRZ speeds: 10G/25G/40G/50G-2/100G-4/200G-8
     * PAM4 speeds: 50G-1/100G-1/100G-2/200G-2/200G-4/400G-4/400G-8
     * Speed-group membership
     * Logical port pool membership (disabled by default)
   """
   def __init__( self, serdesDomain ):
      OctalBase.__init__( self, serdesDomain )

   def getSgRates( self ):
      return { s400g8: sg100g,
               s400g4: sg100g,
               s200g4: sg50g,
               s100g4: sg25g,
               s40g: sg10g,
               s200g2: sg100g,
               s100g2: sg50g,
               s50g2: sg25g,
               s100g1: sg100g,
               s50g1: sg50g,
               s25g: sg25g,
               s10g: sg10g }

   def supportedSpeedsByLaneCount( self, numLane ):
      if numLane == 1:
         speeds = ( s10g, s25g, s50g1, s100g1 )
      elif numLane == 2:
         speeds = ( s50g2, s100g2, s200g2 )
      elif numLane == 4:
         speeds = ( s40g, s100g4, s200g4, s400g4 )
      elif numLane == 8:
         return ( s400g8, )
      else:
         raise ValueError( "only 1, 2, 4, 8 lanes are supported" )

      return [ speed for speed in speeds
               if self.groupLinkModes.hasSpeed( toEthSpeed[ speed ] ) ]

class AdvancedSfp( IntfInteractionReference ):
   """
   Provides reference model for an SFP port that may be a member of a speed-group.

   This reference model can be used for any SFP port, since it includes speeds up
   to 50G-1. For a no-frills SFP+ port, one can use the Sfp class, which
   unconditionally declares an SFP port with 10G/1G/100M and no interactions with
   other ports.

   This class could be consolidated with the Sfp class in the future. Whenever that
   happens, it may be also worth revisiting how <10G speeds are represented.

   Options:
     * 10G/25G/50G-1 speeds
     * If the port supports 100M/1G, it will be reflected in the text render as
       part of the 10G interactions
     * Speed-group membership
     * Logical port pool membership (disabled by default)
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane0" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # For all speeds, there should be no interactions with other interfaces. We
      # will still list out all of the speeds. If the interface is part of a speed
      # group, the appropriate setting will need to be filled in for each speed.
      port = self.port  # Saving character count
      if self.groupLinkModes.mode50GbpsFull1Lane:
         helper.addIntfMode( self.port, s50g1 )
         if self.speedGroupName:
            helper.speedGroupRequirement( port, s50g1, self.speedGroupName, sg50g )
      if self.groupLinkModes.mode25GbpsFull:
         helper.addIntfMode( self.port, s25g )
         if self.speedGroupName:
            helper.speedGroupRequirement( port, s25g, self.speedGroupName, sg25g )
      if self.groupLinkModes.mode10GbpsFull:
         helper.addIntfMode( self.port, s10g )
         if self.speedGroupName:
            helper.speedGroupRequirement( port, s10g, self.speedGroupName, sg10g )
      if self.groupLinkModes.mode1GbpsFull or self.groupLinkModes.mode100MbpsFull:
         helper.intfData( port ).includesSlowerSpeeds10gIs( True )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         # If set, self.logicalPortPoolDesc will be an iterable
         # pylint: disable=not-an-iterable
         helper.logicalPortDescription( self.port, *self.logicalPortPoolDesc )

      return helper.model

class Sfp( IntfInteractionReference ):
   """
   Provides reference model for single SFP+ port

   Options:
    * 100M/1G/10G/25G speeds
    * That's it
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane0" ]

   def model( self ):
      # TODO: this assumes a very simple SFP+ port, since that is the only type
      # of SFP port exposed to the interactions CLI at the moment.
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      if self.groupLinkModes.mode25GbpsFull:
         helper.addIntfMode( self.port, s25g )
      helper.addIntfMode( self.port, s10g )
      helper.addIntfMode( self.port, s1g )
      helper.addIntfMode( self.port, s100m )
      helper.intfData( self.port ).includesSlowerSpeeds10gIs( True )
      return helper.model

class Dsfp( IntfInteractionReference ):
   """
   Provides reference model for single DSFP port

   Options:
    * 10G/25G/50G-2/50G-1/100G-2
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2 ]
      lane1, lane2 = allIntfs

      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )
      for speed in supportedTwoLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, ) )

      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.addIntfMode( lane2, speed )
         helper.speedRequirement( lane2, speed, { lane1: ( s50g1, s25g, s10g ) } )

      return helper.model

class SfpDd( Dsfp ):
   """
   Provides reference model for single SFP-DD port

   Options:
    * 10G/25G/50G-2/50G-1/100G-2
   """
   pass # pylint: disable=unnecessary-pass

class BasicQsfp( IntfInteractionReference ):
   """
   Provides reference model for single QSFP port, not connected to any gearbox.

   Options:
    * (10M/100M/1G/)10G/25G/40G/50G-2/100G-4
    * Logical port pool membership (disabled by default)

   Regarding <10G speeds: these will only affect the output for lane1. There's
   no QSFP module that we support that would allow <10G speeds to run on
   lanes2-4.

   Note
   ----
   Options can be enabled/disabled to represent QSFP+ or QSFP28
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      lane1, lane2, lane3, lane4 = allIntfs
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      for speed in supportedFourLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
      if self.groupLinkModes.mode50GbpsFull:
         helper.addIntfMode( lane1, s50g2 )
         helper.addIntfMode( lane3, s50g2 )
         helper.inactiveIxn( lane1, s50g2, ( lane2, lane4 ) )
         helper.inactiveIxn( lane3, s50g2, ( lane4, ) )
         helper.speedLimitation( lane1, s50g2, { lane3: ( s50g2, ) } )
         helper.primaryIntfSpeedRequirement( lane3, lane1, s50g2 )
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
         for intf in allIntfs[ 1 : ]:
            helper.primaryIntfSpeedRequirement( intf, lane1, speed )
         if len( supportedOneLaneSpeeds ) > 1:
            # Weird conditional. For QSFP28s, since all the ones we have are
            # connected to Falcon-cores, /1 at a 1-lane speed limits the other three
            # to the same speed, so the template describes as such. For QSFP+, when
            # /1 is at 10G, the others are naturally also at 10G. However, because
            # /2-4 in this case can only ever be 10G, they are not "limited" to it.
            # I made the call in the past to omit the speed-limitation line here,
            # and this conditional is accounting for that.
            helper.speedLimitation( lane1, speed,
                                    { intf: ( speed, ) for intf in allIntfs[ 1: ] } )
      # As mentioned earlier, only lane1 needs an indicator that says its <10G
      # interactions are identical to its 10G interactions.
      helper.intfData( lane1 ).includesSlowerSpeeds10gIs( True )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class QsfpMixedRate( IntfInteractionReference ):
   """
   Provides reference model for single QSFP port that supports mixed rates (e.g.
   mix of 25G and 10G across the four interfaces)

   Options:
    * 10G/25G/40G/50G-2/100G-4

   Members:
   blackhawkAutoFecRequirement : dict mapping secondary intfId to primary
      If set, reference will use information to populate blackhawk restrictions.

   Note
   ----
   Options can be enabled/disabled to represent QSFP+ or QSFP28
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.blackhawkAutoFecRequirement = {}
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      lane1, lane2, lane3, lane4 = allIntfs
      supportedFourLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )
      for speed in supportedFourLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
      # Main distinction between this template and BasicQsfp is that this template
      # allows mixing non-four-lane speeds across the interfaces.
      if self.groupLinkModes.mode50GbpsFull:
         helper.addIntfMode( lane1, s50g2 )
         helper.addIntfMode( lane3, s50g2 )
         helper.inactiveIxn( lane1, s50g2, ( lane2, ) )
         helper.inactiveIxn( lane3, s50g2, ( lane4, ) )
         helper.speedRequirement( lane3, s50g2, { lane1: ( s50g2, s25g, s10g ) } )
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
         helper.speedRequirement( lane2, speed, { lane1: ( s25g, s10g ) } )
         helper.speedRequirement( lane3, speed, { lane1: ( s50g2, s25g, s10g ) } )
         helper.speedRequirement( lane4, speed, { lane1: ( s50g2, s25g, s10g ),
                                                  lane3: ( s25g, s10g ) } )
      helper.intfData( lane1 ).includesSlowerSpeeds10gIs( True )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      if self.blackhawkAutoFecRequirement:
         for intf, primaryIntf in self.blackhawkAutoFecRequirement.items():
            helper.blackhawkAutoFecRequirement( intf, primaryIntf )

      return helper.model

class Qsfp56MixedRate( IntfInteractionReference ):
   """
   Reference model for a QSFP56 port (4 lanes of up to 50G-1) that allows
   mixing rates in breakout configurations (e.g. 50-25-25-10)

   Options:
    * 10G/25G/40G/50G-2/50G-1/100G-4/100G-2/200G-4
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.blackhawkAutoFecRequirement = {}
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane2 = serdesDomain.memberIntfs[ "lane2" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]
      self.lane4 = serdesDomain.memberIntfs[ "lane4" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      allIntfs = [ self.lane1, self.lane2, self.lane3, self.lane4 ]
      lane1, lane2, lane3, lane4 = allIntfs

      # Use static PLL-rate mapping
      def _sgRateHelper( _speed ):
         if _speed in ( s50g1, s100g2, s200g4 ):
            return sg50g
         elif _speed in ( s25g, s50g2, s100g4 ):
            return sg25g
         else:
            return sg10g

      # Generate buckets of supported speeds by consumed lanes
      supportedFourLaneSpeeds = []  # 4-lane
      if self.groupLinkModes.mode200GbpsFull4Lane:
         supportedFourLaneSpeeds.append( s200g4 )
      if self.groupLinkModes.mode100GbpsFull:
         supportedFourLaneSpeeds.append( s100g4 )
      if self.groupLinkModes.mode40GbpsFull:
         supportedFourLaneSpeeds.append( s40g )

      supportedTwoLaneSpeeds = []  # 2-lane
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )

      supportedOneLaneSpeeds = []  # 1-lane
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )

      # Populate interactions for each supported speed bucket
      for speed in supportedFourLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, lane3, lane4 ) )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1,
                                          speed,
                                          self.speedGroupName,
                                          _sgRateHelper( speed ) )

      for speed in supportedTwoLaneSpeeds:
         helper.addIntfMode( lane1, speed )
         helper.addIntfMode( lane3, speed )
         helper.inactiveIxn( lane1, speed, ( lane2, ) )
         helper.inactiveIxn( lane3, speed, ( lane4, ) )
         helper.speedRequirement( lane3, speed,
                                  { lane1: tuple( supportedTwoLaneSpeeds +
                                                  supportedOneLaneSpeeds ) } )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1,
                                          speed,
                                          self.speedGroupName,
                                          _sgRateHelper( speed ) )
            helper.speedGroupRequirement( lane3,
                                          speed,
                                          self.speedGroupName,
                                          _sgRateHelper( speed ) )

      for speed in supportedOneLaneSpeeds:
         for intf in allIntfs:
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf,
                                             speed,
                                             self.speedGroupName,
                                             _sgRateHelper( speed ) )
         helper.speedRequirement( lane2, speed,
                                  { lane1: tuple( supportedOneLaneSpeeds ) } )
         helper.speedRequirement( lane3, speed,
                                  { lane1: tuple( supportedTwoLaneSpeeds +
                                                  supportedOneLaneSpeeds ) } )
         helper.speedRequirement( lane4, speed,
                                  { lane1: tuple( supportedTwoLaneSpeeds +
                                                  supportedOneLaneSpeeds ),
                                    lane3: tuple( supportedOneLaneSpeeds ) } )
      helper.intfData( lane1 ).includesSlowerSpeeds10gIs( True )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      if self.blackhawkAutoFecRequirement:
         for intf, primaryIntf in self.blackhawkAutoFecRequirement.items():
            helper.blackhawkAutoFecRequirement( intf, primaryIntf )

      return helper.model

class Qsfp40gOnly( IntfInteractionReference ):
   """
   Provides reference model for a single 40G-only QSFP port. This QSFP port has
   no associated subordinate interfaces (i.e. /2-4 don't exist).
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      # There was a point where 40G-only ports did not include the /1 in the intfId.
      # For more recent products, we include the /1.
      lane = "lane1"
      if lane not in serdesDomain.memberIntfs:
         lane = "lane0"
         assert lane in serdesDomain.memberIntfs
      self.port = serdesDomain.memberIntfs[ lane ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      helper.addIntfMode( self.port, s40g )
      return helper.model

class Sesto2Pair( IntfInteractionReference ):
   """
   Provides reference model for QSFP port pair connected to a Sesto2
   gearbox
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.primary1 = serdesDomain.memberIntfs[ "primary1" ]
      self.primary2 = serdesDomain.memberIntfs[ "primary2" ]
      self.primary3 = serdesDomain.memberIntfs[ "primary3" ]
      self.primary4 = serdesDomain.memberIntfs[ "primary4" ]
      self.secondary1 = serdesDomain.memberIntfs[ "secondary1" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      p1, p2, p3, p4 = [ self.primary1, self.primary2, self.primary3, self.primary4 ]
      s1 = self.secondary1
      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4 interactions:
         # * Applied on M/1; M/2-4,S/1 become inactive
         helper.addIntfMode( p1, s100g4 )
         helper.inactiveIxn( p1, s100g4, ( p2, p3, p4, s1 ) )
      if self.groupLinkModes.mode50GbpsFull:
         # 50G-2 interactions:
         # * Applied on M/1,M/3; M/2,M/4,S/1 become inactive
         # * M/3 is limited to 50G
         for intf in ( p1, p3 ):
            helper.addIntfMode( intf, s50g2 )
         helper.inactiveIxn( p1, s50g2, ( p2, p4, s1 ) )
         helper.inactiveIxn( p3, s50g2, ( p4, ) )
         helper.speedLimitation( p1, s50g2, { p3: ( s50g2, ) } )
         helper.primaryIntfSpeedRequirement( p3, p1, s50g2 )
      if self.groupLinkModes.mode40GbpsFull:
         # 40G interactions:
         # * Applied on M/1,S/1; M/2-4 become inactive
         for intf in ( p1, s1 ):
            helper.addIntfMode( intf, s40g )
         helper.inactiveIxn( p1, s40g, ( p2, p3, p4 ) )
         helper.primaryIntfSpeedRequirement( s1, p1, s40g )
      # 10G/25G interactions are identical, so group them together
      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         # 10G/25G interactions:
         # * Applies on M/1-4; S/1 becomes inactive
         # * M/2-4 require M/1 to be 10G/25G (tightly coupled)
         for intf in ( p1, p2, p3, p4 ):
            helper.addIntfMode( intf, speed )
         helper.inactiveIxn( p1, speed, ( s1, ) )
         helper.speedLimitation( p1, speed, { sub: ( speed, )
                                              for sub in ( p2, p3, p4 ) } )
         for sub in ( p2, p3, p4 ):
            helper.primaryIntfSpeedRequirement( sub, p1, speed )

      return helper.model

class BabbageEnigmaQsfp( IntfInteractionReference ):
   """
   Provides reference model for a single QSFP port that has a Babbage and an Enigma
   in its phy tuple. Despite there being a Babbage, the limitations of the Enigma
   make it so that a QSFP port will only ever take two switch-chip serdes, no matter
   its configuration. Furthermore, these QSFP ports can not be broken out. As a
   result, the associated interface has no interactions with other interfaces.
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane1" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      helper.addIntfMode( self.port, s100g4 )
      helper.addIntfMode( self.port, s40g )
      return helper.model

class SparrowFourWayGearboxQsfp( IntfInteractionReference ):
   """
   Provides reference model for a single QSFP port that has a Sparrow-FourWayGearbox
   in its phy tuple. Sparrow chip, in this hardware configuration only
   supports 100G-4 without any possibilities for a break out. As a
   result, the associated interface has no interactions with other interfaces.
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane1" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      helper.addIntfMode( self.port, s100g4 )
      return helper.model

class MuxQsfp56Pair( IntfInteractionReference ):
   """
   Provides interactions CLI model reference for primary+secondary interfaces of
   a QSFP-DD & QSFP56 mux pair (e.g. Millenio mux port pairs on Brownsville and
   Barchetta2 mux port pairs on Brownsville2)

   Options (enabled by default unless specified):
     * 400G mux mode
     * 40G/100G/200G retimer mode
     * Speed-group membership
     * Hardware port-group membership
   """
   def __init__( self, serdesDomain ):
      """
      Instantiate interaction models provided primary-secondary pair.

      Parameters
      ----------
      primary: string representation of primary port (i.e. EthernetX, without lane)
      secondary: string representation of secondary port (again, without lane)
      """
      IntfInteractionReference.__init__( self )
      self.primary1 = serdesDomain.memberIntfs[ "primary1" ]
      self.primary2 = serdesDomain.memberIntfs[ "primary2" ]
      self.primary3 = serdesDomain.memberIntfs[ "primary3" ]
      self.primary4 = serdesDomain.memberIntfs[ "primary4" ]
      self.primary5 = serdesDomain.memberIntfs[ "primary5" ]
      self.primary6 = serdesDomain.memberIntfs[ "primary6" ]
      self.primary7 = serdesDomain.memberIntfs[ "primary7" ]
      self.primary8 = serdesDomain.memberIntfs[ "primary8" ]
      self.secondary1 = serdesDomain.memberIntfs[ "secondary1" ]
      self.secondary2 = serdesDomain.memberIntfs[ "secondary2" ]
      self.secondary3 = serdesDomain.memberIntfs[ "secondary3" ]
      self.secondary4 = serdesDomain.memberIntfs[ "secondary4" ]

   def model( self ):
      """
      Create models for all member interfaces based on provided options.

      Note
      ----
      To optimize, cache models as class member as well as a dirty flag. Dirty
      flag should be set by default and in base-class's option mutators.
      """
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      p1, p2, p3, p4, p5, p6, p7, p8 = [
         self.primary1, self.primary2, self.primary3, self.primary4,
         self.primary5, self.primary6, self.primary7, self.primary8 ]
      s1, s2, s3, s4 = [
         self.secondary1, self.secondary2, self.secondary3, self.secondary4 ]
      # The two hardware port-group configs are 1x-qsfp-dd* and 2x-qsfp-56*
      # pylint: disable=not-an-iterable
      for mode in self.hardwarePortModes:
         if "1x-qsfp-dd" in mode:
            qsfpDdMode = mode
         if "2x-qsfp-56" in mode:
            qsfp56Mode = mode
      if self.groupLinkModes.mode400GbpsFull8Lane:
         # 400G-8 interactions:
         # * Applied on M/1; M/2-8 become inactive
         helper.addIntfMode( p1, s400g8 )
         if self.speedGroupName:
            helper.speedGroupRequirement( p1, s400g8,
                                          self.speedGroupName, sg50g )
         helper.inactiveIxn( p1, s400g8, ( p2, p3, p4, p5, p6, p7, p8 ) )
         helper.hardwarePortGroupRequirement( p1, s400g8,
                                              self.hardwarePortGroup, qsfpDdMode )
      if self.groupLinkModes.mode200GbpsFull4Lane:
         # 200G-4 interactions:
         # * Applied on M1; M/2-4 become inactive
         # * Applied on M5; M/6-8 become inactive
         #   and M5 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * Applied on S1; S/2-4 become inactive
         for intf in ( p1, p5, s1 ):
            helper.addIntfMode( intf, s200g4 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s200g4,
                                             self.speedGroupName, sg50g )
         helper.inactiveIxn( p1, s200g4, ( p2, p3, p4 ) )
         helper.inactiveIxn( p5, s200g4, ( p6, p7, p8 ) )
         helper.speedRequirement(
            p5, s200g4, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p5, s200g4,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( s1, s200g4, ( s2, s3, s4 ) )
         helper.hardwarePortGroupRequirement( s1, s200g4,
                                              self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4 interactions:
         # * Applied on M1; M/2-4 become inactive
         # * Applied on S1; S/2-4 become inactive
         for intf in ( p1, s1 ):
            helper.addIntfMode( intf, s100g4 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g4,
                                             self.speedGroupName, sg25g )
         helper.inactiveIxn( p1, s100g4, ( p2, p3, p4 ) )
         helper.inactiveIxn( s1, s100g4, ( s2, s3, s4 ) )
         helper.hardwarePortGroupRequirement( s1, s100g4,
                                              self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode100GbpsFull2Lane:
         # 100G-2 interactions:
         # * Applied on M1/M3/M5/M7/S1/S3;
         #      M2/M4/M6/M8/S2/S4 become inactive, respectively
         # * M5 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * M7 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * S3 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         for intf in ( p1, p3, p5, p7, s1, s3 ):
            helper.addIntfMode( intf, s100g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s100g2,
                                             self.speedGroupName, sg50g )
         helper.inactiveIxn( p1, s100g2, ( p2, ) )
         helper.inactiveIxn( p3, s100g2, ( p4, ) )
         helper.inactiveIxn( p5, s100g2, ( p6, ) )
         helper.speedRequirement(
            p5, s100g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p5, s100g2,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( p7, s100g2, ( p8, ) )
         helper.speedRequirement(
            p7, s100g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p7, s100g2,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( s1, s100g2, ( s2, ) )
         helper.hardwarePortGroupRequirement( s1, s100g2,
                                              self.hardwarePortGroup, qsfp56Mode )
         helper.inactiveIxn( s3, s100g2, ( s4, ) )
         helper.speedRequirement(
            s3, s100g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( s3, s100g2,
                                              self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode50GbpsFull:
         # 50G-2 interactions:
         # * Applied on M1/M3/M5/M7/S1/S3;
         #      M2/M4/M6/M8/S2/S4 become inactive, respectively
         # * M3 requires M1 to be at 50G-2
         # * M5 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * M7 requires M5 to be at 50G-2
         # * M7 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * S3 requires S1 to be at 50G-2
         # * S3 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         for intf in ( p1, p3, p5, p7, s1, s3 ):
            helper.addIntfMode( intf, s50g2 )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s50g2,
                                             self.speedGroupName, sg25g )
         helper.inactiveIxn( p1, s50g2, ( p2, ) )
         helper.inactiveIxn( p3, s50g2, ( p4, ) )
         helper.speedRequirement( p3, s50g2, { p1: ( s50g2, ) } )
         helper.inactiveIxn( p5, s50g2, ( p6, ) )
         helper.speedRequirement(
            p5, s50g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p5, s50g2,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( p7, s50g2, ( p8, ) )
         helper.speedRequirement( p7, s50g2, { p5: ( s50g2, ) } )
         helper.speedRequirement(
            p7, s50g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p7, s50g2,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( s1, s50g2, ( s2, ) )
         helper.hardwarePortGroupRequirement( s1, s50g2,
                                              self.hardwarePortGroup, qsfp56Mode )
         helper.inactiveIxn( s3, s50g2, ( s4, ) )
         helper.speedRequirement( s3, s50g2, { s1: ( s50g2, ) } )
         helper.speedRequirement(
            s3, s50g2, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( s3, s50g2,
                                              self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode50GbpsFull1Lane:
         # 50G-1 interactions:
         # * Applied on M/1-4,6,8 and S/2,4
         # * M/2-4 require M/1 to be 50G-1
         # * M/6,8 require M/5 to be 50G-1
         # * S/2,4 require S/1 to be 50G-1
         for intf in ( p1, p2, p3, p4, p6, p8, s2, s4 ):
            helper.addIntfMode( intf, s50g1 )
            if self.speedGroupName:
               helper.speedGroupRequirement(
                  intf, s50g1, self.speedGroupName, sg50g )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s50g1, { p1: ( s50g1, ) } )
         for intf in ( p6, p8 ):
            helper.speedRequirement( intf, s50g1, { p5: ( s50g1, ) } )
            helper.hardwarePortGroupRequirement( intf, s50g1,
                                                 self.hardwarePortGroup, qsfpDdMode )
         for intf in ( s2, s4 ):
            helper.speedRequirement( intf, s50g1, { s1: ( s50g1, ) } )
            helper.hardwarePortGroupRequirement( intf, s50g1,
                                                 self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode40GbpsFull:
         # 40G interactions:
         # * Applied on M1; M/2-4 become inactive
         # * Applied on M5; M/6-8 become inactive
         #   and M5 requires M1 to be either 200G-4/100G-2/100G-4/50G-2/40G
         # * Applied on S1; S/2-4 become inactive
         for intf in ( p1, p5, s1 ):
            helper.addIntfMode( intf, s40g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s40g,
                                             self.speedGroupName, sg10g )
         helper.inactiveIxn( p1, s40g, ( p2, p3, p4 ) )
         helper.inactiveIxn( p5, s40g, ( p6, p7, p8 ) )
         helper.speedRequirement(
            p5, s40g, { p1: ( s200g4, s100g2, s100g4, s50g2, s40g ) } )
         helper.hardwarePortGroupRequirement( p5, s40g,
                                              self.hardwarePortGroup, qsfpDdMode )
         helper.inactiveIxn( s1, s40g, ( s2, s3, s4 ) )
         helper.hardwarePortGroupRequirement( s1, s40g,
                                              self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode25GbpsFull:
         # 25G interactions:
         # * Applied on M/1-4,6,8 and S/2,4
         # * M/2-4 require M/1 to be 25G
         # * M/6,8 require M/5 to be 25G
         # * S/2,4 require S/1 to be 25G
         for intf in ( p1, p2, p3, p4, p6, p8, s2, s4 ):
            helper.addIntfMode( intf, s25g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s25g, self.speedGroupName, sg25g )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s25g, { p1: ( s25g, ) } )
         for intf in ( p6, p8 ):
            helper.speedRequirement( intf, s25g, { p5: ( s25g, ) } )
            helper.hardwarePortGroupRequirement( intf, s25g,
                                                 self.hardwarePortGroup, qsfpDdMode )
         for intf in ( s2, s4 ):
            helper.speedRequirement( intf, s25g, { s1: ( s25g, ) } )
            helper.hardwarePortGroupRequirement( intf, s25g,
                                                 self.hardwarePortGroup, qsfp56Mode )
      if self.groupLinkModes.mode10GbpsFull:
         # 10G interactions:
         # * Applied on M/1-4,6,8 and S/2,4
         # * M/2-4 require M/1 to be 10G
         # * M/6,8 require M/5 to be 10G
         # * S/2,4 require S/1 to be 10G
         for intf in ( p1, p2, p3, p4, p6, p8, s2, s4 ):
            helper.addIntfMode( intf, s10g )
            if self.speedGroupName:
               helper.speedGroupRequirement( intf, s10g, self.speedGroupName, sg10g )
         for intf in ( p2, p3, p4 ):
            helper.speedRequirement( intf, s10g, { p1: ( s10g, ) } )
         for intf in ( p6, p8 ):
            helper.speedRequirement( intf, s10g, { p5: ( s10g, ) } )
            helper.hardwarePortGroupRequirement( intf, s10g,
                                                 self.hardwarePortGroup, qsfpDdMode )
         for intf in ( s2, s4 ):
            helper.speedRequirement( intf, s10g, { s1: ( s10g, ) } )
            helper.hardwarePortGroupRequirement( intf, s10g,
                                                 self.hardwarePortGroup, qsfp56Mode )
      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in ( p1, p2, p3, p4, p5, p6, p7, p8, s1, s2, s3, s4 ):
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class Rj45( IntfInteractionReference ):
   """
   Provides reference model for single RJ45 (fixed BASE-T) port

   Options:
    * Fixed BASE-T ports across our products (currently) have no interactions with
      other interfaces. We can blindly advertise no interactions, and we will have
      update this template if that ever happens.
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane0" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      if self.groupLinkModes.mode100MbpsFull:
         helper.addIntfMode( self.port, s100m )
      if self.groupLinkModes.mode1GbpsFull:
         helper.addIntfMode( self.port, s1g )
      if self.groupLinkModes.mode10GbpsFull:
         helper.addIntfMode( self.port, s10g )
      if self.groupLinkModes.mode1GbpsFull or self.groupLinkModes.mode100MbpsFull:
         helper.intfData( self.port ).includesSlowerSpeeds10gIs( True )

      return helper.model

class SpeedGroupRj45( IntfInteractionReference ):
   """
   Provides reference model for single RJ45 (fixed BASE-T) port that is backed by
   a core that uses speed-groups

   Note:
   This situation is somewhat of an oddity. SierravilleT is one instance where
   a Blackhawk core drives a QSFP port and a number of BASE-T ports. However, there
   are efforts going on now to make that speed-group SW transparent, after which
   what gets rendered here will be identical to the standard Rj45 template.
   """
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.port = serdesDomain.memberIntfs[ "lane0" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      if self.groupLinkModes.mode10GbpsFull:
         helper.addIntfMode( self.port, s10g )
         if self.speedGroupName:
            helper.speedGroupRequirement( self.port, s10g,
                                          self.speedGroupName, sg10g )
      return helper.model

class BlancoSecondary( IntfInteractionReference ):
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.lane1 = serdesDomain.memberIntfs[ "lane1" ]
      self.lane3 = serdesDomain.memberIntfs[ "lane3" ]

   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      # Only Secondary/1 and Secondary/3 are active when blanco is not installed.
      # The capabilities of both interfaces are moved to lane5 and lane7 of
      # the corresponding primary port when we insert a Blanco module.
      allIntfs = [ self.lane1, self.lane3 ]
      lane1, lane3 = allIntfs

      if self.groupLinkModes.mode200GbpsFull4Lane:
         # 200G-4 interactions:
         # * Applied on Secondary/1; Secondary/3 becomes inactive
         helper.addIntfMode( lane1, s200g4 )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1, s200g4,
                                          self.speedGroupName, sg50g )
         helper.inactiveIxn( lane1, s200g4, ( lane3, ) )

      supportedTwoLaneSpeeds = []
      if self.groupLinkModes.mode100GbpsFull2Lane:
         supportedTwoLaneSpeeds.append( s100g2 )
      if self.groupLinkModes.mode50GbpsFull:
         supportedTwoLaneSpeeds.append( s50g2 )
      for speed in supportedTwoLaneSpeeds:
         for intf in ( lane1, lane3 ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s100g2: sg50g,
                          s50g2: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         helper.speedRequirement( lane3, speed, 
                                  { lane1: ( s10g, s25g, s50g1, s50g2, s100g2 ) } )

      if self.groupLinkModes.mode100GbpsFull:
         # 100G-4
         # * Applied on Secondary/1; Secondary/3 becomes inactive
         helper.addIntfMode( lane1, s100g4 )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1, s100g4,
                                          self.speedGroupName, sg25g )
         helper.inactiveIxn( lane1, s100g4, ( lane3, ) )

      supportedOneLaneSpeeds = []
      if self.groupLinkModes.mode50GbpsFull1Lane:
         supportedOneLaneSpeeds.append( s50g1 )
      if self.groupLinkModes.mode25GbpsFull:
         supportedOneLaneSpeeds.append( s25g )
      if self.groupLinkModes.mode10GbpsFull:
         supportedOneLaneSpeeds.append( s10g )
      for speed in supportedOneLaneSpeeds:
         for intf in ( lane1, lane3 ):
            helper.addIntfMode( intf, speed )
            if self.speedGroupName:
               sgRate = { s50g1: sg50g,
                          s25g: sg25g,
                          s10g: sg25g }[ speed ]
               helper.speedGroupRequirement( intf, speed, self.speedGroupName,
                                             sgRate )
         helper.speedRequirement( lane3, speed, 
                                  { lane1: ( s10g, s25g, s50g1, s50g2, s100g2 ) } )

      if self.groupLinkModes.mode40GbpsFull:
         # 40G-4
         # * Applied on Secondary/1; Secondary/3 becomes inactive
         helper.addIntfMode( lane1, s40g )
         if self.speedGroupName:
            helper.speedGroupRequirement( lane1, s40g,
                                          self.speedGroupName, sg25g )
         helper.inactiveIxn( lane1, s40g, ( lane3, ) )

      # Populate logical port description, if applicable
      if self.logicalPortPoolDesc:
         for intf in allIntfs:
            # If set, self.logicalPortPoolDesc will be an iterable
            # pylint: disable=not-an-iterable
            helper.logicalPortDescription( intf, *self.logicalPortPoolDesc )

      return helper.model

class NoPhyTemplate( IntfInteractionReference ):
   def __init__( self, serdesDomain ):
      IntfInteractionReference.__init__( self )
      self.allIntfs = self._notNoneIntfs( serdesDomain.memberIntfs.values() )
      
   def model( self ):
      helper = IntfInteractionsModelHelper( self.groupLinkModes )
      for intf in self.allIntfs:
         # No spped configuration is available, the intf is inactive
         s1 = self.getModuleIntf()
         helper.inactiveReason( intf, f"QDD-BP-200 installed on {s1}" )

      return helper.model

   def getModuleIntf( self ):
      intfs = [ intf for intf in self.allIntfs if intf[ -1 ] == "1" ]
      return intfs[ 0 ]
