#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import Tac
from Toggles.SwagCoreToggleLib import toggleSwagPhase1Enabled

hwCapabilities = None
PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
PortChannelNumMin = PortChannelNum.min
PortChannelNumMax = PortChannelNum.max

LimitedChannelNum = Tac.Type( "Lag::LimitedChannelNum" )
PortChannelExtendedMin = LimitedChannelNum.extendedSubMin
PortChannelExtendedMax = LimitedChannelNum.extendedSubMax

SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
SubExtendedMin = SubIntfRangeDetails.subIntfExtendedIdMin
SubExtendedMax = SubIntfRangeDetails.subIntfExtendedIdMax

def portChannelRangeFn( mode, context=None ):
   # Allow up to extendedMax while loading startup-config as hwCapabilities
   # is not yet available. If both hwCapabilities and mode are unavailable,
   # restrict to original max.
   if ( ( mode and mode.session_.startupConfig() and
          not mode.session_.guardsEnabled() ) or
        ( hwCapabilities and hwCapabilities.extendedLagIdSupported ) ):
      return ( PortChannelNum.min, PortChannelNum.extendedMax )
   return ( PortChannelNum.min, PortChannelNum.max )

def portChannelSubRangeFn():
   if hwCapabilities and hwCapabilities.extendedLagIdSupported:
      return ( PortChannelNum.min, PortChannelNum.extendedMax )
   return ( PortChannelNum.min, PortChannelNum.max )

# With dynamic ifIndex values, all port-channels use the extended sub-interface
# range: subIntfExtendedIdMin(1) - subIntfExtendedIdMax(1048575)
# Interfaces in the Lag::LimitedChannelNum have static ifIndex values
# associated with them; beyond that range the ifIndex values are dynamically
# allocated - more details are in Ebra/SubIntf.tin, ifIndexForExtendedSubIntf()

LagAutoIntfType = IntfRange.GenericRangeIntfType(
   portChannelRangeFn,
   "Port-Channel", "Po", "Lag interface",
   subSupported=True, subIntfGuard=IntfRange.subintfSupportedGuard,
   dynamicRange=True, lazyRange=True, intfSubLowerBound=SubExtendedMin,
   intfSubUpperBound=SubExtendedMax )

intfOptions = IntfRange.IntfOptions( ipSupport=True, routingProtoSupport=True )
IntfRange.registerIntfType( LagAutoIntfType, intfOptions )

def configuredLagCallback( requireMounts, explicitIntfTypes ):
   if explicitIntfTypes and 'Port-Channel' not in explicitIntfTypes:
      return []
   if not toggleSwagPhase1Enabled():
      return requireMounts.getValue( 'interface/config/eth/lag' ).intfConfig
   else:
      # If fabric-channel interfaces aren't excluded, then default
      # config across bridging/routing features will make their way in
      # 'show running-config all'. Replaying the saveAll config will
      # lead to invalid input error as the fabric-channel interfaces
      # don't support bridging/routing config for example.
      excludeTypes = [ 'Fabric-Channel' ]
      ethLagIntfConfigDir = requireMounts.getValue( 'interface/config/eth/lag' )
      return filter( lambda x: not any( x.startswith( t ) for t in excludeTypes ),
                     ethLagIntfConfigDir.intfConfig )

IntfRange.registerConfiguredIntfCallback( configuredLagCallback )

def Plugin( entityManager ):
   global hwCapabilities
   mg = entityManager.mountGroup()
   hwCapabilities = mg.mount( 'bridging/hwcapabilities',
                              'Bridging::HwCapabilities', 'r' )
   mg.close( callback=None, blocking=False )
