# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools

import Fru
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import LazyMount

entmibStatus = None
inventoryBaseDir = None

class XcvrSlot( IntfRange.AutoIntfType ):
   def __init__( self, name, helpText ):
      IntfRange.AutoIntfType.__init__( self, name, name, helpText,
                                       [ "Transceiver slot number" ] )

   def collection( self ):
      # Iterate over entmibStatus to identify all transceiver slot numbers. For
      # modular systems, need to iterate over all cards, and slot numbers will be
      # prepended with card number.
      if entmibStatus.fixedSystem:
         # Collection items must be prepended with "port" (i.e. self.tagLong")
         # to satisfy AutoIntfType requirements.
         # pylint: disable-next=consider-using-f-string
         return ( "%s%s" % ( self.tagLong, i )
                  for i in entmibStatus.fixedSystem.xcvrSlot )

      if entmibStatus.chassis:
         def _cardXcvrSlots( _cardSlot ):
            cardNumber = _cardSlot.relPos
            # pylint: disable-next=consider-using-f-string
            return [ "%s%s/%s" % ( self.tagLong, cardNumber,
                                   slotNum )
                     for slotNum in _cardSlot.card.xcvrSlot ]
         return itertools.chain( *( _cardXcvrSlots( c )
                                 for c in entmibStatus.chassis.cardSlot.values()
                                 if c.card ) )

      return []

   def collectionVersion( self ):
      versionList = []

      # For modular-systems, use Fru.powerGenerationId for linecard inventories.
      invModSys = inventoryBaseDir.get( "modularSystem" )
      if invModSys:
         for invCard in sorted( invModSys.card.values(), key=lambda x: x.slot ):
            versionList.append( Fru.powerGenerationId( Fru.fruBase( invCard ) ) )
         return versionList

      # For fixed-systems, arbitrarily use the number of ports as a version number.
      # The important part here is to generate a different "version" from when the
      # inventory isn't fully populated yet (e.g. startup time).
      invFixedSys = inventoryBaseDir.get( "fixedSystem" )
      if invFixedSys:
         versionList = [ len( invFixedSys.component[ "xcvrSlotDir" ].xcvrSlot ) ]
         return versionList

      # If we get here, the inventory for the system hasn't been populated yet. Use
      # an arbitrary version list, which will be overridden when the collection is
      # re-evaluated after the inventory is populated.
      return [ 0 ]

XcvrSlotType = XcvrSlot( "port", "Line system port settings" )
IntfRange.registerGroupType( XcvrSlotType )

def Plugin( em ):
   global entmibStatus
   global inventoryBaseDir
   entmibStatus = LazyMount.mount( em, "hardware/entmib",
                                   "EntityMib::Status", "r" )
   inventoryBaseDir = LazyMount.mount( em, "hardware/inventory", "Tac::Dir", "ri" )
