#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
from ArnetModel import IpGenericPrefix, IpGenericAddress, MacAddress
from IntfModels import Interface
from CliModel import DeferredModel, Submodel, Model
from CliModel import List, Bool, Dict, Enum, Int, Str, Float
from CliModel import GeneratorDict
from DeviceNameLib import intfLongName
from CliPlugin.TunnelModels import (
      TunnelInfo,
      TunnelViaInfo,
      TunnelId )
from CliPlugin.IpRibLibCliModels import ResolutionRibProfile, ViaFlagDetails
from TableOutput import createTable, Format

routingProtocol = Tac.Type( "Routing::Rib::RoutingProtocol" )
TacTunnelId = Tac.Type( "Tunnel::TunnelTable::TunnelId" )
TacTunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
TacDynTunIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
# In sync with the list in gated/dget_rib_route.c
ribdSupportProtos = [ routingProtocol.connected, routingProtocol.staticConfig,
                      routingProtocol.bgp, routingProtocol.ospf,
                      routingProtocol.ospf3, routingProtocol.isis ]
ViaType = Tac.Type( "Routing::Rib::ViaType" )
protocolString = Tac.newInstance( "Routing::Rib::RoutingProtocolString" )

# In sync with the list in gated/dget_rib_route.c, gated/dget_rib_resolution_route.c
ribdViaType = [ protocolString.viaSetTypeCode( ViaType.tunnel ), 
      protocolString.viaSetTypeCode( ViaType.nextHopGroup ) ]

resolvability = Tac.Type( 'Routing::Rib::Resolvability' )
resolvabilityAttr = resolvability.attributes
ribdSupportResolvability = [ resolvability.reject, resolvability.accept ]

def getTunnelFib():
   # pylint: disable-next=import-outside-toplevel
   from CliPlugin.IpRibCliShow import getCliMounter
   cliMounter = getCliMounter( 'rib' )
   return cliMounter.getTunnelFib()

class ResolvedIpNextHop( DeferredModel ):
   '''An immediate next-hop for a route over Ipv4/Ipv6 transport'''

   nextHop = IpGenericAddress( help="Next hop router address", optional=True )
   interface =  Interface( help="Interface over which the next-hop router is "
                           "reachable" )
   nexthopGroupName = Str( help="Nexthop Group name over which via resolves",
                           optional=True )
   weight = Float( help="UCMP weight of the via", optional=True )
   arpResolved = Bool( help="Designates if nextHop is resolved in ARP/ND",
                       optional=True )
   tunnelInterface = Submodel( valueType=TunnelInfo, help="Tunnel end point details",
                                 optional=True )
   tunnelVias = List( help="LSP's for the tunnel", valueType=TunnelViaInfo,
                      optional=True )
   label = Int( help="Mpls label", optional=True )
   fallbackVrf = Str( optional=True, help="Fallback VRF name" )
   decapActionType = Enum( values=( 'none', 'ipip' ),
                           help="Decapsulation type", optional=True )
   viaType = Enum( help="Type of via, either active or backup", optional=True,
                   values=( 'backup', 'active' ) )

   # Remainder only used in single agent mode (RibCapi)
   _tunnelId = Submodel( valueType=TunnelId, help="Top level tunnel ID" )

   def processData( self, data ):
      self.nextHop = data.pop( 'nhopv4', None )
      if self.nextHop is None:
         nextHop = data.pop( 'nhopv6', None )
         if nextHop:
            if '%' in nextHop:
               nextHop, _ = nextHop.split( '%' )
               # Note: we lose the link-local interface name
            self.nextHop = nextHop
      intf = data.pop( 'interface', None )
      if intf is not None:
         self.interface = intfLongName( intf )
      self.nexthopGroupName = data.pop( 'nhg_name', None)
      topLevelTunnelId = data.pop( 'tunnel_id', None )
      if topLevelTunnelId is not None:
         self._tunnelId = TunnelId()
         self._tunnelId.fromRawValue( topLevelTunnelId )
         self.processTunnelResolvingOverTunnel( topLevelTunnelId )

   def processTunnelResolvingOverTunnel( self, topLevelTunnelId ):
      # set interface as DynamicTunnelXXXX
      self.interface = TacDynTunIntfId.tunnelIdToIntfId( topLevelTunnelId )

      # update tunnelInterface model
      tunnel = TacTunnelId( topLevelTunnelId )
      self.tunnelInterface = TunnelInfo()
      self.tunnelInterface.tunnelType = tunnel.typeCliStr()
      self.tunnelInterface.tunnelIndex = tunnel.tunnelIndex()

      # Check if tunnelFibEntry exists in tunnelFib
      tunnelFib = getTunnelFib()
      tunnelFibEntry = tunnelFib.entry.get( topLevelTunnelId )
      if not tunnelFibEntry:
         return

      # update tunnelVias of tunnelInterface model
      for via in tunnelFibEntry.tunnelVia.values():
         tunnelVias = TunnelViaInfo()
         labelStackEncap = tunnelFib.labelStackEncap.get( via.encapId )
         if labelStackEncap and labelStackEncap.labelStack.stackSize != 0:
            labelOp = labelStackEncap.labelStack
            labelStack = [ labelOp.labelStack( idx )
                           for idx in range( labelOp.stackSize - 1, -1, -1 ) ]
            tunnelVias.labelStack = labelStack
         if TacDynTunIntfId.isDynamicTunnelIntfId( via.intfId ):
            nTunnelId = TacDynTunIntfId.tunnelId( via.intfId )
            nTunnel = TacTunnelId( nTunnelId )
            tunnelVias.resolvingTunnel = TunnelId( index=nTunnel.tunnelIndex(),
                                                   type=nTunnel.typeCliStr() )
         else:
            tunnelVias.nexthop = via.nexthop
            tunnelVias.interface = via.intfId
         self.tunnelInterface.tunnelVias.append( tunnelVias )

class ShamLinkNextHop( DeferredModel ):
   '''A sham-link interface next-hop for a route'''

   nextHop = IpGenericAddress( help="Next hop router address", optional=True )
   interface = Str( help="Interface over which the next-hop router is "
                           "reachable" )

   def processData( self, data ):
      self.nextHop = data.pop( 'nhopv4', None )
      if self.nextHop is None:
         nextHop = data.pop( 'nhopv6', None )
         if nextHop:
            if '%' in nextHop:
               nextHop, _ = nextHop.split( '%' )
               # Note: we lose the link-local interface name
            self.nextHop = nextHop
      self.interface = data.pop( 'interface', None )

class EvpnNextHop( DeferredModel ):
   '''A next-hop for a route over Evpn transport'''

   vtep = IpGenericAddress( help="Vtep address" )
   vni = Int( help="Vni associated with this next-hop" )
   routerMac = MacAddress( help="Router mac address of the vtep" )
   localInterface = Interface( help="Name of the source interface", optional=True )

class ResolvedNextHop( DeferredModel ):
   '''Immediate next-hop for a route, i.e the adjacent router on the path to the
   destination prefix'''

   transportType = Enum( help="The type of transport used to reach the destination",
                         values=( 'ipv4', 'ipv6', 'evpn' ) )

   ipNextHop = Submodel( help="Immediate next-hop router used to reach a destination"
                         " over ipv4/ipv6 transport", valueType=ResolvedIpNextHop,
                         optional=True )

   ospfShamLinkNextHop = Submodel( help="Sham-link interface used as next hop to"
                               " reach the destination", valueType=ShamLinkNextHop,
                               optional=True )

   evpnNextHop = Submodel( help="Vtep used to reach a destination over"
                           "evpn transport", valueType=EvpnNextHop,
                           optional=True )
   vrfName = Str( help="Name of VRF in which this next-hop was learned", 
                  optional=True)
   notInFec = Bool( help="Not in FEC", optional=True )

   # Remainder only used in single agent mode (RibCapi ) or for shamlink hidden
   # route cli - "show ip ospf sham route" which only is present in multi-agent
   def processData( self, data ):
      self.transportType = 'ipv6' if ord( data.pop( 'ipv6' ) ) else 'ipv4'
      if 'OSPF_SL' in data[ 'interface' ] or \
         'OSPF3_SL' in data[ 'interface' ]:
         self.ospfShamLinkNextHop = ShamLinkNextHop()
         self.ospfShamLinkNextHop.processData( data )
      else:
         self.ipNextHop = ResolvedIpNextHop()
         self.ipNextHop.processData( data )

   def renderResolvedNh( self, indent='' ):
      if not self.ipNextHop and not self.ospfShamLinkNextHop:
         return
      if self.ipNextHop:
         nh = self.ipNextHop
         nhIntf = nh.interface.stringValue
         # pylint: disable-msg=protected-access
         # mask pylint error when accessing nh._tunnelId
         if nh.nexthopGroupName:
            print( '         {}via NexthopGroup {}'.format( indent,
                                                        nh.nexthopGroupName ) )
         elif nh._tunnelId is not None:
            tunnelFib = getTunnelFib()
            print( self.getRenderedTunnelInfoBuffer( indent, nh._tunnelId ) )
            self.recursivelyRenderTunnelVias( indent + 3 * ' ', nh._tunnelId,
                                              tunnelFib, {} )
         elif nh.nextHop:
            print( f'         {indent}via {nh.nextHop}, {nhIntf}' )
         else:
            print( f'         {indent}via {nhIntf}, directly connected' )
         # pylint: enable-msg=protected-access
      elif self.ospfShamLinkNextHop:
         nh = self.ospfShamLinkNextHop
         nhIntf = nh.interface
         print( f'         {indent}via {nh.nextHop}, {nhIntf}' )

   def getRenderedTunnelInfoBuffer( self, indent, tunnelIdModel ):
      tunnelInfo = tunnelIdModel.renderTunnelIdStr( tunStr="Tunnel index" )
      return f'         {indent}via {tunnelInfo}'

   def getRenderedLabelStackBuffer( self, encapId, tunnelFib ):
      labelStackEncap = tunnelFib.labelStackEncap
      labelOp = labelStackEncap[ encapId ].labelStack
      if labelOp.stackSize != 0:
         labelStack = [ str( labelOp.labelStack( idx ) )
                        for idx in range( labelOp.stackSize - 1, -1, -1 ) ]
         return ' label %s' % ( " ".join( labelStack ) )
      return ''

   def recursivelyRenderTunnelVias( self, indent, tunnelIdModel, tunnelFib,
                                    renderedTunnelIds ):
      tunnelFibEntry = tunnelFib.entry.get( tunnelIdModel.toRawValue() )
      if not tunnelFibEntry:
         return

      renderedTunnelIds[ tunnelIdModel.toRawValue() ] = True

      for via in tunnelFibEntry.tunnelVia.values():
         if TacDynTunIntfId.isDynamicTunnelIntfId( via.intfId ):
            nTunnelId = TacDynTunIntfId.tunnelId( via.intfId )
            nTunnelIdModel = TunnelId()
            nTunnelIdModel.fromRawValue( nTunnelId )
            viaTunBuf = self.getRenderedTunnelInfoBuffer( indent,
                                                          nTunnelIdModel )
            labelStackBuf = self.getRenderedLabelStackBuffer( via.encapId,
                                                              tunnelFib )
            print( f'{viaTunBuf}{labelStackBuf}' )
            # pylint: disable-msg=unsupported-membership-test
            if nTunnelId in renderedTunnelIds:
               # cycle detected, avoid recursion now
               return
            # pylint: enable-msg=unsupported-membership-test
            self.recursivelyRenderTunnelVias( indent + 3 * ' ', nTunnelIdModel,
                                              tunnelFib, renderedTunnelIds )
         else:
            print( '         %svia %s, %s%s' %
                   ( indent, via.nexthop, via.intfId,
                     self.getRenderedLabelStackBuffer( via.encapId, tunnelFib ) ) )

      # print backup via info for tilfa tunnel
      if tunnelFibEntry.backupTunnelVia:
         tunnel = TacTunnelId( tunnelIdModel.toRawValue() )
         assert tunnel.tunnelType() == TacTunnelType.tiLfaTunnel
         for via in tunnelFibEntry.backupTunnelVia.values():
            print( '         %sbackup via %s, %s%s' %
                   ( indent, via.nexthop, via.intfId,
                     self.getRenderedLabelStackBuffer( via.encapId, tunnelFib ) ) )

class ColorAndCoBits( DeferredModel ):
   '''SRTE color and the CO bits for a recursive nexthop that is used to match with 
   active SRTE policies to update the FIB with SR policy paths'''
   value = Int( help="Color value associated with the nexthop" )
   coBits = Enum( help="The type of SR-TE policy matching to be done for this color",
                  values=( 'EM', 'NM', 'AM' ) )

class RecursiveNextHop( DeferredModel ):
   '''A next-hop for a route that may or may not be the adjacent router on the path
   to the destination prefix. A recursive next-hop is reachable via one or more
   resolved next-hops'''
   __revision__ = 2

   nextHop = IpGenericAddress( help="Next hop router address", optional=True )
   transportType = Enum( help="The type of transport used to reach the destination",
                         values=( 'ipv4', 'ipv6', 'evpn', 'mpls' ) )

   loopedNextHop = Bool( help="Set to true if this next-hop is part of a recursive "
                         "route resolution loop, false otherwise", optional=True )
   unresolvedNextHop = Bool( help="Set to true if this next-hop is unresolved, "
                             "false otherwise", optional=True )
   colors = List( help="The list of SR-TE color values and the CO bits "
                       "for the recursive nexthop",
                       valueType=ColorAndCoBits, optional=True )
   linkBandwidth = Float( help="The link bandwith associated with this next-hop "
                          "in bytes per second", optional=True )
   weight = Float( help="The weight associated with this next-hop", optional=True )
   preference = Int( help="The administrative distance of the route used to reach "
                     "this next-hop", optional=True )
   metric = Int( help="The routing protocols internal metric (cost) for the route "
                 "used to reach this next-hop", optional=True )
   metricType = Enum( help="The type of metric", optional=True,
                      values=( 'metric', 'MED', 'AIGP' ) )
   resolvedNextHops = List( help="The list of immediate next-hop routers through "
                            "which this recursive next-hop is reachable",
                            valueType=ResolvedNextHop, optional=True )
   viaSetType = Enum( help="Type of vias in the via set", optional=True,
                      values=( 'tunnel', 'nexthop-group', 'mpls',
                               'ipv4', 'ipv6', 'evpn', 'invalid' ) )
   viaType = Enum( help="Type of via, either active or backup", optional=True,
                   values=( 'backup', 'active' ) )

   resolutionRibs = Submodel( valueType=ResolutionRibProfile, optional=True,
                              help="Resolution RIBs." )

   flags = Submodel( valueType=ViaFlagDetails, optional=True, help="Via Flags" )

   def getFlattenedIpVias( self, resolvedNextHop ):
      flatIpVias = []
      tunnelFib = getTunnelFib()
      tunnelResolver = Tac.newInstance( "Tunnel::TunnelFib::TunnelResolver",
                                        tunnelFib )
      # pylint: disable-msg=protected-access
      tunnelIdModel = resolvedNextHop.ipNextHop._tunnelId
      tunnel = TacTunnelId( tunnelIdModel.toRawValue() )
      tunnelResolver.updateFlattenedInfo( tunnel )
      consEntry = tunnelResolver.consolidatedEntry.get( tunnel )
      if not consEntry:
         return flatIpVias
      for viaInfo in consEntry.viaInfo:
         flatTunVia = {}
         flatTunVia[ 'nextHop' ] = viaInfo.nexthop.stringValue
         flatTunVia[ 'interface' ] = viaInfo.intfId
         flatTunVia[ 'tunVias' ] = []
         flatIpVias.append( { 'ipNextHop' : flatTunVia,
                             'transportType' : resolvedNextHop.transportType } )
      return flatIpVias

   def degrade( self, dictRepr, revision ):
      tunViaFound = False
      if revision == 1:
         # In revision 1, only return final interface and nexthop information instead
         # of information about resolving tunnel.
         if self.resolvedNextHops:
            flatIpNextHops = []
            for resolvedNextHop in self.resolvedNextHops:
               # pylint: disable-msg=protected-access
               if resolvedNextHop.ipNextHop and resolvedNextHop.ipNextHop._tunnelId:
                  tunViaFound = True
                  tunVias = self.getFlattenedIpVias( resolvedNextHop )
                  flatIpNextHops.extend( tunVias )
            if tunViaFound:
               dictRepr[ 'resolvedNextHops' ] = flatIpNextHops
      return dictRepr

   def processData( self, data ):
      # Remainder only used in single agent mode (RibCapi)
      self.nextHop = data.pop( 'nhopv4', None ) or data.pop( 'nhopv6', None )
      self.transportType = 'ipv6' if ord( data.pop( 'ipv6' ) ) else 'ipv4'
      self.loopedNextHop = bool( ord( data.pop( 'looped' ) ) )
      self.preference = data.pop( 'preference' )
      self.metric = data.pop( 'metric' )
      viaType = ord( data.pop( 'type', '\xff' ) )
      if viaType < len( ribdViaType ):
         self.viaSetType = ribdViaType[ viaType ]

   def renderRecursiveNh( self ):
      loopedStr = ' L*' if self.loopedNextHop else ''
      resolvedStr = '' if self.resolvedNextHops or self.loopedNextHop else ' *'
      viaTypeStr = ''
      if self.viaSetType is not None:
         viaTypeStr = 'type %s' % self.viaSetType
      print( '         via %s [%d/%d]%s%s %s' % ( self.nextHop, self.preference,
                                                 self.metric, loopedStr, resolvedStr,
                                                 viaTypeStr ) )

      for resolvedNh in self.resolvedNextHops:
         resolvedNh.renderResolvedNh( indent='   ' )

class RibRoute( DeferredModel ):
   '''A route learned through a routing protocol that is considered for installation
   in the forwarding table'''
   # revision 3 for removing timestamp and adding lastUpdateEpochTime attribute
   __revision__ = 3

   preference = Int( help="The administrative distance of the route" )
   metric = Int( help="The routing protocols internal metric (cost) for this route" )
   metricType = Enum( help="The type of metric", values=( 'metric', 'MED', 'AIGP' ) )

   bestRoute = Bool( help="Set to true if this route was selected for installation "
                     "into the forwarding table, false otherwise" )

   loopedRoute = Bool( help="Set to true if this route is part of a recursive "
                       "route resolution loop, false otherwise" )

   recmpEligible = Bool( help="Set to true if this route is marked eligible for "
                         "RECMP", optional=True )

   # Bgp/Static: defined as List since protocols can publish multiple next hops with
   # the same ip address but different resolution profile rules.
   recursiveNextHops = List( help="List of next-hop routers through which this "
                             "route is reachable",
                             valueType=RecursiveNextHop, optional=True )

   # Connected/Static/IGPs
   resolvedNextHops = List( help="List of immediate next-hop routers through which "
                            "this route is reachable", valueType=ResolvedNextHop,
                            optional=True )
   srcVrfName = Str( help="Name of VRF from which this route was leaked",
                     optional=True )

   srcVrfId = Int( help="ID of VRF from which this route was leaked",
                   optional=True )

   # Remainder only used in single agent mode (RibCapi)
   _recursiveIndex = Int( help="The index of current recursive via from RibCapi, "
                        "if any" )

   lastUpdateEpochTime = Int( help="Timestamp in UTC time of the last time the "
                              "route has been updated" )

   tag = Int( help="The tag", optional=True )

   communityList = List( help="BGP communities", valueType=str )
   pendingAttributes = Bool( help="Set to true if the tag or communityList is "
                             "pending" )
   routePriority = Enum( values=( 'low', 'medium', 'high' ),
                         help="Route processing priority",
                         optional=True )

   def getKey( self, data ):
      addr = data.get( 'addrv4' )
      if addr is None:
         addr = data.get( 'addrv6' )
      return addr + '/' + str( data.get( 'mlen' ) )

   def overrideHierarchy( self, data ):
      readNext = True

      if 'active' in data:
         # RibRoute
         if self.preference is not None:
            # This is a new RibRoute entry.  Return False to create a new model
            # instance.
            return ( data, False )

         data.pop( 'addrv6', None )
         data.pop( 'addrv4', None )
         data.pop( 'mlen', None )

         self.preference = data.pop( 'preference' )
         self.metric = data.pop( 'metric' )
         self.bestRoute = bool( ord( data.pop( 'active' ) ) )
         self.loopedRoute = bool( ord( data.pop( 'looped' ) ) )

      else:
         assert 'ipv6' in data
         if 'looped' in data:
            # RecursiveNextHop
            via = RecursiveNextHop()
            via.processData( data )
            self._recursiveIndex = len( self.recursiveNextHops )
            self.recursiveNextHops.append( via )

         else:
            # ResolvedNextHop
            via = ResolvedNextHop()
            via.processData( data )
            # Disable warning "Anomalous backslash in string"
            # pylint: disable-msg=W1401
            if ord( data.get( 'recursive', '\0' ) ):
               assert self._recursiveIndex >= 0
               self.recursiveNextHops[ self._recursiveIndex ].\
                  resolvedNextHops.append( via )
            else:
               self._recursiveIndex = -1
               self.resolvedNextHops.append( via )

      return ( data, readNext )

   def renderRoute( self ):
      for resolvedNh in self.resolvedNextHops:
         resolvedNh.renderResolvedNh()

      for recursiveNh in self.recursiveNextHops:
         recursiveNh.renderRecursiveNh()

class RibRoutes( DeferredModel ):
   '''For a given prefix, routes learned through routing protocols considered for
   installation to the forwarding table. The best (lowest cost) route from among
   the candidates is installed in the forwarding table'''

   # allow model.toDict() to create a streamable object
   __streamable__ = True

   # revision 2 for removing timestamp and adding lastUpdateEpochTime attribute
   #            to the submodel RibRoute
   __revision__ = 2

   ribRoutes = GeneratorDict( help="List of routes learned through routing protocols"
                              "for a given prefix", keyType=IpGenericPrefix,
                              valueType=RibRoute )

   # Remainder only used in single agent mode (RibCapi)

   # Private attribute to get the current protocol enum value
   _protoIdx = Int( help="The route protocol index in @ribdSupportProtos" )

   def getKey( self, data ):
      self._protoIdx = ord( data.pop( 'protocol' ) )
      return ribdSupportProtos[ self._protoIdx ]

   def renderRoutes( self ):
      for pfx, ribRoute in self.ribRoutes:
         srcVrfStr = ''
         if ribRoute.srcVrfId:
            srcVrfStr = ' %s(%d)' % ( ribRoute.srcVrfName, ribRoute.srcVrfId )
         print( '%s%s    %s [%d/%d]%s%s' % \
                ( '>' if ribRoute.bestRoute else ' ',
                  protocolString.protocolCode( ribdSupportProtos[ self._protoIdx ] ),
                pfx, ribRoute.preference, ribRoute.metric,
                ' L' if ribRoute.loopedRoute else '', srcVrfStr ) )

         # Print all vias
         ribRoute.renderRoute()

class RibRoutesByProtocol( DeferredModel ):
   '''Map of routes learned via routing protocols that are considered for
   installation into the forwarding table'''

   # allow model.toDict() to create a streamable object
   __streamable__ = True

   # revision 1 for printing flattened via when tunnel resolve over another tunnel
   # revision 3 for removing timestamp and adding lastUpdateEpochTime attribute
   #            to the submodel RibRoute
   __revision__ = 3

   # BUG281506 - Convert keyType from Str to RoutingProtocol Enum once CAPI supports
   #             Dict with Enum key types
   ribRoutesByProtocol = GeneratorDict( help="Map of routes learned through routing "
                                        "protocols that are considered for "
                                        "installation into the forwarding table",
                                        keyType=str, valueType=RibRoutes,
                                        optional=True )

   # Remainder only used in single agent mode (RibCapi)
   def renderProtocols( self, vrfName, prefixProto=None ):
      def renderVrfHeader( vrfName, proto ):
         # don't print protocol if it wasn't supplied at CLI and thus ==
         # 'routingProtocols'
         if proto == routingProtocol.routingProtocols:
            print( 'VRF: %s' % ( vrfName ) )
         else:
            proto = protocolString.internalToExternal( proto )
            print( f'VRF: {vrfName}, Protocol: {proto}' )

      def renderLegend():
         print( 'Codes: C - Connected, S - Static, P - Route Input' )
         print( '       B - BGP, O - OSPF, O3 - OSPF3, I - IS-IS' )
         print( '       > - Best Route, * - Unresolved Nexthop' )
         print( '       L - Part of a recursive route resolution loop' )

      # match output of multi-agent code, print legend once 
      # prefixProto holds protocol and indicates that prefix was specified
      if prefixProto:
         renderVrfHeader( vrfName, prefixProto )
         renderLegend()
         for proto, ribRoutes in self.ribRoutesByProtocol:
            ribRoutes.renderRoutes()
      else:
         for proto, ribRoutes in self.ribRoutesByProtocol:
            renderVrfHeader( vrfName, proto )
            renderLegend()
            ribRoutes.renderRoutes()

class RibRoutesByVrf( DeferredModel ):
   '''For each VRF, map of routes learned via routing protocols that are considered
   for installation into the forwarding table'''

   # allow model.toDict() to create a streamable object
   __streamable__ = True

   # revision 1 for printing flattened via when tunnel resolve over another tunnel
   # revision 2 for printing only single vrf in the format of RibRoutesByProtocol
   #            i.e. without [ 'vrfs' ][ vrfName ]
   # revision 4 for removing timestamp and adding lastUpdateEpochTime attribute
   #            to the submodel RibRoute
   __revision__ = 4

   # Keys are vrf names
   vrfs = GeneratorDict( help='For each VRF, map of routes learned through routing '
                              'protocols that are considered for installation into '
                              'the forwarding table',
                              keyType=str, valueType=RibRoutesByProtocol,
                              optional=True )

class RibRouteSummaryForVrf( Model ):
   routeCountPerProtocol = Dict( keyType=str, valueType=int,
                                 help='Total number of routes for each '
                                 'routing protocol in the RIB' )

   def render( self ):
      table = createTable( ( 'Route Source', 'Number Of Routes' ) )
      justifyLeft = Format( justify='left' )
      justifyLeft.noPadLeftIs( True )
      justifyRight = Format( justify='right' )
      table.formatColumns( justifyLeft, justifyRight )
      for protocol, numberOfRoutes in \
            sorted( self.routeCountPerProtocol.items() ):
         table.newRow( protocol, numberOfRoutes )
      print( table.output() )

class RibRouteSummary( Model ):
   ribRouteSummary = Dict( keyType=str, valueType=RibRouteSummaryForVrf,
                           help='Summary of routes in the RIB of each VRF' )

   def render( self ):
      for vrfName, ribRouteSummaryForVrf in self.ribRouteSummary.items():
         print( 'VRF: %s' % vrfName )
         ribRouteSummaryForVrf.render()

class LoopedRoute( DeferredModel ):
   '''A looped route learned through a routing protocol and loop detection'''

   # BUG281506 - Convert keyType from Str to RoutingProtocol Enum
   protocol = Str( help="The protocol for the looped route" )
   allNextHopsPartOfLoop = Bool( help="Set to true if all next-hops of this route " 
                                 "are part of recursive route resolution loops" )

class LoopedRoutes( DeferredModel ):
   '''The collection of looped routes from all available routes'''

   loopedRoutes = Dict( help="Map of routes involved in recursive route resolution "
                        "loops", keyType=IpGenericPrefix, valueType=LoopedRoute )

class ResolutionRoute( DeferredModel ):
   __revision__ = 2
   preference = Int( help="The administrative distance of this route" )
   metric = Int( help="The routing protocol internal metric (cost) for this route" )
   metricType = Enum( help="The type of metric", values=( 'metric', 'MED', 'AIGP' ) )
   protocol = Enum( help="The routing protocol for this route",
                    values=routingProtocol.attributes )
   resolvability = Enum( help="The resolvability of this route",
                         values=resolvabilityAttr )

   resolvedNextHops = List( help="List of immediate next-hop routers through which "
                            "this route is reachable", valueType=ResolvedNextHop,
                            optional=True )
   # Bgp/Static
   recursiveNextHops = List( help="List of next-hop routers through which routes in "
                             "the RIB are reachable",
                             valueType=RecursiveNextHop, optional=True )
   # Remainder only used in single agent mode (RibCapi)
   _recursiveIndex = Int( help="The index of current recursive via from RibCapi, "
                        "if any" )

   recmpEligible = Bool( help="Set to true if this route is marked eligible for "
                         "RECMP", optional=True )
   lastUpdateEpochTime = Int( help="Timestamp in UTC time of the last time the "
                              "route has been updated" )
   tag = Int( help="The tag", optional=True )

   communityList = List( help="BGP communities", valueType=str )
   pendingAttributes = Bool( help="Set to true if the tag or communityList is "
                             "pending" )

   def getKey( self, data ):
      addr = data.get( 'addrv4' ) or data.get( 'addrv6' )
      return addr + '/' + str( data.get( 'mlen' ) )

   def overrideHierarchy( self, data ):
      # Copy from RibRoute. Here this will populate ResolutionRoutes Model.
      readNext = True
      if 'resolvability' in data:
         # ResolutionRoute
         if self.preference is not None:
            # This is a new ResolutionRoute entry. Return False to create a new model
            # instance.
            return ( data, False )

         data.pop( 'addrv6', None )
         data.pop( 'addrv4', None )
         data.pop( 'mlen', None )

         self.preference = data.pop( 'preference' )
         self.metric = data.pop( 'metric' )
         self.protocol = ribdSupportProtos[ ord( data.pop( 'protocol' ) ) ]
         self.resolvability = ribdSupportResolvability[ \
                                       ord( data.pop( 'resolvability' ) ) ]

      else:
         # Populating RecursiveNextHop/ResolvedNextHop.
         if 'looped' in data:
            # RecursiveNextHop
            via = RecursiveNextHop()
            via.processData( data )
            self._recursiveIndex = len( self.recursiveNextHops )
            self.recursiveNextHops.append( via )

         else:
            # ResolvedNextHop
            via = ResolvedNextHop()
            via.processData( data )
            # Disable warning "Anomalous backslash in string"
            # pylint: disable-msg=W1401
            if ord( data.get( 'recursive', '\0' ) ):
               assert self._recursiveIndex >= 0
               self.recursiveNextHops[ self._recursiveIndex ].\
                  resolvedNextHops.append( via )
            else:
               self._recursiveIndex = -1
               self.resolvedNextHops.append( via )

      return ( data, readNext )

   def renderResolutionRoute( self ):
      for resolvedNh in self.resolvedNextHops:
         resolvedNh.renderResolvedNh()

      for recursiveNh in self.recursiveNextHops:
         recursiveNh.renderRecursiveNh()

class ResolutionRoutes( DeferredModel ):
   __revision__ = 2
   # allow model.toDict() to create a streamable object
   __streamable__ = True
   routeMap = Str( help="RIB recursive resolution policy", optional=True )

   routes = GeneratorDict( help="Map of routes to their recursive resolution status",
                           keyType=IpGenericPrefix, valueType=ResolutionRoute,
                           optional=True )

   # Remainder only used in single agent mode (RibCapi)
   def renderResolutionRoutes( self, af, vrfName ):
      if self.routeMap:
         rmap = self.routeMap
      else:
         rmap = "Not configured"
      # Below output needs to be in sync with multiagent output for same command.
      print( f'VRF: {vrfName}, Resolution policy: {rmap}' )
      print( 'Codes: C - Connected, S - Static, P - Route Input' )
      print( '       B - BGP, O - OSPF, O3 - OSPF3, I - IS-IS' )
      print( '       % - Recursive resolution denied' )
      if self.routeMap is None:
         print( '! No %s recursive resolution policy is configured. ' \
               'Any route may be used for recursive resolution.' % ( af ) )
         return
      for pfx, route in self.routes:
         print( '%s%s    %s [%d/%d]' % \
               ( ' ' if route.resolvability is resolvability.accept else '%',
                  protocolString.protocolCode( route.protocol ),
                  pfx, route.preference, route.metric ) )
         route.renderResolutionRoute()

class RibNextHopRibNextHop( DeferredModel ):
   __revision__ = 2
   recursiveNextHops = List( help="Map of next-hop routers through which routes in "
                            "the RIB are reachable",
                            valueType=RecursiveNextHop, optional=True )
   resolvedNextHops = List( help="List of immediate next-hop routers through which "
                            "routes in the RIB are reachable", 
                            valueType=ResolvedNextHop, optional=True )

class RibNextHop( DeferredModel ):
   '''A rib next-hop learned through a routing protocol'''
   __revision__ = 2

   ribNextHop = Submodel( valueType=RibNextHopRibNextHop, help="RIB nexthop",
                                 optional=True )

   # Bgp/Static
   recursiveNextHops = List( help="List of next-hop routers through which routes in "
                            "the RIB are reachable",
                            valueType=RecursiveNextHop, optional=True )

   # Connected/Static/IGPs
   resolvedNextHops = List( help="List of immediate next-hop routers through which "
                            "routes in the RIB are reachable", 
                            valueType=ResolvedNextHop, optional=True )

class RibNextHopsByProtocol( DeferredModel ):
   '''Map of rib next-hops learned through routing protocols and categorized 
   by protocol'''
   __revision__ = 2

   # BUG281506 - Convert keyType from Str to RoutingProtocol Enum once CAPI supports
   #             Dict with Enum key types
   ribNextHopsByProtocol = Dict( help="Map of next-hop vias for each routing "
                                 "protocol", keyType=str, valueType=RibNextHop )

class RecursiveNhDependency( DeferredModel ):
   nextHop = IpGenericAddress( help="Next hop router address", optional=True )
   viaId = Int( help="ID of Via", optional=True )
   vrfName = Str( help="Name of VRF from which this next-hop was requested", 
                  optional=True)
   resolvingRoute = IpGenericPrefix( help="Covering route for this next-hop",
                                     optional=True )
   # any new routing protocols must be subsequently added to the routeType Enum
   routeType = Enum( help="Protocol through which resolving route was learned", 
                     optional=True, values=( 'reserved', 'connected', 'staticConfig',
                     'bgp', 'routeInput', 'ospf', 'ospf3', 'isis', 'dynamicPolicy',
                     'vrfLeak', 'rip', 'routingProtocols' ) )
                     
   resolvedNextHops = List( help="The list of immediate next-hop routers through "
                            "which this recursive next-hop is reachable",
                            valueType=ResolvedNextHop, optional=True )
   loopedNextHop = Bool( help="Set to true if this next-hop is part of a recursive "
                         "route resolution loop, false otherwise", optional=True )

RecursiveNhDependency.__attributes__[ "recursiveNextHops" ] \
      = List( help="The list of LPM next-hop routers through which this recursive"
              " next-hop is reachable", valueType=RecursiveNhDependency, 
              optional=True )

class RecursiveNhDependencies( DeferredModel ):
   nextHops = List( help="List of next-hops and their dependencies", 
                    valueType=RecursiveNhDependency,
                    optional=True )

class RecursiveNhDependencyByProtocol( DeferredModel ):
   ribNhDependencyByProtocol = GeneratorDict( help="Next-hop dependencies for all "
                                              "protocols in this vrf", keyType=str,
                                              valueType=RecursiveNhDependencies,
                                              optional=True )

class RecursiveNhDependencyByVrf( DeferredModel ):
   ribNhDependencyByVrf = GeneratorDict( help="Next-hop dependencies for all vrfs", 
                                         keyType=str,
                                         valueType=RecursiveNhDependencyByProtocol,
                                         optional=True )
