# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time
from EepromDecoder import EepromDecoder, DecoderException
from types import MethodType
from io import BytesIO
import json
import pycurl

class Decoder( EepromDecoder ):
   def __init__( self, url=None, interface=None, filename=None, retries=None,
                 retryDelay=None ):
      ''' This constructor is not meant to be called directly.  The class methods
          below are used to create instances of this decoder class with different
          functionalities.'''
      super().__init__()
      self.url = url
      self.interface = interface
      self.retries = retries
      self.retryDelay = retryDelay
      self.filename = filename
      self.json_ = None

   @classmethod
   def fromRestApi( cls, url="http://[fe80::1]:8080/api/sys/mb/fruid",
                    interface="usb0", retries=10, retryDelay=1 ):
      # pylint: disable-msg=W0212, protected-access
      self = cls( url=url, interface=interface, retries=retries,
                  retryDelay=retryDelay )
      self._read = MethodType( cls._readFromRestApi, self )
      return self

   @classmethod
   def fromJsonDumpFile( cls, filename="/mnt/flash/eeprom.json" ):
      # pylint: disable-msg=W0212, protected-access
      self = cls( filename=filename )
      self._read = MethodType( cls._readFromFile, self )
      return self

   def _read( self ): # pylint: disable-msg=E0202, method-hidden
      ''' The implementation of this function will be set by the "constructor"
          class methods above
      '''
      raise NotImplementedError

   def _readFromRestApi( self ):
      rawJson = BytesIO()
      c = pycurl.Curl() # pylint: disable=c-extension-no-member
      c.setopt( c.URL, self.url )
      c.setopt( c.WRITEFUNCTION, rawJson.write )
      if self.interface:
         c.setopt( c.INTERFACE, self.interface )

      for _ in range( self.retries ):
         try:
            c.perform()
            break
         except pycurl.error: # pylint: disable=c-extension-no-member
            time.sleep( self.retryDelay )
      else:
         raise DecoderException( "Failed to read from REST API" )

      self.json_ = json.loads( rawJson.getvalue() )
      self._parse()

   def _readFromFile( self ):
      with open( self.filename ) as f:
         self.json_ = json.load( f )
      self._parse()

   def _parse( self ):
      def hwRev( j ): # pylint: disable=inconsistent-return-statements
         major = j.get( "Product Version" )
         minor = j.get( "Product Sub-Version", "0" )
         if major:
            return f"{int( major ):02d}.{int( minor ):02d}"

      def numMacs( j ): # pylint: disable=inconsistent-return-statements
         size = j.get( "Extended MAC Address Size" )
         if size:
            # Local MAC not included in Extended MAC Addr Size
            return f"{int( size ) + 1:d}"

      def mfgTime( j ): # pylint: disable=inconsistent-return-statements
         # In format is "MM-DD-YY", out format is "YYYYMMDDHHNNSS"
         date = j.get( "System Manufacturing Date" )
         if date:
            exploded = [ int( x ) for x in date.split( "-" ) ]
            # pylint: disable-next=consider-using-f-string
            return "{:04d}{:02d}{:02d}000000".format( exploded[ 2 ], exploded[ 0 ],
                                                      exploded[ 1 ] )

      def dataFormat( j ): # pylint: disable=inconsistent-return-statements
         version = j.get( "Version" )
         if version:
            return f"{int( version ):04d}"

      def keyTranslate( key ):
         return lambda j: j.get( key )

      translations = (
      #  ( Arista prefdl key, function to generate value )
         ( "SKU", keyTranslate( "Product Name" ) ),
         ( "MacAddrBase", keyTranslate( "Local MAC" ) ),
         ( "SerialNumber", keyTranslate( "Product Serial Number" ) ),
         ( "ServiceTag", keyTranslate( "Product Asset Tag" ) ),
         ( "PartNumber", keyTranslate( "Product Part Number" ) ),
         ( "Manufacturer", keyTranslate( "System Manufacturer" ) ),
         ( "HwRev", hwRev ),
         ( "NumMacs", numMacs ),
         ( "MfgTime", mfgTime ),
         ( "DataFormat", dataFormat ),
      )

      info = self.json_[ "Information" ]

      for preFdlKey, fn in translations:
         newValue = fn( info )
         if newValue:
            self.preFdl[ preFdlKey ] = newValue
