#!/usr/bin/env arista-python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Isolate is the utility to run another program under isolated namespace,
 either by using netnsd or docker, depending upon the arguments passed to
 the command line.
 All stdout or stderr output of the program is prefixed.
 This is useful for running stests, automake's 'make check' tests, etc in
 parallel. It does not work at all for interactive programs.
"""

import errno
import getpass
import os
import os.path
import sys

__version__ = "2.0.0"
pathjoin = os.path.join

def runAsRoot( args ):
   # Backup effective user in enviroment variable ISOLATE_USER
   os.environ[ "ISOLATE_USER" ] = getpass.getuser()
   if "P4USER" not in os.environ and "LOGNAME" in os.environ:
      os.environ[ "P4USER" ] = os.environ[ "LOGNAME" ]
   preserve = [ "LD_LIBRARY_PATH", "PATH" ]
   if "ARSUDO_SAVE_ENV" in os.environ:
      preserve += os.environ[ "ARSUDO_SAVE_ENV" ].split( ":" )
   variables = [ "%s=%s" % ( var, os.environ.get( var, "" ) ) for var in preserve ]
   cmd = [ "/usr/bin/sudo", "/usr/bin/env" ] + variables + args
   os.execv( cmd[ 0 ], cmd )

# Some of the syscalls used in the tool required root privileges. Re-running the
# process as a root will solve the upfront requirements.
if os.geteuid() != 0:
   runAsRoot( sys.argv )

# pylint: disable-msg=wrong-import-position
import argparse
import ctypes
import ctypes.util
import logging
import pwd
import re
import shutil
import subprocess

from abc import ABCMeta, abstractmethod
# pylint: enable-msg=wrong-import-position

logging.basicConfig( level=logging.INFO,
                     format="%(asctime)s %(levelname)s %(funcName)20s() %(message)s",
                     datefmt="%Y-%m-%d %H:%M:%S" )

if "ISOLATE_USER" not in os.environ:
   os.environ[ "ISOLATE_USER" ] = getpass.getuser()

# There is no built-in mount function. Using ctypes as it provides C compatible data
# types, and allows calling functions in DLLs or shared libraries.
libc = ctypes.CDLL( ctypes.util.find_library( 'c' ), use_errno=True )
# mount( source, target, filesystemtype, mountflags, data )
# For more info: http://man7.org/linux/man-pages/man2/mount.2.html
# To create a bind mount: mountflags includes MS_BIND
# #include <sys/mount.h>
# int main(){
#   printf( "%d", MS_BIND ); // 4096
#   return 0;
# }
# Bind mount example: mount(source, target, 0, MS_BIND, "bind")
MOUNT_BIND = 4096
libc.mount.argtypes = ( ctypes.c_char_p, ctypes.c_char_p, ctypes.c_char_p,
                        ctypes.c_ulong, ctypes.c_char_p )

def safeEnsureBinary( val ):
   return val.encode() if isinstance( val, str ) else val

def mount( source, target, fs, flags=0, options='', createTargetDir=False ):
   logging.debug( "(source=%s, target=%s, fs=%s, options=%s)",
                   source, target, fs, options )
   if createTargetDir and not os.path.exists( target ):
      makedirs( target )
   ret = libc.mount( safeEnsureBinary( source ), safeEnsureBinary( target ),
      safeEnsureBinary( fs ), flags, safeEnsureBinary( options ) )
   if ret < 0:
      err = ctypes.get_errno()
      raise OSError( err, "Error mounting {} ({}) on {} with options '{}': {}".
                     format( source, fs, target, options, os.strerror( err ) ) )

class MetaConst( type ):
   def __getattr__( cls, key ):
      return cls[ key ] # pylint: disable=unsubscriptable-object
   def __setattr__( cls, key, value ):
      raise TypeError( "Modifying global constants not allowed" )

class Const( metaclass=MetaConst ):
   def __getattr__( self, name ):
      return self[ name ]
   def __setattr__( self, name, value ):
      raise TypeError( "Modifying global constants not allowed" )

class MyConst( Const ):
   DEFAULT_DIRS_TO_ISOLATE = [
      "/tmp",
      "/var/lib/rpm",
      "/var/run",
      "/var/tmp",
      "/var/log/qt",
      "/var/core",
      "/coveragetmp",
   ]
   DEFAULT_DIRS_TO_TMPFS = [
      "/var/shmem",
   ]

   # /isolate is a reserved directory to be used by isolate to share
   # mounts and directories between host and containers.
   ISOLATE_COMMON = "/isolate"
   DEFAULT_SHARED_ROOT = pathjoin( ISOLATE_COMMON, "isolate" )
   DEFAULT_SHARED_ROOT_TMPFS = pathjoin( ISOLATE_COMMON, "isolate-tmpfs" )
   DEFAULT_TMP_MOUNTS = pathjoin( ISOLATE_COMMON, "mounts-tmp" )
   HIDDEN_NETWORK_NAMESPACE_PATH = "/isolate/mounts-tmp/hiddenNetns"
   HIDDEN_NETWORK_NAMESPACE = pathjoin( HIDDEN_NETWORK_NAMESPACE_PATH,
                                        "isolateOuterNetNamespace" )

   # global constants for environment variables used across the code.
   USER = "USER"
   NSNAME = "NSNAME"
   LOGNAME = "LOGNAME"
   A4_CHROOT = "A4_CHROOT"
   INSIDE_ISOLATE = "INSIDE_ISOLATE"
   ISOLATE_USER = "ISOLATE_USER"
   IN_MNT_NAMESPACE = "IN_MNT_NAMESPACE"
   QUICKTRACEDIR = "QUICKTRACEDIR"
   ARTEST_NOCHECK = "ARTEST_NOCHECK"
   ISOLATE_DIRS = "ISOLATE_DIRS"
   ISOLATE_DIRS_TO_EXCLUDE = "ISOLATE_DIRS_TO_EXCLUDE"
   FSTRACE_MASTER = "FSTRACE_MASTER"

def makedirs( path ):
   logging.debug( path )
   try:
      os.makedirs( path )
   except OSError as e:
      if e.errno != errno.EEXIST or not os.path.isdir( path ):
         raise

def mknod( fileName ):
   logging.debug( fileName )
   try:
      os.mknod( fileName )
   except OSError as e:
      if e.errno != errno.EEXIST or not os.path.isfile( fileName ):
         raise

def symlink( source, target ):
   logging.debug( "%s %s", str( source ), str( target ) )
   try:
      os.symlink( source, target )
   except OSError as e:
      if e.errno != errno.EEXIST or not os.path.islink( target ):
         raise

def isSupersetPath( path, d ):
   relPath = os.path.relpath( path, d )
   return not relPath.startswith( ".." )

def relpath( path ):
   """Return path relative to root.
   """
   return os.path.relpath( path, "/" )

def parseOptions():
   class Formatter( argparse.ArgumentDefaultsHelpFormatter,
                    argparse.RawDescriptionHelpFormatter ):
      pass
   parser = argparse.ArgumentParser(
      formatter_class=Formatter,
      description=( """
      Isolate is the utility to run another program under isolated namespace,
either by using netnsd or docker, depending upon the arguments passed to
the command line.
All stdout or stderr output of the program is prefixed.
It does not work at all for interactive programs.""" ),
      epilog=( "By default, tool isolate the following directories:\n"
               + " ".join( MyConst.DEFAULT_DIRS_TO_ISOLATE ) )
      + """
Use Environment variable ISOLATE_DIRS_TO_EXCLUDE to exclude some directories
from isolation.
Ex: ISOLATE_DIRS_TO_EXCLUDE='/tmp /var/tmp'""",
   )

   parser.add_argument( "--verbosity", action="store_true",
                        help="increase output verbosity" )
   parser.add_argument( "--interface", "-i", action="store_true",
                        help="need an interface inside the namespace" )
   parser.add_argument( "--chroot", help="(experimental) "
                        "Change root to give workspace." )
   parser.add_argument( "--namespace", "-n", default=os.getpid(),
                        help="custom namespace name. (default: PID of script)" )
   parser.add_argument( "--preload", "-l", action='append',
                        help="preload a library on netnsd startup. "
                        "(can be specified multiple times)" )
   parser.add_argument( "--postload", "-L", action='append',
                        help="postload a library after netnsd has created "
                        "namespace and chrooted. "
                        "(can be specified multiple times)" )
   parser.add_argument( "--prefix", "-p",
                        help="custom line prefix. "
                        "(default: '[basename namespace]') " )
   parser.add_argument( "--volume", "-v", action='append',
                        help="bind mount a volume "
                        "(can be specified multiple times)" )
   parser.add_argument( "--version", "-V", action="store_true",
                        help="print the version" )
   parser.add_argument( "--shared-root", dest="sharedRoot", action="store_true",
                        default=False, help="enable single source directory on host "
                        "for mounting all isolated directories within a namespace." )
   parser.add_argument( "--use-tmpfs", "-t", dest="useTmpfs", action="store_true",
                        default=False, help="mount the shared root  as tmpfs. "
                        "Must be used with --sharedRoot." )
   parser.add_argument( "--disable-cleanup", dest="disableCleanup",
                        action="store_true", default=False,
                        help="Used to disable cleaning up container." )
   parser.add_argument( "command", nargs=argparse.REMAINDER,
                        help="command under isolation" )

   args = parser.parse_args()

   if args.version:
      print( "Version:", __version__ )
      sys.exit( 0 )

   if not args.command:
      print( "Error: command to run under isolate not found.", file=sys.stderr )
      parser.print_usage( sys.stderr )
      sys.exit( 1 )

   if args.disableCleanup and not args.interface:
      print( "Error: --interface(-i) must be passed with --disableCleanup.\n",
             file=sys.stderr )
      parser.print_usage( sys.stderr )
      sys.exit( 1 )

   if args.chroot and not args.interface:
      print( "Error: --interface(-i) must be passed with --chroot.\n" )
      parser.print_usage( sys.stderr )
      sys.exit( 1 )

   if args.useTmpfs and not args.sharedRoot:
      print( "Error: --useTmpfs must be used with --sharedRoot.", file=sys.stderr )
      parser.print_usage( sys.stderr )
      sys.exit( 1 )

   if args.verbosity:
      logging.getLogger().setLevel( logging.DEBUG )

   return args

def createSharedDirectory():
   # <commonPath> -> <commonPath>/isolate
   # if --useTmpfs, <commonPath> -> <commonPath>/isolate-tmpfs

   makedirs( MyConst.DEFAULT_SHARED_ROOT )
   makedirs( MyConst.DEFAULT_SHARED_ROOT_TMPFS )
   def mountedAsTmpfs( path ):
      with open( '/proc/self/mounts' ) as mounts:
         # Sample /proc/self/mount lines:
         # proc /proc proc rw,nosuid,nodev,noexec,relatime 0 0
         # /dev/md126 /src/netns ext4 rw,relatime,data=writeback 0 0
         # tmpfs /isolate/isolate-tmpfs tmpfs rw,relatime 0 0
         for mountPoint in mounts:
            p = mountPoint.split()
            if p[ 0 ] == "tmpfs" and p[ 1 ] == path:
               return True
      return False
   if not mountedAsTmpfs( MyConst.DEFAULT_SHARED_ROOT_TMPFS ):
      mount( "tmpfs", MyConst.DEFAULT_SHARED_ROOT_TMPFS, "tmpfs", 0, None )

class PreIsolate( metaclass=ABCMeta ):
   """Abstract class for setting up pre isolation environment
   """
   def __init__( self, args ):
      self.isolateUser = os.environ.get( MyConst.ISOLATE_USER )
      self.my_env = os.environ.copy()
      self.args = args
      self.netnsNameOrPid = str( self.args.namespace or os.getpid() )
      self.prefix = os.path.basename( self.netnsNameOrPid )
      self.exitStatus = 0
      self.dirsToIsolate = MyConst.DEFAULT_DIRS_TO_ISOLATE
      self.dirPathMap = {}
      # --chroot is passed, "/" will be replaced with new workspace path.
      self.root = "/"
      self.mountsToRestore = {}

   def cmdWithPrefix( self, command ):
      # Consider empty string to be a valid prefix if it was passed explicitly.
      if self.args.prefix is not None:
         self.prefix = self.args.prefix
      # change prefix -> [prefix] only if prefix isn't an empty string.
      if self.prefix:
         self.prefix = str( "[" + self.prefix + "] " )
      if isinstance( command, ( list, ) ):
         return [ "/usr/bin/prefix", self.prefix ] + command
      return command

   def runInNetns( self, command ):
      if isinstance( command, ( list, ) ):
         return [ "netns", self.netnsNameOrPid ] + command
      return command

   def setupDefaultEnvironment( self ):
      self.my_env[ MyConst.INSIDE_ISOLATE ] = "1"
      # Set QUICKTRACEDIR so that agents of parallel tests don't use the
      # same .qt file.
      self.my_env[ MyConst.QUICKTRACEDIR ] = "/var/log/qt"
      self.my_env[ MyConst.ARTEST_NOCHECK ] = "1"

   def createHiddenNamespace( self ):
      """This allows applications running inside isolate to punch a hole out
      of the isolate namespace and be able to make network accesses.
      /proc/self/ns/net of host is bind mounted to a visible path in the
      container. setns() to this file could enable the host networking inside the
      container.
      """
      if MyConst.INSIDE_ISOLATE in os.environ:
         return
      makedirs( MyConst.HIDDEN_NETWORK_NAMESPACE_PATH )
      mount( "tmpfs", MyConst.HIDDEN_NETWORK_NAMESPACE_PATH, "tmpfs", 0, None )
      outerNamespace = "/proc/self/ns/net"
      mknod( MyConst.HIDDEN_NETWORK_NAMESPACE )
      mount( outerNamespace, MyConst.HIDDEN_NETWORK_NAMESPACE, None, MOUNT_BIND,
             "bind" )

   def excludeMountDirectories( self ):
      dirsToExclude = os.environ.get( MyConst.ISOLATE_DIRS_TO_EXCLUDE, '' ).split()
      self.dirsToIsolate = list( set( self.dirsToIsolate ) - set( dirsToExclude ) )

   def setupEnvironmentForIsolateDirs( self ):
      """ISOLATE_DIR_* is used by recursive isolate to bind mount
      the directory under isolation in callers path. (Note: Only for non-interface
      isoalte call).
      ISOLATE_REALDIR_* is the original location isolated bind mounted directory
      on host.
      """
      for directory, path in self.dirPathMap.items():
         if not self.args.interface:
            envpath = "ISOLATE_DIR" + directory.replace( "/", "_" )
            self.my_env[ envpath ] = path

         envpath = "ISOLATE_REALDIR" + directory.replace( "/", "_" )
         # Case of recursive isolate calls.
         # Example:
         # First isolate: /tmp will be bind mounted to /tmp/isolate/<netns_1> of host
         # Second isolate: /tmp will be bind mounted to /tmp/<netns_2> of first
         # isolate and so on..
         # So, path on host of file in /tmp of 2nd isolate will be
         # /tmp/isolate/<netns_1>/<netns_2>
         # Environment variable ISOLATE_REALDIR_* will give this path to command
         # running under 2nd isolate.
         if envpath in os.environ:
            relPath = os.path.relpath( path, directory )
            # Make sure path is a full superset of directory
            assert not relPath.startswith( '..' )
            path = pathjoin( os.environ[ envpath ], relPath )
         self.my_env[ envpath ] = path

   def isolateHostRootDirs( self ):
      """Create directories that needs isolation on the host.
      Example:
      /tmp/       -> /tmp/isolate/<netnsName>/
      /var/tmp/   -> /var/tmp/isolate/<netnsName>/
      """
      for d in self.dirsToIsolate:
         # .isolate let the command under isolation knows the
         # directory <m> is in isolated child process namespace.
         if os.path.isfile( pathjoin( d, ".isolate" ) ):
            path = pathjoin( d, self.netnsNameOrPid )
         else:
            path = pathjoin( d, "isolate", self.netnsNameOrPid )
         if os.path.isdir( path ):
            shutil.rmtree( path )
         makedirs( path )
         os.chmod( path, 0o777 )
         mknod( pathjoin( path, ".isolate" ) )
         self.dirPathMap[ d ] = path

   def isolateSharedRootDirs( self ):
      """Create directories that needs isolation on the host.
      Example:
      /tmp/     ->(DEFAULT_SHARED_ROOT/DEFAULT_SHARED_ROOT_TMPFS)/<netnsName>/tmp/
      /var/tmp/ ->(DEFAULT_SHARED_ROOT/DEFAULT_SHARED_ROOT_TMPFS)<netnsName>/var/tmp/
      """
      sharedDir = MyConst.DEFAULT_SHARED_ROOT
      if self.args.useTmpfs:
         sharedDir = MyConst.DEFAULT_SHARED_ROOT_TMPFS

      baseDir = pathjoin( sharedDir, self.netnsNameOrPid )

      # clear old namespace artifacts
      if os.path.isdir( baseDir ):
         shutil.rmtree( baseDir )

      for d in self.dirsToIsolate:
         path = pathjoin( baseDir, d[ 1: ] )
         makedirs( path )
         os.chmod( path, 0o777 )
         # .isolate let the command under isolation knows the
         # directory <m> is in isolated child process namespace.
         mknod( pathjoin( path, ".isolate" ) )
         self.dirPathMap[ d ] = path

   def isolateDirectories( self ):
      """Originally, the mounts and directories that needed isolation
      are bind mounted to the host by the following way.
      /tmp/     -> /tmp/isolate/<netnsName>/
      /var/tmp/ -> /var/tmp/isolate/<netnsName>/

      New way is have a shared root for all mounts and directories.
      /tmp/     ->(DEFAULT_SHARED_ROOT/DEFAULT_SHARED_ROOT_TMPFS)/<netnsName>/tmp/
      /var/tmp/ ->(DEFAULT_SHARED_ROOT/DEFAULT_SHARED_ROOT_TMPFS)<netnsName>/var/tmp/

      Recursive isolate will continue to follow original way.
      """
      # If Enviornment variable "ISOLATE_DIRS_TO_EXCLUDE" is set,
      # Remove all exclude dirs from dirsToIsolate.
      self.excludeMountDirectories()
      if MyConst.INSIDE_ISOLATE not in os.environ and self.args.sharedRoot:
         self.isolateSharedRootDirs()
      else:
         self.isolateHostRootDirs()
      self.my_env[ MyConst.ISOLATE_DIRS ] = " ".join( self.dirsToIsolate )

   def parseSharedObjectOptions( self ):
      options = []
      for so in self.args.postload or []:
         options += [ "-L", so ]
      for so in self.args.preload or []:
         options += [ "-l", so ]
      return options

   @abstractmethod
   def run( self ):
      logging.debug( "Running pre isolation commands" )
      self.createHiddenNamespace()
      self.setupDefaultEnvironment()
      self.isolateDirectories()
      self.setupEnvironmentForIsolateDirs()

   @abstractmethod
   def runCommand( self ):
      pass

class WithInterface( PreIsolate ):
   """Following class runs a netnsd container in daemon mode and
   configure a virtual interface inside the container. This interface is
   a macvlan bond that exposes host default interfaces directly to
   the container. The command under isolation is executed using netns client
   under the container namespace. The netnsd container is cleanedup after
   the completion of the task, depending on the --disableCleanup flag.
   """
   def daemonizeNetnsd( self, netnsdCmd=None ):
      command = None
      if netnsdCmd is None:
         command = ( [ "/usr/bin/netnsd", "-d", "-u", self.isolateUser ] +
                     self.parseSharedObjectOptions() +
                     [ "-f", "mnhpi", self.netnsNameOrPid ] )
      else:
         command = netnsdCmd
      logging.debug( "Running netnsd in daemon mode: %s", str( command ) )
      # Create namespace and daemonize it. This can fail in an Abuild environment.
      # See BUG167888. We retry this 3 times before giving up.
      for _ in range( 0, 3 ):
         p = subprocess.Popen( command, close_fds=True )
         if not p.wait():
            break

      # Check status of netnsd container
      if p.poll():
         self.exitStatus = p.returncode
         logging.error( "Failed to start netnsd daemon.!!" )
         sys.exit( 1 )

   def restoreVarRun( self ):
      """Most of the services like mysql, httpd, etc stores it's sock or pid file in
      this location. Backup and restore directories from host /var/run to /var/run
      in new mount namespace for clean working of these services.
      """
      d = "/var/run"
      # No need to restore /var/run if it is not in the list of directories under
      # isolation.
      if d not in self.dirPathMap:
         return

      for o in os.listdir( d ):
         if os.path.isdir( pathjoin( d, o ) ):
            makedirs( pathjoin( self.dirPathMap[ d ], o ) )
      os.chmod( pathjoin( self.dirPathMap[ d ], "agents" ), 0o777 )
      mknod( pathjoin( self.dirPathMap[ d ], "seqno" ) )
      os.chmod( pathjoin( self.dirPathMap[ d ], "seqno" ), 0o666 )

   def backupInodesAndMounts( self ):
      """Function to backup inodes or mounts from directories under isolation.
      For ex: /var/run is a default directory for isolation.
      To backup /var/run/docker.sock, the function bind mount it to
      DEFAULT_TMP_MOUNTS/var/run/docker.sock. After isolating /var/run, it re-mounts
      DEFAULT_TMP_MOUNTS/var/run/docker.sock to /var/run/docker.sock in new mount
      namespace.
      """
      preservePath = pathjoin( MyConst.DEFAULT_TMP_MOUNTS, self.netnsNameOrPid )

      if self.args.volume:
         # Bind-Mount a file from one of the original directories into its overlay
         # Only mount files that are inside one of the overlayed directories.
         # In case of --chroot, mount them all.
         for volume in self.args.volume:
            assert os.path.isabs( volume )
            if self.args.chroot:
               self.mountsToRestore[ volume ] = pathjoin( preservePath,
                                                          relpath( volume ) )
            else:
               for d in self.dirsToIsolate:
                  if isSupersetPath( volume, d ):
                     self.mountsToRestore[ volume ] = pathjoin( preservePath,
                                                         relpath( volume ) )

         for v, e in self.mountsToRestore.items():
            if os.path.isdir( v ):
               makedirs( e )
            else:
               makedirs( pathjoin( preservePath,
                                   relpath( os.path.dirname( v ) ) ) )
               mknod( e )
            mount( v, e, None, MOUNT_BIND, "bind" )

   def bindMountDirectories( self ):
      # Hack for solving issue with recursive mounts
      # For ex: /tmp/var must be mounted before /tmp
      dirs = sorted( self.dirPathMap, key=len, reverse=True )
      for d in dirs:
         assert os.path.isabs( d )
         mount( self.dirPathMap[ d ], pathjoin( self.root, relpath( d ) ),
                None, MOUNT_BIND, "bind", createTargetDir=True )

   def restoreInodesAndMounts( self ):
      """Fuction to restore the backed up mounts in new mount namespace.
      For ex: DEFAULT_TMP_MOUNTS/var/run/docker.sock to /var/run/docker.sock in new
      mount namspace.
      """
      for e, v in self.mountsToRestore.items():
         assert os.path.isabs( e )
         path = pathjoin( self.root, relpath( e ) )
         if os.path.isdir( v ):
            makedirs( path )
         else:
            makedirs( os.path.dirname( path ) )
            mknod( path )
         mount( v, path, None, MOUNT_BIND, "bind" )

      # The older tests depend on /var/run/hiddenNetns for hidden
      # network namespace. Creating a mount to support the old
      # behavior.
      hiddenPath = pathjoin( self.root, "var/run/hiddenNetns" )
      makedirs( hiddenPath )
      hiddenNamespace = pathjoin( hiddenPath, "isolateOuterNetNamespace" )
      mknod( hiddenNamespace )
      mount( MyConst.HIDDEN_NETWORK_NAMESPACE, hiddenNamespace, None, MOUNT_BIND,
             "bind" )

   def mountsToTmpfs( self ):
      for d in MyConst.DEFAULT_DIRS_TO_TMPFS:
         assert os.path.isabs( d )
         path = pathjoin( self.root, relpath( d ) )
         makedirs( path )
         mount( "tmpfs", path, "tmpfs", 0, None )

   def createInterface( self ):
      """We create an interface inside the namespace if available. This is not
      really useful for breadth tests but is useful for stests ( ScheduleStests
      and a4 make )
      This is carried out in 3 steps:
      Step 1: Search for network namespace PID
      Step 2: Search for default interface on host
      Step 3: Create a macvlan bond from host default interface to newly created
              network namespace
      """
      # `netns -q <name>` prints the namespace pid
      nsidCommand = [ "/usr/bin/netns", "-q", self.netnsNameOrPid ]
      logging.debug( "Find pid of netnsd container: %s", str( nsidCommand ) )
      try:
         nsid = subprocess.check_output( nsidCommand )
      except subprocess.CalledProcessError as e:
         logging.error( "netns [%s] not running. Returncode=%s", self.netnsNameOrPid,
                        e.returncode )
         sys.exit( 1 )
      nsid = nsid.split()[ 0 ]
      logging.debug( "Pid of netnsd container: %s", str( nsid ) )
      ldevCommand = [ "/usr/sbin/ip", "route", "show", "default" ]
      logging.debug( "Find default interface of host: %s", str( ldevCommand ) )
      ldev = ""
      try:
         routes = subprocess.check_output( ldevCommand, text=True )
      except subprocess.CalledProcessError as e:
         logging.error( "Unable to find host's default interface: %s", str( e ) )
         sys.exit( 1 )
      logging.debug( "Available default routes: %s", str( routes ) )
      # Sample `ip route show default` lines:
      # default via 172.25.230.1 dev e-a4c-e572dd0f
      # 172.25.230.0/23 dev e-a4c-e572dd0f proto kernel scope link src 172.25.230.70
      for route in routes.split( "\n" ) :
         if route.startswith( "default" ):
            ldev = route.split()[ 4 ]
      logging.debug( "Default interface: %s", ldev )

      # We want $ldev-e0 as the new link name but its total length must be limited
      # to 15 characters to avoid errors due to length of the interface name
      newldev = ( ldev + "-e0" )[ -15: ]

      createInterface = [ "/usr/sbin/ip", "link", "add", "link", ldev, "name",
                          newldev, "netns", nsid, "type", "macvlan", "mode",
                          "bridge" ]
      logging.debug( "Creating interface: %s", str( createInterface ) )
      try:
         subprocess.check_call( createInterface )
      except subprocess.CalledProcessError as e:
         logging.error( "Error creating macvlan bond: %s", str( e ) )
         sys.exit( 1 )

      self.my_env[ "NS_INTERFACE" ] = newldev

   def postIsolation( self ):
      addRoute = self.runInNetns( [ "ip", "route", "add", "224/4", "dev", "lo" ] )
      logging.debug( "Adding route on local interface: %s", str( addRoute ) )
      try:
         subprocess.call( addRoute )
      except subprocess.CalledProcessError as e:
         logging.error( "Failed to add route on local interface: %s", str( e ) )
      pw = pwd.getpwnam( self.isolateUser )
      # getpass.getuser() check's LOGNAME. If we don't set this,
      # ArtoolsTestLib can't spin up a MySQL instance
      self.my_env[ "LOGNAME" ] = pw.pw_name

   def runCommand( self ):
      command = self.cmdWithPrefix( [ "sudo", "-u", self.isolateUser,
                                      "/usr/bin/netns",
                                      self.netnsNameOrPid ] + self.args.command )
      logging.debug( "Running command with interface: %s",
                     str( command ) )
      p = subprocess.Popen( command, env=self.my_env )
      p.wait()
      self.exitStatus = p.returncode

   def cleanupContainer( self ):
      try:
         command = [ "/usr/bin/netns", "-k", self.netnsNameOrPid ]
         logging.debug( "Killing netnsd container: %s", str( command ) )
         subprocess.check_call( command, stdout=subprocess.PIPE )
      except subprocess.CalledProcessError as e:
         logging.error( e )

   def run( self ):
      super().run()
      logging.debug( "Running pre isolation commands" )
      self.restoreVarRun()
      self.backupInodesAndMounts()
      self.bindMountDirectories()
      self.restoreInodesAndMounts()
      self.mountsToTmpfs()
      self.daemonizeNetnsd()
      self.postIsolation()
      try:
         self.createInterface()
      except SystemExit:
         self.cleanupContainer()
         raise
      self.runCommand()
      if not self.args.disableCleanup:
         self.cleanupContainer()

class WithInterfaceAndChroot( WithInterface ):
   def __init__( self, args ):
      super().__init__( args )
      self.root = self.args.chroot

   def setupDefaultEnvironment( self ):
      super().setupDefaultEnvironment()
      # BUG358387: MountProfileHdlr depends on the availability of env A4_CHROOT
      # for loading ConfigMountProfile.
      self.my_env[ MyConst.A4_CHROOT ] = self.root

   def findWorkspaceArch( self ):
      defaultArch = "x86_64"
      platformPath = os.path.join( self.root, "usr", "share", "Artools", "platform" )
      if not os.path.exists( platformPath ):
         logging.error( "Unable to parse platform from %s. Using default "
                        "architecture( x86_64 )", str( platformPath ) )
         return defaultArch
      p = open( platformPath ).read()
      arch, _ = re.match( "(.*)_(.*)", p ).groups()
      if arch == "i386":
         return "i686"
      if arch == "x86_64":
         return "x86_64"
      if arch == 'aarch64':
         return "aarch64"
      else:
         assert False, "Unrecognized platform: " + p
      return "x86_64"

   def daemonizeNetnsd( self, netnsdCmd=None ):
      netnsdArgs = [ "--chroot", self.root ]
      mounts = """
/usr/bin/sudo /bin/mount -n --rbind %(x)s %(x)s;
/usr/bin/sudo /bin/mount -n -t tmpfs none %(x)s/dev;
/usr/bin/sudo /bin/cp -ax /dev/. %(x)s/dev;
/usr/bin/sudo /bin/mount -n -t tmpfs tmpfs %(x)s/var/run/netns;
/usr/bin/sudo /bin/mount -n --bind /home/arastra %(x)s/home/arastra;
/usr/bin/sudo /bin/mount -n --bind /dev/pts %(x)s/dev/pts;
/usr/bin/sudo /bin/mount -n --bind /proc %(x)s/proc;
/usr/bin/sudo /bin/mount -n --bind /sys %(x)s/sys;
/usr/bin/sudo /bin/mount -n --rbind /sys/fs/cgroup %(x)s/sys/fs/cgroup;
""".replace( "\n", "" ).replace( "%(x)s", self.root )
      netnsdArgs += [ "--pre-chroot-cmd", mounts ]
      # BUG358385: ArosTest plugins depends on workspace arch in the isolate
      # environment. Usually /usr/share/Artools/platform stores information
      # about platform inside the workspace.
      archstr = self.findWorkspaceArch()
      setarchCmd = [ "/usr/bin/setarch", archstr ]
      netnsdCmd = ( [ "/usr/bin/netnsd", "-d", "-u", self.isolateUser ] +
                  self.parseSharedObjectOptions() +
                  netnsdArgs +
                  [ "-f", "mnhpi", self.netnsNameOrPid ] )
      command = setarchCmd + netnsdCmd
      super().daemonizeNetnsd( command )

   def run( self ):
      logging.debug( "Running pre isolation commands with interface and chroot" )
      super().run()

class WithoutInterface( PreIsolate ):
   """Following class execute the command under isolation in a netnsd container.
   The container is automatically cleared once the command returns.
   """
   def setupDefaultEnvironment( self ):
      super().setupDefaultEnvironment()
      # "IN_MNT_NAMESPACE" is used to change the behavior of isolate in recursive
      # call.
      self.my_env[ MyConst.IN_MNT_NAMESPACE ] = "1"

   def runCommand( self ):
      command = self.cmdWithPrefix( [ "/usr/bin/netnsd", "-u", self.isolateUser ] +
                                    self.parseSharedObjectOptions() +
                                    [ "-f", "mnhpi", self.netnsNameOrPid ] +
                                    sys.argv )
      logging.debug( "Running command without interface: %s", str( command ) )
      p = subprocess.Popen( command, env=self.my_env )
      p.wait()
      self.exitStatus = p.returncode

   def run( self ):
      super().run()
      self.runCommand()

class PostIsolate:
   def __init__( self, args ):
      self.isolateUser = os.environ.get( MyConst.ISOLATE_USER )
      self.args = args
      self.mountPoints = {}
      self.backupDirectories = []

   def bindMountDirectories( self ):
      # isolate passes the user - visible pathnames down to us
      # as ISOLATE_DIR_<path>
      # We find all of them, and if any of the environment
      # variables match the path, we bind mount the original path with
      # the given path
      pathMap = {}
      for e, v in os.environ.items():
         if e.startswith( 'ISOLATE_DIR_' ):
            pathMap[ e.replace( 'ISOLATE_DIR', '' ).replace( '_', '/' ) ] = v
      # Hack for solving issue with recursive mounts
      dirs = sorted( pathMap.keys(), key=len, reverse=True )
      for d in dirs:
         mount( pathMap[ d ], d, None, MOUNT_BIND, "bind" )

   def mountsToTmpfs( self ):
      for d in MyConst.DEFAULT_DIRS_TO_TMPFS:
         makedirs( d )
         mount( "tmpfs", d, "tmpfs", 0, None )

   def backupInodesAndMounts( self ):
      """Function to backup inodes or mounts from directories under isolation.
      For ex: /var/run is a default directory for isolation.
      To backup /var/run/docker.sock, the function bind mount it to
      DEFAULT_TMP_MOUNTS/var/run/docker.sock. After isolating /var/run, it re-mounts
      DEFAULT_TMP_MOUNTS/var/run/docker.sock to /var/run/docker.sock in new mount
      namespace.
      """
      # Most of the services like mysql, httpd, etc stores it's sock or pid file in
      # this location. Backup and restore directories from host /var/run to /var/run
      # in new mount namespace for clean working of these services.
      d = "/var/run"
      self.backupDirectories = [ pathjoin( d, o ) for o in os.listdir( d )
                                 if os.path.isdir( pathjoin( d, o ) ) ]

      preservePath = pathjoin( MyConst.DEFAULT_TMP_MOUNTS,
                               os.environ[ MyConst.NSNAME ] )
      isolatedDirs = os.environ.get( MyConst.ISOLATE_DIRS, '' ).split()

      if self.args.volume:
         # Bind-Mount a file from one of the original directories into its overlay
         # Only mount files that are inside one of the overlayed directories
         for volume in self.args.volume:
            for d in isolatedDirs:
               if isSupersetPath( volume, d ):
                  self.mountPoints[ volume ] = preservePath + volume

         for v, e in self.mountPoints.items():
            if os.path.isdir( v ):
               makedirs( e )
            else:
               makedirs( preservePath + os.path.dirname( v ) )
               mknod( e )
            mount( v, e, None, MOUNT_BIND, "bind" )

   def restoreInodesAndMounts( self ):
      _ = [ makedirs( d ) for d in self.backupDirectories ]
      # Fix agents directory permissions if such dir exists
      agentsDir = "/var/run/agents"
      if os.path.isdir( agentsDir ):
         os.chmod( agentsDir, 0o777 )
      mknod( "/var/run/seqno" )
      os.chmod( "/var/run/seqno", 0o666 )

      # Restore volumes
      for e, v in self.mountPoints.items():
         if os.path.isdir( v ):
            makedirs( e )
         else:
            makedirs( os.path.dirname( e ) )
            mknod( e )
         mount( v, e, None, MOUNT_BIND, "bind" )

      # The older tests depend on /var/run/hiddenNetns for hidden
      # network namespace. Creating a mount to support the old
      # behavior.
      hiddenPath = "/var/run/hiddenNetns"
      makedirs( hiddenPath )
      symlink( MyConst.HIDDEN_NETWORK_NAMESPACE,
               pathjoin( hiddenPath, "isolateOuterNetNamespace" ) )

   def run( self ):
      logging.debug( "Running post isolation commands" )
      self.backupInodesAndMounts()
      self.bindMountDirectories()
      self.mountsToTmpfs()
      self.restoreInodesAndMounts()
      # setsockopt(IP_ADOD_MEMBERSHIP) picks an interface by looking
      # where a packet sent to the multicast group would go.  Normally
      # this is the default route.  If there is no route matching a
      # multicast address, the IP_ADD_MEMBERSHIP of that address fails.
      # So if we want host-local multicast applications to work, we need
      # to add a multicast route.
      addRoute = [ "/usr/sbin/ip", "route", "add", "224/4", "dev", "lo" ]
      logging.debug( "Adding route on local interface: %s", str( addRoute ) )
      try:
         subprocess.call( addRoute )
      except subprocess.CalledProcessError as e:
         logging.error( "Failed to add route on local interface: %s", str( e ) )
      pw = pwd.getpwnam( self.isolateUser )
      os.setgid( pw.pw_gid )
      os.setuid( pw.pw_uid )
      os.environ[ MyConst.USER ] = pw.pw_name
      # getpass.getuser() check's LOGNAME. If we don't set this,
      # ArtoolsTestLib can't spin up a MySQL instance
      os.environ[ MyConst.LOGNAME ] = pw.pw_name
      del os.environ[ MyConst.IN_MNT_NAMESPACE ]
      cmd = self.args.command
      # fstrace based on fanotify Linux framework cannot see
      # accesses inside private mount namespaces, We need to launch it in
      # slave mode if master fstrace session is active
      if MyConst.FSTRACE_MASTER in os.environ:
         cmd = [ 'fstrace', '-s', '--' ] + list( cmd )
      logging.debug( "Running command with isolation: %s", str( cmd ) )
      os.execvp( cmd[ 0 ], cmd )

def main():
   args = parseOptions()
   instance = None
   # isolate without `-i` run the command under isolation directly with
   # netnsd, instead of running netnsd in daemon mode and send the command
   # to running daemon using netns client. After running the netnsd, some post
   # isolation work is required before running the actual command. Recursive running
   # of isolate script with different behavior helps to achieve the goal.
   # Process tree example:
   # root    5229  5228 python /usr/bin/isolate2 sleep 100
   # root    5231  5229 /usr/bin/prefix [5229]  /usr/bin/netnsd -u arastra -f mnhpi \
   #                    5229 /usr/bin/isolate2 sleep 100
   # root    5232  5231 /usr/bin/netnsd -u arastra -f mnhpi 5229 /usr/bin/isolate2 \
   #                    sleep 100
   # root    5233  5232 netnsd-server   -u arastra -f mnhpi 5229 /usr/bin/isolate2 \
   #                    sleep 100
   # root    5234  5233 /usr/bin/sudo <env> /usr/bin/isolate2 sleep 100
   # arastra 5235  5234 sleep 100
   # From the above example, isolate runs the netnsd and rerun the command with
   # isolate in new namespace for post isolation work.
   # Environment variable "IN_MNT_NAMESPACE" is used to change the behavior of
   # isolate script in pre-post isolation.
   # If the env is set, the isolate script run the post isolation job.
   if MyConst.IN_MNT_NAMESPACE in os.environ:
      instance = PostIsolate( args )
      # exit status of post isolate totally depends on the exit status of
      # command under isolation.
      instance.run()

   if args.interface:
      instance = WithInterfaceAndChroot( args ) if args.chroot \
                 else WithInterface( args )
   else:
      instance = WithoutInterface( args )
   # Useful to avoid creating shared directory in recursive isolate calls.
   if MyConst.INSIDE_ISOLATE not in os.environ:
      createSharedDirectory()
      import netns
      netns.unshare( netns.CLONE_NEWNS ) # pylint: disable=c-extension-no-member
   instance.run()
   sys.exit( instance.exitStatus )
